/**
 * TinyMCE version 8.3.1 (2025-12-17)
 */

(function () {
    'use strict';

    var typeOf$1 = function (x) {
        if (x === null) {
            return 'null';
        }
        if (x === undefined) {
            return 'undefined';
        }
        var t = typeof x;
        if (t === 'object' && (Array.prototype.isPrototypeOf(x) || x.constructor && x.constructor.name === 'Array')) {
            return 'array';
        }
        if (t === 'object' && (String.prototype.isPrototypeOf(x) || x.constructor && x.constructor.name === 'String')) {
            return 'string';
        }
        return t;
    };
    var isEquatableType = function (x) {
        return ['undefined', 'boolean', 'number', 'string', 'function', 'xml', 'null'].indexOf(x) !== -1;
    };

    var sort$1 = function (xs, compareFn) {
        var clone = Array.prototype.slice.call(xs);
        return clone.sort(compareFn);
    };

    var contramap = function (eqa, f) {
        return eq$2(function (x, y) { return eqa.eq(f(x), f(y)); });
    };
    var eq$2 = function (f) {
        return ({ eq: f });
    };
    var tripleEq = eq$2(function (x, y) { return x === y; });
    var eqString = tripleEq;
    var eqArray = function (eqa) { return eq$2(function (x, y) {
        if (x.length !== y.length) {
            return false;
        }
        var len = x.length;
        for (var i = 0; i < len; i++) {
            if (!eqa.eq(x[i], y[i])) {
                return false;
            }
        }
        return true;
    }); };
    // TODO: Make an Ord typeclass
    var eqSortedArray = function (eqa, compareFn) {
        return contramap(eqArray(eqa), function (xs) { return sort$1(xs, compareFn); });
    };
    var eqRecord = function (eqa) { return eq$2(function (x, y) {
        var kx = Object.keys(x);
        var ky = Object.keys(y);
        if (!eqSortedArray(eqString).eq(kx, ky)) {
            return false;
        }
        var len = kx.length;
        for (var i = 0; i < len; i++) {
            var q = kx[i];
            if (!eqa.eq(x[q], y[q])) {
                return false;
            }
        }
        return true;
    }); };
    var eqAny = eq$2(function (x, y) {
        if (x === y) {
            return true;
        }
        var tx = typeOf$1(x);
        var ty = typeOf$1(y);
        if (tx !== ty) {
            return false;
        }
        if (isEquatableType(tx)) {
            return x === y;
        }
        else if (tx === 'array') {
            return eqArray(eqAny).eq(x, y);
        }
        else if (tx === 'object') {
            return eqRecord(eqAny).eq(x, y);
        }
        return false;
    });

    /* eslint-disable @typescript-eslint/no-wrapper-object-types */
    const getPrototypeOf$2 = Object.getPrototypeOf;
    const hasProto = (v, constructor, predicate) => {
        if (predicate(v, constructor.prototype)) {
            return true;
        }
        else {
            // String-based fallback time
            return v.constructor?.name === constructor.name;
        }
    };
    const typeOf = (x) => {
        const t = typeof x;
        if (x === null) {
            return 'null';
        }
        else if (t === 'object' && Array.isArray(x)) {
            return 'array';
        }
        else if (t === 'object' && hasProto(x, String, (o, proto) => proto.isPrototypeOf(o))) {
            return 'string';
        }
        else {
            return t;
        }
    };
    const isType$1 = (type) => (value) => typeOf(value) === type;
    const isSimpleType = (type) => (value) => typeof value === type;
    const eq$1 = (t) => (a) => t === a;
    const is$5 = (value, constructor) => isObject(value) && hasProto(value, constructor, (o, proto) => getPrototypeOf$2(o) === proto);
    const isString = isType$1('string');
    const isObject = isType$1('object');
    const isPlainObject = (value) => is$5(value, Object);
    const isArray$1 = isType$1('array');
    const isNull = eq$1(null);
    const isBoolean = isSimpleType('boolean');
    const isUndefined = eq$1(undefined);
    const isNullable = (a) => a === null || a === undefined;
    const isNonNullable = (a) => !isNullable(a);
    const isFunction = isSimpleType('function');
    const isNumber = isSimpleType('number');
    const isArrayOf = (value, pred) => {
        if (isArray$1(value)) {
            for (let i = 0, len = value.length; i < len; ++i) {
                if (!(pred(value[i]))) {
                    return false;
                }
            }
            return true;
        }
        return false;
    };

    const noop = () => { };
    /** Compose a unary function with an n-ary function */
    const compose = (fa, fb) => {
        return (...args) => {
            return fa(fb.apply(null, args));
        };
    };
    /** Compose two unary functions. Similar to compose, but avoids using Function.prototype.apply. */
    const compose1 = (fbc, fab) => (a) => fbc(fab(a));
    const constant = (value) => {
        return () => {
            return value;
        };
    };
    const identity = (x) => {
        return x;
    };
    const tripleEquals = (a, b) => {
        return a === b;
    };
    function curry(fn, ...initialArgs) {
        return (...restArgs) => {
            const all = initialArgs.concat(restArgs);
            return fn.apply(null, all);
        };
    }
    const not = (f) => (t) => !f(t);
    const die = (msg) => {
        return () => {
            throw new Error(msg);
        };
    };
    const apply$1 = (f) => {
        return f();
    };
    const call = (f) => {
        f();
    };
    const never = constant(false);
    const always = constant(true);

    /**
     * The `Optional` type represents a value (of any type) that potentially does
     * not exist. Any `Optional<T>` can either be a `Some<T>` (in which case the
     * value does exist) or a `None` (in which case the value does not exist). This
     * module defines a whole lot of FP-inspired utility functions for dealing with
     * `Optional` objects.
     *
     * Comparison with null or undefined:
     * - We don't get fancy null coalescing operators with `Optional`
     * - We do get fancy helper functions with `Optional`
     * - `Optional` support nesting, and allow for the type to still be nullable (or
     * another `Optional`)
     * - There is no option to turn off strict-optional-checks like there is for
     * strict-null-checks
     */
    class Optional {
        tag;
        value;
        // Sneaky optimisation: every instance of Optional.none is identical, so just
        // reuse the same object
        static singletonNone = new Optional(false);
        // The internal representation has a `tag` and a `value`, but both are
        // private: able to be console.logged, but not able to be accessed by code
        constructor(tag, value) {
            this.tag = tag;
            this.value = value;
        }
        // --- Identities ---
        /**
         * Creates a new `Optional<T>` that **does** contain a value.
         */
        static some(value) {
            return new Optional(true, value);
        }
        /**
         * Create a new `Optional<T>` that **does not** contain a value. `T` can be
         * any type because we don't actually have a `T`.
         */
        static none() {
            return Optional.singletonNone;
        }
        /**
         * Perform a transform on an `Optional` type. Regardless of whether this
         * `Optional` contains a value or not, `fold` will return a value of type `U`.
         * If this `Optional` does not contain a value, the `U` will be created by
         * calling `onNone`. If this `Optional` does contain a value, the `U` will be
         * created by calling `onSome`.
         *
         * For the FP enthusiasts in the room, this function:
         * 1. Could be used to implement all of the functions below
         * 2. Forms a catamorphism
         */
        fold(onNone, onSome) {
            if (this.tag) {
                return onSome(this.value);
            }
            else {
                return onNone();
            }
        }
        /**
         * Determine if this `Optional` object contains a value.
         */
        isSome() {
            return this.tag;
        }
        /**
         * Determine if this `Optional` object **does not** contain a value.
         */
        isNone() {
            return !this.tag;
        }
        // --- Functor (name stolen from Haskell / maths) ---
        /**
         * Perform a transform on an `Optional` object, **if** there is a value. If
         * you provide a function to turn a T into a U, this is the function you use
         * to turn an `Optional<T>` into an `Optional<U>`. If this **does** contain
         * a value then the output will also contain a value (that value being the
         * output of `mapper(this.value)`), and if this **does not** contain a value
         * then neither will the output.
         */
        map(mapper) {
            if (this.tag) {
                return Optional.some(mapper(this.value));
            }
            else {
                return Optional.none();
            }
        }
        // --- Monad (name stolen from Haskell / maths) ---
        /**
         * Perform a transform on an `Optional` object, **if** there is a value.
         * Unlike `map`, here the transform itself also returns an `Optional`.
         */
        bind(binder) {
            if (this.tag) {
                return binder(this.value);
            }
            else {
                return Optional.none();
            }
        }
        // --- Traversable (name stolen from Haskell / maths) ---
        /**
         * For a given predicate, this function finds out if there **exists** a value
         * inside this `Optional` object that meets the predicate. In practice, this
         * means that for `Optional`s that do not contain a value it returns false (as
         * no predicate-meeting value exists).
         */
        exists(predicate) {
            return this.tag && predicate(this.value);
        }
        /**
         * For a given predicate, this function finds out if **all** the values inside
         * this `Optional` object meet the predicate. In practice, this means that
         * for `Optional`s that do not contain a value it returns true (as all 0
         * objects do meet the predicate).
         */
        forall(predicate) {
            return !this.tag || predicate(this.value);
        }
        filter(predicate) {
            if (!this.tag || predicate(this.value)) {
                return this;
            }
            else {
                return Optional.none();
            }
        }
        // --- Getters ---
        /**
         * Get the value out of the inside of the `Optional` object, using a default
         * `replacement` value if the provided `Optional` object does not contain a
         * value.
         */
        getOr(replacement) {
            return this.tag ? this.value : replacement;
        }
        /**
         * Get the value out of the inside of the `Optional` object, using a default
         * `replacement` value if the provided `Optional` object does not contain a
         * value.  Unlike `getOr`, in this method the `replacement` object is also
         * `Optional` - meaning that this method will always return an `Optional`.
         */
        or(replacement) {
            return this.tag ? this : replacement;
        }
        /**
         * Get the value out of the inside of the `Optional` object, using a default
         * `replacement` value if the provided `Optional` object does not contain a
         * value. Unlike `getOr`, in this method the `replacement` value is
         * "thunked" - that is to say that you don't pass a value to `getOrThunk`, you
         * pass a function which (if called) will **return** the `value` you want to
         * use.
         */
        getOrThunk(thunk) {
            return this.tag ? this.value : thunk();
        }
        /**
         * Get the value out of the inside of the `Optional` object, using a default
         * `replacement` value if the provided Optional object does not contain a
         * value.
         *
         * Unlike `or`, in this method the `replacement` value is "thunked" - that is
         * to say that you don't pass a value to `orThunk`, you pass a function which
         * (if called) will **return** the `value` you want to use.
         *
         * Unlike `getOrThunk`, in this method the `replacement` value is also
         * `Optional`, meaning that this method will always return an `Optional`.
         */
        orThunk(thunk) {
            return this.tag ? this : thunk();
        }
        /**
         * Get the value out of the inside of the `Optional` object, throwing an
         * exception if the provided `Optional` object does not contain a value.
         *
         * WARNING:
         * You should only be using this function if you know that the `Optional`
         * object **is not** empty (otherwise you're throwing exceptions in production
         * code, which is bad).
         *
         * In tests this is more acceptable.
         *
         * Prefer other methods to this, such as `.each`.
         */
        getOrDie(message) {
            if (!this.tag) {
                throw new Error(message ?? 'Called getOrDie on None');
            }
            else {
                return this.value;
            }
        }
        // --- Interop with null and undefined ---
        /**
         * Creates an `Optional` value from a nullable (or undefined-able) input.
         * Null, or undefined, is converted to `None`, and anything else is converted
         * to `Some`.
         */
        static from(value) {
            return isNonNullable(value) ? Optional.some(value) : Optional.none();
        }
        /**
         * Converts an `Optional` to a nullable type, by getting the value if it
         * exists, or returning `null` if it does not.
         */
        getOrNull() {
            return this.tag ? this.value : null;
        }
        /**
         * Converts an `Optional` to an undefined-able type, by getting the value if
         * it exists, or returning `undefined` if it does not.
         */
        getOrUndefined() {
            return this.value;
        }
        // --- Utilities ---
        /**
         * If the `Optional` contains a value, perform an action on that value.
         * Unlike the rest of the methods on this type, `.each` has side-effects. If
         * you want to transform an `Optional<T>` **into** something, then this is not
         * the method for you. If you want to use an `Optional<T>` to **do**
         * something, then this is the method for you - provided you're okay with not
         * doing anything in the case where the `Optional` doesn't have a value inside
         * it. If you're not sure whether your use-case fits into transforming
         * **into** something or **doing** something, check whether it has a return
         * value. If it does, you should be performing a transform.
         */
        each(worker) {
            if (this.tag) {
                worker(this.value);
            }
        }
        /**
         * Turn the `Optional` object into an array that contains all of the values
         * stored inside the `Optional`. In practice, this means the output will have
         * either 0 or 1 elements.
         */
        toArray() {
            return this.tag ? [this.value] : [];
        }
        /**
         * Turn the `Optional` object into a string for debugging or printing. Not
         * recommended for production code, but good for debugging. Also note that
         * these days an `Optional` object can be logged to the console directly, and
         * its inner value (if it exists) will be visible.
         */
        toString() {
            return this.tag ? `some(${this.value})` : 'none()';
        }
    }

    const nativeSlice = Array.prototype.slice;
    const nativeIndexOf = Array.prototype.indexOf;
    const nativePush = Array.prototype.push;
    const rawIndexOf = (ts, t) => nativeIndexOf.call(ts, t);
    const indexOf$1 = (xs, x) => {
        // The rawIndexOf method does not wrap up in an option. This is for performance reasons.
        const r = rawIndexOf(xs, x);
        return r === -1 ? Optional.none() : Optional.some(r);
    };
    const contains$2 = (xs, x) => rawIndexOf(xs, x) > -1;
    const exists = (xs, pred) => {
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            if (pred(x, i)) {
                return true;
            }
        }
        return false;
    };
    const map$3 = (xs, f) => {
        // pre-allocating array size when it's guaranteed to be known
        // http://jsperf.com/push-allocated-vs-dynamic/22
        const len = xs.length;
        const r = new Array(len);
        for (let i = 0; i < len; i++) {
            const x = xs[i];
            r[i] = f(x, i);
        }
        return r;
    };
    // Unwound implementing other functions in terms of each.
    // The code size is roughly the same, and it should allow for better optimisation.
    // const each = function<T, U>(xs: T[], f: (x: T, i?: number, xs?: T[]) => void): void {
    const each$e = (xs, f) => {
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            f(x, i);
        }
    };
    const eachr = (xs, f) => {
        for (let i = xs.length - 1; i >= 0; i--) {
            const x = xs[i];
            f(x, i);
        }
    };
    const partition$2 = (xs, pred) => {
        const pass = [];
        const fail = [];
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            const arr = pred(x, i) ? pass : fail;
            arr.push(x);
        }
        return { pass, fail };
    };
    const filter$5 = (xs, pred) => {
        const r = [];
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            if (pred(x, i)) {
                r.push(x);
            }
        }
        return r;
    };
    /*
     * Groups an array into contiguous arrays of like elements. Whether an element is like or not depends on f.
     *
     * f is a function that derives a value from an element - e.g. true or false, or a string.
     * Elements are like if this function generates the same value for them (according to ===).
     *
     *
     * Order of the elements is preserved. Arr.flatten() on the result will return the original list, as with Haskell groupBy function.
     *  For a good explanation, see the group function (which is a special case of groupBy)
     *  http://hackage.haskell.org/package/base-4.7.0.0/docs/Data-List.html#v:group
     */
    const groupBy = (xs, f) => {
        if (xs.length === 0) {
            return [];
        }
        else {
            let wasType = f(xs[0]); // initial case for matching
            const r = [];
            let group = [];
            for (let i = 0, len = xs.length; i < len; i++) {
                const x = xs[i];
                const type = f(x);
                if (type !== wasType) {
                    r.push(group);
                    group = [];
                }
                wasType = type;
                group.push(x);
            }
            if (group.length !== 0) {
                r.push(group);
            }
            return r;
        }
    };
    const foldr = (xs, f, acc) => {
        eachr(xs, (x, i) => {
            acc = f(acc, x, i);
        });
        return acc;
    };
    const foldl = (xs, f, acc) => {
        each$e(xs, (x, i) => {
            acc = f(acc, x, i);
        });
        return acc;
    };
    const findUntil$1 = (xs, pred, until) => {
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            if (pred(x, i)) {
                return Optional.some(x);
            }
            else if (until(x, i)) {
                break;
            }
        }
        return Optional.none();
    };
    const find$2 = (xs, pred) => {
        return findUntil$1(xs, pred, never);
    };
    const findIndex$2 = (xs, pred) => {
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            if (pred(x, i)) {
                return Optional.some(i);
            }
        }
        return Optional.none();
    };
    const findLastIndex = (arr, pred) => {
        for (let i = arr.length - 1; i >= 0; i--) {
            if (pred(arr[i], i)) {
                return Optional.some(i);
            }
        }
        return Optional.none();
    };
    const flatten$1 = (xs) => {
        // Note, this is possible because push supports multiple arguments:
        // http://jsperf.com/concat-push/6
        // Note that in the past, concat() would silently work (very slowly) for array-like objects.
        // With this change it will throw an error.
        const r = [];
        for (let i = 0, len = xs.length; i < len; ++i) {
            // Ensure that each value is an array itself
            if (!isArray$1(xs[i])) {
                throw new Error('Arr.flatten item ' + i + ' was not an array, input: ' + xs);
            }
            nativePush.apply(r, xs[i]);
        }
        return r;
    };
    const bind$3 = (xs, f) => flatten$1(map$3(xs, f));
    const forall = (xs, pred) => {
        for (let i = 0, len = xs.length; i < len; ++i) {
            const x = xs[i];
            if (pred(x, i) !== true) {
                return false;
            }
        }
        return true;
    };
    const reverse = (xs) => {
        const r = nativeSlice.call(xs, 0);
        r.reverse();
        return r;
    };
    const difference = (a1, a2) => filter$5(a1, (x) => !contains$2(a2, x));
    const mapToObject = (xs, f) => {
        const r = {};
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            r[String(x)] = f(x, i);
        }
        return r;
    };
    const sort = (xs, comparator) => {
        const copy = nativeSlice.call(xs, 0);
        copy.sort(comparator);
        return copy;
    };
    const get$b = (xs, i) => i >= 0 && i < xs.length ? Optional.some(xs[i]) : Optional.none();
    const head = (xs) => get$b(xs, 0);
    const last$2 = (xs) => get$b(xs, xs.length - 1);
    const from = isFunction(Array.from) ? Array.from : (x) => nativeSlice.call(x);
    const findMap = (arr, f) => {
        for (let i = 0; i < arr.length; i++) {
            const r = f(arr[i], i);
            if (r.isSome()) {
                return r;
            }
        }
        return Optional.none();
    };
    const unique$1 = (xs, comparator) => {
        const r = [];
        const isDuplicated = isFunction(comparator) ?
            (x) => exists(r, (i) => comparator(i, x)) :
            (x) => contains$2(r, x);
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            if (!isDuplicated(x)) {
                r.push(x);
            }
        }
        return r;
    };

    // There are many variations of Object iteration that are faster than the 'for-in' style:
    // http://jsperf.com/object-keys-iteration/107
    //
    // Use the native keys if it is available (IE9+), otherwise fall back to manually filtering
    const keys = Object.keys;
    const hasOwnProperty$1 = Object.hasOwnProperty;
    const each$d = (obj, f) => {
        const props = keys(obj);
        for (let k = 0, len = props.length; k < len; k++) {
            const i = props[k];
            const x = obj[i];
            f(x, i);
        }
    };
    const map$2 = (obj, f) => {
        return tupleMap(obj, (x, i) => ({
            k: i,
            v: f(x, i)
        }));
    };
    const tupleMap = (obj, f) => {
        const r = {};
        each$d(obj, (x, i) => {
            const tuple = f(x, i);
            r[tuple.k] = tuple.v;
        });
        return r;
    };
    const objAcc = (r) => (x, i) => {
        r[i] = x;
    };
    const internalFilter = (obj, pred, onTrue, onFalse) => {
        each$d(obj, (x, i) => {
            (pred(x, i) ? onTrue : onFalse)(x, i);
        });
    };
    const bifilter = (obj, pred) => {
        const t = {};
        const f = {};
        internalFilter(obj, pred, objAcc(t), objAcc(f));
        return { t, f };
    };
    const filter$4 = (obj, pred) => {
        const t = {};
        internalFilter(obj, pred, objAcc(t), noop);
        return t;
    };
    const mapToArray = (obj, f) => {
        const r = [];
        each$d(obj, (value, name) => {
            r.push(f(value, name));
        });
        return r;
    };
    const values = (obj) => {
        return mapToArray(obj, identity);
    };
    const get$a = (obj, key) => {
        return has$2(obj, key) ? Optional.from(obj[key]) : Optional.none();
    };
    const has$2 = (obj, key) => hasOwnProperty$1.call(obj, key);
    const hasNonNullableKey = (obj, key) => has$2(obj, key) && obj[key] !== undefined && obj[key] !== null;
    const equal$1 = (a1, a2, eq = eqAny) => eqRecord(eq).eq(a1, a2);

    /*
     * Generates a church encoded ADT (https://en.wikipedia.org/wiki/Church_encoding)
     * For syntax and use, look at the test code.
     */
    const generate$1 = (cases) => {
        // validation
        if (!isArray$1(cases)) {
            throw new Error('cases must be an array');
        }
        if (cases.length === 0) {
            throw new Error('there must be at least one case');
        }
        const constructors = [];
        // adt is mutated to add the individual cases
        const adt = {};
        each$e(cases, (acase, count) => {
            const keys$1 = keys(acase);
            // validation
            if (keys$1.length !== 1) {
                throw new Error('one and only one name per case');
            }
            const key = keys$1[0];
            const value = acase[key];
            // validation
            if (adt[key] !== undefined) {
                throw new Error('duplicate key detected:' + key);
            }
            else if (key === 'cata') {
                throw new Error('cannot have a case named cata (sorry)');
            }
            else if (!isArray$1(value)) {
                // this implicitly checks if acase is an object
                throw new Error('case arguments must be an array');
            }
            constructors.push(key);
            //
            // constructor for key
            //
            adt[key] = (...args) => {
                const argLength = args.length;
                // validation
                if (argLength !== value.length) {
                    throw new Error('Wrong number of arguments to case ' + key + '. Expected ' + value.length + ' (' + value + '), got ' + argLength);
                }
                const match = (branches) => {
                    const branchKeys = keys(branches);
                    if (constructors.length !== branchKeys.length) {
                        throw new Error('Wrong number of arguments to match. Expected: ' + constructors.join(',') + '\nActual: ' + branchKeys.join(','));
                    }
                    const allReqd = forall(constructors, (reqKey) => {
                        return contains$2(branchKeys, reqKey);
                    });
                    if (!allReqd) {
                        throw new Error('Not all branches were specified when using match. Specified: ' + branchKeys.join(', ') + '\nRequired: ' + constructors.join(', '));
                    }
                    return branches[key].apply(null, args);
                };
                //
                // the fold function for key
                //
                return {
                    fold: (...foldArgs) => {
                        // runtime validation
                        if (foldArgs.length !== cases.length) {
                            throw new Error('Wrong number of arguments to fold. Expected ' + cases.length + ', got ' + foldArgs.length);
                        }
                        const target = foldArgs[count];
                        return target.apply(null, args);
                    },
                    match,
                    // NOTE: Only for debugging.
                    log: (label) => {
                        // eslint-disable-next-line no-console
                        console.log(label, {
                            constructors,
                            constructor: key,
                            params: args
                        });
                    }
                };
            };
        });
        return adt;
    };
    const Adt = {
        generate: generate$1
    };

    const Cell = (initial) => {
        let value = initial;
        const get = () => {
            return value;
        };
        const set = (v) => {
            value = v;
        };
        return {
            get,
            set
        };
    };

    /**
     * Creates a new `Result<T, E>` that **does** contain a value.
     */
    const value$2 = (value) => {
        const applyHelper = (fn) => fn(value);
        const constHelper = constant(value);
        const outputHelper = () => output;
        const output = {
            // Debug info
            tag: true,
            inner: value,
            // Actual Result methods
            fold: (_onError, onValue) => onValue(value),
            isValue: always,
            isError: never,
            map: (mapper) => Result.value(mapper(value)),
            mapError: outputHelper,
            bind: applyHelper,
            exists: applyHelper,
            forall: applyHelper,
            getOr: constHelper,
            or: outputHelper,
            getOrThunk: constHelper,
            orThunk: outputHelper,
            getOrDie: constHelper,
            each: (fn) => {
                // Can't write the function inline because we don't want to return something by mistake
                fn(value);
            },
            toOptional: () => Optional.some(value),
        };
        return output;
    };
    /**
     * Creates a new `Result<T, E>` that **does not** contain a value, and therefore
     * contains an error.
     */
    const error = (error) => {
        const outputHelper = () => output;
        const output = {
            // Debug info
            tag: false,
            inner: error,
            // Actual Result methods
            fold: (onError, _onValue) => onError(error),
            isValue: never,
            isError: always,
            map: outputHelper,
            mapError: (mapper) => Result.error(mapper(error)),
            bind: outputHelper,
            exists: never,
            forall: always,
            getOr: identity,
            or: identity,
            getOrThunk: apply$1,
            orThunk: apply$1,
            getOrDie: die(String(error)),
            each: noop,
            toOptional: Optional.none,
        };
        return output;
    };
    /**
     * Creates a new `Result<T, E>` from an `Optional<T>` and an `E`. If the
     * `Optional` contains a value, so will the outputted `Result`. If it does not,
     * the outputted `Result` will contain an error (and that error will be the
     * error passed in).
     */
    const fromOption = (optional, err) => optional.fold(() => error(err), value$2);
    const Result = {
        value: value$2,
        error,
        fromOption
    };

    // Use window object as the global if it's available since CSP will block script evals
    // eslint-disable-next-line @typescript-eslint/no-implied-eval
    const Global = typeof window !== 'undefined' ? window : Function('return this;')();

    /* eslint-disable no-bitwise */
    const uuidV4Bytes = () => {
        const bytes = window.crypto.getRandomValues(new Uint8Array(16));
        // https://tools.ietf.org/html/rfc4122#section-4.1.3
        // This will first bit mask away the most significant 4 bits (version octet)
        // then mask in the v4 number we only care about v4 random version at this point so (byte & 0b00001111 | 0b01000000)
        bytes[6] = bytes[6] & 15 | 64;
        // https://tools.ietf.org/html/rfc4122#section-4.1.1
        // This will first bit mask away the highest two bits then masks in the highest bit so (byte & 0b00111111 | 0b10000000)
        // So it will set the Msb0=1 & Msb1=0 described by the "The variant specified in this document." row in the table
        bytes[8] = bytes[8] & 63 | 128;
        return bytes;
    };
    const uuidV4String = () => {
        const uuid = uuidV4Bytes();
        const getHexRange = (startIndex, endIndex) => {
            let buff = '';
            for (let i = startIndex; i <= endIndex; ++i) {
                const hexByte = uuid[i].toString(16).padStart(2, '0');
                buff += hexByte;
            }
            return buff;
        };
        // RFC 4122 UUID format: xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx
        return `${getHexRange(0, 3)}-${getHexRange(4, 5)}-${getHexRange(6, 7)}-${getHexRange(8, 9)}-${getHexRange(10, 15)}`;
    };

    /**
     * Adds two numbers, and wrap to a range.
     * If the result overflows to the right, snap to the left.
     * If the result overflows to the left, snap to the right.
     */
    // ASSUMPTION: Max will always be larger than min
    const clamp$2 = (value, min, max) => Math.min(Math.max(value, min), max);
    // the division is meant to get a number between 0 and 1 for more information check this discussion: https://stackoverflow.com/questions/58285941/how-to-replace-math-random-with-crypto-getrandomvalues-and-keep-same-result
    const random = () => window.crypto.getRandomValues(new Uint32Array(1))[0] / 4294967295;

    /**
     * Generate a unique identifier.
     *
     * The unique portion of the identifier only contains an underscore
     * and digits, so that it may safely be used within HTML attributes.
     *
     * The chance of generating a non-unique identifier has been minimized
     * by combining the current time, a random number and a one-up counter.
     *
     * generate :: String -> String
     */
    let unique = 0;
    const generate = (prefix) => {
        const date = new Date();
        const time = date.getTime();
        const random$1 = Math.floor(random() * 1000000000);
        unique++;
        return prefix + '_' + random$1 + unique + String(time);
    };
    /**
     * Generate a uuidv4 string
     * In accordance with RFC 4122 (https://datatracker.ietf.org/doc/html/rfc4122)
     */
    const uuidV4 = () => {
        if (window.isSecureContext) {
            return window.crypto.randomUUID();
        }
        else {
            return uuidV4String();
        }
    };

    const shallow$1 = (old, nu) => {
        return nu;
    };
    const deep$1 = (old, nu) => {
        const bothObjects = isPlainObject(old) && isPlainObject(nu);
        return bothObjects ? deepMerge(old, nu) : nu;
    };
    const baseMerge = (merger) => {
        return (...objects) => {
            if (objects.length === 0) {
                throw new Error(`Can't merge zero objects`);
            }
            const ret = {};
            for (let j = 0; j < objects.length; j++) {
                const curObject = objects[j];
                for (const key in curObject) {
                    if (has$2(curObject, key)) {
                        ret[key] = merger(ret[key], curObject[key]);
                    }
                }
            }
            return ret;
        };
    };
    const deepMerge = baseMerge(deep$1);
    const merge$1 = baseMerge(shallow$1);

    /**
     * **Is** the value stored inside this Optional object equal to `rhs`?
     */
    const is$4 = (lhs, rhs, comparator = tripleEquals) => lhs.exists((left) => comparator(left, rhs));
    /**
     * Are these two Optional objects equal? Equality here means either they're both
     * `Some` (and the values are equal under the comparator) or they're both `None`.
     */
    const equals = (lhs, rhs, comparator = tripleEquals) => lift2(lhs, rhs, comparator).getOr(lhs.isNone() && rhs.isNone());
    const cat = (arr) => {
        const r = [];
        const push = (x) => {
            r.push(x);
        };
        for (let i = 0; i < arr.length; i++) {
            arr[i].each(push);
        }
        return r;
    };
    /*
    Notes on the lift functions:
    - We used to have a generic liftN, but we were concerned about its type-safety, and the below variants were faster in microbenchmarks.
    - The getOrDie calls are partial functions, but are checked beforehand. This is faster and more convenient (but less safe) than folds.
    - && is used instead of a loop for simplicity and performance.
    */
    const lift2 = (oa, ob, f) => oa.isSome() && ob.isSome() ? Optional.some(f(oa.getOrDie(), ob.getOrDie())) : Optional.none();
    const lift3 = (oa, ob, oc, f) => oa.isSome() && ob.isSome() && oc.isSome() ? Optional.some(f(oa.getOrDie(), ob.getOrDie(), oc.getOrDie())) : Optional.none();
    const flatten = (oot) => oot.bind(identity);
    // This can help with type inference, by specifying the type param on the none case, so the caller doesn't have to.
    const someIf = (b, a) => b ? Optional.some(a) : Optional.none();

    /** path :: ([String], JsObj?) -> JsObj */
    const path = (parts, scope) => {
        let o = scope !== undefined && scope !== null ? scope : Global;
        for (let i = 0; i < parts.length && o !== undefined && o !== null; ++i) {
            o = o[parts[i]];
        }
        return o;
    };
    /** resolve :: (String, JsObj?) -> JsObj */
    const resolve$3 = (p, scope) => {
        const parts = p.split('.');
        return path(parts, scope);
    };

    Adt.generate([
        { bothErrors: ['error1', 'error2'] },
        { firstError: ['error1', 'value2'] },
        { secondError: ['value1', 'error2'] },
        { bothValues: ['value1', 'value2'] }
    ]);
    /** partition :: [Result a] -> { errors: [String], values: [a] } */
    const partition$1 = (results) => {
        const errors = [];
        const values = [];
        each$e(results, (result) => {
            result.fold((err) => {
                errors.push(err);
            }, (value) => {
                values.push(value);
            });
        });
        return { errors, values };
    };

    const singleton = (doRevoke) => {
        const subject = Cell(Optional.none());
        const revoke = () => subject.get().each(doRevoke);
        const clear = () => {
            revoke();
            subject.set(Optional.none());
        };
        const isSet = () => subject.get().isSome();
        const get = () => subject.get();
        const set = (s) => {
            revoke();
            subject.set(Optional.some(s));
        };
        return {
            clear,
            isSet,
            get,
            set
        };
    };
    const repeatable = (delay) => {
        const intervalId = Cell(Optional.none());
        const revoke = () => intervalId.get().each((id) => clearInterval(id));
        const clear = () => {
            revoke();
            intervalId.set(Optional.none());
        };
        const isSet = () => intervalId.get().isSome();
        const get = () => intervalId.get();
        const set = (functionToRepeat) => {
            revoke();
            intervalId.set(Optional.some(setInterval(functionToRepeat, delay)));
        };
        return {
            clear,
            isSet,
            get,
            set,
        };
    };
    const value$1 = () => {
        const subject = singleton(noop);
        const on = (f) => subject.get().each(f);
        return {
            ...subject,
            on
        };
    };

    const removeFromStart = (str, numChars) => {
        return str.substring(numChars);
    };

    const checkRange = (str, substr, start) => substr === '' || str.length >= substr.length && str.substr(start, start + substr.length) === substr;
    const removeLeading = (str, prefix) => {
        return startsWith(str, prefix) ? removeFromStart(str, prefix.length) : str;
    };
    const contains$1 = (str, substr, start = 0, end) => {
        const idx = str.indexOf(substr, start);
        if (idx !== -1) {
            return isUndefined(end) ? true : idx + substr.length <= end;
        }
        else {
            return false;
        }
    };
    /** Does 'str' start with 'prefix'?
     *  Note: all strings start with the empty string.
     *        More formally, for all strings x, startsWith(x, "").
     *        This is so that for all strings x and y, startsWith(y + x, y)
     */
    const startsWith = (str, prefix) => {
        return checkRange(str, prefix, 0);
    };
    /** Does 'str' end with 'suffix'?
     *  Note: all strings end with the empty string.
     *        More formally, for all strings x, endsWith(x, "").
     *        This is so that for all strings x and y, endsWith(x + y, y)
     */
    const endsWith = (str, suffix) => {
        return checkRange(str, suffix, str.length - suffix.length);
    };
    const blank = (r) => (s) => s.replace(r, '');
    /** removes all leading and trailing spaces */
    const trim$4 = blank(/^\s+|\s+$/g);
    const lTrim = blank(/^\s+/g);
    const rTrim = blank(/\s+$/g);
    const isNotEmpty = (s) => s.length > 0;
    const isEmpty$5 = (s) => !isNotEmpty(s);
    const repeat = (s, count) => count <= 0 ? '' : new Array(count + 1).join(s);
    const toInt = (value, radix = 10) => {
        const num = parseInt(value, radix);
        return isNaN(num) ? Optional.none() : Optional.some(num);
    };

    // Run a function fn after rate ms. If another invocation occurs
    // during the time it is waiting, ignore it completely.
    const first$1 = (fn, rate) => {
        let timer = null;
        const cancel = () => {
            if (!isNull(timer)) {
                clearTimeout(timer);
                timer = null;
            }
        };
        const throttle = (...args) => {
            if (isNull(timer)) {
                timer = setTimeout(() => {
                    timer = null;
                    fn.apply(null, args);
                }, rate);
            }
        };
        return {
            cancel,
            throttle
        };
    };
    // Run a function fn after rate ms. If another invocation occurs
    // during the time it is waiting, reschedule the function again
    // with the new arguments.
    const last$1 = (fn, rate) => {
        let timer = null;
        const cancel = () => {
            if (!isNull(timer)) {
                clearTimeout(timer);
                timer = null;
            }
        };
        const throttle = (...args) => {
            cancel();
            timer = setTimeout(() => {
                timer = null;
                fn.apply(null, args);
            }, rate);
        };
        return {
            cancel,
            throttle
        };
    };

    const cached = (f) => {
        let called = false;
        let r;
        return (...args) => {
            if (!called) {
                called = true;
                r = f.apply(null, args);
            }
            return r;
        };
    };

    const zeroWidth = '\uFEFF';
    const nbsp = '\u00A0';
    const ellipsis = '\u2026';
    const isZwsp$2 = (char) => char === zeroWidth;
    const removeZwsp = (s) => s.replace(/\uFEFF/g, '');

    const stringArray = (a) => {
        const all = {};
        each$e(a, (key) => {
            all[key] = {};
        });
        return keys(all);
    };

    const isArrayLike = (o) => o.length !== undefined;
    const isArray = Array.isArray;
    const toArray$1 = (obj) => {
        if (!isArray(obj)) {
            const array = [];
            for (let i = 0, l = obj.length; i < l; i++) {
                array[i] = obj[i];
            }
            return array;
        }
        else {
            return obj;
        }
    };
    const each$c = (o, cb, s) => {
        if (!o) {
            return false;
        }
        s = s || o;
        if (isArrayLike(o)) {
            // Indexed arrays, needed for Safari
            for (let n = 0, l = o.length; n < l; n++) {
                if (cb.call(s, o[n], n, o) === false) {
                    return false;
                }
            }
        }
        else {
            // Hashtables
            for (const n in o) {
                if (has$2(o, n)) {
                    if (cb.call(s, o[n], n, o) === false) {
                        return false;
                    }
                }
            }
        }
        return true;
    };
    const map$1 = (array, callback) => {
        const out = [];
        each$c(array, (item, index) => {
            out.push(callback(item, index, array));
        });
        return out;
    };
    const filter$3 = (a, f) => {
        const o = [];
        each$c(a, (v, index) => {
            if (!f || f(v, index, a)) {
                o.push(v);
            }
        });
        return o;
    };
    const indexOf = (a, v) => {
        if (a) {
            for (let i = 0, l = a.length; i < l; i++) {
                if (a[i] === v) {
                    return i;
                }
            }
        }
        return -1;
    };
    const reduce = (collection, iteratee, accumulator, thisArg) => {
        let acc = isUndefined(accumulator) ? collection[0] : accumulator;
        for (let i = 0; i < collection.length; i++) {
            acc = iteratee.call(thisArg, acc, collection[i], i);
        }
        return acc;
    };
    const findIndex$1 = (array, predicate, thisArg) => {
        for (let i = 0, l = array.length; i < l; i++) {
            if (predicate.call(thisArg, array[i], i, array)) {
                return i;
            }
        }
        return -1;
    };
    const last = (collection) => collection[collection.length - 1];

    const DeviceType = (os, browser, userAgent, mediaMatch) => {
        const isiPad = os.isiOS() && /ipad/i.test(userAgent) === true;
        const isiPhone = os.isiOS() && !isiPad;
        const isMobile = os.isiOS() || os.isAndroid();
        const isTouch = isMobile || mediaMatch('(pointer:coarse)');
        const isTablet = isiPad || !isiPhone && isMobile && mediaMatch('(min-device-width:768px)');
        const isPhone = isiPhone || isMobile && !isTablet;
        const iOSwebview = browser.isSafari() && os.isiOS() && /safari/i.test(userAgent) === false;
        const isDesktop = !isPhone && !isTablet && !iOSwebview;
        return {
            isiPad: constant(isiPad),
            isiPhone: constant(isiPhone),
            isTablet: constant(isTablet),
            isPhone: constant(isPhone),
            isTouch: constant(isTouch),
            isAndroid: os.isAndroid,
            isiOS: os.isiOS,
            isWebView: constant(iOSwebview),
            isDesktop: constant(isDesktop)
        };
    };

    const firstMatch = (regexes, s) => {
        for (let i = 0; i < regexes.length; i++) {
            const x = regexes[i];
            if (x.test(s)) {
                return x;
            }
        }
        return undefined;
    };
    const find$1 = (regexes, agent) => {
        const r = firstMatch(regexes, agent);
        if (!r) {
            return { major: 0, minor: 0 };
        }
        const group = (i) => {
            return Number(agent.replace(r, '$' + i));
        };
        return nu$3(group(1), group(2));
    };
    const detect$4 = (versionRegexes, agent) => {
        const cleanedAgent = String(agent).toLowerCase();
        if (versionRegexes.length === 0) {
            return unknown$2();
        }
        return find$1(versionRegexes, cleanedAgent);
    };
    const unknown$2 = () => {
        return nu$3(0, 0);
    };
    const nu$3 = (major, minor) => {
        return { major, minor };
    };
    const Version = {
        nu: nu$3,
        detect: detect$4,
        unknown: unknown$2
    };

    const detectBrowser$1 = (browsers, userAgentData) => {
        return findMap(userAgentData.brands, (uaBrand) => {
            const lcBrand = uaBrand.brand.toLowerCase();
            return find$2(browsers, (browser) => lcBrand === browser.brand?.toLowerCase())
                .map((info) => ({
                current: info.name,
                version: Version.nu(parseInt(uaBrand.version, 10), 0)
            }));
        });
    };

    const detect$3 = (candidates, userAgent) => {
        const agent = String(userAgent).toLowerCase();
        return find$2(candidates, (candidate) => {
            return candidate.search(agent);
        });
    };
    // They (browser and os) are the same at the moment, but they might
    // not stay that way.
    const detectBrowser = (browsers, userAgent) => {
        return detect$3(browsers, userAgent).map((browser) => {
            const version = Version.detect(browser.versionRegexes, userAgent);
            return {
                current: browser.name,
                version
            };
        });
    };
    const detectOs = (oses, userAgent) => {
        return detect$3(oses, userAgent).map((os) => {
            const version = Version.detect(os.versionRegexes, userAgent);
            return {
                current: os.name,
                version
            };
        });
    };

    const normalVersionRegex = /.*?version\/\ ?([0-9]+)\.([0-9]+).*/;
    const checkContains = (target) => {
        return (uastring) => {
            return contains$1(uastring, target);
        };
    };
    const browsers = [
        // This is legacy Edge
        {
            name: 'Edge',
            versionRegexes: [/.*?edge\/ ?([0-9]+)\.([0-9]+)$/],
            search: (uastring) => {
                return contains$1(uastring, 'edge/') && contains$1(uastring, 'chrome') && contains$1(uastring, 'safari') && contains$1(uastring, 'applewebkit');
            }
        },
        // This is Google Chrome and Chromium Edge
        {
            name: 'Chromium',
            brand: 'Chromium',
            versionRegexes: [/.*?chrome\/([0-9]+)\.([0-9]+).*/, normalVersionRegex],
            search: (uastring) => {
                return contains$1(uastring, 'chrome') && !contains$1(uastring, 'chromeframe');
            }
        },
        {
            name: 'IE',
            versionRegexes: [/.*?msie\ ?([0-9]+)\.([0-9]+).*/, /.*?rv:([0-9]+)\.([0-9]+).*/],
            search: (uastring) => {
                return contains$1(uastring, 'msie') || contains$1(uastring, 'trident');
            }
        },
        // INVESTIGATE: Is this still the Opera user agent?
        {
            name: 'Opera',
            versionRegexes: [normalVersionRegex, /.*?opera\/([0-9]+)\.([0-9]+).*/],
            search: checkContains('opera')
        },
        {
            name: 'Firefox',
            versionRegexes: [/.*?firefox\/\ ?([0-9]+)\.([0-9]+).*/],
            search: checkContains('firefox')
        },
        {
            name: 'Safari',
            versionRegexes: [normalVersionRegex, /.*?cpu os ([0-9]+)_([0-9]+).*/],
            search: (uastring) => {
                return (contains$1(uastring, 'safari') || contains$1(uastring, 'mobile/')) && contains$1(uastring, 'applewebkit');
            }
        }
    ];
    const oses = [
        {
            name: 'Windows',
            search: checkContains('win'),
            versionRegexes: [/.*?windows\ nt\ ?([0-9]+)\.([0-9]+).*/]
        },
        {
            name: 'iOS',
            search: (uastring) => {
                return contains$1(uastring, 'iphone') || contains$1(uastring, 'ipad');
            },
            versionRegexes: [/.*?version\/\ ?([0-9]+)\.([0-9]+).*/, /.*cpu os ([0-9]+)_([0-9]+).*/, /.*cpu iphone os ([0-9]+)_([0-9]+).*/]
        },
        {
            name: 'Android',
            search: checkContains('android'),
            versionRegexes: [/.*?android\ ?([0-9]+)\.([0-9]+).*/]
        },
        {
            name: 'macOS',
            search: checkContains('mac os x'),
            versionRegexes: [/.*?mac\ os\ x\ ?([0-9]+)_([0-9]+).*/]
        },
        {
            name: 'Linux',
            search: checkContains('linux'),
            versionRegexes: []
        },
        { name: 'Solaris',
            search: checkContains('sunos'),
            versionRegexes: []
        },
        {
            name: 'FreeBSD',
            search: checkContains('freebsd'),
            versionRegexes: []
        },
        {
            name: 'ChromeOS',
            search: checkContains('cros'),
            versionRegexes: [/.*?chrome\/([0-9]+)\.([0-9]+).*/]
        }
    ];
    const PlatformInfo = {
        browsers: constant(browsers),
        oses: constant(oses)
    };

    const edge = 'Edge';
    const chromium = 'Chromium';
    const ie = 'IE';
    const opera = 'Opera';
    const firefox = 'Firefox';
    const safari = 'Safari';
    const unknown$1 = () => {
        return nu$2({
            current: undefined,
            version: Version.unknown()
        });
    };
    const nu$2 = (info) => {
        const current = info.current;
        const version = info.version;
        const isBrowser = (name) => () => current === name;
        return {
            current,
            version,
            isEdge: isBrowser(edge),
            isChromium: isBrowser(chromium),
            // NOTE: isIe just looks too weird
            isIE: isBrowser(ie),
            isOpera: isBrowser(opera),
            isFirefox: isBrowser(firefox),
            isSafari: isBrowser(safari)
        };
    };
    const Browser = {
        unknown: unknown$1,
        nu: nu$2,
        edge: constant(edge),
        chromium: constant(chromium),
        ie: constant(ie),
        opera: constant(opera),
        firefox: constant(firefox),
        safari: constant(safari)
    };

    const windows = 'Windows';
    const ios = 'iOS';
    const android = 'Android';
    const linux = 'Linux';
    const macos = 'macOS';
    const solaris = 'Solaris';
    const freebsd = 'FreeBSD';
    const chromeos = 'ChromeOS';
    // Though there is a bit of dupe with this and Browser, trying to
    // reuse code makes it much harder to follow and change.
    const unknown = () => {
        return nu$1({
            current: undefined,
            version: Version.unknown()
        });
    };
    const nu$1 = (info) => {
        const current = info.current;
        const version = info.version;
        const isOS = (name) => () => current === name;
        return {
            current,
            version,
            isWindows: isOS(windows),
            // TODO: Fix capitalisation
            isiOS: isOS(ios),
            isAndroid: isOS(android),
            isMacOS: isOS(macos),
            isLinux: isOS(linux),
            isSolaris: isOS(solaris),
            isFreeBSD: isOS(freebsd),
            isChromeOS: isOS(chromeos)
        };
    };
    const OperatingSystem = {
        unknown,
        nu: nu$1,
        windows: constant(windows),
        ios: constant(ios),
        android: constant(android),
        linux: constant(linux),
        macos: constant(macos),
        solaris: constant(solaris),
        freebsd: constant(freebsd),
        chromeos: constant(chromeos)
    };

    const detect$2 = (userAgent, userAgentDataOpt, mediaMatch) => {
        const browsers = PlatformInfo.browsers();
        const oses = PlatformInfo.oses();
        const browser = userAgentDataOpt.bind((userAgentData) => detectBrowser$1(browsers, userAgentData))
            .orThunk(() => detectBrowser(browsers, userAgent))
            .fold(Browser.unknown, Browser.nu);
        const os = detectOs(oses, userAgent).fold(OperatingSystem.unknown, OperatingSystem.nu);
        const deviceType = DeviceType(os, browser, userAgent, mediaMatch);
        return {
            browser,
            os,
            deviceType
        };
    };
    const PlatformDetection = {
        detect: detect$2
    };

    const mediaMatch = (query) => window.matchMedia(query).matches;
    // IMPORTANT: Must be in a thunk, otherwise rollup thinks calling this immediately
    // causes side effects and won't tree shake this away
    // Note: navigator.userAgentData is not part of the native typescript types yet
    let platform$4 = cached(() => PlatformDetection.detect(window.navigator.userAgent, Optional.from((window.navigator.userAgentData)), mediaMatch));
    const detect$1 = () => platform$4();

    const unsafe = (name, scope) => {
        return resolve$3(name, scope);
    };
    const getOrDie = (name, scope) => {
        const actual = unsafe(name, scope);
        if (actual === undefined || actual === null) {
            throw new Error(name + ' not available on this browser');
        }
        return actual;
    };

    const getPrototypeOf$1 = Object.getPrototypeOf;
    /*
     * IE9 and above
     *
     * MDN no use on this one, but here's the link anyway:
     * https://developer.mozilla.org/en/docs/Web/API/HTMLElement
     */
    const sandHTMLElement = (scope) => {
        return getOrDie('HTMLElement', scope);
    };
    const isPrototypeOf = (x) => {
        // use Resolve to get the window object for x and just return undefined if it can't find it.
        // undefined scope later triggers using the global window.
        const scope = resolve$3('ownerDocument.defaultView', x);
        // TINY-7374: We can't rely on looking at the owner window HTMLElement as the element may have
        // been constructed in a different window and then appended to the current window document.
        return isObject(x) && (sandHTMLElement(scope).prototype.isPrototypeOf(x) || /^HTML\w*Element$/.test(getPrototypeOf$1(x).constructor.name));
    };

    /**
     * This class contains various environment constants like browser versions etc.
     * Normally you don't want to sniff specific browser versions but sometimes you have
     * to when it's impossible to feature detect. So use this with care.
     *
     * @class tinymce.Env
     * @static
     */
    const userAgent = window.navigator.userAgent;
    const platform$3 = detect$1();
    const browser$3 = platform$3.browser;
    const os$1 = platform$3.os;
    const deviceType = platform$3.deviceType;
    const windowsPhone = userAgent.indexOf('Windows Phone') !== -1;
    const Env = {
        /**
         * Transparent image data url.
         *
         * @property transparentSrc
         * @type Boolean
         * @final
         */
        transparentSrc: 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7',
        /**
         * Returns the IE document mode. For non IE browsers, this will fake IE 10 document mode.
         *
         * @property documentMode
         * @type Number
         */
        documentMode: browser$3.isIE() ? (document.documentMode || 7) : 10,
        cacheSuffix: null,
        container: null,
        /**
         * Constant if CSP mode is possible or not. Meaning we can't use script urls for the iframe.
         */
        canHaveCSP: !browser$3.isIE(),
        windowsPhone,
        /**
         * @include ../../../../../tools/docs/tinymce.Env.js
         */
        browser: {
            current: browser$3.current,
            version: browser$3.version,
            isChromium: browser$3.isChromium,
            isEdge: browser$3.isEdge,
            isFirefox: browser$3.isFirefox,
            isIE: browser$3.isIE,
            isOpera: browser$3.isOpera,
            isSafari: browser$3.isSafari
        },
        os: {
            current: os$1.current,
            version: os$1.version,
            isAndroid: os$1.isAndroid,
            isChromeOS: os$1.isChromeOS,
            isFreeBSD: os$1.isFreeBSD,
            isiOS: os$1.isiOS,
            isLinux: os$1.isLinux,
            isMacOS: os$1.isMacOS,
            isSolaris: os$1.isSolaris,
            isWindows: os$1.isWindows
        },
        deviceType: {
            isDesktop: deviceType.isDesktop,
            isiPad: deviceType.isiPad,
            isiPhone: deviceType.isiPhone,
            isPhone: deviceType.isPhone,
            isTablet: deviceType.isTablet,
            isTouch: deviceType.isTouch,
            isWebView: deviceType.isWebView
        }
    };

    /**
     * This class contains various utility functions. These are also exposed
     * directly on the tinymce namespace.
     *
     * @class tinymce.util.Tools
     */
    /**
     * Removes whitespace from the beginning and end of a string.
     *
     * @method trim
     * @param {String} s String to remove whitespace from.
     * @return {String} New string with removed whitespace.
     */
    const whiteSpaceRegExp$1 = /^\s*|\s*$/g;
    const trim$3 = (str) => {
        return isNullable(str) ? '' : ('' + str).replace(whiteSpaceRegExp$1, '');
    };
    /**
     * Checks if a object is of a specific type for example an array.
     *
     * @method is
     * @param {Object} obj Object to check type of.
     * @param {String} type Optional type to check for.
     * @return {Boolean} true/false if the object is of the specified type.
     */
    const is$3 = (obj, type) => {
        if (!type) {
            return obj !== undefined;
        }
        if (type === 'array' && isArray(obj)) {
            return true;
        }
        return typeof obj === type;
    };
    /**
     * Makes a name/object map out of an array with names.
     *
     * @method makeMap
     * @param {Array/String} items Items to make map out of.
     * @param {String} delim Optional delimiter to split string by.
     * @param {Object} map Optional map to add items to.
     * @return {Object} Name/value map of items.
     */
    const makeMap$4 = (items, delim, map = {}) => {
        const resolvedItems = isString(items) ? items.split(delim || ',') : (items || []);
        let i = resolvedItems.length;
        while (i--) {
            map[resolvedItems[i]] = {};
        }
        return map;
    };
    /**
     * JavaScript does not protect hasOwnProperty method, so it is possible to overwrite it. This is
     * an object independent version.
     * Checks if the input object "<code>obj</code>" has the property "<code>prop</code>".
     *
     * @method hasOwnProperty
     * @param {Object} obj Object to check if the property exists.
     * @param {String} prop Name of a property on the object.
     * @returns {Boolean} true if the object has the specified property.
     */
    const hasOwnProperty = has$2;
    const extend$3 = (obj, ...exts) => {
        for (let i = 0; i < exts.length; i++) {
            const ext = exts[i];
            for (const name in ext) {
                if (has$2(ext, name)) {
                    const value = ext[name];
                    if (value !== undefined) {
                        obj[name] = value;
                    }
                }
            }
        }
        return obj;
    };
    /**
     * Executed the specified function for each item in a object tree.
     *
     * @method walk
     * @param {Object} o Object tree to walk though.
     * @param {Function} f Function to call for each item.
     * @param {String} n Optional name of collection inside the objects to walk for example childNodes.
     * @param {String} s Optional scope to execute the function in.
     */
    const walk$4 = function (o, f, n, s) {
        s = s || this;
        if (o) {
            if (n) {
                o = o[n];
            }
            each$c(o, (o, i) => {
                if (f.call(s, o, i, n) === false) {
                    return false;
                }
                else {
                    walk$4(o, f, n, s);
                    return true;
                }
            });
        }
    };
    /**
     * Resolves a string and returns the object from a specific structure.
     *
     * @method resolve
     * @param {String} n Path to resolve for example a.b.c.d.
     * @param {Object} o Optional object to search though, defaults to window.
     * @return {Object} Last object in path or null if it couldn't be resolved.
     * @example
     * // Resolve a path into an object reference
     * const obj = tinymce.resolve('a.b.c.d');
     */
    const resolve$2 = (n, o = window) => {
        const path = n.split('.');
        for (let i = 0, l = path.length; i < l; i++) {
            o = o[path[i]];
            if (!o) {
                break;
            }
        }
        return o;
    };
    /**
     * Splits a string but removes the whitespace before and after each value.
     *
     * @method explode
     * @param {String} s String to split.
     * @param {String} d Delimiter to split by.
     * @example
     * // Split a string into an array with a,b,c
     * const arr = tinymce.explode('a, b,   c');
     */
    const explode$3 = (s, d) => {
        if (isArray$1(s)) {
            return s;
        }
        else if (s === '') {
            return [];
        }
        else {
            return map$1(s.split(d || ','), trim$3);
        }
    };
    const _addCacheSuffix = (url) => {
        const cacheSuffix = Env.cacheSuffix;
        if (cacheSuffix) {
            url += (url.indexOf('?') === -1 ? '?' : '&') + cacheSuffix;
        }
        return url;
    };
    const Tools = {
        trim: trim$3,
        /**
         * Returns true/false if the object is an array or not.
         *
         * @method isArray
         * @param {Object} obj Object to check.
         * @return {Boolean} true/false state if the object is an array or not.
         */
        isArray: isArray,
        is: is$3,
        /**
         * Converts the specified object into a real JavaScript array.
         *
         * @method toArray
         * @param {Object} obj Object to convert into array.
         * @return {Array} Array object based in input.
         */
        toArray: toArray$1,
        makeMap: makeMap$4,
        /**
         * Performs an iteration of all items in a collection such as an object or array. This method will execute the
         * callback function for each item in the collection, if the callback returns false the iteration will terminate.
         * The callback has the following format: `cb(value, key_or_index)`.
         *
         * @method each
         * @param {Object} o Collection to iterate.
         * @param {Function} cb Callback function to execute for each item.
         * @param {Object} s Optional scope to execute the callback in.
         * @example
         * // Iterate an array
         * tinymce.each([ 1,2,3 ], (v, i) => {
         *   console.debug("Value: " + v + ", Index: " + i);
         * });
         *
         * // Iterate an object
         * tinymce.each({ a: 1, b: 2, c: 3 }, (v, k) => {
         *   console.debug("Value: " + v + ", Key: " + k);
         * });
         */
        each: each$c,
        /**
         * Creates a new array by the return value of each iteration function call. This enables you to convert
         * one array list into another.
         *
         * @method map
         * @param {Array} array Array of items to iterate.
         * @param {Function} callback Function to call for each item. It's return value will be the new value.
         * @return {Array} Array with new values based on function return values.
         */
        map: map$1,
        /**
         * Filters out items from the input array by calling the specified function for each item.
         * If the function returns false the item will be excluded if it returns true it will be included.
         *
         * @method grep
         * @param {Array} a Array of items to loop though.
         * @param {Function} f Function to call for each item. Include/exclude depends on it's return value.
         * @return {Array} New array with values imported and filtered based in input.
         * @example
         * // Filter out some items, this will return an array with 4 and 5
         * const items = tinymce.grep([ 1,2,3,4,5 ], (v) => v > 3);
         */
        grep: filter$3,
        /**
         * Returns an index of the item or -1 if item is not present in the array.
         *
         * @method inArray
         * @param {any} item Item to search for.
         * @param {Array} arr Array to search in.
         * @return {Number} index of the item or -1 if item was not found.
         */
        inArray: indexOf,
        hasOwn: hasOwnProperty,
        extend: extend$3,
        walk: walk$4,
        resolve: resolve$2,
        explode: explode$3,
        _addCacheSuffix
    };

    const fromHtml$1 = (html, scope) => {
        const doc = scope || document;
        const div = doc.createElement('div');
        div.innerHTML = html;
        if (!div.hasChildNodes() || div.childNodes.length > 1) {
            const message = 'HTML does not have a single root node';
            // eslint-disable-next-line no-console
            console.error(message, html);
            throw new Error(message);
        }
        return fromDom$2(div.childNodes[0]);
    };
    const fromTag = (tag, scope) => {
        const doc = scope || document;
        const node = doc.createElement(tag);
        return fromDom$2(node);
    };
    const fromText = (text, scope) => {
        const doc = scope || document;
        const node = doc.createTextNode(text);
        return fromDom$2(node);
    };
    const fromDom$2 = (node) => {
        // TODO: Consider removing this check, but left atm for safety
        if (node === null || node === undefined) {
            throw new Error('Node cannot be null or undefined');
        }
        return {
            dom: node
        };
    };
    const fromPoint$2 = (docElm, x, y) => Optional.from(docElm.dom.elementFromPoint(x, y)).map(fromDom$2);
    // tslint:disable-next-line:variable-name
    const SugarElement = {
        fromHtml: fromHtml$1,
        fromTag,
        fromText,
        fromDom: fromDom$2,
        fromPoint: fromPoint$2
    };

    // NOTE: Mutates the range.
    const setStart = (rng, situ) => {
        situ.fold((e) => {
            rng.setStartBefore(e.dom);
        }, (e, o) => {
            rng.setStart(e.dom, o);
        }, (e) => {
            rng.setStartAfter(e.dom);
        });
    };
    const setFinish = (rng, situ) => {
        situ.fold((e) => {
            rng.setEndBefore(e.dom);
        }, (e, o) => {
            rng.setEnd(e.dom, o);
        }, (e) => {
            rng.setEndAfter(e.dom);
        });
    };
    const relativeToNative = (win, startSitu, finishSitu) => {
        const range = win.document.createRange();
        setStart(range, startSitu);
        setFinish(range, finishSitu);
        return range;
    };
    const exactToNative = (win, start, soffset, finish, foffset) => {
        const rng = win.document.createRange();
        rng.setStart(start.dom, soffset);
        rng.setEnd(finish.dom, foffset);
        return rng;
    };

    const adt$3 = Adt.generate([
        { ltr: ['start', 'soffset', 'finish', 'foffset'] },
        { rtl: ['start', 'soffset', 'finish', 'foffset'] }
    ]);
    const fromRange = (win, type, range) => type(SugarElement.fromDom(range.startContainer), range.startOffset, SugarElement.fromDom(range.endContainer), range.endOffset);
    const getRanges$1 = (win, selection) => selection.match({
        domRange: (rng) => {
            return {
                ltr: constant(rng),
                rtl: Optional.none
            };
        },
        relative: (startSitu, finishSitu) => {
            return {
                ltr: cached(() => relativeToNative(win, startSitu, finishSitu)),
                rtl: cached(() => Optional.some(relativeToNative(win, finishSitu, startSitu)))
            };
        },
        exact: (start, soffset, finish, foffset) => {
            return {
                ltr: cached(() => exactToNative(win, start, soffset, finish, foffset)),
                rtl: cached(() => Optional.some(exactToNative(win, finish, foffset, start, soffset)))
            };
        }
    });
    const doDiagnose = (win, ranges) => {
        // If we cannot create a ranged selection from start > finish, it could be RTL
        const rng = ranges.ltr();
        if (rng.collapsed) {
            // Let's check if it's RTL ... if it is, then reversing the direction will not be collapsed
            const reversed = ranges.rtl().filter((rev) => rev.collapsed === false);
            return reversed.map((rev) => 
            // We need to use "reversed" here, because the original only has one point (collapsed)
            adt$3.rtl(SugarElement.fromDom(rev.endContainer), rev.endOffset, SugarElement.fromDom(rev.startContainer), rev.startOffset)).getOrThunk(() => fromRange(win, adt$3.ltr, rng));
        }
        else {
            return fromRange(win, adt$3.ltr, rng);
        }
    };
    const diagnose = (win, selection) => {
        const ranges = getRanges$1(win, selection);
        return doDiagnose(win, ranges);
    };
    adt$3.ltr;
    adt$3.rtl;

    const COMMENT = 8;
    const DOCUMENT = 9;
    const DOCUMENT_FRAGMENT = 11;
    const ELEMENT = 1;
    const TEXT = 3;

    const is$2 = (element, selector) => {
        const dom = element.dom;
        if (dom.nodeType !== ELEMENT) {
            return false;
        }
        else {
            const elem = dom;
            if (elem.matches !== undefined) {
                return elem.matches(selector);
            }
            else if (elem.msMatchesSelector !== undefined) {
                return elem.msMatchesSelector(selector);
            }
            else if (elem.webkitMatchesSelector !== undefined) {
                return elem.webkitMatchesSelector(selector);
            }
            else if (elem.mozMatchesSelector !== undefined) {
                // cast to any as mozMatchesSelector doesn't exist in TS DOM lib
                return elem.mozMatchesSelector(selector);
            }
            else {
                throw new Error('Browser lacks native selectors');
            } // unfortunately we can't throw this on startup :(
        }
    };
    const bypassSelector = (dom) => 
    // Only elements, documents and shadow roots support querySelector
    // shadow root element type is DOCUMENT_FRAGMENT
    dom.nodeType !== ELEMENT && dom.nodeType !== DOCUMENT && dom.nodeType !== DOCUMENT_FRAGMENT ||
        // IE fix for complex queries on empty nodes: http://jsfiddle.net/spyder/fv9ptr5L/
        dom.childElementCount === 0;
    const all = (selector, scope) => {
        const base = scope === undefined ? document : scope.dom;
        return bypassSelector(base) ? [] : map$3(base.querySelectorAll(selector), SugarElement.fromDom);
    };
    const one = (selector, scope) => {
        const base = scope === undefined ? document : scope.dom;
        return bypassSelector(base) ? Optional.none() : Optional.from(base.querySelector(selector)).map(SugarElement.fromDom);
    };

    const eq = (e1, e2) => e1.dom === e2.dom;
    // Returns: true if node e1 contains e2, otherwise false.
    // (returns false if e1===e2: A node does not contain itself).
    const contains = (e1, e2) => {
        const d1 = e1.dom;
        const d2 = e2.dom;
        return d1 === d2 ? false : d1.contains(d2);
    };
    const is$1 = is$2;

    /**
     * Applies f repeatedly until it completes (by returning Optional.none()).
     *
     * Normally would just use recursion, but JavaScript lacks tail call optimisation.
     *
     * This is what recursion looks like when manually unravelled :)
     */
    const toArray = (target, f) => {
        const r = [];
        const recurse = (e) => {
            r.push(e);
            return f(e);
        };
        let cur = f(target);
        do {
            cur = cur.bind(recurse);
        } while (cur.isSome());
        return r;
    };

    const name = (element) => {
        const r = element.dom.nodeName;
        return r.toLowerCase();
    };
    const type$1 = (element) => element.dom.nodeType;
    const isType = (t) => (element) => type$1(element) === t;
    const isComment$1 = (element) => type$1(element) === COMMENT || name(element) === '#comment';
    const isHTMLElement$1 = (element) => isElement$8(element) && isPrototypeOf(element.dom);
    const isElement$8 = isType(ELEMENT);
    const isText$c = isType(TEXT);
    const isDocument$2 = isType(DOCUMENT);
    const isDocumentFragment$1 = isType(DOCUMENT_FRAGMENT);
    const isTag = (tag) => (e) => isElement$8(e) && name(e) === tag;

    /**
     * The document associated with the current element
     * NOTE: this will throw if the owner is null.
     */
    const owner$1 = (element) => SugarElement.fromDom(element.dom.ownerDocument);
    /**
     * If the element is a document, return it. Otherwise, return its ownerDocument.
     * @param dos
     */
    const documentOrOwner = (dos) => isDocument$2(dos) ? dos : owner$1(dos);
    const documentElement = (element) => SugarElement.fromDom(documentOrOwner(element).dom.documentElement);
    /**
     * The window element associated with the element
     * NOTE: this will throw if the defaultView is null.
     */
    const defaultView = (element) => SugarElement.fromDom(documentOrOwner(element).dom.defaultView);
    const parent = (element) => Optional.from(element.dom.parentNode).map(SugarElement.fromDom);
    const parentElement = (element) => Optional.from(element.dom.parentElement).map(SugarElement.fromDom);
    const parents$1 = (element, isRoot) => {
        const stop = isFunction(isRoot) ? isRoot : never;
        // This is used a *lot* so it needs to be performant, not recursive
        let dom = element.dom;
        const ret = [];
        while (dom.parentNode !== null && dom.parentNode !== undefined) {
            const rawParent = dom.parentNode;
            const p = SugarElement.fromDom(rawParent);
            ret.push(p);
            if (stop(p) === true) {
                break;
            }
            else {
                dom = rawParent;
            }
        }
        return ret;
    };
    const siblings = (element) => {
        // TODO: Refactor out children so we can just not add self instead of filtering afterwards
        const filterSelf = (elements) => filter$5(elements, (x) => !eq(element, x));
        return parent(element).map(children$1).map(filterSelf).getOr([]);
    };
    const prevSibling = (element) => Optional.from(element.dom.previousSibling).map(SugarElement.fromDom);
    const nextSibling = (element) => Optional.from(element.dom.nextSibling).map(SugarElement.fromDom);
    // This one needs to be reversed, so they're still in DOM order
    const prevSiblings = (element) => reverse(toArray(element, prevSibling));
    const nextSiblings = (element) => toArray(element, nextSibling);
    const children$1 = (element) => map$3(element.dom.childNodes, SugarElement.fromDom);
    const child$1 = (element, index) => {
        const cs = element.dom.childNodes;
        return Optional.from(cs[index]).map(SugarElement.fromDom);
    };
    const firstChild = (element) => child$1(element, 0);
    const lastChild = (element) => child$1(element, element.dom.childNodes.length - 1);
    const childNodesCount = (element) => element.dom.childNodes.length;

    const getHead = (doc) => {
        /*
         * IE9 and above per
         * https://developer.mozilla.org/en-US/docs/Web/API/Document/head
         */
        const b = doc.dom.head;
        if (b === null || b === undefined) {
            throw new Error('Head is not available yet');
        }
        return SugarElement.fromDom(b);
    };

    /**
     * Is the element a ShadowRoot?
     *
     * Note: this is insufficient to test if any element is a shadow root, but it is sufficient to differentiate between
     * a Document and a ShadowRoot.
     */
    const isShadowRoot = (dos) => isDocumentFragment$1(dos) && isNonNullable(dos.dom.host);
    const getRootNode = (e) => SugarElement.fromDom(e.dom.getRootNode());
    /** Where style tags need to go. ShadowRoot or document head */
    const getStyleContainer = (dos) => isShadowRoot(dos) ? dos : getHead(documentOrOwner(dos));
    /** Where content needs to go. ShadowRoot or document body */
    const getContentContainer = (dos) => 
    // Can't use SugarBody.body without causing a circular module reference (since SugarBody.inBody uses SugarShadowDom)
    isShadowRoot(dos) ? dos : SugarElement.fromDom(documentOrOwner(dos).dom.body);
    /** If this element is in a ShadowRoot, return it. */
    const getShadowRoot = (e) => {
        const r = getRootNode(e);
        return isShadowRoot(r) ? Optional.some(r) : Optional.none();
    };
    /** Return the host of a ShadowRoot.
     *
     * This function will throw if Shadow DOM is unsupported in the browser, or if the host is null.
     * If you actually have a ShadowRoot, this shouldn't happen.
     */
    const getShadowHost = (e) => SugarElement.fromDom(e.dom.host);
    /**
     * When Events bubble up through a ShadowRoot, the browser changes the target to be the shadow host.
     * This function gets the "original" event target if possible.
     * This only works if the shadow tree is open - if the shadow tree is closed, event.target is returned.
     * See: https://developers.google.com/web/fundamentals/web-components/shadowdom#events
     */
    const getOriginalEventTarget = (event) => {
        if (isNonNullable(event.target)) {
            const el = SugarElement.fromDom(event.target);
            if (isElement$8(el) && isOpenShadowHost(el)) {
                // When target element is inside Shadow DOM we need to take first element from composedPath
                // otherwise we'll get Shadow Root parent, not actual target element.
                if (event.composed && event.composedPath) {
                    const composedPath = event.composedPath();
                    if (composedPath) {
                        return head(composedPath);
                    }
                }
            }
        }
        return Optional.from(event.target);
    };
    /** Return true if the element is a host of an open shadow root.
     *  Return false if the element is a host of a closed shadow root, or if the element is not a host.
     */
    const isOpenShadowHost = (element) => isNonNullable(element.dom.shadowRoot);

    const mkEvent = (target, x, y, stop, prevent, kill, raw) => ({
        target,
        x,
        y,
        stop,
        prevent,
        kill,
        raw
    });
    /** Wraps an Event in an EventArgs structure.
     * The returned EventArgs structure has its target set to the "original" target if possible.
     * See SugarShadowDom.getOriginalEventTarget
     */
    const fromRawEvent = (rawEvent) => {
        const target = SugarElement.fromDom(getOriginalEventTarget(rawEvent).getOr(rawEvent.target));
        const stop = () => rawEvent.stopPropagation();
        const prevent = () => rawEvent.preventDefault();
        const kill = compose(prevent, stop); // more of a sequence than a compose, but same effect
        // FIX: Don't just expose the raw event. Need to identify what needs standardisation.
        return mkEvent(target, rawEvent.clientX, rawEvent.clientY, stop, prevent, kill, rawEvent);
    };
    const handle$1 = (filter, handler) => (rawEvent) => {
        if (filter(rawEvent)) {
            handler(fromRawEvent(rawEvent));
        }
    };
    const binder = (element, event, filter, handler, useCapture) => {
        const wrapped = handle$1(filter, handler);
        // IE9 minimum
        element.dom.addEventListener(event, wrapped, useCapture);
        return {
            unbind: curry(unbind, element, event, wrapped, useCapture)
        };
    };
    const bind$2 = (element, event, filter, handler) => binder(element, event, filter, handler, false);
    const unbind = (element, event, handler, useCapture) => {
        // IE9 minimum
        element.dom.removeEventListener(event, handler, useCapture);
    };

    const filter$2 = always; // no filter on plain DomEvents
    const bind$1 = (element, event, handler) => bind$2(element, event, filter$2, handler);

    const getDocument = () => SugarElement.fromDom(document);

    const focus$1 = (element, preventScroll = false) => element.dom.focus({ preventScroll });
    const hasFocus$1 = (element) => {
        const root = getRootNode(element).dom;
        return element.dom === root.activeElement;
    };
    // Note: assuming that activeElement will always be a HTMLElement (maybe we should add a runtime check?)
    const active = (root = getDocument()) => Optional.from(root.dom.activeElement).map(SugarElement.fromDom);
    /**
     * Return the descendant element that has focus.
     * Use instead of SelectorFind.descendant(container, ':focus')
     *  because the :focus selector relies on keyboard focus.
     */
    const search = (element) => active(getRootNode(element))
        .filter((e) => element.dom.contains(e.dom));

    const before$4 = (marker, element) => {
        const parent$1 = parent(marker);
        parent$1.each((v) => {
            v.dom.insertBefore(element.dom, marker.dom);
        });
    };
    const after$4 = (marker, element) => {
        const sibling = nextSibling(marker);
        sibling.fold(() => {
            const parent$1 = parent(marker);
            parent$1.each((v) => {
                append$1(v, element);
            });
        }, (v) => {
            before$4(v, element);
        });
    };
    const prepend = (parent, element) => {
        const firstChild$1 = firstChild(parent);
        firstChild$1.fold(() => {
            append$1(parent, element);
        }, (v) => {
            parent.dom.insertBefore(element.dom, v.dom);
        });
    };
    const append$1 = (parent, element) => {
        parent.dom.appendChild(element.dom);
    };
    const wrap$2 = (element, wrapper) => {
        before$4(element, wrapper);
        append$1(wrapper, element);
    };

    const before$3 = (marker, elements) => {
        each$e(elements, (x) => {
            before$4(marker, x);
        });
    };
    const after$3 = (marker, elements) => {
        each$e(elements, (x, i) => {
            const e = i === 0 ? marker : elements[i - 1];
            after$4(e, x);
        });
    };
    const append = (parent, elements) => {
        each$e(elements, (x) => {
            append$1(parent, x);
        });
    };

    const rawSet = (dom, key, value) => {
        /*
         * JQuery coerced everything to a string, and silently did nothing on text node/null/undefined.
         *
         * We fail on those invalid cases, only allowing numbers and booleans.
         */
        if (isString(value) || isBoolean(value) || isNumber(value)) {
            dom.setAttribute(key, value + '');
        }
        else {
            // eslint-disable-next-line no-console
            console.error('Invalid call to Attribute.set. Key ', key, ':: Value ', value, ':: Element ', dom);
            throw new Error('Attribute value was not simple');
        }
    };
    const set$4 = (element, key, value) => {
        rawSet(element.dom, key, value);
    };
    const setAll$1 = (element, attrs) => {
        const dom = element.dom;
        each$d(attrs, (v, k) => {
            rawSet(dom, k, v);
        });
    };
    const get$9 = (element, key) => {
        const v = element.dom.getAttribute(key);
        // undefined is the more appropriate value for JS, and this matches JQuery
        return v === null ? undefined : v;
    };
    const getOpt = (element, key) => Optional.from(get$9(element, key));
    const has$1 = (element, key) => {
        const dom = element.dom;
        // return false for non-element nodes, no point in throwing an error
        return dom && dom.hasAttribute ? dom.hasAttribute(key) : false;
    };
    const remove$9 = (element, key) => {
        element.dom.removeAttribute(key);
    };
    const hasNone = (element) => {
        const attrs = element.dom.attributes;
        return attrs === undefined || attrs === null || attrs.length === 0;
    };
    const clone$4 = (element) => foldl(element.dom.attributes, (acc, attr) => {
        acc[attr.name] = attr.value;
        return acc;
    }, {});

    const empty = (element) => {
        // shortcut "empty node" trick. Requires IE 9.
        element.dom.textContent = '';
        // If the contents was a single empty text node, the above doesn't remove it. But, it's still faster in general
        // than removing every child node manually.
        // The following is (probably) safe for performance as 99.9% of the time the trick works and
        // Traverse.children will return an empty array.
        each$e(children$1(element), (rogue) => {
            remove$8(rogue);
        });
    };
    const remove$8 = (element) => {
        const dom = element.dom;
        if (dom.parentNode !== null) {
            dom.parentNode.removeChild(dom);
        }
    };
    const unwrap = (wrapper) => {
        const children = children$1(wrapper);
        if (children.length > 0) {
            after$3(wrapper, children);
        }
        remove$8(wrapper);
    };

    const clone$3 = (original, isDeep) => SugarElement.fromDom(original.dom.cloneNode(isDeep));
    /** Shallow clone - just the tag, no children */
    const shallow = (original) => clone$3(original, false);
    /** Deep clone - everything copied including children */
    const deep = (original) => clone$3(original, true);
    /** Shallow clone, with a new tag */
    const shallowAs = (original, tag) => {
        const nu = SugarElement.fromTag(tag);
        const attributes = clone$4(original);
        setAll$1(nu, attributes);
        return nu;
    };
    /** Change the tag name, but keep all children */
    const mutate = (original, tag) => {
        const nu = shallowAs(original, tag);
        after$4(original, nu);
        const children = children$1(original);
        append(nu, children);
        remove$8(original);
        return nu;
    };

    const fromHtml = (html, scope) => {
        const doc = scope || document;
        const div = doc.createElement('div');
        div.innerHTML = html;
        return children$1(SugarElement.fromDom(div));
    };
    const fromDom$1 = (nodes) => map$3(nodes, SugarElement.fromDom);

    const get$8 = (element) => element.dom.innerHTML;
    const set$3 = (element, content) => {
        const owner = owner$1(element);
        const docDom = owner.dom;
        // FireFox has *terrible* performance when using innerHTML = x
        const fragment = SugarElement.fromDom(docDom.createDocumentFragment());
        const contentElements = fromHtml(content, docDom);
        append(fragment, contentElements);
        empty(element);
        append$1(element, fragment);
    };
    const getOuter = (element) => {
        const container = SugarElement.fromTag('div');
        const clone = SugarElement.fromDom(element.dom.cloneNode(true));
        append$1(container, clone);
        return get$8(container);
    };

    // some elements, such as mathml, don't have style attributes
    // others, such as angular elements, have style attributes that aren't a CSSStyleDeclaration
    const isSupported = (dom) => dom.style !== undefined && isFunction(dom.style.getPropertyValue);

    // Node.contains() is very, very, very good performance
    // http://jsperf.com/closest-vs-contains/5
    const inBody = (element) => {
        // Technically this is only required on IE, where contains() returns false for text nodes.
        // But it's cheap enough to run everywhere and Sugar doesn't have platform detection (yet).
        const dom = isText$c(element) ? element.dom.parentNode : element.dom;
        // use ownerDocument.body to ensure this works inside iframes.
        // Normally contains is bad because an element "contains" itself, but here we want that.
        if (dom === undefined || dom === null || dom.ownerDocument === null) {
            return false;
        }
        const doc = dom.ownerDocument;
        return getShadowRoot(SugarElement.fromDom(dom)).fold(() => doc.body.contains(dom), compose1(inBody, getShadowHost));
    };

    const internalSet = (dom, property, value) => {
        // This is going to hurt. Apologies.
        // JQuery coerces numbers to pixels for certain property names, and other times lets numbers through.
        // we're going to be explicit; strings only.
        if (!isString(value)) {
            // eslint-disable-next-line no-console
            console.error('Invalid call to CSS.set. Property ', property, ':: Value ', value, ':: Element ', dom);
            throw new Error('CSS value must be a string: ' + value);
        }
        // removed: support for dom().style[property] where prop is camel case instead of normal property name
        if (isSupported(dom)) {
            dom.style.setProperty(property, value);
        }
    };
    const internalRemove = (dom, property) => {
        /*
         * IE9 and above - MDN doesn't have details, but here's a couple of random internet claims
         *
         * http://help.dottoro.com/ljopsjck.php
         * http://stackoverflow.com/a/7901886/7546
         */
        if (isSupported(dom)) {
            dom.style.removeProperty(property);
        }
    };
    const set$2 = (element, property, value) => {
        const dom = element.dom;
        internalSet(dom, property, value);
    };
    const setAll = (element, css) => {
        const dom = element.dom;
        each$d(css, (v, k) => {
            internalSet(dom, k, v);
        });
    };
    /*
     * NOTE: For certain properties, this returns the "used value" which is subtly different to the "computed value" (despite calling getComputedStyle).
     * Blame CSS 2.0.
     *
     * https://developer.mozilla.org/en-US/docs/Web/CSS/used_value
     */
    const get$7 = (element, property) => {
        const dom = element.dom;
        /*
         * IE9 and above per
         * https://developer.mozilla.org/en/docs/Web/API/window.getComputedStyle
         *
         * Not in numerosity, because it doesn't memoize and looking this up dynamically in performance critical code would be horrendous.
         *
         * JQuery has some magic here for IE popups, but we don't really need that.
         * It also uses element.ownerDocument.defaultView to handle iframes but that hasn't been required since FF 3.6.
         */
        const styles = window.getComputedStyle(dom);
        const r = styles.getPropertyValue(property);
        // jquery-ism: If r is an empty string, check that the element is not in a document. If it isn't, return the raw value.
        // Turns out we do this a lot.
        return (r === '' && !inBody(element)) ? getUnsafeProperty(dom, property) : r;
    };
    // removed: support for dom().style[property] where prop is camel case instead of normal property name
    // empty string is what the browsers (IE11 and Chrome) return when the propertyValue doesn't exists.
    const getUnsafeProperty = (dom, property) => isSupported(dom) ? dom.style.getPropertyValue(property) : '';
    /*
     * Gets the raw value from the style attribute. Useful for retrieving "used values" from the DOM:
     * https://developer.mozilla.org/en-US/docs/Web/CSS/used_value
     *
     * Returns NONE if the property isn't set, or the value is an empty string.
     */
    const getRaw$1 = (element, property) => {
        const dom = element.dom;
        const raw = getUnsafeProperty(dom, property);
        return Optional.from(raw).filter((r) => r.length > 0);
    };
    const getAllRaw = (element) => {
        const css = {};
        const dom = element.dom;
        if (isSupported(dom)) {
            for (let i = 0; i < dom.style.length; i++) {
                const ruleName = dom.style.item(i);
                css[ruleName] = dom.style[ruleName];
            }
        }
        return css;
    };
    const remove$7 = (element, property) => {
        const dom = element.dom;
        internalRemove(dom, property);
        if (is$4(getOpt(element, 'style').map(trim$4), '')) {
            // No more styles left, remove the style attribute as well
            remove$9(element, 'style');
        }
    };
    /* NOTE: This function is here for the side effect it triggers.
    The value itself is not used.
    Be sure to not use the return value, and that it is not removed by a minifier.
     */
    const reflow = (e) => e.dom.offsetWidth;

    const Dimension = (name, getOffset) => {
        const set = (element, h) => {
            if (!isNumber(h) && !h.match(/^[0-9]+$/)) {
                throw new Error(name + '.set accepts only positive integer values. Value was ' + h);
            }
            const dom = element.dom;
            if (isSupported(dom)) {
                dom.style[name] = h + 'px';
            }
        };
        /*
         * jQuery supports querying width and height on the document and window objects.
         *
         * TBIO doesn't do this, so the code is removed to save space, but left here just in case.
         */
        /*
        var getDocumentWidth = (element) => {
          var dom = element.dom;
          if (Node.isDocument(element)) {
            var body = dom.body;
            var doc = dom.documentElement;
            return Math.max(
              body.scrollHeight,
              doc.scrollHeight,
              body.offsetHeight,
              doc.offsetHeight,
              doc.clientHeight
            );
          }
        };
      
        var getWindowWidth = (element) => {
          var dom = element.dom;
          if (dom.window === dom) {
            // There is no offsetHeight on a window, so use the clientHeight of the document
            return dom.document.documentElement.clientHeight;
          }
        };
      */
        const get = (element) => {
            const r = getOffset(element);
            // zero or null means non-standard or disconnected, fall back to CSS
            if (r <= 0 || r === null) {
                const css = get$7(element, name);
                // ugh this feels dirty, but it saves cycles
                return parseFloat(css) || 0;
            }
            return r;
        };
        // in jQuery, getOuter replicates (or uses) box-sizing: border-box calculations
        // although these calculations only seem relevant for quirks mode, and edge cases TBIO doesn't rely on
        const getOuter = get;
        const aggregate = (element, properties) => foldl(properties, (acc, property) => {
            const val = get$7(element, property);
            const value = val === undefined ? 0 : parseInt(val, 10);
            return isNaN(value) ? acc : acc + value;
        }, 0);
        const max = (element, value, properties) => {
            const cumulativeInclusions = aggregate(element, properties);
            // if max-height is 100px and your cumulativeInclusions is 150px, there is no way max-height can be 100px, so we return 0.
            const absoluteMax = value > cumulativeInclusions ? value - cumulativeInclusions : 0;
            return absoluteMax;
        };
        return {
            set,
            get,
            getOuter,
            aggregate,
            max
        };
    };

    const api$1 = Dimension('height', (element) => {
        // getBoundingClientRect gives better results than offsetHeight for tables with captions on Firefox
        const dom = element.dom;
        return inBody(element) ? dom.getBoundingClientRect().height : dom.offsetHeight;
    });
    const get$6 = (element) => api$1.get(element);

    const r = (left, top) => {
        const translate = (x, y) => r(left + x, top + y);
        return {
            left,
            top,
            translate
        };
    };
    // tslint:disable-next-line:variable-name
    const SugarPosition = r;

    const boxPosition = (dom) => {
        const box = dom.getBoundingClientRect();
        return SugarPosition(box.left, box.top);
    };
    // Avoids falsy false fallthrough
    const firstDefinedOrZero = (a, b) => {
        if (a !== undefined) {
            return a;
        }
        else {
            return b !== undefined ? b : 0;
        }
    };
    const absolute = (element) => {
        const doc = element.dom.ownerDocument;
        const body = doc.body;
        const win = doc.defaultView;
        const html = doc.documentElement;
        if (body === element.dom) {
            return SugarPosition(body.offsetLeft, body.offsetTop);
        }
        const scrollTop = firstDefinedOrZero(win?.pageYOffset, html.scrollTop);
        const scrollLeft = firstDefinedOrZero(win?.pageXOffset, html.scrollLeft);
        const clientTop = firstDefinedOrZero(html.clientTop, body.clientTop);
        const clientLeft = firstDefinedOrZero(html.clientLeft, body.clientLeft);
        return viewport(element).translate(scrollLeft - clientLeft, scrollTop - clientTop);
    };
    const viewport = (element) => {
        const dom = element.dom;
        const doc = dom.ownerDocument;
        const body = doc.body;
        if (body === dom) {
            return SugarPosition(body.offsetLeft, body.offsetTop);
        }
        if (!inBody(element)) {
            return SugarPosition(0, 0);
        }
        return boxPosition(dom);
    };

    // get scroll position (x,y) relative to document _doc (or global if not supplied)
    const get$5 = (_DOC) => {
        const doc = _DOC !== undefined ? _DOC.dom : document;
        // ASSUMPTION: This is for cross-browser support, body works for Safari & EDGE, and when we have an iframe body scroller
        const x = doc.body.scrollLeft || doc.documentElement.scrollLeft;
        const y = doc.body.scrollTop || doc.documentElement.scrollTop;
        return SugarPosition(x, y);
    };
    // Scroll content to (x,y) relative to document _doc (or global if not supplied)
    const to = (x, y, _DOC) => {
        const doc = _DOC !== undefined ? _DOC.dom : document;
        const win = doc.defaultView;
        if (win) {
            win.scrollTo(x, y);
        }
    };
    // TBIO-4472 Safari 10 - Scrolling typeahead with keyboard scrolls page
    const intoView = (element, alignToTop) => {
        const isSafari = detect$1().browser.isSafari();
        // this method isn't in TypeScript
        if (isSafari && isFunction(element.dom.scrollIntoViewIfNeeded)) {
            element.dom.scrollIntoViewIfNeeded(false); // false=align to nearest edge
        }
        else {
            element.dom.scrollIntoView(alignToTop); // true=to top, false=to bottom
        }
    };

    const NodeValue = (is, name) => {
        const get = (element) => {
            if (!is(element)) {
                throw new Error('Can only get ' + name + ' value of a ' + name + ' node');
            }
            return getOption(element).getOr('');
        };
        const getOption = (element) => is(element) ? Optional.from(element.dom.nodeValue) : Optional.none();
        const set = (element, value) => {
            if (!is(element)) {
                throw new Error('Can only set raw ' + name + ' value of a ' + name + ' node');
            }
            element.dom.nodeValue = value;
        };
        return {
            get,
            getOption,
            set
        };
    };

    const fromElements = (elements, scope) => {
        const doc = scope || document;
        const fragment = doc.createDocumentFragment();
        each$e(elements, (element) => {
            fragment.appendChild(element.dom);
        });
        return SugarElement.fromDom(fragment);
    };

    const api = NodeValue(isText$c, 'text');
    const get$4 = (element) => api.get(element);
    const getOption = (element) => api.getOption(element);
    const set$1 = (element, value) => api.set(element, value);

    // Methods for handling attributes that contain a list of values <div foo="alpha beta theta">
    const read$4 = (element, attr) => {
        const value = get$9(element, attr);
        return value === undefined || value === '' ? [] : value.split(' ');
    };
    const add$4 = (element, attr, id) => {
        const old = read$4(element, attr);
        const nu = old.concat([id]);
        set$4(element, attr, nu.join(' '));
        return true;
    };
    const remove$6 = (element, attr, id) => {
        const nu = filter$5(read$4(element, attr), (v) => v !== id);
        if (nu.length > 0) {
            set$4(element, attr, nu.join(' '));
        }
        else {
            remove$9(element, attr);
        }
        return false;
    };

    var ClosestOrAncestor = (is, ancestor, scope, a, isRoot) => {
        if (is(scope, a)) {
            return Optional.some(scope);
        }
        else if (isFunction(isRoot) && isRoot(scope)) {
            return Optional.none();
        }
        else {
            return ancestor(scope, a, isRoot);
        }
    };

    const ancestor$5 = (scope, predicate, isRoot) => {
        let element = scope.dom;
        const stop = isFunction(isRoot) ? isRoot : never;
        while (element.parentNode) {
            element = element.parentNode;
            const el = SugarElement.fromDom(element);
            if (predicate(el)) {
                return Optional.some(el);
            }
            else if (stop(el)) {
                break;
            }
        }
        return Optional.none();
    };
    const closest$5 = (scope, predicate, isRoot) => {
        // This is required to avoid ClosestOrAncestor passing the predicate to itself
        const is = (s, test) => test(s);
        return ClosestOrAncestor(is, ancestor$5, scope, predicate, isRoot);
    };
    const sibling$1 = (scope, predicate) => {
        const element = scope.dom;
        if (!element.parentNode) {
            return Optional.none();
        }
        return child(SugarElement.fromDom(element.parentNode), (x) => !eq(scope, x) && predicate(x));
    };
    const child = (scope, predicate) => {
        const pred = (node) => predicate(SugarElement.fromDom(node));
        const result = find$2(scope.dom.childNodes, pred);
        return result.map(SugarElement.fromDom);
    };
    const descendant$2 = (scope, predicate) => {
        const descend = (node) => {
            // tslint:disable-next-line:prefer-for-of
            for (let i = 0; i < node.childNodes.length; i++) {
                const child = SugarElement.fromDom(node.childNodes[i]);
                if (predicate(child)) {
                    return Optional.some(child);
                }
                const res = descend(node.childNodes[i]);
                if (res.isSome()) {
                    return res;
                }
            }
            return Optional.none();
        };
        return descend(scope.dom);
    };

    const ancestor$4 = (scope, selector, isRoot) => ancestor$5(scope, (e) => is$2(e, selector), isRoot);
    const descendant$1 = (scope, selector) => one(selector, scope);
    // Returns Some(closest ancestor element (sugared)) matching 'selector' up to isRoot, or None() otherwise
    const closest$4 = (scope, selector, isRoot) => {
        const is = (element, selector) => is$2(element, selector);
        return ClosestOrAncestor(is, ancestor$4, scope, selector, isRoot);
    };

    // IE11 Can return undefined for a classList on elements such as math, so we make sure it's not undefined before attempting to use it.
    const supports = (element) => element.dom.classList !== undefined;
    const get$3 = (element) => read$4(element, 'class');
    const add$3 = (element, clazz) => add$4(element, 'class', clazz);
    const remove$5 = (element, clazz) => remove$6(element, 'class', clazz);
    const toggle$2 = (element, clazz) => {
        if (contains$2(get$3(element), clazz)) {
            return remove$5(element, clazz);
        }
        else {
            return add$3(element, clazz);
        }
    };

    /*
     * ClassList is IE10 minimum:
     * https://developer.mozilla.org/en-US/docs/Web/API/Element.classList
     *
     * Note that IE doesn't support the second argument to toggle (at all).
     * If it did, the toggler could be better.
     */
    const add$2 = (element, clazz) => {
        if (supports(element)) {
            element.dom.classList.add(clazz);
        }
        else {
            add$3(element, clazz);
        }
    };
    const cleanClass = (element) => {
        const classList = supports(element) ? element.dom.classList : get$3(element);
        // classList is a "live list", so this is up to date already
        if (classList.length === 0) {
            // No more classes left, remove the class attribute as well
            remove$9(element, 'class');
        }
    };
    const remove$4 = (element, clazz) => {
        if (supports(element)) {
            const classList = element.dom.classList;
            classList.remove(clazz);
        }
        else {
            remove$5(element, clazz);
        }
        cleanClass(element);
    };
    const toggle$1 = (element, clazz) => {
        const result = supports(element) ? element.dom.classList.toggle(clazz) : toggle$2(element, clazz);
        cleanClass(element);
        return result;
    };
    const has = (element, clazz) => supports(element) && element.dom.classList.contains(clazz);

    /*
     * ClassList is IE10 minimum:
     * https://developer.mozilla.org/en-US/docs/Web/API/Element.classList
     */
    const add$1 = (element, classes) => {
        each$e(classes, (x) => {
            add$2(element, x);
        });
    };
    const remove$3 = (element, classes) => {
        each$e(classes, (x) => {
            remove$4(element, x);
        });
    };

    const closest$3 = (target) => closest$4(target, '[contenteditable]');
    const isEditable$2 = (element, assumeEditable = false) => {
        if (inBody(element)) {
            return element.dom.isContentEditable;
        }
        else {
            // Find the closest contenteditable element and check if it's editable
            return closest$3(element).fold(constant(assumeEditable), (editable) => getRaw(editable) === 'true');
        }
    };
    const getRaw = (element) => element.dom.contentEditable;
    const set = (element, editable) => {
        element.dom.contentEditable = editable ? 'true' : 'false';
    };

    const ancestors$1 = (scope, predicate, isRoot) => filter$5(parents$1(scope, isRoot), predicate);
    const children = (scope, predicate) => filter$5(children$1(scope), predicate);
    const descendants$1 = (scope, predicate) => {
        let result = [];
        // Recurse.toArray() might help here
        each$e(children$1(scope), (x) => {
            if (predicate(x)) {
                result = result.concat([x]);
            }
            result = result.concat(descendants$1(x, predicate));
        });
        return result;
    };

    // For all of the following:
    //
    // jQuery does siblings of firstChild. IE9+ supports scope.dom.children (similar to Traverse.children but elements only).
    // Traverse should also do this (but probably not by default).
    //
    const ancestors = (scope, selector, isRoot) => 
    // It may surprise you to learn this is exactly what JQuery does
    // TODO: Avoid all this wrapping and unwrapping
    ancestors$1(scope, (e) => is$2(e, selector), isRoot);
    const descendants = (scope, selector) => all(selector, scope);

    const ancestor$3 = (scope, predicate, isRoot) => ancestor$5(scope, predicate, isRoot).isSome();
    const sibling = (scope, predicate) => sibling$1(scope, predicate).isSome();
    const descendant = (scope, predicate) => descendant$2(scope, predicate).isSome();

    const ancestor$2 = (element, target) => ancestor$3(element, curry(eq, target));

    const ancestor$1 = (scope, selector, isRoot) => ancestor$4(scope, selector, isRoot).isSome();
    const closest$2 = (scope, selector, isRoot) => closest$4(scope, selector, isRoot).isSome();

    const ensureIsRoot = (isRoot) => isFunction(isRoot) ? isRoot : never;
    const ancestor = (scope, transform, isRoot) => {
        let element = scope.dom;
        const stop = ensureIsRoot(isRoot);
        while (element.parentNode) {
            element = element.parentNode;
            const el = SugarElement.fromDom(element);
            const transformed = transform(el);
            if (transformed.isSome()) {
                return transformed;
            }
            else if (stop(el)) {
                break;
            }
        }
        return Optional.none();
    };
    const closest$1 = (scope, transform, isRoot) => {
        const current = transform(scope);
        const stop = ensureIsRoot(isRoot);
        return current.orThunk(() => stop(scope) ? Optional.none() : ancestor(scope, transform, stop));
    };

    const isTextNodeWithCursorPosition = (el) => getOption(el).filter((text) => 
    // For the purposes of finding cursor positions only allow text nodes with content,
    // but trim removes &nbsp; and that's allowed
    text.trim().length !== 0 || text.indexOf(nbsp) > -1).isSome();
    const isContentEditableFalse$b = (elem) => isHTMLElement$1(elem) && (get$9(elem, 'contenteditable') === 'false');
    const elementsWithCursorPosition = ['img', 'br'];
    const isCursorPosition = (elem) => {
        const hasCursorPosition = isTextNodeWithCursorPosition(elem);
        return hasCursorPosition || contains$2(elementsWithCursorPosition, name(elem)) || isContentEditableFalse$b(elem);
    };

    const first = (element) => descendant$2(element, isCursorPosition);

    const create$c = (start, soffset, finish, foffset) => ({
        start,
        soffset,
        finish,
        foffset
    });
    // tslint:disable-next-line:variable-name
    const SimRange = {
        create: create$c
    };

    const adt$2 = Adt.generate([
        { before: ['element'] },
        { on: ['element', 'offset'] },
        { after: ['element'] }
    ]);
    // Probably don't need this given that we now have "match"
    const cata = (subject, onBefore, onOn, onAfter) => subject.fold(onBefore, onOn, onAfter);
    const getStart$2 = (situ) => situ.fold(identity, identity, identity);
    const before$2 = adt$2.before;
    const on = adt$2.on;
    const after$2 = adt$2.after;
    // tslint:disable-next-line:variable-name
    const Situ = {
        before: before$2,
        on,
        after: after$2,
        cata,
        getStart: getStart$2
    };

    // Consider adding a type for "element"
    const adt$1 = Adt.generate([
        { domRange: ['rng'] },
        { relative: ['startSitu', 'finishSitu'] },
        { exact: ['start', 'soffset', 'finish', 'foffset'] }
    ]);
    const exactFromRange = (simRange) => adt$1.exact(simRange.start, simRange.soffset, simRange.finish, simRange.foffset);
    const getStart$1 = (selection) => selection.match({
        domRange: (rng) => SugarElement.fromDom(rng.startContainer),
        relative: (startSitu, _finishSitu) => Situ.getStart(startSitu),
        exact: (start, _soffset, _finish, _foffset) => start
    });
    const domRange = adt$1.domRange;
    const relative = adt$1.relative;
    const exact = adt$1.exact;
    const getWin = (selection) => {
        const start = getStart$1(selection);
        return defaultView(start);
    };
    // This is out of place but it's API so I can't remove it
    const range = SimRange.create;
    // tslint:disable-next-line:variable-name
    const SimSelection = {
        domRange,
        relative,
        exact,
        exactFromRange,
        getWin,
        range
    };

    const caretPositionFromPoint = (doc, x, y) => Optional.from(doc.caretPositionFromPoint?.(x, y))
        .bind((pos) => {
        // It turns out that Firefox can return null for pos.offsetNode
        if (pos.offsetNode === null) {
            return Optional.none();
        }
        const r = doc.createRange();
        r.setStart(pos.offsetNode, pos.offset);
        r.collapse();
        return Optional.some(r);
    });
    const caretRangeFromPoint = (doc, x, y) => Optional.from(doc.caretRangeFromPoint?.(x, y));
    const availableSearch = (doc, x, y) => {
        if (doc.caretPositionFromPoint) {
            return caretPositionFromPoint(doc, x, y); // defined standard, firefox only
        }
        else if (doc.caretRangeFromPoint) {
            return caretRangeFromPoint(doc, x, y); // webkit/blink implementation
        }
        else {
            return Optional.none(); // unsupported browser
        }
    };
    const fromPoint$1 = (win, x, y) => {
        const doc = win.document;
        return availableSearch(doc, x, y).map((rng) => SimRange.create(SugarElement.fromDom(rng.startContainer), rng.startOffset, SugarElement.fromDom(rng.endContainer), rng.endOffset));
    };

    const beforeSpecial = (element, offset) => {
        // From memory, we don't want to use <br> directly on Firefox because it locks the keyboard input.
        // It turns out that <img> directly on IE locks the keyboard as well.
        // If the offset is 0, use before. If the offset is 1, use after.
        // TBIO-3889: Firefox Situ.on <input> results in a child of the <input>; Situ.before <input> results in platform inconsistencies
        const name$1 = name(element);
        if ('input' === name$1) {
            return Situ.after(element);
        }
        else if (!contains$2(['br', 'img'], name$1)) {
            return Situ.on(element, offset);
        }
        else {
            return offset === 0 ? Situ.before(element) : Situ.after(element);
        }
    };
    const preprocessRelative = (startSitu, finishSitu) => {
        const start = startSitu.fold(Situ.before, beforeSpecial, Situ.after);
        const finish = finishSitu.fold(Situ.before, beforeSpecial, Situ.after);
        return SimSelection.relative(start, finish);
    };
    const preprocessExact = (start, soffset, finish, foffset) => {
        const startSitu = beforeSpecial(start, soffset);
        const finishSitu = beforeSpecial(finish, foffset);
        return SimSelection.relative(startSitu, finishSitu);
    };
    const preprocess = (selection) => selection.match({
        domRange: (rng) => {
            const start = SugarElement.fromDom(rng.startContainer);
            const finish = SugarElement.fromDom(rng.endContainer);
            return preprocessExact(start, rng.startOffset, finish, rng.endOffset);
        },
        relative: preprocessRelative,
        exact: preprocessExact
    });

    const toNative = (selection) => {
        const win = SimSelection.getWin(selection).dom;
        const getDomRange = (start, soffset, finish, foffset) => exactToNative(win, start, soffset, finish, foffset);
        const filtered = preprocess(selection);
        return diagnose(win, filtered).match({
            ltr: getDomRange,
            rtl: getDomRange
        });
    };
    const getAtPoint = (win, x, y) => fromPoint$1(win, x, y);

    const get$2 = (_win) => {
        const win = _win === undefined ? window : _win;
        if (detect$1().browser.isFirefox()) {
            // TINY-7984: Firefox 91 is returning incorrect values for visualViewport.pageTop, so disable it for now
            return Optional.none();
        }
        else {
            return Optional.from(win.visualViewport);
        }
    };
    const bounds = (x, y, width, height) => ({
        x,
        y,
        width,
        height,
        right: x + width,
        bottom: y + height
    });
    const getBounds = (_win) => {
        const win = _win === undefined ? window : _win;
        const doc = win.document;
        const scroll = get$5(SugarElement.fromDom(doc));
        return get$2(win).fold(() => {
            const html = win.document.documentElement;
            // Don't use window.innerWidth/innerHeight here, as we don't want to include scrollbars
            // since the right/bottom position is based on the edge of the scrollbar not the window
            const width = html.clientWidth;
            const height = html.clientHeight;
            return bounds(scroll.left, scroll.top, width, height);
        }, (visualViewport) => 
        // iOS doesn't update the pageTop/pageLeft when element.scrollIntoView() is called, so we need to fallback to the
        // scroll position which will always be less than the page top/left values when page top/left are accurate/correct.
        bounds(Math.max(visualViewport.pageLeft, scroll.left), Math.max(visualViewport.pageTop, scroll.top), visualViewport.width, visualViewport.height));
    };

    /**
     * TreeWalker class enables you to walk the DOM in a linear manner.
     *
     * @class tinymce.dom.TreeWalker
     * @example
     * const walker = new tinymce.dom.TreeWalker(startNode);
     *
     * do {
     *   console.log(walker.current());
     * } while (walker.next());
     */
    class DomTreeWalker {
        rootNode;
        node;
        constructor(startNode, rootNode) {
            this.node = startNode;
            this.rootNode = rootNode;
            // This is a bit hacky but needed to ensure the 'this' variable
            // always references the instance and not the caller scope
            this.current = this.current.bind(this);
            this.next = this.next.bind(this);
            this.prev = this.prev.bind(this);
            this.prev2 = this.prev2.bind(this);
        }
        /**
         * Returns the current node.
         *
         * @method current
         * @return {Node/undefined} Current node where the walker is, or undefined if the walker has reached the end.
         */
        current() {
            return this.node;
        }
        /**
         * Walks to the next node in tree.
         *
         * @method next
         * @return {Node/undefined} Current node where the walker is after moving to the next node, or undefined if the walker has reached the end.
         */
        next(shallow) {
            this.node = this.findSibling(this.node, 'firstChild', 'nextSibling', shallow);
            return this.node;
        }
        /**
         * Walks to the previous node in tree.
         *
         * @method prev
         * @return {Node/undefined} Current node where the walker is after moving to the previous node, or undefined if the walker has reached the end.
         */
        prev(shallow) {
            this.node = this.findSibling(this.node, 'lastChild', 'previousSibling', shallow);
            return this.node;
        }
        prev2(shallow) {
            this.node = this.findPreviousNode(this.node, shallow);
            return this.node;
        }
        findSibling(node, startName, siblingName, shallow) {
            if (node) {
                // Walk into nodes if it has a start
                if (!shallow && node[startName]) {
                    return node[startName];
                }
                // Return the sibling if it has one
                if (node !== this.rootNode) {
                    let sibling = node[siblingName];
                    if (sibling) {
                        return sibling;
                    }
                    // Walk up the parents to look for siblings
                    for (let parent = node.parentNode; parent && parent !== this.rootNode; parent = parent.parentNode) {
                        sibling = parent[siblingName];
                        if (sibling) {
                            return sibling;
                        }
                    }
                }
            }
            return undefined;
        }
        findPreviousNode(node, shallow) {
            if (node) {
                const sibling = node.previousSibling;
                if (this.rootNode && sibling === this.rootNode) {
                    return;
                }
                if (sibling) {
                    if (!shallow) {
                        // Walk down to the most distant child
                        for (let child = sibling.lastChild; child; child = child.lastChild) {
                            if (!child.lastChild) {
                                return child;
                            }
                        }
                    }
                    return sibling;
                }
                const parent = node.parentNode;
                if (parent && parent !== this.rootNode) {
                    return parent;
                }
            }
            return undefined;
        }
    }

    const whiteSpaceRegExp = /^[ \t\r\n]*$/;
    const isWhitespaceText = (text) => whiteSpaceRegExp.test(text);
    const isZwsp$1 = (text) => {
        for (const c of text) {
            if (!isZwsp$2(c)) {
                return false;
            }
        }
        return true;
    };
    // Don't compare other unicode spaces here, as we're only concerned about whitespace the browser would collapse
    const isCollapsibleWhitespace$1 = (c) => ' \f\t\v'.indexOf(c) !== -1;
    const isNewLineChar = (c) => c === '\n' || c === '\r';
    const isNewline = (text, idx) => (idx < text.length && idx >= 0) ? isNewLineChar(text[idx]) : false;
    // Converts duplicate whitespace to alternating space/nbsps and tabs to spaces
    const normalize$4 = (text, tabSpaces = 4, isStartOfContent = true, isEndOfContent = true) => {
        // Replace tabs with a variable amount of spaces
        // Note: We don't use an actual tab character here, as it only works when in a "whitespace: pre" element,
        // which will cause other issues, such as trying to type the content will also be treated as being in a pre.
        const tabSpace = repeat(' ', tabSpaces);
        const normalizedText = text.replace(/\t/g, tabSpace);
        const result = foldl(normalizedText, (acc, c) => {
            // Are we dealing with a char other than some collapsible whitespace or nbsp? if so then just use it as is
            if (isCollapsibleWhitespace$1(c) || c === nbsp) {
                // If the previous char is a space, we are at the start or end, or if the next char is a new line char, then we need
                // to convert the space to a nbsp
                if (acc.pcIsSpace || (acc.str === '' && isStartOfContent) || (acc.str.length === normalizedText.length - 1 && isEndOfContent) || isNewline(normalizedText, acc.str.length + 1)) {
                    return { pcIsSpace: false, str: acc.str + nbsp };
                }
                else {
                    return { pcIsSpace: true, str: acc.str + ' ' };
                }
            }
            else {
                // Treat newlines as being a space, since we'll need to convert any leading spaces to nsbps
                return { pcIsSpace: isNewLineChar(c), str: acc.str + c };
            }
        }, { pcIsSpace: false, str: '' });
        return result.str;
    };

    const isNodeType = (type) => {
        return (node) => {
            return !!node && node.nodeType === type;
        };
    };
    // Firefox can allow you to get a selection on a restricted node, such as file/number inputs. These nodes
    // won't implement the Object prototype, so Object.getPrototypeOf() will return null or something similar.
    const isRestrictedNode = (node) => !!node && !Object.getPrototypeOf(node);
    const isElement$7 = isNodeType(1);
    const isHTMLElement = (node) => isElement$7(node) && isHTMLElement$1(SugarElement.fromDom(node));
    const isSVGElement = (node) => isElement$7(node) && node.namespaceURI === 'http://www.w3.org/2000/svg';
    const matchNodeName$1 = (name) => {
        const lowerCasedName = name.toLowerCase();
        return (node) => isNonNullable(node) && node.nodeName.toLowerCase() === lowerCasedName;
    };
    const matchNodeNames$1 = (names) => {
        const lowerCasedNames = names.map((s) => s.toLowerCase());
        return (node) => {
            if (node && node.nodeName) {
                const nodeName = node.nodeName.toLowerCase();
                return contains$2(lowerCasedNames, nodeName);
            }
            return false;
        };
    };
    const matchStyleValues = (name, values) => {
        const items = values.toLowerCase().split(' ');
        return (node) => {
            if (isElement$7(node)) {
                const win = node.ownerDocument.defaultView;
                if (win) {
                    for (let i = 0; i < items.length; i++) {
                        const computed = win.getComputedStyle(node, null);
                        const cssValue = computed ? computed.getPropertyValue(name) : null;
                        if (cssValue === items[i]) {
                            return true;
                        }
                    }
                }
            }
            return false;
        };
    };
    const hasAttribute = (attrName) => {
        return (node) => {
            return isElement$7(node) && node.hasAttribute(attrName);
        };
    };
    const isBogus$1 = (node) => isElement$7(node) && node.hasAttribute('data-mce-bogus');
    const isBogusAll = (node) => isElement$7(node) && node.getAttribute('data-mce-bogus') === 'all';
    const isTable$2 = (node) => isElement$7(node) && node.tagName === 'TABLE';
    const hasContentEditableState = (value) => {
        return (node) => {
            if (isHTMLElement(node)) {
                if (node.contentEditable === value) {
                    return true;
                }
                if (node.getAttribute('data-mce-contenteditable') === value) {
                    return true;
                }
            }
            return false;
        };
    };
    const isTextareaOrInput = matchNodeNames$1(['textarea', 'input']);
    const isText$b = isNodeType(3);
    const isCData = isNodeType(4);
    const isPi = isNodeType(7);
    const isComment = isNodeType(8);
    const isDocument$1 = isNodeType(9);
    const isDocumentFragment = isNodeType(11);
    const isBr$7 = matchNodeName$1('br');
    const isImg = matchNodeName$1('img');
    const isAnchor = matchNodeName$1('a');
    const isContentEditableTrue$3 = hasContentEditableState('true');
    const isContentEditableFalse$a = hasContentEditableState('false');
    const isEditingHost = (node) => isHTMLElement(node) && node.isContentEditable && isNonNullable(node.parentElement) && !node.parentElement.isContentEditable;
    const isTableCell$3 = matchNodeNames$1(['td', 'th']);
    const isTableCellOrCaption = matchNodeNames$1(['td', 'th', 'caption']);
    const isTemplate = matchNodeName$1('template');
    const isMedia$2 = matchNodeNames$1(['video', 'audio', 'object', 'embed']);
    const isListItem$3 = matchNodeName$1('li');
    const isDetails = matchNodeName$1('details');
    const isSummary$1 = matchNodeName$1('summary');
    const ucVideoNodeName = 'uc-video';
    const isUcVideo = (el) => el.nodeName.toLowerCase() === ucVideoNodeName;

    const defaultOptionValues = {
        skipBogus: true,
        includeZwsp: false,
        checkRootAsContent: false,
    };
    const hasWhitespacePreserveParent = (node, rootNode, schema) => {
        const rootElement = SugarElement.fromDom(rootNode);
        const startNode = SugarElement.fromDom(node);
        const whitespaceElements = schema.getWhitespaceElements();
        const predicate = (node) => has$2(whitespaceElements, name(node));
        return ancestor$3(startNode, predicate, curry(eq, rootElement));
    };
    const isNamedAnchor = (node) => {
        return isElement$7(node) && node.nodeName === 'A' && !node.hasAttribute('href') && (node.hasAttribute('name') || node.hasAttribute('id'));
    };
    const isNonEmptyElement$1 = (node, schema) => {
        return isElement$7(node) && has$2(schema.getNonEmptyElements(), node.nodeName);
    };
    const isBookmark = hasAttribute('data-mce-bookmark');
    const hasNonEditableParent = (node) => parentElement(SugarElement.fromDom(node)).exists((parent) => !isEditable$2(parent));
    const isWhitespace$1 = (node, rootNode, schema) => isWhitespaceText(node.data)
        && !hasWhitespacePreserveParent(node, rootNode, schema);
    const isText$a = (node, rootNode, schema, options) => isText$b(node)
        && !isWhitespace$1(node, rootNode, schema)
        && (!options.includeZwsp || !isZwsp$1(node.data));
    const isContentNode = (schema, node, rootNode, options) => {
        return isFunction(options.isContent) && options.isContent(node)
            || isNonEmptyElement$1(node, schema)
            || isBookmark(node)
            || isNamedAnchor(node)
            || isText$a(node, rootNode, schema, options)
            || isContentEditableFalse$a(node)
            || isContentEditableTrue$3(node) && hasNonEditableParent(node);
    };
    const isEmptyNode = (schema, targetNode, opts) => {
        const options = { ...defaultOptionValues, ...opts };
        if (options.checkRootAsContent) {
            if (isContentNode(schema, targetNode, targetNode, options)) {
                return false;
            }
        }
        let node = targetNode.firstChild;
        let brCount = 0;
        if (!node) {
            return true;
        }
        const walker = new DomTreeWalker(node, targetNode);
        do {
            if (options.skipBogus && isElement$7(node)) {
                const bogusValue = node.getAttribute('data-mce-bogus');
                if (bogusValue) {
                    node = walker.next(bogusValue === 'all');
                    continue;
                }
            }
            if (isComment(node)) {
                node = walker.next(true);
                continue;
            }
            if (isBr$7(node)) {
                brCount++;
                node = walker.next();
                continue;
            }
            if (isContentNode(schema, node, targetNode, options)) {
                return false;
            }
            node = walker.next();
        } while (node);
        return brCount <= 1;
    };
    const isEmpty$4 = (schema, elm, options) => {
        return isEmptyNode(schema, elm.dom, { checkRootAsContent: true, ...options });
    };
    const isContent$1 = (schema, node, options) => {
        return isContentNode(schema, node, node, { includeZwsp: defaultOptionValues.includeZwsp, ...options });
    };

    const nodeNameToNamespaceType = (name) => {
        const lowerCaseName = name.toLowerCase();
        if (lowerCaseName === 'svg') {
            return 'svg';
        }
        else if (lowerCaseName === 'math') {
            return 'math';
        }
        else {
            return 'html';
        }
    };
    const isNonHtmlElementRootName = (name) => nodeNameToNamespaceType(name) !== 'html';
    const isNonHtmlElementRoot = (node) => isNonHtmlElementRootName(node.nodeName);
    const toScopeType = (node) => nodeNameToNamespaceType(node.nodeName);
    const namespaceElements = ['svg', 'math'];
    const createNamespaceTracker = () => {
        const currentScope = value$1();
        const current = () => currentScope.get().map(toScopeType).getOr('html');
        const track = (node) => {
            if (isNonHtmlElementRoot(node)) {
                currentScope.set(node);
            }
            else if (currentScope.get().exists((scopeNode) => !scopeNode.contains(node))) {
                currentScope.clear();
            }
            return current();
        };
        const reset = () => {
            currentScope.clear();
        };
        return {
            track,
            current,
            reset
        };
    };

    const transparentBlockAttr = 'data-mce-block';
    // Returns the lowercase element names form a SchemaMap by excluding anyone that has uppercase letters.
    // This method is to avoid having to specify all possible valid characters other than lowercase a-z such as '-' or ':' etc.
    const elementNames = (map) => filter$5(keys(map), (key) => !/[A-Z]/.test(key));
    const makeSelectorFromSchemaMap = (map) => map$3(elementNames(map), (name) => {
        // Exclude namespace elements from processing
        const escapedName = CSS.escape(name);
        return `${escapedName}:` + map$3(namespaceElements, (ns) => `not(${ns} ${escapedName})`).join(':');
    }).join(',');
    const updateTransparent = (blocksSelector, transparent) => {
        if (isNonNullable(transparent.querySelector(blocksSelector))) {
            transparent.setAttribute(transparentBlockAttr, 'true');
            if (transparent.getAttribute('data-mce-selected') === 'inline-boundary') {
                transparent.removeAttribute('data-mce-selected');
            }
            return true;
        }
        else {
            transparent.removeAttribute(transparentBlockAttr);
            return false;
        }
    };
    const updateBlockStateOnChildren = (schema, scope) => {
        const transparentSelector = makeSelectorFromSchemaMap(schema.getTransparentElements());
        const blocksSelector = makeSelectorFromSchemaMap(schema.getBlockElements());
        return filter$5(scope.querySelectorAll(transparentSelector), (transparent) => updateTransparent(blocksSelector, transparent));
    };
    const trimEdge = (schema, el, leftSide) => {
        const childPropertyName = leftSide ? 'lastChild' : 'firstChild';
        for (let child = el[childPropertyName]; child; child = child[childPropertyName]) {
            if (isEmptyNode(schema, child, { checkRootAsContent: true })) {
                child.parentNode?.removeChild(child);
                return;
            }
        }
    };
    const split$2 = (schema, parentElm, splitElm) => {
        const range = document.createRange();
        const parentNode = parentElm.parentNode;
        if (parentNode) {
            range.setStartBefore(parentElm);
            range.setEndBefore(splitElm);
            const beforeFragment = range.extractContents();
            trimEdge(schema, beforeFragment, true);
            range.setStartAfter(splitElm);
            range.setEndAfter(parentElm);
            const afterFragment = range.extractContents();
            trimEdge(schema, afterFragment, false);
            if (!isEmptyNode(schema, beforeFragment, { checkRootAsContent: true })) {
                parentNode.insertBefore(beforeFragment, parentElm);
            }
            if (!isEmptyNode(schema, splitElm, { checkRootAsContent: true })) {
                parentNode.insertBefore(splitElm, parentElm);
            }
            if (!isEmptyNode(schema, afterFragment, { checkRootAsContent: true })) {
                parentNode.insertBefore(afterFragment, parentElm);
            }
            parentNode.removeChild(parentElm);
        }
    };
    // This will find invalid blocks wrapped in anchors and split them out so for example
    // <h1><a href="#"><h2>x</h2></a></h1> will find that h2 is invalid inside the H1 and split that out.
    // This is a simplistic apporach so it's likely not covering all the cases but it's a start.
    const splitInvalidChildren = (schema, scope, transparentBlocks) => {
        const blocksElements = schema.getBlockElements();
        const rootNode = SugarElement.fromDom(scope);
        const isBlock = (el) => name(el) in blocksElements;
        const isRoot = (el) => eq(el, rootNode);
        each$e(fromDom$1(transparentBlocks), (transparentBlock) => {
            ancestor$5(transparentBlock, isBlock, isRoot).each((parentBlock) => {
                const invalidChildren = children(transparentBlock, (el) => isBlock(el) && !schema.isValidChild(name(parentBlock), name(el)));
                if (invalidChildren.length > 0) {
                    const stateScope = parentElement(parentBlock);
                    each$e(invalidChildren, (child) => {
                        ancestor$5(child, isBlock, isRoot).each((parentBlock) => {
                            split$2(schema, parentBlock.dom, child.dom);
                        });
                    });
                    stateScope.each((scope) => updateBlockStateOnChildren(schema, scope.dom));
                }
            });
        });
    };
    const unwrapInvalidChildren = (schema, scope, transparentBlocks) => {
        each$e([...transparentBlocks, ...(isTransparentBlock(schema, scope) ? [scope] : [])], (block) => each$e(descendants(SugarElement.fromDom(block), block.nodeName.toLowerCase()), (elm) => {
            if (isTransparentInline(schema, elm.dom)) {
                unwrap(elm);
            }
        }));
    };
    const updateChildren = (schema, scope) => {
        const transparentBlocks = updateBlockStateOnChildren(schema, scope);
        splitInvalidChildren(schema, scope, transparentBlocks);
        unwrapInvalidChildren(schema, scope, transparentBlocks);
    };
    const updateElement = (schema, target) => {
        if (isTransparentElement(schema, target)) {
            const blocksSelector = makeSelectorFromSchemaMap(schema.getBlockElements());
            updateTransparent(blocksSelector, target);
        }
    };
    const updateCaret = (schema, root, caretParent) => {
        const isRoot = (el) => eq(el, SugarElement.fromDom(root));
        const parents = parents$1(SugarElement.fromDom(caretParent), isRoot);
        // Check the element just above below the root so in if caretParent is I in this
        // case <body><p><b><i>|</i></b></p></body> it would use the P as the scope
        get$b(parents, parents.length - 2).filter(isElement$8).fold(() => updateChildren(schema, root), (scope) => updateChildren(schema, scope.dom));
    };
    const hasBlockAttr = (el) => el.hasAttribute(transparentBlockAttr);
    const isTransparentElementName = (schema, name) => has$2(schema.getTransparentElements(), name);
    const isTransparentElement = (schema, node) => isElement$7(node) && isTransparentElementName(schema, node.nodeName);
    const isTransparentBlock = (schema, node) => isTransparentElement(schema, node) && hasBlockAttr(node);
    const isTransparentInline = (schema, node) => isTransparentElement(schema, node) && !hasBlockAttr(node);
    const isTransparentAstBlock = (schema, node) => node.type === 1 && isTransparentElementName(schema, node.name) && isString(node.attr(transparentBlockAttr));

    const browser$2 = detect$1().browser;
    const firstElement = (nodes) => find$2(nodes, isElement$8);
    // Firefox has a bug where caption height is not included correctly in offset calculations of tables
    // this tries to compensate for that by detecting if that offsets are incorrect and then remove the height
    const getTableCaptionDeltaY = (elm) => {
        if (browser$2.isFirefox() && name(elm) === 'table') {
            return firstElement(children$1(elm)).filter((elm) => {
                return name(elm) === 'caption';
            }).bind((caption) => {
                return firstElement(nextSiblings(caption)).map((body) => {
                    const bodyTop = body.dom.offsetTop;
                    const captionTop = caption.dom.offsetTop;
                    const captionHeight = caption.dom.offsetHeight;
                    return bodyTop <= captionTop ? -captionHeight : 0;
                });
            }).getOr(0);
        }
        else {
            return 0;
        }
    };
    const hasChild = (elm, child) => elm.children && contains$2(elm.children, child);
    const getPos = (body, elm, rootElm) => {
        let x = 0, y = 0;
        const doc = body.ownerDocument;
        rootElm = rootElm ? rootElm : body;
        if (elm) {
            // Use getBoundingClientRect if it exists since it's faster than looping offset nodes
            // Fallback to offsetParent calculations if the body isn't static better since it stops at the body root
            if (rootElm === body && elm.getBoundingClientRect && get$7(SugarElement.fromDom(body), 'position') === 'static') {
                const pos = elm.getBoundingClientRect();
                // Add scroll offsets from documentElement or body since IE with the wrong box model will use d.body and so do WebKit
                // Also remove the body/documentelement clientTop/clientLeft on IE 6, 7 since they offset the position
                x = pos.left + (doc.documentElement.scrollLeft || body.scrollLeft) - doc.documentElement.clientLeft;
                y = pos.top + (doc.documentElement.scrollTop || body.scrollTop) - doc.documentElement.clientTop;
                return { x, y };
            }
            let offsetParent = elm;
            while (offsetParent && offsetParent !== rootElm && offsetParent.nodeType && !hasChild(offsetParent, rootElm)) {
                const castOffsetParent = offsetParent;
                x += castOffsetParent.offsetLeft || 0;
                y += castOffsetParent.offsetTop || 0;
                offsetParent = castOffsetParent.offsetParent;
            }
            offsetParent = elm.parentNode;
            while (offsetParent && offsetParent !== rootElm && offsetParent.nodeType && !hasChild(offsetParent, rootElm)) {
                x -= offsetParent.scrollLeft || 0;
                y -= offsetParent.scrollTop || 0;
                offsetParent = offsetParent.parentNode;
            }
            y += getTableCaptionDeltaY(SugarElement.fromDom(elm));
        }
        return { x, y };
    };

    const getCrossOrigin$1 = (url, settings) => {
        const crossOriginFn = settings.crossOrigin;
        if (settings.contentCssCors) {
            return 'anonymous';
        }
        else if (isFunction(crossOriginFn)) {
            return crossOriginFn(url);
        }
        else {
            return undefined;
        }
    };
    const StyleSheetLoader = (documentOrShadowRoot, settings = {}) => {
        let idCount = 0;
        const loadedStates = {};
        const edos = SugarElement.fromDom(documentOrShadowRoot);
        const doc = documentOrOwner(edos);
        const _setReferrerPolicy = (referrerPolicy) => {
            settings.referrerPolicy = referrerPolicy;
        };
        const _setContentCssCors = (contentCssCors) => {
            settings.contentCssCors = contentCssCors;
        };
        const _setCrossOrigin = (crossOrigin) => {
            settings.crossOrigin = crossOrigin;
        };
        const addStyle = (element) => {
            append$1(getStyleContainer(edos), element);
        };
        const removeStyle = (id) => {
            const styleContainer = getStyleContainer(edos);
            descendant$1(styleContainer, '#' + id).each(remove$8);
        };
        const getOrCreateState = (url) => get$a(loadedStates, url).getOrThunk(() => ({
            id: 'mce-u' + (idCount++),
            passed: [],
            failed: [],
            count: 0
        }));
        /**
         * Loads the specified CSS file and returns a Promise that will resolve when the stylesheet is loaded successfully or reject if it failed to load.
         *
         * @method load
         * @param {String} url Url to be loaded.
         * @return {Promise} A Promise that will resolve or reject when the stylesheet is loaded.
         */
        const load = (url) => new Promise((success, failure) => {
            let link;
            const urlWithSuffix = Tools._addCacheSuffix(url);
            const state = getOrCreateState(urlWithSuffix);
            loadedStates[urlWithSuffix] = state;
            state.count++;
            const resolve = (callbacks, status) => {
                each$e(callbacks, call);
                state.status = status;
                state.passed = [];
                state.failed = [];
                if (link) {
                    link.onload = null;
                    link.onerror = null;
                    link = null;
                }
            };
            const passed = () => resolve(state.passed, 2);
            const failed = () => resolve(state.failed, 3);
            if (success) {
                state.passed.push(success);
            }
            if (failure) {
                state.failed.push(failure);
            }
            // Is loading wait for it to pass
            if (state.status === 1) {
                return;
            }
            // Has finished loading and was success
            if (state.status === 2) {
                passed();
                return;
            }
            // Has finished loading and was a failure
            if (state.status === 3) {
                failed();
                return;
            }
            // Start loading
            state.status = 1;
            const linkElem = SugarElement.fromTag('link', doc.dom);
            setAll$1(linkElem, {
                rel: 'stylesheet',
                type: 'text/css',
                id: state.id
            });
            const crossorigin = getCrossOrigin$1(url, settings);
            if (crossorigin !== undefined) {
                set$4(linkElem, 'crossOrigin', crossorigin);
            }
            if (settings.referrerPolicy) {
                // Note: Don't use link.referrerPolicy = ... here as it doesn't work on Safari
                set$4(linkElem, 'referrerpolicy', settings.referrerPolicy);
            }
            link = linkElem.dom;
            link.onload = passed;
            link.onerror = failed;
            addStyle(linkElem);
            set$4(linkElem, 'href', urlWithSuffix);
        });
        /**
         * Loads the specified css string in as a style element with an unique key.
         *
         * @method loadRawCss
         * @param {String} key Unique key for the style element.
         * @param {String} css Css style content to add.
         */
        const loadRawCss = (key, css) => {
            const state = getOrCreateState(key);
            loadedStates[key] = state;
            state.count++;
            // Start loading
            const styleElem = SugarElement.fromTag('style', doc.dom);
            setAll$1(styleElem, {
                'rel': 'stylesheet',
                'type': 'text/css',
                'id': state.id,
                'data-mce-key': key
            });
            styleElem.dom.innerHTML = css;
            addStyle(styleElem);
        };
        /**
         * Loads the specified CSS files and returns a Promise that is resolved when all stylesheets are loaded or rejected if any failed to load.
         *
         * @method loadAll
         * @param {Array} urls URLs to be loaded.
         * @return {Promise} A Promise that will resolve or reject when all stylesheets are loaded.
         */
        const loadAll = (urls) => {
            const loadedUrls = Promise.allSettled(map$3(urls, (url) => load(url).then(constant(url))));
            return loadedUrls.then((results) => {
                const parts = partition$2(results, (r) => r.status === 'fulfilled');
                if (parts.fail.length > 0) {
                    return Promise.reject(map$3(parts.fail, (result) => result.reason));
                }
                else {
                    return map$3(parts.pass, (result) => result.value);
                }
            });
        };
        /**
         * Unloads the specified CSS file if no resources currently depend on it.
         *
         * @method unload
         * @param {String} url URL to unload or remove.
         */
        const unload = (url) => {
            const urlWithSuffix = Tools._addCacheSuffix(url);
            get$a(loadedStates, urlWithSuffix).each((state) => {
                const count = --state.count;
                if (count === 0) {
                    delete loadedStates[urlWithSuffix];
                    removeStyle(state.id);
                }
            });
        };
        /**
         * Unloads the specified CSS style element by key.
         *
         * @method unloadRawCss
         * @param {String} key Key of CSS style resource to unload.
         */
        const unloadRawCss = (key) => {
            get$a(loadedStates, key).each((state) => {
                const count = --state.count;
                if (count === 0) {
                    delete loadedStates[key];
                    removeStyle(state.id);
                }
            });
        };
        /**
         * Unloads each specified CSS file if no resources currently depend on it.
         *
         * @method unloadAll
         * @param {Array} urls URLs to unload or remove.
         */
        const unloadAll = (urls) => {
            each$e(urls, (url) => {
                unload(url);
            });
        };
        return {
            load,
            loadRawCss,
            loadAll,
            unload,
            unloadRawCss,
            unloadAll,
            _setReferrerPolicy,
            _setContentCssCors,
            _setCrossOrigin
        };
    };

    /**
     * This function is exported for testing purposes only - please use StyleSheetLoader.instance in production code.
     */
    const create$b = () => {
        const map = new WeakMap();
        const forElement = (referenceElement, settings) => {
            const root = getRootNode(referenceElement);
            const rootDom = root.dom;
            return Optional.from(map.get(rootDom)).getOrThunk(() => {
                const sl = StyleSheetLoader(rootDom, settings);
                map.set(rootDom, sl);
                return sl;
            });
        };
        return {
            forElement
        };
    };
    const instance = create$b();

    const isSpan = (node) => node.nodeName.toLowerCase() === 'span';
    const isInlineContent = (node, schema) => isNonNullable(node) && (isContent$1(schema, node) || schema.isInline(node.nodeName.toLowerCase()));
    const surroundedByInlineContent = (node, root, schema) => {
        const prev = new DomTreeWalker(node, root).prev(false);
        const next = new DomTreeWalker(node, root).next(false);
        // Check if the next/previous is either inline content or the start/end (eg is undefined)
        const prevIsInline = isUndefined(prev) || isInlineContent(prev, schema);
        const nextIsInline = isUndefined(next) || isInlineContent(next, schema);
        return prevIsInline && nextIsInline;
    };
    const isBookmarkNode$2 = (node) => isSpan(node) && node.getAttribute('data-mce-type') === 'bookmark';
    // Keep text nodes with only spaces if surrounded by spans.
    // eg. "<p><span>a</span> <span>b</span></p>" should keep space between a and b
    const isKeepTextNode = (node, root, schema) => isText$b(node) && node.data.length > 0 && surroundedByInlineContent(node, root, schema);
    // Keep elements as long as they have any children
    const isKeepElement = (node) => isElement$7(node) ? node.childNodes.length > 0 : false;
    const isDocument = (node) => isDocumentFragment(node) || isDocument$1(node);
    // W3C valid browsers tend to leave empty nodes to the left/right side of the contents - this makes sense
    // but we don't want that in our code since it serves no purpose for the end user
    // For example splitting this html at the bold element:
    //   <p>text 1<span><b>CHOP</b></span>text 2</p>
    // would produce:
    //   <p>text 1<span></span></p><b>CHOP</b><p><span></span>text 2</p>
    // this function will then trim off empty edges and produce:
    //   <p>text 1</p><b>CHOP</b><p>text 2</p>
    const trimNode = (dom, node, schema, root) => {
        const rootNode = root || node;
        if (isElement$7(node) && isBookmarkNode$2(node)) {
            return node;
        }
        const children = node.childNodes;
        for (let i = children.length - 1; i >= 0; i--) {
            trimNode(dom, children[i], schema, rootNode);
        }
        // If the only child is a bookmark then move it up
        if (isElement$7(node)) {
            const currentChildren = node.childNodes;
            if (currentChildren.length === 1 && isBookmarkNode$2(currentChildren[0])) {
                node.parentNode?.insertBefore(currentChildren[0], node);
            }
        }
        // Remove any empty nodes
        if (!isDocument(node) && !isContent$1(schema, node) && !isKeepElement(node) && !isKeepTextNode(node, rootNode, schema)) {
            dom.remove(node);
        }
        return node;
    };

    /**
     * Entity encoder class.
     *
     * @class tinymce.html.Entities
     * @static
     * @version 3.4
     */
    const makeMap$3 = Tools.makeMap;
    const attrsCharsRegExp = /[&<>\"\u0060\u007E-\uD7FF\uE000-\uFFEF]|[\uD800-\uDBFF][\uDC00-\uDFFF]/g;
    const textCharsRegExp = /[<>&\u007E-\uD7FF\uE000-\uFFEF]|[\uD800-\uDBFF][\uDC00-\uDFFF]/g;
    const rawCharsRegExp = /[<>&\"\']/g;
    const entityRegExp = /&#([a-z0-9]+);?|&([a-z0-9]+);/gi;
    const asciiMap = {
        128: '\u20AC', 130: '\u201A', 131: '\u0192', 132: '\u201E', 133: '\u2026', 134: '\u2020',
        135: '\u2021', 136: '\u02C6', 137: '\u2030', 138: '\u0160', 139: '\u2039', 140: '\u0152',
        142: '\u017D', 145: '\u2018', 146: '\u2019', 147: '\u201C', 148: '\u201D', 149: '\u2022',
        150: '\u2013', 151: '\u2014', 152: '\u02DC', 153: '\u2122', 154: '\u0161', 155: '\u203A',
        156: '\u0153', 158: '\u017E', 159: '\u0178'
    };
    // Raw entities
    const baseEntities = {
        '\"': '&quot;', // Needs to be escaped since the YUI compressor would otherwise break the code
        '\'': '&#39;',
        '<': '&lt;',
        '>': '&gt;',
        '&': '&amp;',
        '\u0060': '&#96;'
    };
    // Reverse lookup table for raw entities
    const reverseEntities = {
        '&lt;': '<',
        '&gt;': '>',
        '&amp;': '&',
        '&quot;': '"',
        '&apos;': `'`
    };
    // Decodes text by using the browser
    const nativeDecode = (text) => {
        const elm = SugarElement.fromTag('div').dom;
        elm.innerHTML = text;
        return elm.textContent || elm.innerText || text;
    };
    // Build a two way lookup table for the entities
    const buildEntitiesLookup = (items, radix) => {
        const lookup = {};
        if (items) {
            const itemList = items.split(',');
            radix = radix || 10;
            // Build entities lookup table
            for (let i = 0; i < itemList.length; i += 2) {
                const chr = String.fromCharCode(parseInt(itemList[i], radix));
                // Only add non base entities
                if (!baseEntities[chr]) {
                    const entity = '&' + itemList[i + 1] + ';';
                    lookup[chr] = entity;
                    lookup[entity] = chr;
                }
            }
            return lookup;
        }
        else {
            return undefined;
        }
    };
    // Unpack entities lookup where the numbers are in radix 32 to reduce the size
    const namedEntities = buildEntitiesLookup('50,nbsp,51,iexcl,52,cent,53,pound,54,curren,55,yen,56,brvbar,57,sect,58,uml,59,copy,' +
        '5a,ordf,5b,laquo,5c,not,5d,shy,5e,reg,5f,macr,5g,deg,5h,plusmn,5i,sup2,5j,sup3,5k,acute,' +
        '5l,micro,5m,para,5n,middot,5o,cedil,5p,sup1,5q,ordm,5r,raquo,5s,frac14,5t,frac12,5u,frac34,' +
        '5v,iquest,60,Agrave,61,Aacute,62,Acirc,63,Atilde,64,Auml,65,Aring,66,AElig,67,Ccedil,' +
        '68,Egrave,69,Eacute,6a,Ecirc,6b,Euml,6c,Igrave,6d,Iacute,6e,Icirc,6f,Iuml,6g,ETH,6h,Ntilde,' +
        '6i,Ograve,6j,Oacute,6k,Ocirc,6l,Otilde,6m,Ouml,6n,times,6o,Oslash,6p,Ugrave,6q,Uacute,' +
        '6r,Ucirc,6s,Uuml,6t,Yacute,6u,THORN,6v,szlig,70,agrave,71,aacute,72,acirc,73,atilde,74,auml,' +
        '75,aring,76,aelig,77,ccedil,78,egrave,79,eacute,7a,ecirc,7b,euml,7c,igrave,7d,iacute,7e,icirc,' +
        '7f,iuml,7g,eth,7h,ntilde,7i,ograve,7j,oacute,7k,ocirc,7l,otilde,7m,ouml,7n,divide,7o,oslash,' +
        '7p,ugrave,7q,uacute,7r,ucirc,7s,uuml,7t,yacute,7u,thorn,7v,yuml,ci,fnof,sh,Alpha,si,Beta,' +
        'sj,Gamma,sk,Delta,sl,Epsilon,sm,Zeta,sn,Eta,so,Theta,sp,Iota,sq,Kappa,sr,Lambda,ss,Mu,' +
        'st,Nu,su,Xi,sv,Omicron,t0,Pi,t1,Rho,t3,Sigma,t4,Tau,t5,Upsilon,t6,Phi,t7,Chi,t8,Psi,' +
        't9,Omega,th,alpha,ti,beta,tj,gamma,tk,delta,tl,epsilon,tm,zeta,tn,eta,to,theta,tp,iota,' +
        'tq,kappa,tr,lambda,ts,mu,tt,nu,tu,xi,tv,omicron,u0,pi,u1,rho,u2,sigmaf,u3,sigma,u4,tau,' +
        'u5,upsilon,u6,phi,u7,chi,u8,psi,u9,omega,uh,thetasym,ui,upsih,um,piv,812,bull,816,hellip,' +
        '81i,prime,81j,Prime,81u,oline,824,frasl,88o,weierp,88h,image,88s,real,892,trade,89l,alefsym,' +
        '8cg,larr,8ch,uarr,8ci,rarr,8cj,darr,8ck,harr,8dl,crarr,8eg,lArr,8eh,uArr,8ei,rArr,8ej,dArr,' +
        '8ek,hArr,8g0,forall,8g2,part,8g3,exist,8g5,empty,8g7,nabla,8g8,isin,8g9,notin,8gb,ni,8gf,prod,' +
        '8gh,sum,8gi,minus,8gn,lowast,8gq,radic,8gt,prop,8gu,infin,8h0,ang,8h7,and,8h8,or,8h9,cap,8ha,cup,' +
        '8hb,int,8hk,there4,8hs,sim,8i5,cong,8i8,asymp,8j0,ne,8j1,equiv,8j4,le,8j5,ge,8k2,sub,8k3,sup,8k4,' +
        'nsub,8k6,sube,8k7,supe,8kl,oplus,8kn,otimes,8l5,perp,8m5,sdot,8o8,lceil,8o9,rceil,8oa,lfloor,8ob,' +
        'rfloor,8p9,lang,8pa,rang,9ea,loz,9j0,spades,9j3,clubs,9j5,hearts,9j6,diams,ai,OElig,aj,oelig,b0,' +
        'Scaron,b1,scaron,bo,Yuml,m6,circ,ms,tilde,802,ensp,803,emsp,809,thinsp,80c,zwnj,80d,zwj,80e,lrm,' +
        '80f,rlm,80j,ndash,80k,mdash,80o,lsquo,80p,rsquo,80q,sbquo,80s,ldquo,80t,rdquo,80u,bdquo,810,dagger,' +
        '811,Dagger,81g,permil,81p,lsaquo,81q,rsaquo,85c,euro', 32);
    /**
     * Encodes the specified string using raw entities. This means only the required XML base entities will be encoded.
     *
     * @method encodeRaw
     * @param {String} text Text to encode.
     * @param {Boolean} attr Optional flag to specify if the text is attribute contents.
     * @return {String} Entity encoded text.
     */
    const encodeRaw = (text, attr) => text.replace(attr ? attrsCharsRegExp : textCharsRegExp, (chr) => {
        return baseEntities[chr] || chr;
    });
    /**
     * Encoded the specified text with both the attributes and text entities. This function will produce larger text contents
     * since it doesn't know if the context is within an attribute or text node. This was added for compatibility
     * and is exposed as the `DOMUtils.encode` function.
     *
     * @method encodeAllRaw
     * @param {String} text Text to encode.
     * @return {String} Entity encoded text.
     */
    const encodeAllRaw = (text) => ('' + text).replace(rawCharsRegExp, (chr) => {
        return baseEntities[chr] || chr;
    });
    /**
     * Encodes the specified string using numeric entities. The core entities will be
     * encoded as named ones but all non lower ascii characters will be encoded into numeric entities.
     *
     * @method encodeNumeric
     * @param {String} text Text to encode.
     * @param {Boolean} attr Optional flag to specify if the text is attribute contents.
     * @return {String} Entity encoded text.
     */
    const encodeNumeric = (text, attr) => text.replace(attr ? attrsCharsRegExp : textCharsRegExp, (chr) => {
        // Multi byte sequence convert it to a single entity
        if (chr.length > 1) {
            return '&#' + (((chr.charCodeAt(0) - 0xD800) * 0x400) + (chr.charCodeAt(1) - 0xDC00) + 0x10000) + ';';
        }
        return baseEntities[chr] || '&#' + chr.charCodeAt(0) + ';';
    });
    /**
     * Encodes the specified string using named entities. The core entities will be encoded
     * as named ones but all non lower ascii characters will be encoded into named entities.
     *
     * @method encodeNamed
     * @param {String} text Text to encode.
     * @param {Boolean} attr Optional flag to specify if the text is attribute contents.
     * @param {Object} entities Optional parameter with entities to use.
     * @return {String} Entity encoded text.
     */
    const encodeNamed = (text, attr, entities) => {
        const resolveEntities = entities || namedEntities;
        return text.replace(attr ? attrsCharsRegExp : textCharsRegExp, (chr) => {
            return baseEntities[chr] || resolveEntities[chr] || chr;
        });
    };
    /**
     * Returns an encode function based on the name(s) and it's optional entities.
     *
     * @method getEncodeFunc
     * @param {String} name Comma separated list of encoders for example named,numeric.
     * @param {String} entities Optional parameter with entities to use instead of the built in set.
     * @return {Function} Encode function to be used.
     */
    const getEncodeFunc = (name, entities) => {
        const entitiesMap = buildEntitiesLookup(entities) || namedEntities;
        const encodeNamedAndNumeric = (text, attr) => text.replace(attr ? attrsCharsRegExp : textCharsRegExp, (chr) => {
            if (baseEntities[chr] !== undefined) {
                return baseEntities[chr];
            }
            if (entitiesMap[chr] !== undefined) {
                return entitiesMap[chr];
            }
            // Convert multi-byte sequences to a single entity.
            if (chr.length > 1) {
                return '&#' + (((chr.charCodeAt(0) - 0xD800) * 0x400) + (chr.charCodeAt(1) - 0xDC00) + 0x10000) + ';';
            }
            return '&#' + chr.charCodeAt(0) + ';';
        });
        const encodeCustomNamed = (text, attr) => {
            return encodeNamed(text, attr, entitiesMap);
        };
        // Replace + with , to be compatible with previous TinyMCE versions
        const nameMap = makeMap$3(name.replace(/\+/g, ','));
        // Named and numeric encoder
        if (nameMap.named && nameMap.numeric) {
            return encodeNamedAndNumeric;
        }
        // Named encoder
        if (nameMap.named) {
            // Custom names
            if (entities) {
                return encodeCustomNamed;
            }
            return encodeNamed;
        }
        // Numeric
        if (nameMap.numeric) {
            return encodeNumeric;
        }
        // Raw encoder
        return encodeRaw;
    };
    /**
     * Decodes the specified string, this will replace entities with raw UTF characters.
     *
     * @method decode
     * @param {String} text Text to entity decode.
     * @return {String} Entity decoded string.
     */
    const decode = (text) => text.replace(entityRegExp, (all, numeric) => {
        if (numeric) {
            if (numeric.charAt(0).toLowerCase() === 'x') {
                numeric = parseInt(numeric.substr(1), 16);
            }
            else {
                numeric = parseInt(numeric, 10);
            }
            // Support upper UTF
            if (numeric > 0xFFFF) {
                numeric -= 0x10000;
                // eslint-disable-next-line no-bitwise
                return String.fromCharCode(0xD800 + (numeric >> 10), 0xDC00 + (numeric & 0x3FF));
            }
            return asciiMap[numeric] || String.fromCharCode(numeric);
        }
        return reverseEntities[all] || namedEntities[all] || nativeDecode(all);
    });
    const Entities = {
        encodeRaw,
        encodeAllRaw,
        encodeNumeric,
        encodeNamed,
        getEncodeFunc,
        decode
    };

    const split$1 = (items, delim) => {
        items = Tools.trim(items);
        return items ? items.split(delim || ' ') : [];
    };
    // Converts a wildcard expression string to a regexp for example *a will become /.*a/.
    const patternToRegExp = (str) => new RegExp('^' + str.replace(/([?+*])/g, '.$1') + '$');
    const isRegExp$1 = (obj) => isObject(obj) && obj.source && Object.prototype.toString.call(obj) === '[object RegExp]';
    const deepCloneElementRule = (obj) => {
        const helper = (value) => {
            if (isArray$1(value)) {
                return map$3(value, helper);
            }
            else if (isRegExp$1(value)) {
                return new RegExp(value.source, value.flags);
            }
            else if (isObject(value)) {
                return map$2(value, helper);
            }
            else {
                return value;
            }
        };
        return helper(obj);
    };

    const parseCustomElementsRules = (value) => {
        const customElementRegExp = /^(~)?(.+)$/;
        return bind$3(split$1(value, ','), (rule) => {
            const matches = customElementRegExp.exec(rule);
            if (matches) {
                const inline = matches[1] === '~';
                const cloneName = inline ? 'span' : 'div';
                const name = matches[2];
                return [{ cloneName, name }];
            }
            else {
                return [];
            }
        });
    };

    const getGlobalAttributeSet = (type) => {
        return Object.freeze([
            // Present on all schema types
            'id',
            'accesskey',
            'class',
            'dir',
            'lang',
            'style',
            'tabindex',
            'title',
            'role',
            // html5 and html5-strict extra attributes
            ...(type !== 'html4' ? ['contenteditable', 'contextmenu', 'draggable', 'dropzone', 'hidden', 'spellcheck', 'translate', 'itemprop', 'itemscope', 'itemtype'] : []),
            // html4 and html5 extra attributes
            ...(type !== 'html5-strict' ? ['xml:lang'] : [])
        ]);
    };

    // Missing elements in `phrasing` compared to HTML5 spec at 2024-01-30 (timestamped since spec is constantly evolving)
    //  area - required to be inside a map element so we should not add it to all elements.
    //  link - required to be in the body so we should not add it to all elements.
    //  math - currently not supported.
    //  meta - Only allowed if the `itemprop` attribute is set so very special.
    //  slot - We only want these to be accepted in registered custom components.
    // Extra element in `phrasing`: command keygen
    //
    // Missing elements in `flow` compared to HTML5 spec at 2034-01-30 (timestamped since the spec is constantly evolving)
    //  search - Can be both in a block and inline position but is not a transparent element. So not supported at this time.
    const getElementSetsAsStrings = (type) => {
        let blockContent;
        let phrasingContent;
        // Block content elements
        blockContent =
            'address blockquote div dl fieldset form h1 h2 h3 h4 h5 h6 hr menu ol p pre table ul';
        // Phrasing content elements from the HTML5 spec (inline)
        phrasingContent =
            'a abbr b bdo br button cite code del dfn em embed i iframe img input ins kbd ' +
                'label map noscript object q s samp script select small span strong sub sup ' +
                'textarea u var #text #comment';
        // Add HTML5 items to globalAttributes, blockContent, phrasingContent
        if (type !== 'html4') {
            const transparentContent = 'a ins del canvas map';
            blockContent += ' article aside details dialog figure main header footer hgroup section nav ' + transparentContent;
            phrasingContent += ' audio canvas command data datalist mark meter output picture ' +
                'progress template time wbr video ruby bdi keygen svg';
        }
        // Add HTML4 elements unless it's html5-strict
        if (type !== 'html5-strict') {
            const html4PhrasingContent = 'acronym applet basefont big font strike tt';
            phrasingContent = [phrasingContent, html4PhrasingContent].join(' ');
            const html4BlockContent = 'center dir isindex noframes';
            blockContent = [blockContent, html4BlockContent].join(' ');
        }
        // Flow content elements from the HTML5 spec (block+inline)
        const flowContent = [blockContent, phrasingContent].join(' ');
        return { blockContent, phrasingContent, flowContent };
    };
    const getElementSets = (type) => {
        const { blockContent, phrasingContent, flowContent } = getElementSetsAsStrings(type);
        const toArr = (value) => {
            return Object.freeze(value.split(' '));
        };
        return Object.freeze({
            blockContent: toArr(blockContent),
            phrasingContent: toArr(phrasingContent),
            flowContent: toArr(flowContent)
        });
    };

    const cachedSets = {
        'html4': cached(() => getElementSets('html4')),
        'html5': cached(() => getElementSets('html5')),
        'html5-strict': cached(() => getElementSets('html5-strict'))
    };
    // eslint-disable-next-line @typescript-eslint/no-redundant-type-constituents
    const getElementsPreset = (type, name) => {
        const { blockContent, phrasingContent, flowContent } = cachedSets[type]();
        if (name === 'blocks') {
            return Optional.some(blockContent);
        }
        else if (name === 'phrasing') {
            return Optional.some(phrasingContent);
        }
        else if (name === 'flow') {
            return Optional.some(flowContent);
        }
        else {
            return Optional.none();
        }
    };

    const makeSchema = (type) => {
        const globalAttributes = getGlobalAttributeSet(type);
        const { phrasingContent, flowContent } = getElementSetsAsStrings(type);
        const schema = {};
        const addElement = (name, attributes, children) => {
            schema[name] = {
                attributes: mapToObject(attributes, constant({})),
                attributesOrder: attributes,
                children: mapToObject(children, constant({}))
            };
        };
        const add = (name, attributes = '', children = '') => {
            const childNames = split$1(children);
            const names = split$1(name);
            let ni = names.length;
            const allAttributes = [...globalAttributes, ...split$1(attributes)];
            while (ni--) {
                addElement(names[ni], allAttributes.slice(), childNames);
            }
        };
        const addAttrs = (name, attributes) => {
            const names = split$1(name);
            const attrs = split$1(attributes);
            let ni = names.length;
            while (ni--) {
                const schemaItem = schema[names[ni]];
                for (let i = 0, l = attrs.length; i < l; i++) {
                    schemaItem.attributes[attrs[i]] = {};
                    schemaItem.attributesOrder.push(attrs[i]);
                }
            }
        };
        if (type !== 'html5-strict') {
            const html4PhrasingContent = 'acronym applet basefont big font strike tt';
            each$e(split$1(html4PhrasingContent), (name) => {
                add(name, '', phrasingContent);
            });
            const html4BlockContent = 'center dir isindex noframes';
            each$e(split$1(html4BlockContent), (name) => {
                add(name, '', flowContent);
            });
        }
        // HTML4 base schema TODO: Move HTML5 specific attributes to HTML5 specific if statement
        // Schema items <element name>, <specific attributes>, <children ..>
        add('html', 'manifest', 'head body');
        add('head', '', 'base command link meta noscript script style title');
        add('title hr noscript br');
        add('base', 'href target');
        add('link', 'href rel media hreflang type sizes hreflang');
        add('meta', 'name http-equiv content charset property'); // Property is an RDFa spec attribute.
        add('style', 'media type scoped');
        add('script', 'src async defer type charset');
        add('body', 'onafterprint onbeforeprint onbeforeunload onblur onerror onfocus ' +
            'onhashchange onload onmessage onoffline ononline onpagehide onpageshow ' +
            'onpopstate onresize onscroll onstorage onunload', flowContent);
        add('dd div', '', flowContent);
        add('address dt caption', '', type === 'html4' ? phrasingContent : flowContent);
        add('h1 h2 h3 h4 h5 h6 pre p abbr code var samp kbd sub sup i b u bdo span legend em strong small s cite dfn', '', phrasingContent);
        add('blockquote', 'cite', flowContent);
        add('ol', 'reversed start type', 'li');
        add('ul', '', 'li');
        add('li', 'value', flowContent);
        add('dl', '', 'dt dd');
        add('a', 'href target rel media hreflang type', type === 'html4' ? phrasingContent : flowContent);
        add('q', 'cite', phrasingContent);
        add('ins del', 'cite datetime', flowContent);
        add('img', 'src sizes srcset alt usemap ismap width height');
        add('iframe', 'src name width height', flowContent);
        add('embed', 'src type width height');
        add('object', 'data type typemustmatch name usemap form width height', [flowContent, 'param'].join(' '));
        add('param', 'name value');
        add('map', 'name', [flowContent, 'area'].join(' '));
        add('area', 'alt coords shape href target rel media hreflang type');
        add('table', 'border', 'caption colgroup thead tfoot tbody tr' + (type === 'html4' ? ' col' : ''));
        add('colgroup', 'span', 'col');
        add('col', 'span');
        add('tbody thead tfoot', '', 'tr');
        add('tr', '', 'td th');
        add('td', 'colspan rowspan headers', flowContent);
        add('th', 'colspan rowspan headers scope abbr', flowContent);
        add('form', 'accept-charset action autocomplete enctype method name novalidate target', flowContent);
        add('fieldset', 'disabled form name', [flowContent, 'legend'].join(' '));
        add('label', 'form for', phrasingContent);
        add('input', 'accept alt autocomplete checked dirname disabled form formaction formenctype formmethod formnovalidate ' +
            'formtarget height list max maxlength min multiple name pattern readonly required size src step type value width');
        add('button', 'disabled form formaction formenctype formmethod formnovalidate formtarget name type value', type === 'html4' ? flowContent : phrasingContent);
        add('select', 'disabled form multiple name required size', 'option optgroup');
        add('optgroup', 'disabled label', 'option');
        add('option', 'disabled label selected value');
        add('textarea', 'cols dirname disabled form maxlength name readonly required rows wrap');
        add('menu', 'type label', [flowContent, 'li'].join(' '));
        add('noscript', '', flowContent);
        // Extend with HTML5 elements
        if (type !== 'html4') {
            add('wbr');
            add('ruby', '', [phrasingContent, 'rt rp'].join(' '));
            add('figcaption', '', flowContent);
            add('mark rt rp bdi', '', phrasingContent);
            add('summary', '', [phrasingContent, 'h1 h2 h3 h4 h5 h6'].join(' '));
            add('canvas', 'width height', flowContent);
            add('data', 'value', phrasingContent);
            add('video', 'src crossorigin poster preload autoplay mediagroup loop ' +
                'controlslist disablepictureinpicture disableremoteplayback playsinline ' +
                'muted controls width height buffered', [flowContent, 'track source'].join(' '));
            add('audio', 'src crossorigin preload autoplay mediagroup loop muted controls ' +
                'buffered volume', [flowContent, 'track source'].join(' '));
            add('picture', '', 'img source');
            add('source', 'src srcset type media sizes');
            add('track', 'kind src srclang label default');
            add('datalist', '', [phrasingContent, 'option'].join(' '));
            add('article section nav aside main header footer', '', flowContent);
            add('hgroup', '', 'h1 h2 h3 h4 h5 h6');
            add('figure', '', [flowContent, 'figcaption'].join(' '));
            add('time', 'datetime', phrasingContent);
            add('dialog', 'open', flowContent);
            add('command', 'type label icon disabled checked radiogroup command');
            add('output', 'for form name', phrasingContent);
            add('progress', 'value max', phrasingContent);
            add('meter', 'value min max low high optimum', phrasingContent);
            add('details', 'open', [flowContent, 'summary'].join(' '));
            add('keygen', 'autofocus challenge disabled form keytype name');
            // SVGs only support a subset of the global attributes
            addElement('svg', 'id tabindex lang xml:space class style x y width height viewBox preserveAspectRatio zoomAndPan transform'.split(' '), []);
        }
        // Extend with HTML4 attributes unless it's html5-strict
        if (type !== 'html5-strict') {
            addAttrs('script', 'language xml:space');
            addAttrs('style', 'xml:space');
            addAttrs('object', 'declare classid code codebase codetype archive standby align border hspace vspace');
            addAttrs('embed', 'align name hspace vspace');
            addAttrs('param', 'valuetype type');
            addAttrs('a', 'charset name rev shape coords');
            addAttrs('br', 'clear');
            addAttrs('applet', 'codebase archive code object alt name width height align hspace vspace');
            addAttrs('img', 'name longdesc align border hspace vspace');
            addAttrs('iframe', 'longdesc frameborder marginwidth marginheight scrolling align');
            addAttrs('font basefont', 'size color face');
            addAttrs('input', 'usemap align');
            addAttrs('select');
            addAttrs('textarea');
            addAttrs('h1 h2 h3 h4 h5 h6 div p legend caption', 'align');
            addAttrs('ul', 'type compact');
            addAttrs('li', 'type');
            addAttrs('ol dl menu dir', 'compact');
            addAttrs('pre', 'width xml:space');
            addAttrs('hr', 'align noshade size width');
            addAttrs('isindex', 'prompt');
            addAttrs('table', 'summary width frame rules cellspacing cellpadding align bgcolor');
            addAttrs('col', 'width align char charoff valign');
            addAttrs('colgroup', 'width align char charoff valign');
            addAttrs('thead', 'align char charoff valign');
            addAttrs('tr', 'align char charoff valign bgcolor');
            addAttrs('th', 'axis align char charoff valign nowrap bgcolor width height');
            addAttrs('form', 'accept');
            addAttrs('td', 'abbr axis scope align char charoff valign nowrap bgcolor width height');
            addAttrs('tfoot', 'align char charoff valign');
            addAttrs('tbody', 'align char charoff valign');
            addAttrs('area', 'nohref');
            addAttrs('body', 'background bgcolor text link vlink alink');
        }
        // Extend with HTML5 attributes unless it's html4
        if (type !== 'html4') {
            addAttrs('input button select textarea', 'autofocus');
            addAttrs('input textarea', 'placeholder');
            addAttrs('a', 'download');
            addAttrs('link script img', 'crossorigin');
            addAttrs('img', 'loading');
            addAttrs('iframe', 'sandbox seamless allow allowfullscreen loading referrerpolicy'); // Excluded: srcdoc
        }
        // Special: iframe, ruby, video, audio, label
        if (type !== 'html4') {
            // Video/audio elements cannot have nested children
            each$e([schema.video, schema.audio], (item) => {
                delete item.children.audio;
                delete item.children.video;
            });
        }
        // Delete children of the same name from it's parent
        // For example: form can't have a child of the name form
        each$e(split$1('a form meter progress dfn'), (name) => {
            if (schema[name]) {
                delete schema[name].children[name];
            }
        });
        // Delete header, footer, sectioning and heading content descendants
        /* each('dt th address', function(name) {
         delete schema[name].children[name];
         });*/
        // Caption can't have tables
        delete schema.caption.children.table;
        // Delete scripts by default due to possible XSS
        delete schema.script;
        // TODO: LI:s can only have value if parent is OL
        return schema;
    };

    const prefixToOperation = (prefix) => prefix === '-' ? 'remove' : 'add';
    const parseValidChild = (name) => {
        // see: https://html.spec.whatwg.org/#valid-custom-element-name
        const validChildRegExp = /^(@?)([A-Za-z0-9_\-.\u00b7\u00c0-\u00d6\u00d8-\u00f6\u00f8-\u037d\u037f-\u1fff\u200c-\u200d\u203f-\u2040\u2070-\u218f\u2c00-\u2fef\u3001-\ud7ff\uf900-\ufdcf\ufdf0-\ufffd]+)$/;
        return Optional.from(validChildRegExp.exec(name)).map((matches) => ({
            preset: matches[1] === '@',
            name: matches[2]
        }));
    };
    const parseValidChildrenRules = (value) => {
        // see: https://html.spec.whatwg.org/#valid-custom-element-name
        const childRuleRegExp = /^([+\-]?)([A-Za-z0-9_\-.\u00b7\u00c0-\u00d6\u00d8-\u00f6\u00f8-\u037d\u037f-\u1fff\u200c-\u200d\u203f-\u2040\u2070-\u218f\u2c00-\u2fef\u3001-\ud7ff\uf900-\ufdcf\ufdf0-\ufffd]+)\[([^\]]+)]$/; // from w3c's custom grammar (above)
        return bind$3(split$1(value, ','), (rule) => {
            const matches = childRuleRegExp.exec(rule);
            if (matches) {
                const prefix = matches[1];
                const operation = prefix ? prefixToOperation(prefix) : 'replace';
                const name = matches[2];
                const validChildren = bind$3(split$1(matches[3], '|'), (validChild) => parseValidChild(validChild).toArray());
                return [{ operation, name, validChildren }];
            }
            else {
                return [];
            }
        });
    };

    const parseValidElementsAttrDataIntoElement = (attrData, targetElement) => {
        const attrRuleRegExp = /^([!\-])?(\w+[\\:]:\w+|[^=~<]+)?(?:([=~<])(.*))?$/;
        const hasPatternsRegExp = /[*?+]/;
        const { attributes, attributesOrder } = targetElement;
        return each$e(split$1(attrData, '|'), (rule) => {
            const matches = attrRuleRegExp.exec(rule);
            if (matches) {
                const attr = {};
                const attrType = matches[1];
                const attrName = matches[2].replace(/[\\:]:/g, ':');
                const attrPrefix = matches[3];
                const value = matches[4];
                // Required
                if (attrType === '!') {
                    targetElement.attributesRequired = targetElement.attributesRequired || [];
                    targetElement.attributesRequired.push(attrName);
                    attr.required = true;
                }
                // Denied from global
                if (attrType === '-') {
                    delete attributes[attrName];
                    attributesOrder.splice(Tools.inArray(attributesOrder, attrName), 1);
                    return;
                }
                // Default value
                if (attrPrefix) {
                    if (attrPrefix === '=') { // Default value
                        targetElement.attributesDefault = targetElement.attributesDefault || [];
                        targetElement.attributesDefault.push({ name: attrName, value });
                        attr.defaultValue = value;
                    }
                    else if (attrPrefix === '~') { // Forced value
                        targetElement.attributesForced = targetElement.attributesForced || [];
                        targetElement.attributesForced.push({ name: attrName, value });
                        attr.forcedValue = value;
                    }
                    else if (attrPrefix === '<') { // Required values
                        attr.validValues = Tools.makeMap(value, '?');
                    }
                }
                // Check for attribute patterns
                if (hasPatternsRegExp.test(attrName)) {
                    const attrPattern = attr;
                    targetElement.attributePatterns = targetElement.attributePatterns || [];
                    attrPattern.pattern = patternToRegExp(attrName);
                    targetElement.attributePatterns.push(attrPattern);
                }
                else {
                    // Add attribute to order list if it doesn't already exist
                    if (!attributes[attrName]) {
                        attributesOrder.push(attrName);
                    }
                    attributes[attrName] = attr;
                }
            }
        });
    };
    const cloneAttributesInto = (from, to) => {
        each$d(from.attributes, (value, key) => {
            to.attributes[key] = value;
        });
        to.attributesOrder.push(...from.attributesOrder);
    };
    const parseValidElementsRules = (globalElement, validElements) => {
        const elementRuleRegExp = /^([#+\-])?([^\[!\/]+)(?:\/([^\[!]+))?(?:(!?)\[([^\]]+)])?$/;
        return bind$3(split$1(validElements, ','), (rule) => {
            const matches = elementRuleRegExp.exec(rule);
            if (matches) {
                const prefix = matches[1];
                const elementName = matches[2];
                const outputName = matches[3];
                const attrsPrefix = matches[4];
                const attrData = matches[5];
                const element = {
                    attributes: {},
                    attributesOrder: []
                };
                globalElement.each((el) => cloneAttributesInto(el, element));
                if (prefix === '#') {
                    element.paddEmpty = true;
                }
                else if (prefix === '-') {
                    element.removeEmpty = true;
                }
                if (attrsPrefix === '!') {
                    element.removeEmptyAttrs = true;
                }
                if (attrData) {
                    parseValidElementsAttrDataIntoElement(attrData, element);
                }
                // Handle substitute elements such as b/strong
                if (outputName) {
                    element.outputName = elementName;
                }
                // Mutate the local globalElement option state if we find a global @ rule
                if (elementName === '@') {
                    // We only care about the first one
                    if (globalElement.isNone()) {
                        globalElement = Optional.some(element);
                    }
                    else {
                        return [];
                    }
                }
                return [outputName ? { name: elementName, element, aliasName: outputName } : { name: elementName, element }];
            }
            else {
                return [];
            }
        });
    };

    /**
     * Schema validator class.
     *
     * @class tinymce.html.Schema
     * @version 3.4
     * @example
     * if (tinymce.activeEditor.schema.isValidChild('p', 'span')) {
     *   alert('span is valid child of p.');
     * }
     *
     * if (tinymce.activeEditor.schema.getElementRule('p')) {
     *   alert('P is a valid element.');
     * }
     */
    const mapCache = {};
    const makeMap$2 = Tools.makeMap, each$b = Tools.each, extend$2 = Tools.extend, explode$2 = Tools.explode;
    const createMap = (defaultValue, extendWith = {}) => {
        const value = makeMap$2(defaultValue, ' ', makeMap$2(defaultValue.toUpperCase(), ' '));
        return extend$2(value, extendWith);
    };
    // A curated list using the textBlockElements map and parts of the blockElements map from the schema
    // TODO: TINY-8728 Investigate if the extras can be added directly to the default text block elements
    const getTextRootBlockElements = (schema) => createMap('td th li dt dd figcaption caption details summary', schema.getTextBlockElements());
    const compileElementMap = (value, mode) => {
        if (value) {
            const styles = {};
            if (isString(value)) {
                value = {
                    '*': value
                };
            }
            // Convert styles into a rule list
            each$b(value, (value, key) => {
                styles[key] = styles[key.toUpperCase()] = mode === 'map' ? makeMap$2(value, /[, ]/) : explode$2(value, /[, ]/);
            });
            return styles;
        }
        else {
            return undefined;
        }
    };
    const Schema = (settings = {}) => {
        const elements = {};
        const children = {};
        let patternElements = [];
        const customElementsMap = {};
        const specialElements = {};
        const componentUrls = {};
        // Creates an lookup table map object for the specified option or the default value
        const createLookupTable = (option, defaultValue, extendWith) => {
            const value = settings[option];
            if (!value) {
                // Get cached default map or make it if needed
                let newValue = mapCache[option];
                if (!newValue) {
                    newValue = createMap(defaultValue, extendWith);
                    mapCache[option] = newValue;
                }
                return newValue;
            }
            else {
                // Create custom map
                return makeMap$2(value, /[, ]/, makeMap$2(value.toUpperCase(), /[, ]/));
            }
        };
        const schemaType = settings.schema ?? 'html5';
        const schemaItems = makeSchema(schemaType);
        // Allow all elements and attributes if verify_html is set to false
        if (settings.verify_html === false) {
            settings.valid_elements = '*[*]';
        }
        const validStyles = compileElementMap(settings.valid_styles);
        const invalidStyles = compileElementMap(settings.invalid_styles, 'map');
        const validClasses = compileElementMap(settings.valid_classes, 'map');
        // Setup map objects
        const whitespaceElementsMap = createLookupTable('whitespace_elements', 'pre script noscript style textarea video audio iframe object code');
        const selfClosingElementsMap = createLookupTable('self_closing_elements', 'colgroup dd dt li option p td tfoot th thead tr');
        const voidElementsMap = createLookupTable('void_elements', 'area base basefont br col frame hr img input isindex link ' +
            'meta param embed source wbr track');
        const boolAttrMap = createLookupTable('boolean_attributes', 'checked compact declare defer disabled ismap multiple nohref noresize ' +
            'noshade nowrap readonly selected autoplay loop controls allowfullscreen');
        const nonEmptyOrMoveCaretBeforeOnEnter = 'td th iframe video audio object script code';
        const nonEmptyElementsMap = createLookupTable('non_empty_elements', nonEmptyOrMoveCaretBeforeOnEnter + ' pre svg textarea summary', voidElementsMap);
        const moveCaretBeforeOnEnterElementsMap = createLookupTable('move_caret_before_on_enter_elements', nonEmptyOrMoveCaretBeforeOnEnter + ' table', voidElementsMap);
        const headings = 'h1 h2 h3 h4 h5 h6';
        const textBlockElementsMap = createLookupTable('text_block_elements', headings + ' p div address pre form ' +
            'blockquote center dir fieldset header footer article section hgroup aside main nav figure');
        const blockElementsMap = createLookupTable('block_elements', 'hr table tbody thead tfoot ' +
            'th tr td li ol ul caption dl dt dd noscript menu isindex option ' +
            'datalist select optgroup figcaption details summary html body multicol listing colgroup col', textBlockElementsMap);
        const textInlineElementsMap = createLookupTable('text_inline_elements', 'span strong b em i font s strike u var cite ' +
            'dfn code mark q sup sub samp');
        const transparentElementsMap = createLookupTable('transparent_elements', 'a ins del canvas map');
        const wrapBlockElementsMap = createLookupTable('wrap_block_elements', 'pre ' + headings);
        // See https://html.spec.whatwg.org/multipage/parsing.html#parsing-html-fragments
        each$b(('script noscript iframe noframes noembed title style textarea xmp plaintext').split(' '), (name) => {
            specialElements[name] = new RegExp('<\/' + name + '[^>]*>', 'gi');
        });
        // Parses the specified valid_elements string and adds to the current rules
        const addValidElements = (validElements) => {
            const globalElement = Optional.from(elements['@']);
            const hasPatternsRegExp = /[*?+]/;
            each$e(parseValidElementsRules(globalElement, validElements ?? ''), ({ name, element, aliasName }) => {
                if (aliasName) {
                    elements[aliasName] = element;
                }
                // Add pattern or exact element
                if (hasPatternsRegExp.test(name)) {
                    const patternElement = element;
                    patternElement.pattern = patternToRegExp(name);
                    patternElements.push(patternElement);
                }
                else {
                    elements[name] = element;
                }
            });
        };
        const setValidElements = (validElements) => {
            // Clear any existing rules. Note that since `elements` is exposed we can't
            // overwrite it, so instead we delete all the properties
            patternElements = [];
            each$e(keys(elements), (name) => {
                delete elements[name];
            });
            addValidElements(validElements);
        };
        const addCustomElement = (name, spec) => {
            // Flush cached items since we are altering the default maps
            delete mapCache.text_block_elements;
            delete mapCache.block_elements;
            const inline = spec.extends ? !isBlock(spec.extends) : false;
            const cloneName = spec.extends;
            children[name] = cloneName ? children[cloneName] : {};
            customElementsMap[name] = cloneName ?? name;
            // Treat all custom elements as being non-empty by default
            nonEmptyElementsMap[name.toUpperCase()] = {};
            nonEmptyElementsMap[name] = {};
            // If it's not marked as inline then add it to valid block elements
            if (!inline) {
                blockElementsMap[name.toUpperCase()] = {};
                blockElementsMap[name] = {};
            }
            // Add elements clone if needed
            if (cloneName && !elements[name] && elements[cloneName]) {
                const customRule = deepCloneElementRule(elements[cloneName]);
                delete customRule.removeEmptyAttrs;
                delete customRule.removeEmpty;
                elements[name] = customRule;
            }
            else {
                elements[name] = { attributesOrder: [], attributes: {} };
            }
            // Add custom attributes
            if (isArray$1(spec.attributes)) {
                const processAttrName = (name) => {
                    customRule.attributesOrder.push(name);
                    customRule.attributes[name] = {};
                };
                const customRule = elements[name] ?? {};
                delete customRule.attributesDefault;
                delete customRule.attributesForced;
                delete customRule.attributePatterns;
                delete customRule.attributesRequired;
                customRule.attributesOrder = [];
                customRule.attributes = {};
                each$e(spec.attributes, (attrName) => {
                    const globalAttrs = getGlobalAttributeSet(schemaType);
                    parseValidChild(attrName).each(({ preset, name }) => {
                        if (preset) {
                            if (name === 'global') {
                                each$e(globalAttrs, processAttrName);
                            }
                        }
                        else {
                            processAttrName(name);
                        }
                    });
                });
                elements[name] = customRule;
            }
            // Add custom pad empty rule
            if (isBoolean(spec.padEmpty)) {
                const customRule = elements[name] ?? {};
                customRule.paddEmpty = spec.padEmpty;
                elements[name] = customRule;
            }
            // Add custom children
            if (isArray$1(spec.children)) {
                const customElementChildren = {};
                const processNodeName = (name) => {
                    customElementChildren[name] = {};
                };
                const processPreset = (name) => {
                    getElementsPreset(schemaType, name).each((names) => {
                        each$e(names, processNodeName);
                    });
                };
                each$e(spec.children, (child) => {
                    parseValidChild(child).each(({ preset, name }) => {
                        if (preset) {
                            processPreset(name);
                        }
                        else {
                            processNodeName(name);
                        }
                    });
                });
                children[name] = customElementChildren;
            }
            // Add custom elements at extends positions
            if (cloneName) {
                each$d(children, (element, elmName) => {
                    if (element[cloneName]) {
                        children[elmName] = element = extend$2({}, children[elmName]);
                        element[name] = element[cloneName];
                    }
                });
            }
        };
        const addCustomElementsFromString = (customElements) => {
            each$e(parseCustomElementsRules(customElements ?? ''), ({ name, cloneName }) => {
                addCustomElement(name, { extends: cloneName });
            });
        };
        const addComponentUrl = (elementName, componentUrl) => {
            componentUrls[elementName] = componentUrl;
        };
        const addCustomElements = (customElements) => {
            if (isObject(customElements)) {
                each$d(customElements, (spec, name) => {
                    const componentUrl = spec.componentUrl;
                    if (isString(componentUrl)) {
                        addComponentUrl(name, componentUrl);
                    }
                    addCustomElement(name, spec);
                });
            }
            else if (isString(customElements)) {
                addCustomElementsFromString(customElements);
            }
        };
        // Adds valid children to the schema object
        const addValidChildren = (validChildren) => {
            each$e(parseValidChildrenRules(validChildren ?? ''), ({ operation, name, validChildren }) => {
                const parent = operation === 'replace' ? { '#comment': {} } : children[name];
                const processNodeName = (name) => {
                    if (operation === 'remove') {
                        delete parent[name];
                    }
                    else {
                        parent[name] = {};
                    }
                };
                const processPreset = (name) => {
                    getElementsPreset(schemaType, name).each((names) => {
                        each$e(names, processNodeName);
                    });
                };
                each$e(validChildren, ({ preset, name }) => {
                    if (preset) {
                        processPreset(name);
                    }
                    else {
                        processNodeName(name);
                    }
                });
                children[name] = parent;
            });
        };
        const getElementRule = (name) => {
            const element = elements[name];
            // Exact match found
            if (element) {
                return element;
            }
            // No exact match then try the patterns
            let i = patternElements.length;
            while (i--) {
                const patternElement = patternElements[i];
                if (patternElement.pattern.test(name)) {
                    return patternElement;
                }
            }
            return undefined;
        };
        const setup = () => {
            if (!settings.valid_elements) {
                // No valid elements defined then clone the elements from the schema spec
                each$b(schemaItems, (element, name) => {
                    elements[name] = {
                        attributes: element.attributes,
                        attributesOrder: element.attributesOrder
                    };
                    children[name] = element.children;
                });
                // Prefer strong/em over b/i
                each$b(split$1('strong/b em/i'), (item) => {
                    const items = split$1(item, '/');
                    elements[items[1]].outputName = items[0];
                });
                // Add default alt attribute for images, removed since alt="" is treated as presentational.
                // elements.img.attributesDefault = [{name: 'alt', value: ''}];
                // By default,
                // - padd the text inline element if it is empty and also a child of an empty root block
                // - in all other cases, remove the text inline element if it is empty
                each$b(textInlineElementsMap, (_val, name) => {
                    if (elements[name]) {
                        if (settings.padd_empty_block_inline_children) {
                            elements[name].paddInEmptyBlock = true;
                        }
                        elements[name].removeEmpty = true;
                    }
                });
                // Remove these if they are empty by default
                each$b(split$1('ol ul blockquote a table tbody'), (name) => {
                    if (elements[name]) {
                        elements[name].removeEmpty = true;
                    }
                });
                // Padd these by default
                each$b(split$1('p h1 h2 h3 h4 h5 h6 th td pre div address caption li summary'), (name) => {
                    if (elements[name]) {
                        elements[name].paddEmpty = true;
                    }
                });
                // Remove these if they have no attributes
                each$b(split$1('span'), (name) => {
                    elements[name].removeEmptyAttrs = true;
                });
                // Remove these by default
                // TODO: Reenable in 4.1
                /* each(split('script style'), function(name) {
                 delete elements[name];
                 });*/
            }
            else {
                setValidElements(settings.valid_elements);
                each$b(schemaItems, (element, name) => {
                    children[name] = element.children;
                });
            }
            // Opt in is done with options like `extended_valid_elements`
            delete elements.svg;
            addCustomElements(settings.custom_elements);
            addValidChildren(settings.valid_children);
            addValidElements(settings.extended_valid_elements);
            // Todo: Remove this when we fix list handling to be valid
            addValidChildren('+ol[ul|ol],+ul[ul|ol]');
            // Some elements are not valid by themselves - require parents
            each$b({
                dd: 'dl',
                dt: 'dl',
                li: 'ul ol',
                td: 'tr',
                th: 'tr',
                tr: 'tbody thead tfoot',
                tbody: 'table',
                thead: 'table',
                tfoot: 'table',
                legend: 'fieldset',
                area: 'map',
                param: 'video audio object'
            }, (parents, item) => {
                if (elements[item]) {
                    elements[item].parentsRequired = split$1(parents);
                }
            });
            // Delete invalid elements
            if (settings.invalid_elements) {
                each$b(explode$2(settings.invalid_elements), (item) => {
                    if (elements[item]) {
                        delete elements[item];
                    }
                });
            }
            // If the user didn't allow span only allow internal spans
            if (!getElementRule('span')) {
                addValidElements('span[!data-mce-type|*]');
            }
        };
        /**
         * Name/value map object with valid parents and children to those parents.
         *
         * @field children
         * @type Object
         * @example
         * children = {
         *    div: { p:{}, h1:{} }
         * };
         */
        /**
         * Name/value map object with valid styles for each element.
         *
         * @method getValidStyles
         * @type Object
         */
        const getValidStyles = constant(validStyles);
        /**
         * Name/value map object with valid styles for each element.
         *
         * @method getInvalidStyles
         * @type Object
         */
        const getInvalidStyles = constant(invalidStyles);
        /**
         * Name/value map object with valid classes for each element.
         *
         * @method getValidClasses
         * @type Object
         */
        const getValidClasses = constant(validClasses);
        /**
         * Returns a map with boolean attributes.
         *
         * @method getBoolAttrs
         * @return {Object} Name/value lookup map for boolean attributes.
         */
        const getBoolAttrs = constant(boolAttrMap);
        /**
         * Returns a map with block elements.
         *
         * @method getBlockElements
         * @return {Object} Name/value lookup map for block elements.
         */
        const getBlockElements = constant(blockElementsMap);
        /**
         * Returns a map with text block elements. For example: <code>&#60;p&#62;</code>, <code>&#60;h1&#62;</code> to <code>&#60;h6&#62;</code>, <code>&#60;div&#62;</code> or <code>&#60;address&#62;</code>.
         *
         * @method getTextBlockElements
         * @return {Object} Name/value lookup map for block elements.
         */
        const getTextBlockElements = constant(textBlockElementsMap);
        /**
         * Returns a map of inline text format nodes. For example: <code>&#60;strong&#62;</code>, <code>&#60;span&#62;</code> or <code>&#60;ins&#62;</code>.
         *
         * @method getTextInlineElements
         * @return {Object} Name/value lookup map for text format elements.
         */
        const getTextInlineElements = constant(textInlineElementsMap);
        /**
         * Returns a map with void elements. For example: <code>&#60;br&#62;</code> or <code>&#60;img&#62;</code>.
         *
         * @method getVoidElements
         * @return {Object} Name/value lookup map for void elements.
         */
        const getVoidElements = constant(Object.seal(voidElementsMap));
        /**
         * Returns a map with self closing tags. For example: <code>&#60;li&#62;</code>.
         *
         * @method getSelfClosingElements
         * @return {Object} Name/value lookup map for self closing tags elements.
         */
        const getSelfClosingElements = constant(selfClosingElementsMap);
        /**
         * Returns a map with elements that should be treated as contents regardless if it has text
         * content in them or not. For example: <code>&#60;td&#62;</code>, <code>&#60;video&#62;</code> or <code>&#60;img&#62;</code>.
         *
         * @method getNonEmptyElements
         * @return {Object} Name/value lookup map for non empty elements.
         */
        const getNonEmptyElements = constant(nonEmptyElementsMap);
        /**
         * Returns a map with elements that the caret should be moved in front of after enter is
         * pressed.
         *
         * @method getMoveCaretBeforeOnEnterElements
         * @return {Object} Name/value lookup map for elements to place the caret in front of.
         */
        const getMoveCaretBeforeOnEnterElements = constant(moveCaretBeforeOnEnterElementsMap);
        /**
         * Returns a map with elements where white space is to be preserved. For example: <code>&#60;pre&#62;</code> or <code>&#60;script&#62;</code>.
         *
         * @method getWhitespaceElements
         * @return {Object} Name/value lookup map for white space elements.
         */
        const getWhitespaceElements = constant(whitespaceElementsMap);
        /**
         * Returns a map with elements that should be treated as transparent.
         *
         * @method getTransparentElements
         * @return {Object} Name/value lookup map for special elements.
         */
        const getTransparentElements = constant(transparentElementsMap);
        const getWrapBlockElements = constant(wrapBlockElementsMap);
        /**
         * Returns a map with special elements. These are elements that needs to be parsed
         * in a special way such as script, style, textarea etc. The map object values
         * are regexps used to find the end of the element.
         *
         * @method getSpecialElements
         * @return {Object} Name/value lookup map for special elements.
         */
        const getSpecialElements = constant(Object.seal(specialElements));
        /**
         * Returns true/false if the specified element and it's child is valid or not
         * according to the schema.
         *
         * @method isValidChild
         * @param {String} name Element name to check for.
         * @param {String} child Element child to verify.
         * @return {Boolean} True/false if the element is a valid child of the specified parent.
         */
        const isValidChild = (name, child) => {
            const parent = children[name.toLowerCase()];
            return !!(parent && parent[child.toLowerCase()]);
        };
        /**
         * Returns true/false if the specified element name and optional attribute is
         * valid according to the schema.
         *
         * @method isValid
         * @param {String} name Name of element to check.
         * @param {String} attr Optional attribute name to check for.
         * @return {Boolean} True/false if the element and attribute is valid.
         */
        const isValid = (name, attr) => {
            const rule = getElementRule(name);
            // Check if it's a valid element
            if (rule) {
                if (attr) {
                    // Check if attribute name exists
                    if (rule.attributes[attr]) {
                        return true;
                    }
                    // Check if attribute matches a regexp pattern
                    const attrPatterns = rule.attributePatterns;
                    if (attrPatterns) {
                        let i = attrPatterns.length;
                        while (i--) {
                            if (attrPatterns[i].pattern.test(attr)) {
                                return true;
                            }
                        }
                    }
                }
                else {
                    return true;
                }
            }
            // No match
            return false;
        };
        const isBlock = (name) => has$2(getBlockElements(), name);
        // Check if name starts with # to detect non-element node names like #text and #comment
        const isInline = (name) => !startsWith(name, '#') && isValid(name) && !isBlock(name);
        const isWrapper = (name) => has$2(getWrapBlockElements(), name) || isInline(name);
        /**
         * Returns true/false if the specified element is valid or not
         * according to the schema.
         *
         * @method getElementRule
         * @param {String} name Element name to check for.
         * @return {Object} Element object or undefined if the element isn't valid.
         */
        /**
         * Returns an map object of all custom elements.
         *
         * @method getCustomElements
         * @return {Object} Name/value map object of all custom elements.
         */
        const getCustomElements = constant(customElementsMap);
        /**
         * Parses a valid elements string and adds it to the schema. The valid elements
         * format is for example <code>element[attr=default|otherattr]</code>.
         * Existing rules will be replaced with the ones specified, so this extends the schema.
         *
         * @method addValidElements
         * @param {String} valid_elements String in the valid elements format to be parsed.
         */
        /**
         * Parses a valid elements string and sets it to the schema. The valid elements
         * format is for example <code>element[attr=default|otherattr]</code>.
         * Existing rules will be replaced with the ones specified, so this extends the schema.
         *
         * @method setValidElements
         * @param {String} valid_elements String in the valid elements format to be parsed.
         */
        /**
         * Adds custom non-HTML elements to the schema. For more information about adding custom elements see:
         * <a href="https://www.tiny.cloud/docs/tinymce/latest/content-filtering/#custom_elements">custom_elements</a>
         *
         * @method addCustomElements
         * @param {String/Object} custom_elements Comma separated list or record of custom elements to add.
         */
        /**
         * Parses a valid children string and adds them to the schema structure. The valid children
         * format is for example <code>element[child1|child2]</code>.
         *
         * @method addValidChildren
         * @param {String} valid_children Valid children elements string to parse
         */
        /**
         * Returns an object of all custom elements that have component URLs.
         *
         * @method getComponentUrls
         * @return {Object} Object with where key is the component and the value is the url for that component.
         */
        const getComponentUrls = constant(componentUrls);
        setup();
        return {
            type: schemaType,
            children,
            elements,
            getValidStyles,
            getValidClasses,
            getBlockElements,
            getInvalidStyles,
            getVoidElements,
            getTextBlockElements,
            getTextInlineElements,
            getBoolAttrs,
            getElementRule,
            getSelfClosingElements,
            getNonEmptyElements,
            getMoveCaretBeforeOnEnterElements,
            getWhitespaceElements,
            getTransparentElements,
            getSpecialElements,
            getComponentUrls,
            isValidChild,
            isValid,
            isBlock,
            isInline,
            isWrapper,
            getCustomElements,
            addValidElements,
            setValidElements,
            addCustomElements,
            addValidChildren,
        };
    };

    const hexColour = (value) => ({
        value: normalizeHex(value)
    });
    const normalizeHex = (hex) => removeLeading(hex, '#').toUpperCase();
    const toHex = (component) => {
        const hex = component.toString(16);
        return (hex.length === 1 ? '0' + hex : hex).toUpperCase();
    };
    const fromRgba = (rgbaColour) => {
        const value = toHex(rgbaColour.red) + toHex(rgbaColour.green) + toHex(rgbaColour.blue);
        return hexColour(value);
    };

    const rgbRegex = /^\s*rgb\s*\(\s*(\d+)\s*[,\s]\s*(\d+)\s*[,\s]\s*(\d+)\s*\)\s*$/i;
    // This regex will match rgba(0, 0, 0, 0.5) or rgba(0, 0, 0, 50%) , or without commas
    const rgbaRegex = /^\s*rgba\s*\(\s*(\d+)\s*[,\s]\s*(\d+)\s*[,\s]\s*(\d+)\s*[,\s]\s*((?:\d?\.\d+|\d+)%?)\s*\)\s*$/i;
    const rgbaColour = (red, green, blue, alpha) => ({
        red,
        green,
        blue,
        alpha
    });
    const fromStringValues = (red, green, blue, alpha) => {
        const r = parseInt(red, 10);
        const g = parseInt(green, 10);
        const b = parseInt(blue, 10);
        const a = parseFloat(alpha);
        return rgbaColour(r, g, b, a);
    };
    const getColorFormat = (colorString) => {
        if (rgbRegex.test(colorString)) {
            return 'rgb';
        }
        else if (rgbaRegex.test(colorString)) {
            return 'rgba';
        }
        return 'other';
    };
    const fromString = (rgbaString) => {
        const rgbMatch = rgbRegex.exec(rgbaString);
        if (rgbMatch !== null) {
            return Optional.some(fromStringValues(rgbMatch[1], rgbMatch[2], rgbMatch[3], '1'));
        }
        const rgbaMatch = rgbaRegex.exec(rgbaString);
        if (rgbaMatch !== null) {
            return Optional.some(fromStringValues(rgbaMatch[1], rgbaMatch[2], rgbaMatch[3], rgbaMatch[4]));
        }
        return Optional.none();
    };
    const toString = (rgba) => `rgba(${rgba.red},${rgba.green},${rgba.blue},${rgba.alpha})`;

    const rgbaToHexString = (color) => fromString(color)
        .map(fromRgba)
        .map((h) => '#' + h.value)
        .getOr(color);

    /**
     * This class is used to parse CSS styles. It also compresses styles to reduce the output size.
     *
     * @class tinymce.html.Styles
     * @version 3.4
     * @example
     * const Styles = tinymce.html.Styles({
     *   url_converter: (url) => {
     *     return url;
     *   }
     * });
     *
     * styles = Styles.parse('border: 1px solid red');
     * styles.color = 'red';
     *
     * console.log(tinymce.html.Styles().serialize(styles));
     */
    const Styles = (settings = {}, schema) => {
        /* jshint maxlen:255 */
        /* eslint max-len:0 */
        const urlOrStrRegExp = /(?:url(?:(?:\(\s*\"([^\"]+)\"\s*\))|(?:\(\s*\'([^\']+)\'\s*\))|(?:\(\s*([^)\s]+)\s*\))))|(?:\'([^\']+)\')|(?:\"([^\"]+)\")/gi;
        const styleRegExp = /\s*([^:]+):\s*([^;]+);?/g;
        const trimRightRegExp = /\s+$/;
        const encodingLookup = {};
        let validStyles;
        let invalidStyles;
        const invisibleChar = zeroWidth;
        if (schema) {
            validStyles = schema.getValidStyles();
            invalidStyles = schema.getInvalidStyles();
        }
        const encodingItems = (`\\" \\' \\; \\: ; : ` + invisibleChar).split(' ');
        for (let i = 0; i < encodingItems.length; i++) {
            encodingLookup[encodingItems[i]] = invisibleChar + i;
            encodingLookup[invisibleChar + i] = encodingItems[i];
        }
        // eslint-disable-next-line consistent-this
        const self = {
            /**
             * Parses the specified style value into an object collection. This parser will also
             * merge and remove any redundant items that browsers might have added. URLs inside
             * the styles will also be converted to absolute/relative based on the settings.
             *
             * @method parse
             * @param {String} css Style value to parse. For example: `border:1px solid red;`
             * @return {Object} Object representation of that style. For example: `{ border: '1px solid red' }`
             */
            parse: (css) => {
                const styles = {};
                let isEncoded = false;
                const urlConverter = settings.url_converter;
                const urlConverterScope = settings.url_converter_scope || self;
                const compress = (prefix, suffix, noJoin) => {
                    const top = styles[prefix + '-top' + suffix];
                    if (!top) {
                        return;
                    }
                    const right = styles[prefix + '-right' + suffix];
                    if (!right) {
                        return;
                    }
                    const bottom = styles[prefix + '-bottom' + suffix];
                    if (!bottom) {
                        return;
                    }
                    const left = styles[prefix + '-left' + suffix];
                    if (!left) {
                        return;
                    }
                    const box = [top, right, bottom, left];
                    let i = box.length - 1;
                    while (i--) {
                        if (box[i] !== box[i + 1]) {
                            break;
                        }
                    }
                    if (i > -1 && noJoin) {
                        return;
                    }
                    styles[prefix + suffix] = i === -1 ? box[0] : box.join(' ');
                    delete styles[prefix + '-top' + suffix];
                    delete styles[prefix + '-right' + suffix];
                    delete styles[prefix + '-bottom' + suffix];
                    delete styles[prefix + '-left' + suffix];
                };
                /**
                 * Checks if the specific style can be compressed in other words if all border-width are equal.
                 */
                const canCompress = (key) => {
                    const value = styles[key];
                    if (!value) {
                        return;
                    }
                    // Make sure not to split values like 'rgb(100, 50, 100);
                    const values = value.indexOf(',') > -1 ? [value] : value.split(' ');
                    let i = values.length;
                    while (i--) {
                        if (values[i] !== values[0]) {
                            return false;
                        }
                    }
                    styles[key] = values[0];
                    return true;
                };
                /**
                 * Compresses multiple styles into one style.
                 */
                const compress2 = (target, a, b, c) => {
                    if (!canCompress(a)) {
                        return;
                    }
                    if (!canCompress(b)) {
                        return;
                    }
                    if (!canCompress(c)) {
                        return;
                    }
                    // Compress
                    styles[target] = styles[a] + ' ' + styles[b] + ' ' + styles[c];
                    delete styles[a];
                    delete styles[b];
                    delete styles[c];
                };
                // Encodes the specified string by replacing all \" \' ; : with _<num>
                const encode = (str) => {
                    isEncoded = true;
                    return encodingLookup[str];
                };
                // Decodes the specified string by replacing all _<num> with it's original value \" \' etc
                // It will also decode the \" \' if keepSlashes is set to false or omitted
                const decode = (str, keepSlashes) => {
                    if (isEncoded) {
                        str = str.replace(/\uFEFF[0-9]/g, (str) => {
                            return encodingLookup[str];
                        });
                    }
                    if (!keepSlashes) {
                        str = str.replace(/\\([\'\";:])/g, '$1');
                    }
                    return str;
                };
                const decodeSingleHexSequence = (escSeq) => {
                    return String.fromCharCode(parseInt(escSeq.slice(1), 16));
                };
                const decodeHexSequences = (value) => {
                    return value.replace(/\\[0-9a-f]+/gi, decodeSingleHexSequence);
                };
                const processUrl = (match, url, url2, url3, str, str2) => {
                    str = str || str2;
                    if (str) {
                        str = decode(str);
                        // Force strings into single quote format
                        return `'` + str.replace(/\'/g, `\\'`) + `'`;
                    }
                    url = decode(url || url2 || url3 || '');
                    if (!settings.allow_script_urls) {
                        const scriptUrl = url.replace(/[\s\r\n]+/g, '');
                        if (/(java|vb)script:/i.test(scriptUrl)) {
                            return '';
                        }
                        if (!settings.allow_svg_data_urls && /^data:image\/svg/i.test(scriptUrl)) {
                            return '';
                        }
                    }
                    // Convert the URL to relative/absolute depending on config
                    if (urlConverter) {
                        url = urlConverter.call(urlConverterScope, url, 'style');
                    }
                    // Output new URL format
                    return `url('` + url.replace(/\'/g, `\\'`) + `')`;
                };
                if (css) {
                    css = css.replace(/[\u0000-\u001F]/g, '');
                    // Encode \" \' % and ; and : inside strings so they don't interfere with the style parsing
                    css = css.replace(/\\[\"\';:\uFEFF]/g, encode).replace(/\"[^\"]+\"|\'[^\']+\'/g, (str) => {
                        return str.replace(/[;:]/g, encode);
                    });
                    // Parse styles
                    let matches;
                    while ((matches = styleRegExp.exec(css))) {
                        styleRegExp.lastIndex = matches.index + matches[0].length;
                        let name = matches[1].replace(trimRightRegExp, '').toLowerCase();
                        let value = matches[2].replace(trimRightRegExp, '');
                        if (name && value) {
                            // Decode escaped sequences like \65 -> e
                            name = decodeHexSequences(name);
                            value = decodeHexSequences(value);
                            // Skip properties with double quotes and sequences like \" \' in their names
                            // See 'mXSS Attacks: Attacking well-secured Web-Applications by using innerHTML Mutations'
                            // https://cure53.de/fp170.pdf
                            if (name.indexOf(invisibleChar) !== -1 || name.indexOf('"') !== -1) {
                                continue;
                            }
                            // Don't allow behavior name or expression/comments within the values
                            if (!settings.allow_script_urls && (name === 'behavior' || /expression\s*\(|\/\*|\*\//.test(value))) {
                                continue;
                            }
                            // Opera will produce 700 instead of bold in their style values
                            if (name === 'font-weight' && value === '700') {
                                value = 'bold';
                            }
                            else if (name === 'color' || name === 'background-color') { // Lowercase colors like RED
                                value = value.toLowerCase();
                            }
                            // Convert RGB colors to HEX
                            if (getColorFormat(value) === 'rgb') {
                                fromString(value).each((rgba) => {
                                    value = rgbaToHexString(toString(rgba)).toLowerCase();
                                });
                            }
                            // Convert URLs and force them into url('value') format
                            value = value.replace(urlOrStrRegExp, processUrl);
                            styles[name] = isEncoded ? decode(value, true) : value;
                        }
                    }
                    // Compress the styles to reduce it's size for example IE will expand styles
                    compress('border', '', true);
                    compress('border', '-width');
                    compress('border', '-color');
                    compress('border', '-style');
                    compress('padding', '');
                    compress('margin', '');
                    compress2('border', 'border-width', 'border-style', 'border-color');
                    // Remove pointless border, IE produces these
                    if (styles.border === 'medium none') {
                        delete styles.border;
                    }
                    // IE 11 will produce a border-image: none when getting the style attribute from <p style="border: 1px solid red"></p>
                    // So let us assume it shouldn't be there
                    if (styles['border-image'] === 'none') {
                        delete styles['border-image'];
                    }
                }
                return styles;
            },
            /**
             * Serializes the specified style object into a string.
             *
             * @method serialize
             * @param {Object} styles Object to serialize as string. For example: `{ border: '1px solid red' }`
             * @param {String} elementName Optional element name, if specified only the styles that matches the schema will be serialized.
             * @return {String} String representation of the style object. For example: `border: 1px solid red`
             */
            serialize: (styles, elementName) => {
                let css = '';
                const serializeStyles = (elemName, validStyleList) => {
                    const styleList = validStyleList[elemName];
                    if (styleList) {
                        for (let i = 0, l = styleList.length; i < l; i++) {
                            const name = styleList[i];
                            const value = styles[name];
                            if (value) {
                                css += (css.length > 0 ? ' ' : '') + name + ': ' + value + ';';
                            }
                        }
                    }
                };
                const isValid = (name, elemName) => {
                    if (!invalidStyles || !elemName) {
                        return true;
                    }
                    let styleMap = invalidStyles['*'];
                    if (styleMap && styleMap[name]) {
                        return false;
                    }
                    styleMap = invalidStyles[elemName];
                    return !(styleMap && styleMap[name]);
                };
                // Serialize styles according to schema
                if (elementName && validStyles) {
                    // Serialize global styles and element specific styles
                    serializeStyles('*', validStyles);
                    serializeStyles(elementName, validStyles);
                }
                else {
                    // Output the styles in the order they are inside the object
                    each$d(styles, (value, name) => {
                        if (value && isValid(name, elementName)) {
                            css += (css.length > 0 ? ' ' : '') + name + ': ' + value + ';';
                        }
                    });
                }
                return css;
            }
        };
        return self;
    };

    // Note: The values here aren't used. This is just used as a hash map to see if the key exists
    const deprecated = {
        keyLocation: true,
        layerX: true,
        layerY: true,
        returnValue: true,
        webkitMovementX: true,
        webkitMovementY: true,
        keyIdentifier: true,
        mozPressure: true
    };
    // Note: We can't rely on `instanceof` here as it won't work if the event was fired from another window.
    // Additionally, the constructor name might be `MouseEvent` or similar so we can't rely on the constructor name.
    const isNativeEvent = (event) => event instanceof Event || isFunction(event.initEvent);
    // Checks if it is our own isDefaultPrevented function
    const hasIsDefaultPrevented = (event) => event.isDefaultPrevented === always || event.isDefaultPrevented === never;
    // An event needs normalizing if it doesn't have the prevent default function or if it's a native event
    const needsNormalizing = (event) => isNullable(event.preventDefault) || isNativeEvent(event);
    const clone$2 = (originalEvent, data) => {
        const event = data ?? {};
        // Copy all properties from the original event
        for (const name in originalEvent) {
            // Some properties are deprecated and produces a warning so don't include them
            if (!has$2(deprecated, name)) {
                event[name] = originalEvent[name];
            }
        }
        // The composed path can't be cloned, so delegate instead
        if (isNonNullable(originalEvent.composedPath)) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            event.composedPath = () => originalEvent.composedPath();
        }
        // The getModifierState won't work when cloned, so delegate instead
        if (isNonNullable(originalEvent.getModifierState)) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            event.getModifierState = (keyArg) => originalEvent.getModifierState(keyArg);
        }
        // The getTargetRanges won't work when cloned, so delegate instead
        if (isNonNullable(originalEvent.getTargetRanges)) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            event.getTargetRanges = () => originalEvent.getTargetRanges();
        }
        return event;
    };
    const normalize$3 = (type, originalEvent, fallbackTarget, data) => {
        const event = clone$2(originalEvent, data);
        event.type = type;
        // Normalize target IE uses srcElement
        if (isNullable(event.target)) {
            event.target = event.srcElement ?? fallbackTarget;
        }
        if (needsNormalizing(originalEvent)) {
            // Add preventDefault method
            event.preventDefault = () => {
                event.defaultPrevented = true;
                event.isDefaultPrevented = always;
                // Execute preventDefault on the original event object
                if (isFunction(originalEvent.preventDefault)) {
                    originalEvent.preventDefault();
                }
            };
            // Add stopPropagation
            event.stopPropagation = () => {
                event.cancelBubble = true;
                event.isPropagationStopped = always;
                // Execute stopPropagation on the original event object
                if (isFunction(originalEvent.stopPropagation)) {
                    originalEvent.stopPropagation();
                }
            };
            // Add stopImmediatePropagation
            event.stopImmediatePropagation = () => {
                event.isImmediatePropagationStopped = always;
                event.stopPropagation();
            };
            // Add event delegation states
            if (!hasIsDefaultPrevented(event)) {
                event.isDefaultPrevented = event.defaultPrevented === true ? always : never;
                event.isPropagationStopped = event.cancelBubble === true ? always : never;
                event.isImmediatePropagationStopped = never;
            }
        }
        return event;
    };

    /**
     * This class wraps the browsers native event logic with more convenient methods.
     *
     * @class tinymce.dom.EventUtils
     */
    const eventExpandoPrefix = 'mce-data-';
    const mouseEventRe = /^(?:mouse|contextmenu)|click/;
    /**
     * Binds a native event to a callback on the speified target.
     */
    const addEvent = (target, name, callback, capture) => {
        target.addEventListener(name, callback, capture || false);
    };
    /**
     * Unbinds a native event callback on the specified target.
     */
    const removeEvent = (target, name, callback, capture) => {
        target.removeEventListener(name, callback, capture || false);
    };
    const isMouseEvent = (event) => isNonNullable(event) && mouseEventRe.test(event.type);
    /**
     * Normalizes a native event object or just adds the event specific methods on a custom event.
     */
    const fix = (originalEvent, data) => {
        const event = normalize$3(originalEvent.type, originalEvent, document, data);
        // Calculate pageX/Y if missing and clientX/Y available
        if (isMouseEvent(originalEvent) && isUndefined(originalEvent.pageX) && !isUndefined(originalEvent.clientX)) {
            const eventDoc = event.target.ownerDocument || document;
            const doc = eventDoc.documentElement;
            const body = eventDoc.body;
            const mouseEvent = event;
            mouseEvent.pageX = originalEvent.clientX + (doc && doc.scrollLeft || body && body.scrollLeft || 0) -
                (doc && doc.clientLeft || body && body.clientLeft || 0);
            mouseEvent.pageY = originalEvent.clientY + (doc && doc.scrollTop || body && body.scrollTop || 0) -
                (doc && doc.clientTop || body && body.clientTop || 0);
        }
        return event;
    };
    /**
     * Bind a DOMContentLoaded event across browsers and executes the callback once the page DOM is initialized.
     * It will also set/check the domLoaded state of the event_utils instance so ready isn't called multiple times.
     */
    const bindOnReady = (win, callback, eventUtils) => {
        const doc = win.document, event = { type: 'ready' };
        if (eventUtils.domLoaded) {
            callback(event);
            return;
        }
        const isDocReady = () => {
            // Check complete or interactive state if there is a body
            // element on some iframes IE 8 will produce a null body
            return doc.readyState === 'complete' || (doc.readyState === 'interactive' && doc.body);
        };
        // Gets called when the DOM is ready
        const readyHandler = () => {
            removeEvent(win, 'DOMContentLoaded', readyHandler);
            removeEvent(win, 'load', readyHandler);
            if (!eventUtils.domLoaded) {
                eventUtils.domLoaded = true;
                callback(event);
            }
            // Clean memory for IE
            win = null;
        };
        if (isDocReady()) {
            readyHandler();
        }
        else {
            addEvent(win, 'DOMContentLoaded', readyHandler);
        }
        // Fallback if any of the above methods should fail for some odd reason
        if (!eventUtils.domLoaded) {
            addEvent(win, 'load', readyHandler);
        }
    };
    /**
     * This class enables you to bind/unbind native events to elements and normalize it's behavior across browsers.
     */
    class EventUtils {
        static Event = new EventUtils();
        // State if the DOMContentLoaded was executed or not
        domLoaded = false;
        events = {};
        expando;
        hasFocusIn;
        count = 1;
        constructor() {
            this.expando = eventExpandoPrefix + (+new Date()).toString(32);
            this.hasFocusIn = 'onfocusin' in document.documentElement;
            this.count = 1;
        }
        bind(target, names, callback, scope) {
            const self = this;
            let callbackList;
            const win = window;
            // Native event handler function patches the event and executes the callbacks for the expando
            const defaultNativeHandler = (evt) => {
                self.executeHandlers(fix(evt || win.event), id);
            };
            // Don't bind to text nodes or comments
            if (!target || isText$b(target) || isComment(target)) {
                return callback;
            }
            // Create or get events id for the target
            let id;
            if (!target[self.expando]) {
                id = self.count++;
                target[self.expando] = id;
                self.events[id] = {};
            }
            else {
                id = target[self.expando];
            }
            // Setup the specified scope or use the target as a default
            scope = scope || target;
            // Split names and bind each event, enables you to bind multiple events with one call
            const namesList = names.split(' ');
            let i = namesList.length;
            while (i--) {
                let name = namesList[i];
                let nativeHandler = defaultNativeHandler;
                let capture = false;
                let fakeName = false;
                // Use ready instead of DOMContentLoaded
                if (name === 'DOMContentLoaded') {
                    name = 'ready';
                }
                // DOM is already ready
                if (self.domLoaded && name === 'ready' && target.readyState === 'complete') {
                    callback.call(scope, fix({ type: name }));
                    continue;
                }
                // Fake bubbling of focusin/focusout
                if (!self.hasFocusIn && (name === 'focusin' || name === 'focusout')) {
                    capture = true;
                    fakeName = name === 'focusin' ? 'focus' : 'blur';
                    nativeHandler = (evt) => {
                        const event = fix(evt || win.event);
                        event.type = event.type === 'focus' ? 'focusin' : 'focusout';
                        self.executeHandlers(event, id);
                    };
                }
                // Setup callback list and bind native event
                callbackList = self.events[id][name];
                if (!callbackList) {
                    self.events[id][name] = callbackList = [{ func: callback, scope }];
                    callbackList.fakeName = fakeName;
                    callbackList.capture = capture;
                    // callbackList.callback = callback;
                    // Add the nativeHandler to the callback list so that we can later unbind it
                    callbackList.nativeHandler = nativeHandler;
                    // Check if the target has native events support
                    if (name === 'ready') {
                        bindOnReady(target, nativeHandler, self);
                    }
                    else {
                        addEvent(target, fakeName || name, nativeHandler, capture);
                    }
                }
                else {
                    if (name === 'ready' && self.domLoaded) {
                        callback(fix({ type: name }));
                    }
                    else {
                        // If it already has an native handler then just push the callback
                        callbackList.push({ func: callback, scope });
                    }
                }
            }
            target = callbackList = null; // Clean memory for IE
            return callback;
        }
        unbind(target, names, callback) {
            // Don't bind to text nodes or comments
            if (!target || isText$b(target) || isComment(target)) {
                return this;
            }
            // Unbind event or events if the target has the expando
            const id = target[this.expando];
            if (id) {
                let eventMap = this.events[id];
                // Specific callback
                if (names) {
                    const namesList = names.split(' ');
                    let i = namesList.length;
                    while (i--) {
                        const name = namesList[i];
                        const callbackList = eventMap[name];
                        // Unbind the event if it exists in the map
                        if (callbackList) {
                            // Remove specified callback
                            if (callback) {
                                let ci = callbackList.length;
                                while (ci--) {
                                    if (callbackList[ci].func === callback) {
                                        const nativeHandler = callbackList.nativeHandler;
                                        const fakeName = callbackList.fakeName, capture = callbackList.capture;
                                        // Clone callbackList since unbind inside a callback would otherwise break the handlers loop
                                        const newCallbackList = callbackList.slice(0, ci).concat(callbackList.slice(ci + 1));
                                        newCallbackList.nativeHandler = nativeHandler;
                                        newCallbackList.fakeName = fakeName;
                                        newCallbackList.capture = capture;
                                        eventMap[name] = newCallbackList;
                                    }
                                }
                            }
                            // Remove all callbacks if there isn't a specified callback or there is no callbacks left
                            if (!callback || callbackList.length === 0) {
                                delete eventMap[name];
                                removeEvent(target, callbackList.fakeName || name, callbackList.nativeHandler, callbackList.capture);
                            }
                        }
                    }
                }
                else {
                    // All events for a specific element
                    each$d(eventMap, (callbackList, name) => {
                        removeEvent(target, callbackList.fakeName || name, callbackList.nativeHandler, callbackList.capture);
                    });
                    eventMap = {};
                }
                // Check if object is empty, if it isn't then we won't remove the expando map
                for (const name in eventMap) {
                    if (has$2(eventMap, name)) {
                        return this;
                    }
                }
                // Delete event object
                delete this.events[id];
                // Remove expando from target
                try {
                    // IE will fail here since it can't delete properties from window
                    delete target[this.expando];
                }
                catch {
                    // IE will set it to null
                    target[this.expando] = null;
                }
            }
            return this;
        }
        /**
         * Fires the specified event on the specified target.
         * <br>
         * <em>Deprecated in TinyMCE 6.0 and has been marked for removal in TinyMCE 7.0. Use <code>dispatch</code> instead.</em>
         *
         * @method fire
         * @param {Object} target Target node/window or custom object.
         * @param {String} name Event name to fire.
         * @param {Object} args Optional arguments to send to the observers.
         * @return {EventUtils} Event utils instance.
         * @deprecated Use dispatch() instead
         */
        fire(target, name, args) {
            return this.dispatch(target, name, args);
        }
        /**
         * Dispatches the specified event on the specified target.
         *
         * @method dispatch
         * @param {Node/window} target Target node/window or custom object.
         * @param {String} name Event name to dispatch.
         * @param {Object} args Optional arguments to send to the observers.
         * @return {EventUtils} Event utils instance.
         */
        dispatch(target, name, args) {
            // Don't bind to text nodes or comments
            if (!target || isText$b(target) || isComment(target)) {
                return this;
            }
            // Build event object by patching the args
            const event = fix({ type: name, target }, args);
            do {
                // Found an expando that means there is listeners to execute
                const id = target[this.expando];
                if (id) {
                    this.executeHandlers(event, id);
                }
                // Walk up the DOM
                target = target.parentNode || target.ownerDocument || target.defaultView || target.parentWindow;
            } while (target && !event.isPropagationStopped());
            return this;
        }
        /**
         * Removes all bound event listeners for the specified target. This will also remove any bound
         * listeners to child nodes within that target.
         *
         * @method clean
         * @param {Object} target Target node/window object.
         * @return {EventUtils} Event utils instance.
         */
        clean(target) {
            // Don't bind to text nodes or comments
            if (!target || isText$b(target) || isComment(target)) {
                return this;
            }
            // Unbind any element on the specified target
            if (target[this.expando]) {
                this.unbind(target);
            }
            // Target doesn't have getElementsByTagName it's probably a window object then use it's document to find the children
            if (!target.getElementsByTagName) {
                target = target.document;
            }
            // Remove events from each child element
            if (target && target.getElementsByTagName) {
                this.unbind(target);
                const children = target.getElementsByTagName('*');
                let i = children.length;
                while (i--) {
                    target = children[i];
                    if (target[this.expando]) {
                        this.unbind(target);
                    }
                }
            }
            return this;
        }
        /**
         * Destroys the event object. Call this to remove memory leaks.
         */
        destroy() {
            this.events = {};
        }
        // Legacy function for canceling events
        cancel(e) {
            if (e) {
                e.preventDefault();
                e.stopImmediatePropagation();
            }
            return false;
        }
        /**
         * Executes all event handler callbacks for a specific event.
         *
         * @private
         * @param {Event} evt Event object.
         * @param {String} id Expando id value to look for.
         */
        executeHandlers(evt, id) {
            const container = this.events[id];
            const callbackList = container && container[evt.type];
            if (callbackList) {
                for (let i = 0, l = callbackList.length; i < l; i++) {
                    const callback = callbackList[i];
                    // Check if callback exists might be removed if a unbind is called inside the callback
                    if (callback && callback.func.call(callback.scope, evt) === false) {
                        evt.preventDefault();
                    }
                    // Should we stop propagation to immediate listeners
                    if (evt.isImmediatePropagationStopped()) {
                        return;
                    }
                }
            }
        }
    }

    /**
     * Utility class for various DOM manipulation and retrieval functions.
     *
     * @class tinymce.dom.DOMUtils
     * @example
     * // Add a class to an element by id in the page
     * tinymce.DOM.addClass('someid', 'someclass');
     *
     * // Add a class to an element by id inside the editor
     * tinymce.activeEditor.dom.addClass('someid', 'someclass');
     */
    // Shorten names
    const each$a = Tools.each;
    const grep = Tools.grep;
    const internalStyleName = 'data-mce-style';
    const numericalCssMap = Tools.makeMap('fill-opacity font-weight line-height opacity orphans widows z-index zoom', ' ');
    const legacySetAttribute = (elm, name, value) => {
        if (isNullable(value) || value === '') {
            remove$9(elm, name);
        }
        else {
            set$4(elm, name, value);
        }
    };
    // Convert camel cased names back to hyphenated names
    const camelCaseToHyphens = (name) => name.replace(/[A-Z]/g, (v) => '-' + v.toLowerCase());
    const findNodeIndex = (node, normalized) => {
        let idx = 0;
        if (node) {
            for (let lastNodeType = node.nodeType, tempNode = node.previousSibling; tempNode; tempNode = tempNode.previousSibling) {
                const nodeType = tempNode.nodeType;
                // Normalize text nodes
                if (normalized && isText$b(tempNode)) {
                    if (nodeType === lastNodeType || !tempNode.data.length) {
                        continue;
                    }
                }
                idx++;
                lastNodeType = nodeType;
            }
        }
        return idx;
    };
    const updateInternalStyleAttr = (styles, elm) => {
        const rawValue = get$9(elm, 'style');
        const value = styles.serialize(styles.parse(rawValue), name(elm));
        legacySetAttribute(elm, internalStyleName, value);
    };
    const convertStyleToString = (cssValue, cssName) => {
        if (isNumber(cssValue)) {
            return has$2(numericalCssMap, cssName) ? cssValue + '' : cssValue + 'px';
        }
        else {
            return cssValue;
        }
    };
    const applyStyle$1 = ($elm, cssName, cssValue) => {
        const normalizedName = camelCaseToHyphens(cssName);
        if (isNullable(cssValue) || cssValue === '') {
            remove$7($elm, normalizedName);
        }
        else {
            set$2($elm, normalizedName, convertStyleToString(cssValue, normalizedName));
        }
    };
    const setupAttrHooks = (styles, settings, getContext) => {
        const keepValues = settings.keep_values;
        const keepUrlHook = {
            set: (elm, value, name) => {
                const sugarElm = SugarElement.fromDom(elm);
                if (isFunction(settings.url_converter) && isNonNullable(value)) {
                    value = settings.url_converter.call(settings.url_converter_scope || getContext(), String(value), name, elm);
                }
                const internalName = 'data-mce-' + name;
                legacySetAttribute(sugarElm, internalName, value);
                legacySetAttribute(sugarElm, name, value);
            },
            get: (elm, name) => {
                const sugarElm = SugarElement.fromDom(elm);
                return get$9(sugarElm, 'data-mce-' + name) || get$9(sugarElm, name);
            }
        };
        const attrHooks = {
            style: {
                set: (elm, value) => {
                    const sugarElm = SugarElement.fromDom(elm);
                    if (keepValues) {
                        legacySetAttribute(sugarElm, internalStyleName, value);
                    }
                    remove$9(sugarElm, 'style');
                    // If setting a style then delegate to the css api, otherwise
                    // this will cause issues when using a content security policy
                    if (isString(value)) {
                        setAll(sugarElm, styles.parse(value));
                    }
                },
                get: (elm) => {
                    const sugarElm = SugarElement.fromDom(elm);
                    const value = get$9(sugarElm, internalStyleName) || get$9(sugarElm, 'style');
                    return styles.serialize(styles.parse(value), name(sugarElm));
                }
            }
        };
        if (keepValues) {
            attrHooks.href = attrHooks.src = keepUrlHook;
        }
        return attrHooks;
    };
    /**
     * Constructs a new DOMUtils instance. Consult the TinyMCE Documentation for more details on settings etc for this class.
     *
     * @private
     * @constructor
     * @method DOMUtils
     * @param {Document} doc Document reference to bind the utility class to.
     * @param {settings} settings Optional settings collection.
     */
    const DOMUtils = (doc, settings = {}) => {
        const addedStyles = {};
        const win = window;
        const files = {};
        let counter = 0;
        const stdMode = true;
        const boxModel = true;
        const styleSheetLoader = instance.forElement(SugarElement.fromDom(doc), {
            contentCssCors: settings.contentCssCors,
            referrerPolicy: settings.referrerPolicy,
            crossOrigin: (url) => {
                const crossOrigin = settings.crossOrigin;
                if (isFunction(crossOrigin)) {
                    return crossOrigin(url, 'stylesheet');
                }
                else {
                    return undefined;
                }
            }
        });
        const boundEvents = [];
        const schema = settings.schema ? settings.schema : Schema({});
        const styles = Styles({
            url_converter: settings.url_converter,
            url_converter_scope: settings.url_converter_scope,
        }, settings.schema);
        const events = settings.ownEvents ? new EventUtils() : EventUtils.Event;
        const blockElementsMap = schema.getBlockElements();
        /**
         * Returns true/false if the specified element is a block element or not.
         *
         * @method isBlock
         * @param {Node/String} node Element/Node to check.
         * @return {Boolean} True/False state if the node is a block element or not.
         */
        const isBlock = (node) => {
            if (isString(node)) {
                return has$2(blockElementsMap, node);
            }
            else {
                return isElement$7(node) && (has$2(blockElementsMap, node.nodeName) || isTransparentBlock(schema, node));
            }
        };
        const get = (elm) => elm && doc && isString(elm)
            ? doc.getElementById(elm)
            : elm;
        const _get = (elm) => {
            const value = get(elm);
            return isNonNullable(value) ? SugarElement.fromDom(value) : null;
        };
        const getAttrib = (elm, name, defaultVal = '') => {
            let value;
            const $elm = _get(elm);
            if (isNonNullable($elm) && isElement$8($elm)) {
                const hook = attrHooks[name];
                if (hook && hook.get) {
                    value = hook.get($elm.dom, name);
                }
                else {
                    value = get$9($elm, name);
                }
            }
            return isNonNullable(value) ? value : defaultVal;
        };
        const getAttribs = (elm) => {
            const node = get(elm);
            return isNullable(node) ? [] : node.attributes;
        };
        const setAttrib = (elm, name, value) => {
            run(elm, (e) => {
                if (isElement$7(e)) {
                    const $elm = SugarElement.fromDom(e);
                    const val = value === '' ? null : value;
                    const originalValue = get$9($elm, name);
                    const hook = attrHooks[name];
                    if (hook && hook.set) {
                        hook.set($elm.dom, val, name);
                    }
                    else {
                        legacySetAttribute($elm, name, val);
                    }
                    if (originalValue !== val && settings.onSetAttrib) {
                        settings.onSetAttrib({
                            attrElm: $elm.dom, // We lie here to not break backwards compatibility
                            attrName: name,
                            attrValue: val
                        });
                    }
                }
            });
        };
        const clone = (node, deep) => {
            return node.cloneNode(deep);
        };
        const getRoot = () => settings.root_element || doc.body;
        const getViewPort = (argWin) => {
            const vp = getBounds(argWin);
            // Returns viewport size excluding scrollbars
            return {
                x: vp.x,
                y: vp.y,
                w: vp.width,
                h: vp.height
            };
        };
        const getPos$1 = (elm, rootElm) => getPos(doc.body, get(elm), rootElm);
        const setStyle = (elm, name, value) => {
            run(elm, (e) => {
                const $elm = SugarElement.fromDom(e);
                applyStyle$1($elm, name, value);
                if (settings.update_styles) {
                    updateInternalStyleAttr(styles, $elm);
                }
            });
        };
        const setStyles = (elm, stylesArg) => {
            run(elm, (e) => {
                const $elm = SugarElement.fromDom(e);
                each$d(stylesArg, (v, n) => {
                    applyStyle$1($elm, n, v);
                });
                if (settings.update_styles) {
                    updateInternalStyleAttr(styles, $elm);
                }
            });
        };
        const getStyle = (elm, name, computed) => {
            const $elm = get(elm);
            if (isNullable($elm) || (!isHTMLElement($elm) && !isSVGElement($elm))) {
                return undefined;
            }
            if (computed) {
                return get$7(SugarElement.fromDom($elm), camelCaseToHyphens(name));
            }
            else {
                // Camelcase it, if needed
                name = name.replace(/-(\D)/g, (a, b) => b.toUpperCase());
                if (name === 'float') {
                    name = 'cssFloat';
                }
                return $elm.style ? $elm.style[name] : undefined;
            }
        };
        const getSize = (elm) => {
            const $elm = get(elm);
            if (!$elm) {
                return { w: 0, h: 0 };
            }
            let w = getStyle($elm, 'width');
            let h = getStyle($elm, 'height');
            // Non pixel value, then force offset/clientWidth
            if (!w || w.indexOf('px') === -1) {
                w = '0';
            }
            // Non pixel value, then force offset/clientWidth
            if (!h || h.indexOf('px') === -1) {
                h = '0';
            }
            return {
                w: parseInt(w, 10) || $elm.offsetWidth || $elm.clientWidth,
                h: parseInt(h, 10) || $elm.offsetHeight || $elm.clientHeight
            };
        };
        const getRect = (elm) => {
            const $elm = get(elm);
            const pos = getPos$1($elm);
            const size = getSize($elm);
            return {
                x: pos.x, y: pos.y,
                w: size.w, h: size.h
            };
        };
        const is = (elm, selector) => {
            if (!elm) {
                return false;
            }
            const elms = isArray$1(elm) ? elm : [elm];
            return exists(elms, (e) => {
                return is$2(SugarElement.fromDom(e), selector);
            });
        };
        const getParents = (elm, selector, root, collect) => {
            const result = [];
            let node = get(elm);
            collect = collect === undefined;
            // Default root on inline mode
            const resolvedRoot = root || (getRoot().nodeName !== 'BODY' ? getRoot().parentNode : null);
            // Wrap node name as func
            if (isString(selector)) {
                if (selector === '*') {
                    selector = isElement$7;
                }
                else {
                    const selectorVal = selector;
                    selector = (node) => is(node, selectorVal);
                }
            }
            while (node) {
                // TODO: Remove nullable check once TINY-6599 is complete
                if (node === resolvedRoot || isNullable(node.nodeType) || isDocument$1(node) || isDocumentFragment(node)) {
                    break;
                }
                if (!selector || selector(node)) {
                    if (collect) {
                        result.push(node);
                    }
                    else {
                        return [node];
                    }
                }
                node = node.parentNode;
            }
            return collect ? result : null;
        };
        const getParent = (node, selector, root) => {
            const parents = getParents(node, selector, root, false);
            return parents && parents.length > 0 ? parents[0] : null;
        };
        const _findSib = (node, selector, name) => {
            let func = selector;
            if (node) {
                // If expression make a function of it using is
                if (isString(selector)) {
                    func = (node) => {
                        return is(node, selector);
                    };
                }
                // Loop all siblings
                for (let tempNode = node[name]; tempNode; tempNode = tempNode[name]) {
                    if (isFunction(func) && func(tempNode)) {
                        return tempNode;
                    }
                }
            }
            return null;
        };
        const getNext = (node, selector) => _findSib(node, selector, 'nextSibling');
        const getPrev = (node, selector) => _findSib(node, selector, 'previousSibling');
        const isParentNode = (node) => isFunction(node.querySelectorAll);
        const select = (selector, scope) => {
            const elm = get(scope) ?? settings.root_element ?? doc;
            return isParentNode(elm) ? from(elm.querySelectorAll(selector)) : [];
        };
        const run = function (elm, func, scope) {
            const context = scope ?? this;
            if (isArray$1(elm)) {
                const result = [];
                each$a(elm, (e, i) => {
                    const node = get(e);
                    if (node) {
                        result.push(func.call(context, node, i));
                    }
                });
                return result;
            }
            else {
                const node = get(elm);
                return !node ? false : func.call(context, node);
            }
        };
        const setAttribs = (elm, attrs) => {
            run(elm, ($elm) => {
                each$d(attrs, (value, name) => {
                    setAttrib($elm, name, value);
                });
            });
        };
        const setHTML = (elm, html) => {
            run(elm, (e) => {
                const $elm = SugarElement.fromDom(e);
                set$3($elm, html);
            });
        };
        const add = (parentElm, name, attrs, html, create) => run(parentElm, (parentElm) => {
            const newElm = isString(name) ? doc.createElement(name) : name;
            if (isNonNullable(attrs)) {
                setAttribs(newElm, attrs);
            }
            if (html) {
                if (!isString(html) && html.nodeType) {
                    newElm.appendChild(html);
                }
                else if (isString(html)) {
                    setHTML(newElm, html);
                }
            }
            return !create ? parentElm.appendChild(newElm) : newElm;
        });
        const create = (name, attrs, html) => add(doc.createElement(name), name, attrs, html, true);
        const decode = Entities.decode;
        const encode = Entities.encodeAllRaw;
        const createHTML = (name, attrs, html = '') => {
            let outHtml = '<' + name;
            for (const key in attrs) {
                if (hasNonNullableKey(attrs, key)) {
                    outHtml += ' ' + key + '="' + encode(attrs[key]) + '"';
                }
            }
            if (isEmpty$5(html) && has$2(schema.getVoidElements(), name)) {
                return outHtml + ' />';
            }
            else {
                return outHtml + '>' + html + '</' + name + '>';
            }
        };
        const createFragment = (html) => {
            const container = doc.createElement('div');
            const frag = doc.createDocumentFragment();
            // Append the container to the fragment so as to remove it from
            // the current document context
            frag.appendChild(container);
            if (html) {
                container.innerHTML = html;
            }
            let node;
            while ((node = container.firstChild)) {
                frag.appendChild(node);
            }
            // Remove the container now that all the children have been transferred
            frag.removeChild(container);
            return frag;
        };
        const remove = (node, keepChildren) => {
            return run(node, (n) => {
                const $node = SugarElement.fromDom(n);
                if (keepChildren) {
                    // Unwrap but don't keep any empty text nodes
                    each$e(children$1($node), (child) => {
                        if (isText$c(child) && child.dom.length === 0) {
                            remove$8(child);
                        }
                        else {
                            before$4($node, child);
                        }
                    });
                }
                remove$8($node);
                return $node.dom;
            });
        };
        const removeAllAttribs = (e) => run(e, (e) => {
            const attrs = e.attributes;
            for (let i = attrs.length - 1; i >= 0; i--) {
                e.removeAttributeNode(attrs.item(i));
            }
        });
        const parseStyle = (cssText) => styles.parse(cssText);
        const serializeStyle = (stylesArg, name) => styles.serialize(stylesArg, name);
        const addStyle = (cssText) => {
            // Prevent inline from loading the same styles twice
            if (self !== DOMUtils.DOM && doc === document) {
                if (addedStyles[cssText]) {
                    return;
                }
                addedStyles[cssText] = true;
            }
            // Create style element if needed
            let styleElm = doc.getElementById('mceDefaultStyles');
            if (!styleElm) {
                styleElm = doc.createElement('style');
                styleElm.id = 'mceDefaultStyles';
                styleElm.type = 'text/css';
                const head = doc.head;
                if (head.firstChild) {
                    head.insertBefore(styleElm, head.firstChild);
                }
                else {
                    head.appendChild(styleElm);
                }
            }
            // Append style data to old or new style element
            if (styleElm.styleSheet) {
                styleElm.styleSheet.cssText += cssText;
            }
            else {
                styleElm.appendChild(doc.createTextNode(cssText));
            }
        };
        const loadCSS = (urls) => {
            if (!urls) {
                urls = '';
            }
            each$e(urls.split(','), (url) => {
                files[url] = true;
                styleSheetLoader.load(url).catch(noop);
            });
        };
        const toggleClass = (elm, cls, state) => {
            run(elm, (e) => {
                if (isElement$7(e)) {
                    const $elm = SugarElement.fromDom(e);
                    // TINY-4520: DomQuery used to handle specifying multiple classes and the
                    // formatter relies on it due to the changes made for TINY-7227
                    const classes = cls.split(' ');
                    each$e(classes, (c) => {
                        if (isNonNullable(state)) {
                            const fn = state ? add$2 : remove$4;
                            fn($elm, c);
                        }
                        else {
                            toggle$1($elm, c);
                        }
                    });
                }
            });
        };
        const addClass = (elm, cls) => {
            toggleClass(elm, cls, true);
        };
        const removeClass = (elm, cls) => {
            toggleClass(elm, cls, false);
        };
        const hasClass = (elm, cls) => {
            const $elm = _get(elm);
            // TINY-4520: DomQuery used to handle specifying multiple classes and the
            // formatter relies on it due to the changes made for TINY-7227
            const classes = cls.split(' ');
            return isNonNullable($elm) && forall(classes, (c) => has($elm, c));
        };
        const show = (elm) => {
            run(elm, (e) => remove$7(SugarElement.fromDom(e), 'display'));
        };
        const hide = (elm) => {
            run(elm, (e) => set$2(SugarElement.fromDom(e), 'display', 'none'));
        };
        const isHidden = (elm) => {
            const $elm = _get(elm);
            return isNonNullable($elm) && is$4(getRaw$1($elm, 'display'), 'none');
        };
        const uniqueId = (prefix) => (!prefix ? 'mce_' : prefix) + (counter++);
        const getOuterHTML = (elm) => {
            const $elm = _get(elm);
            if (isNonNullable($elm)) {
                return isElement$7($elm.dom) ? $elm.dom.outerHTML : getOuter($elm);
            }
            else {
                return '';
            }
        };
        const setOuterHTML = (elm, html) => {
            run(elm, ($elm) => {
                if (isElement$7($elm)) {
                    $elm.outerHTML = html;
                }
            });
        };
        const insertAfter = (node, reference) => {
            const referenceNode = get(reference);
            return run(node, (node) => {
                const parent = referenceNode?.parentNode;
                const nextSibling = referenceNode?.nextSibling;
                if (parent) {
                    if (nextSibling) {
                        parent.insertBefore(node, nextSibling);
                    }
                    else {
                        parent.appendChild(node);
                    }
                }
                return node;
            });
        };
        const replace = (newElm, oldElm, keepChildren) => run(oldElm, (elm) => {
            const replacee = isArray$1(oldElm) ? newElm.cloneNode(true) : newElm;
            if (keepChildren) {
                each$a(grep(elm.childNodes), (node) => {
                    replacee.appendChild(node);
                });
            }
            elm.parentNode?.replaceChild(replacee, elm);
            return elm;
        });
        const rename = (elm, name) => {
            if (elm.nodeName !== name.toUpperCase()) {
                // Rename block element
                const newElm = create(name);
                // Copy attribs to new block
                each$a(getAttribs(elm), (attrNode) => {
                    setAttrib(newElm, attrNode.nodeName, getAttrib(elm, attrNode.nodeName));
                });
                // Replace block
                replace(newElm, elm, true);
                return newElm;
            }
            else {
                return elm;
            }
        };
        const findCommonAncestor = (a, b) => {
            let ps = a;
            while (ps) {
                let pe = b;
                while (pe && ps !== pe) {
                    pe = pe.parentNode;
                }
                if (ps === pe) {
                    break;
                }
                ps = ps.parentNode;
            }
            if (!ps && a.ownerDocument) {
                return a.ownerDocument.documentElement;
            }
            else {
                return ps;
            }
        };
        const isEmpty = (node, elements, options) => {
            if (isPlainObject(elements)) {
                const isContent = (node) => {
                    const name = node.nodeName.toLowerCase();
                    return Boolean(elements[name]);
                };
                return isEmptyNode(schema, node, { ...options, isContent });
            }
            else {
                return isEmptyNode(schema, node, options);
            }
        };
        const createRng = () => doc.createRange();
        const split = (parentElm, splitElm, replacementElm) => {
            let range = createRng();
            let beforeFragment;
            let afterFragment;
            if (parentElm && splitElm && parentElm.parentNode && splitElm.parentNode) {
                const parentNode = parentElm.parentNode;
                // Get before chunk
                range.setStart(parentNode, findNodeIndex(parentElm));
                range.setEnd(splitElm.parentNode, findNodeIndex(splitElm));
                beforeFragment = range.extractContents();
                // Get after chunk
                range = createRng();
                range.setStart(splitElm.parentNode, findNodeIndex(splitElm) + 1);
                range.setEnd(parentNode, findNodeIndex(parentElm) + 1);
                afterFragment = range.extractContents();
                // Insert before chunk
                parentNode.insertBefore(trimNode(self, beforeFragment, schema), parentElm);
                // Insert middle chunk
                if (replacementElm) {
                    parentNode.insertBefore(replacementElm, parentElm);
                    // pa.replaceChild(replacementElm, splitElm);
                }
                else {
                    parentNode.insertBefore(splitElm, parentElm);
                }
                // Insert after chunk
                parentNode.insertBefore(trimNode(self, afterFragment, schema), parentElm);
                remove(parentElm);
                return replacementElm || splitElm;
            }
            else {
                return undefined;
            }
        };
        const bind = (target, name, func, scope) => {
            if (isArray$1(target)) {
                let i = target.length;
                const rv = [];
                while (i--) {
                    rv[i] = bind(target[i], name, func, scope);
                }
                return rv;
            }
            else {
                // Collect all window/document events bound by editor instance
                if (settings.collect && (target === doc || target === win)) {
                    boundEvents.push([target, name, func, scope]);
                }
                return events.bind(target, name, func, scope || self);
            }
        };
        const unbind = (target, name, func) => {
            if (isArray$1(target)) {
                let i = target.length;
                const rv = [];
                while (i--) {
                    rv[i] = unbind(target[i], name, func);
                }
                return rv;
            }
            else {
                // Remove any bound events matching the input
                if (boundEvents.length > 0 && (target === doc || target === win)) {
                    let i = boundEvents.length;
                    while (i--) {
                        const [boundTarget, boundName, boundFunc] = boundEvents[i];
                        if (target === boundTarget && (!name || name === boundName) && (!func || func === boundFunc)) {
                            events.unbind(boundTarget, boundName, boundFunc);
                        }
                    }
                }
                return events.unbind(target, name, func);
            }
        };
        const dispatch = (target, name, evt) => events.dispatch(target, name, evt);
        const fire = (target, name, evt) => events.dispatch(target, name, evt);
        const getContentEditable = (node) => {
            if (node && isHTMLElement(node)) {
                // Check for fake content editable
                const contentEditable = node.getAttribute('data-mce-contenteditable');
                if (contentEditable && contentEditable !== 'inherit') {
                    return contentEditable;
                }
                // Check for real content editable
                return node.contentEditable !== 'inherit' ? node.contentEditable : null;
            }
            else {
                return null;
            }
        };
        const getContentEditableParent = (node) => {
            const root = getRoot();
            let state = null;
            for (let tempNode = node; tempNode && tempNode !== root; tempNode = tempNode.parentNode) {
                state = getContentEditable(tempNode);
                if (state !== null) {
                    break;
                }
            }
            return state;
        };
        const isEditable = (node) => {
            if (isNonNullable(node)) {
                const scope = isElement$7(node) ? node : node.parentElement;
                return isNonNullable(scope) && isHTMLElement(scope) && isEditable$2(SugarElement.fromDom(scope));
            }
            else {
                return false;
            }
        };
        const destroy = () => {
            // Unbind all events bound to window/document by editor instance
            if (boundEvents.length > 0) {
                let i = boundEvents.length;
                while (i--) {
                    const [boundTarget, boundName, boundFunc] = boundEvents[i];
                    events.unbind(boundTarget, boundName, boundFunc);
                }
            }
            // Remove CSS files added to the dom
            each$d(files, (_, url) => {
                styleSheetLoader.unload(url);
                delete files[url];
            });
        };
        const isChildOf = (node, parent) => {
            return node === parent || parent.contains(node);
        };
        const dumpRng = (r) => ('startContainer: ' + r.startContainer.nodeName +
            ', startOffset: ' + r.startOffset +
            ', endContainer: ' + r.endContainer.nodeName +
            ', endOffset: ' + r.endOffset);
        // eslint-disable-next-line consistent-this
        const self = {
            doc,
            settings,
            win,
            files,
            stdMode,
            boxModel,
            styleSheetLoader,
            boundEvents,
            styles,
            schema,
            events,
            isBlock: isBlock,
            root: null,
            clone,
            /**
             * Returns the root node of the document. This is normally the body but might be a DIV. Parents like getParent will not
             * go above the point of this root node.
             *
             * @method getRoot
             * @return {Element} Root element for the utility class.
             */
            getRoot,
            /**
             * Returns the viewport of the window.
             *
             * @method getViewPort
             * @param {Window} win Optional window to get viewport of.
             * @return {Object} Viewport object with fields x, y, w and h.
             */
            getViewPort,
            /**
             * Returns the rectangle for a specific element.
             *
             * @method getRect
             * @param {Element/String} elm Element object or element ID to get rectangle from.
             * @return {Object} Rectangle for specified element object with x, y, w, h fields.
             */
            getRect,
            /**
             * Returns the size dimensions of the specified element.
             *
             * @method getSize
             * @param {Element/String} elm Element object or element ID to get rectangle from.
             * @return {Object} Rectangle for specified element object with w, h fields.
             */
            getSize,
            /**
             * Returns a node by the specified selector function. This function will
             * loop through all parent nodes and call the specified function for each node.
             * If the function then returns true indicating that it has found what it was looking for, the loop execution will then end
             * and the node it found will be returned.
             *
             * @method getParent
             * @param {Node/String} node DOM node to search parents on or ID string.
             * @param {Function} selector Selection function or CSS selector to execute on each node.
             * @param {Node} root Optional root element, never go beyond this point.
             * @return {Node} DOM Node or null if it wasn't found.
             */
            getParent,
            /**
             * Returns a node list of all parents matching the specified selector function or pattern.
             * If the function then returns true indicating that it has found what it was looking for and that node will be collected.
             *
             * @method getParents
             * @param {Node/String} node DOM node to search parents on or ID string.
             * @param {Function} selector Selection function to execute on each node or CSS pattern.
             * @param {Node} root Optional root element, never go beyond this point.
             * @return {Array} Array of nodes or null if it wasn't found.
             */
            getParents: getParents,
            /**
             * Returns the specified element by ID or the input element if it isn't a string.
             *
             * @method get
             * @param {String/Element} n Element id to look for or element to just pass though.
             * @return {Element} Element matching the specified id or null if it wasn't found.
             */
            get,
            /**
             * Returns the next node that matches selector or function
             *
             * @method getNext
             * @param {Node} node Node to find siblings from.
             * @param {String/function} selector Selector CSS expression or function.
             * @return {Node} Next node item matching the selector or null if it wasn't found.
             */
            getNext,
            /**
             * Returns the previous node that matches selector or function
             *
             * @method getPrev
             * @param {Node} node Node to find siblings from.
             * @param {String/function} selector Selector CSS expression or function.
             * @return {Node} Previous node item matching the selector or null if it wasn't found.
             */
            getPrev,
            // #ifndef jquery
            /**
             * Returns a list of the elements specified by the given CSS selector. For example: `div#a1 p.test`
             *
             * @method select
             * @param {String} selector Target CSS selector.
             * @param {Object} scope Optional root element/scope element to search in.
             * @return {Array} Array with all matched elements.
             * @example
             * // Adds a class to all paragraphs in the currently active editor
             * tinymce.activeEditor.dom.addClass(tinymce.activeEditor.dom.select('p'), 'someclass');
             *
             * // Adds a class to all spans that have the test class in the currently active editor
             * tinymce.activeEditor.dom.addClass(tinymce.activeEditor.dom.select('span.test'), 'someclass')
             */
            select,
            /**
             * Returns true/false if the specified element matches the specified css pattern.
             *
             * @method is
             * @param {Node/NodeList} elm DOM node to match or an array of nodes to match.
             * @param {String} selector CSS pattern to match the element against.
             */
            is,
            // #endif
            /**
             * Adds the specified element to another element or elements.
             *
             * @method add
             * @param {String/Element/Array} parentElm Element id string, DOM node element or array of ids or elements to add to.
             * @param {String/Element} name Name of new element to add or existing element to add.
             * @param {Object} attrs Optional object collection with arguments to add to the new element(s).
             * @param {String} html Optional inner HTML contents to add for each element.
             * @param {Boolean} create Optional flag if the element should be created or added.
             * @return {Element/Array} Element that got created, or an array of created elements if multiple input elements
             * were passed in.
             * @example
             * // Adds a new paragraph to the end of the active editor
             * tinymce.activeEditor.dom.add(tinymce.activeEditor.getBody(), 'p', { title: 'my title' }, 'Some content');
             */
            add,
            /**
             * Creates a new element.
             *
             * @method create
             * @param {String} name Name of new element.
             * @param {Object} attrs Optional object name/value collection with element attributes.
             * @param {String} html Optional HTML string to set as inner HTML of the element.
             * @return {Element} HTML DOM node element that got created.
             * @example
             * // Adds an element where the caret/selection is in the active editor
             * var el = tinymce.activeEditor.dom.create('div', { id: 'test', 'class': 'myclass' }, 'some content');
             * tinymce.activeEditor.selection.setNode(el);
             */
            create,
            /**
             * Creates HTML string for element. The element will be closed unless an empty inner HTML string is passed in.
             *
             * @method createHTML
             * @param {String} name Name of new element.
             * @param {Object} attrs Optional object name/value collection with element attributes.
             * @param {String} html Optional HTML string to set as inner HTML of the element.
             * @return {String} String with new HTML element, for example: <a href="#">test</a>.
             * @example
             * // Creates a html chunk and inserts it at the current selection/caret location
             * tinymce.activeEditor.insertContent(tinymce.activeEditor.dom.createHTML('a', { href: 'test.html' }, 'some line'));
             */
            createHTML,
            /**
             * Creates a document fragment out of the specified HTML string.
             *
             * @method createFragment
             * @param {String} html Html string to create fragment from.
             * @return {DocumentFragment} Document fragment node.
             */
            createFragment,
            /**
             * Removes/deletes the specified element(s) from the DOM.
             *
             * @method remove
             * @param {String/Element/Array} node ID of element or DOM element object or array containing multiple elements/ids.
             * @param {Boolean} keepChildren Optional state to keep children or not. If set to true all children will be
             * placed at the location of the removed element.
             * @return {Element/Array} HTML DOM element that got removed, or an array of removed elements if multiple input elements
             * were passed in.
             * @example
             * // Removes all paragraphs in the active editor
             * tinymce.activeEditor.dom.remove(tinymce.activeEditor.dom.select('p'));
             *
             * // Removes an element by id in the document
             * tinymce.DOM.remove('mydiv');
             */
            remove,
            /**
             * Sets the CSS style value on a HTML element. The name can be a camelcase string
             * or the CSS style name like background-color.
             *
             * @method setStyle
             * @param {String/Element/Array} elm HTML element/Array of elements to set CSS style value on.
             * @param {String} name Name of the style value to set.
             * @param {String} value Value to set on the style.
             * @example
             * // Sets a style value on all paragraphs in the currently active editor
             * tinymce.activeEditor.dom.setStyle(tinymce.activeEditor.dom.select('p'), 'background-color', 'red');
             *
             * // Sets a style value to an element by id in the current document
             * tinymce.DOM.setStyle('mydiv', 'background-color', 'red');
             */
            setStyle,
            /**
             * Returns the current style or runtime/computed value of an element.
             *
             * @method getStyle
             * @param {String/Element} elm HTML element or element id string to get style from.
             * @param {String} name Style name to return.
             * @param {Boolean} computed Computed style.
             * @return {String} Current style or computed style value of an element.
             */
            getStyle: getStyle,
            /**
             * Sets multiple styles on the specified element(s).
             *
             * @method setStyles
             * @param {Element/String/Array} elm DOM element, element id string or array of elements/ids to set styles on.
             * @param {Object} styles Name/Value collection of style items to add to the element(s).
             * @example
             * // Sets styles on all paragraphs in the currently active editor
             * tinymce.activeEditor.dom.setStyles(tinymce.activeEditor.dom.select('p'), { 'background-color': 'red', 'color': 'green' });
             *
             * // Sets styles to an element by id in the current document
             * tinymce.DOM.setStyles('mydiv', { 'background-color': 'red', 'color': 'green' });
             */
            setStyles,
            /**
             * Removes all attributes from an element or elements.
             *
             * @method removeAllAttribs
             * @param {Element/String/Array} e DOM element, element id string or array of elements/ids to remove attributes from.
             */
            removeAllAttribs,
            /**
             * Sets the specified attribute of an element or elements.
             *
             * @method setAttrib
             * @param {Element/String/Array} elm DOM element, element id string or array of elements/ids to set attribute on.
             * @param {String} name Name of attribute to set.
             * @param {String} value Value to set on the attribute - if this value is falsy like null, 0 or '' it will remove
             * the attribute instead.
             * @example
             * // Sets class attribute on all paragraphs in the active editor
             * tinymce.activeEditor.dom.setAttrib(tinymce.activeEditor.dom.select('p'), 'class', 'myclass');
             *
             * // Sets class attribute on a specific element in the current page
             * tinymce.dom.setAttrib('mydiv', 'class', 'myclass');
             */
            setAttrib,
            /**
             * Sets two or more specified attributes of an element or elements.
             *
             * @method setAttribs
             * @param {Element/String/Array} elm DOM element, element id string or array of elements/ids to set attributes on.
             * @param {Object} attrs Name/Value collection of attribute items to add to the element(s).
             * @example
             * // Sets class and title attributes on all paragraphs in the active editor
             * tinymce.activeEditor.dom.setAttribs(tinymce.activeEditor.dom.select('p'), { 'class': 'myclass', title: 'some title' });
             *
             * // Sets class and title attributes on a specific element in the current page
             * tinymce.DOM.setAttribs('mydiv', { 'class': 'myclass', title: 'some title' });
             */
            setAttribs,
            /**
             * Returns the specified attribute by name.
             *
             * @method getAttrib
             * @param {String/Element} elm Element string id or DOM element to get attribute from.
             * @param {String} name Name of attribute to get.
             * @param {String} defaultVal Optional default value to return if the attribute didn't exist.
             * @return {String} Attribute value string, default value or null if the attribute wasn't found.
             */
            getAttrib,
            /**
             * Returns the absolute x, y position of a node. The position will be returned in an object with x, y fields.
             *
             * @method getPos
             * @param {Element/String} elm HTML element or element id to get x, y position from.
             * @param {Element} rootElm Optional root element to stop calculations at.
             * @return {Object} Absolute position of the specified element object with x, y fields.
             */
            getPos: getPos$1,
            /**
             * Parses the specified style value into an object collection. This parser will also
             * merge and remove any redundant items that browsers might have added. It will also convert non-hex
             * colors to hex values. Urls inside the styles will also be converted to absolute/relative based on settings.
             *
             * @method parseStyle
             * @param {String} cssText Style value to parse, for example: border:1px solid red;.
             * @return {Object} Object representation of that style, for example: {border: '1px solid red'}
             */
            parseStyle,
            /**
             * Serializes the specified style object into a string.
             *
             * @method serializeStyle
             * @param {Object} styles Object to serialize as string, for example: {border: '1px solid red'}
             * @param {String} name Optional element name.
             * @return {String} String representation of the style object, for example: border: 1px solid red.
             */
            serializeStyle,
            /**
             * Adds a style element at the top of the document with the specified cssText content.
             *
             * @method addStyle
             * @param {String} cssText CSS Text style to add to top of head of document.
             */
            addStyle,
            /**
             * Imports/loads the specified CSS file into the document bound to the class.
             *
             * @method loadCSS
             * @param {String} url URL to CSS file to load.
             * @example
             * // Loads a CSS file dynamically into the current document
             * tinymce.DOM.loadCSS('somepath/some.css');
             *
             * // Loads a CSS file into the currently active editor instance
             * tinymce.activeEditor.dom.loadCSS('somepath/some.css');
             *
             * // Loads a CSS file into an editor instance by id
             * tinymce.get('someid').dom.loadCSS('somepath/some.css');
             *
             * // Loads multiple CSS files into the current document
             * tinymce.DOM.loadCSS('somepath/some.css,somepath/someother.css');
             */
            loadCSS,
            /**
             * Adds a class to the specified element or elements.
             *
             * @method addClass
             * @param {String/Element/Array} elm Element ID string or DOM element or array with elements or IDs.
             * @param {String} cls Class name to add to each element.
             * @return {String/Array} String with new class value or array with new class values for all elements.
             * @example
             * // Adds a class to all paragraphs in the active editor
             * tinymce.activeEditor.dom.addClass(tinymce.activeEditor.dom.select('p'), 'myclass');
             *
             * // Adds a class to a specific element in the current page
             * tinymce.DOM.addClass('mydiv', 'myclass');
             */
            addClass,
            /**
             * Removes a class from the specified element or elements.
             *
             * @method removeClass
             * @param {String/Element/Array} elm Element ID string or DOM element or array with elements or IDs.
             * @param {String} cls Class name to remove from each element.
             * @return {String/Array} String of remaining class name(s), or an array of strings if multiple input elements
             * were passed in.
             * @example
             * // Removes a class from all paragraphs in the active editor
             * tinymce.activeEditor.dom.removeClass(tinymce.activeEditor.dom.select('p'), 'myclass');
             *
             * // Removes a class from a specific element in the current page
             * tinymce.DOM.removeClass('mydiv', 'myclass');
             */
            removeClass,
            /**
             * Returns true if the specified element has the specified class.
             *
             * @method hasClass
             * @param {String/Element} elm HTML element or element id string to check CSS class on.
             * @param {String} cls CSS class to check for.
             * @return {Boolean} true/false if the specified element has the specified class.
             */
            hasClass,
            /**
             * Toggles the specified class on/off.
             *
             * @method toggleClass
             * @param {Element} elm Element to toggle class on.
             * @param {String} cls Class to toggle on/off.
             * @param {Boolean} state Optional state to set.
             */
            toggleClass,
            /**
             * Shows the specified element(s) by ID by setting the "display" style.
             *
             * @method show
             * @param {String/Element/Array} elm ID of DOM element or DOM element or array with elements or IDs to show.
             */
            show,
            /**
             * Hides the specified element(s) by ID by setting the "display" style.
             *
             * @method hide
             * @param {String/Element/Array} elm ID of DOM element or DOM element or array with elements or IDs to hide.
             * @example
             * // Hides an element by id in the document
             * tinymce.DOM.hide('myid');
             */
            hide,
            /**
             * Returns true/false if the element is hidden or not by checking the "display" style.
             *
             * @method isHidden
             * @param {String/Element} elm Id or element to check display state on.
             * @return {Boolean} true/false if the element is hidden or not.
             */
            isHidden,
            /**
             * Returns a unique id. This can be useful when generating elements on the fly.
             * This method will not check if the element already exists.
             *
             * @method uniqueId
             * @param {String} prefix Optional prefix to add in front of all ids - defaults to "mce_".
             * @return {String} Unique id.
             */
            uniqueId,
            /**
             * Sets the specified HTML content inside the element or elements. The HTML will first be processed. This means
             * URLs will get converted, hex color values fixed etc. Check processHTML for details.
             *
             * @method setHTML
             * @param {Element/String/Array} elm DOM element, element id string or array of elements/ids to set HTML inside of.
             * @param {String} html HTML content to set as inner HTML of the element.
             * @example
             * // Sets the inner HTML of all paragraphs in the active editor
             * tinymce.activeEditor.dom.setHTML(tinymce.activeEditor.dom.select('p'), 'some inner html');
             *
             * // Sets the inner HTML of an element by id in the document
             * tinymce.DOM.setHTML('mydiv', 'some inner html');
             */
            setHTML,
            /**
             * Returns the outer HTML of an element.
             *
             * @method getOuterHTML
             * @param {String/Element} elm Element ID or element object to get outer HTML from.
             * @return {String} Outer HTML string.
             * @example
             * tinymce.DOM.getOuterHTML(editorElement);
             * tinymce.activeEditor.getOuterHTML(tinymce.activeEditor.getBody());
             */
            getOuterHTML,
            /**
             * Sets the specified outer HTML on an element or elements.
             *
             * @method setOuterHTML
             * @param {Element/String/Array} elm DOM element, element id string or array of elements/ids to set outer HTML on.
             * @param {Object} html HTML code to set as outer value for the element.
             * @example
             * // Sets the outer HTML of all paragraphs in the active editor
             * tinymce.activeEditor.dom.setOuterHTML(tinymce.activeEditor.dom.select('p'), '<div>some html</div>');
             *
             * // Sets the outer HTML of an element by id in the document
             * tinymce.DOM.setOuterHTML('mydiv', '<div>some html</div>');
             */
            setOuterHTML,
            /**
             * Entity decodes a string. This method decodes any HTML entities, such as `&amp;aring;`.
             *
             * @method decode
             * @param {String} s String to decode entities on.
             * @return {String} Entity decoded string.
             */
            decode,
            /**
             * Entity encodes a string. This method encodes the most common entities, such as `<`, `>`, `"` and `&`.
             *
             * @method encode
             * @param {String} text String to encode with entities.
             * @return {String} Entity encoded string.
             */
            encode,
            /**
             * Inserts an element after the reference element.
             *
             * @method insertAfter
             * @param {Element} node Element to insert after the reference.
             * @param {Element/String/Array} referenceNode Reference element, element id or array of elements to insert after.
             * @return {Element/Array} Element that got added or an array with elements.
             */
            insertAfter,
            /**
             * Replaces the specified element or elements with the new element specified. The new element will
             * be cloned if multiple input elements are passed in.
             *
             * @method replace
             * @param {Element} newElm New element to replace old ones with.
             * @param {Element/String/Array} oldElm Element DOM node, element id or array of elements or ids to replace.
             * @param {Boolean} keepChildren Optional keep children state, if set to true child nodes from the old object will be added
             * to new ones.
             */
            replace,
            /**
             * Renames the specified element and keeps its attributes and children.
             *
             * @method rename
             * @param {Element} elm Element to rename.
             * @param {String} name Name of the new element.
             * @return {Element} New element or the old element if it needed renaming.
             */
            rename,
            /**
             * Find the common ancestor of two elements. This is a shorter method than using the DOM Range logic.
             *
             * @method findCommonAncestor
             * @param {Element} a Element to find common ancestor of.
             * @param {Element} b Element to find common ancestor of.
             * @return {Element} Common ancestor element of the two input elements.
             */
            findCommonAncestor,
            /**
             * Executes the specified function on the element by id or dom element node or array of elements/id.
             *
             * @method run
             * @param {String/Element/Array} elm ID or DOM element object or array with ids or elements.
             * @param {Function} func Function to execute for each item.
             * @param {Object} scope Optional scope to execute the function in.
             * @return {Object/Array} Single object, or an array of objects if multiple input elements were passed in.
             */
            run,
            /**
             * Returns a NodeList with attributes for the element.
             *
             * @method getAttribs
             * @param {HTMLElement/string} elm Element node or string id to get attributes from.
             * @return {NodeList} NodeList with attributes.
             */
            getAttribs,
            /**
             * Returns true/false if the specified node is to be considered empty or not.
             *
             * @method isEmpty
             * @param {Node} node The target node to check if it's empty.
             * @param {Object} elements Optional name/value object with elements that are automatically treated as non-empty elements.
             * @return {Boolean} true/false if the node is empty or not.
             * @example
             * tinymce.DOM.isEmpty(node, { img: true });
             */
            isEmpty,
            /**
             * Creates a new DOM Range object. This will use the native DOM Range API if it's
             * available. If it's not, it will fall back to the custom TinyMCE implementation.
             *
             * @method createRng
             * @return {DOMRange} DOM Range object.
             * @example
             * const rng = tinymce.DOM.createRng();
             * alert(rng.startContainer + "," + rng.startOffset);
             */
            createRng,
            /**
             * Returns the index of the specified node within its parent.
             *
             * @method nodeIndex
             * @param {Node} node Node to look for.
             * @param {Boolean} normalized Optional true/false state if the index is what it would be after a normalization.
             * @return {Number} Index of the specified node.
             */
            nodeIndex: findNodeIndex,
            /**
             * Splits an element into two new elements and places the specified split
             * element or elements between the new ones. For example splitting the paragraph at the bold element in
             * this example `<p>abc<b>abc</b>123</p>` would produce `<p>abc</p><b>abc</b><p>123</p>`.
             *
             * @method split
             * @param {Element} parentElm Parent element to split.
             * @param {Element} splitElm Element to split at.
             * @param {Element} replacementElm Optional replacement element to replace the split element with.
             * @return {Element} Returns the split element or the replacement element if that is specified.
             */
            split,
            /**
             * Adds an event handler to the specified object.
             *
             * @method bind
             * @param {Element/Document/Window/Array} target Target element to bind events to.
             * handler to or an array of elements/ids/documents.
             * @param {String} name Name of event handler to add, for example: click.
             * @param {Function} func Function to execute when the event occurs.
             * @param {Object} scope Optional scope to execute the function in.
             * @return {Function} Function callback handler the same as the one passed in.
             */
            bind: bind,
            /**
             * Removes the specified event handler by name and function from an element or collection of elements.
             *
             * @method unbind
             * @param {Element/Document/Window/Array} target Target element to unbind events on.
             * @param {String} name Event handler name, for example: "click"
             * @param {Function} func Function to remove.
             * @return {Boolean/Array} Bool state of true if the handler was removed, or an array of states if multiple input elements
             * were passed in.
             */
            unbind: unbind,
            /**
             * Fires the specified event name and optional object on the specified target.
             * <br>
             * <em>Deprecated in TinyMCE 6.0 and has been marked for removal in TinyMCE 7.0. Use <code>dispatch</code> instead.</em>
             *
             * @method fire
             * @param {Node/Document/Window} target Target element or object to fire event on.
             * @param {String} name Event name to fire.
             * @param {Object} evt Event object to send.
             * @return {Event} Event object.
             * @deprecated Use dispatch() instead
             */
            fire,
            /**
             * Dispatches the specified event name and optional object on the specified target.
             *
             * @method dispatch
             * @param {Node/Document/Window} target Target element or object to dispatch event on.
             * @param {String} name Name of the event to fire.
             * @param {Object} evt Event object to send.
             * @return {Event} Event object.
             */
            dispatch,
            // Returns the content editable state of a node
            getContentEditable,
            getContentEditableParent,
            /**
             * Checks if the specified node is editable within the given context of its parents.
             *
             * @method isEditable
             * @param {Node} node Node to check if it's editable.
             * @return {Boolean} Will be true if the node is editable and false if it's not editable.
             */
            isEditable,
            /**
             * Destroys all internal references to the DOM to solve memory leak issues.
             *
             * @method destroy
             */
            destroy,
            isChildOf,
            dumpRng
        };
        const attrHooks = setupAttrHooks(styles, settings, constant(self));
        return self;
    };
    /**
     * Instance of DOMUtils for the current document.
     *
     * @static
     * @property DOM
     * @type tinymce.dom.DOMUtils
     * @example
     * // Example of how to add a class to some element by id
     * tinymce.DOM.addClass('someid', 'someclass');
     */
    DOMUtils.DOM = DOMUtils(document);
    DOMUtils.nodeIndex = findNodeIndex;

    /**
     * This class handles asynchronous/synchronous loading of JavaScript files it will execute callbacks
     * when various items gets loaded. This class is useful to load external JavaScript files.
     *
     * @class tinymce.dom.ScriptLoader
     * @example
     * // Load a script from a specific URL using the global script loader
     * tinymce.ScriptLoader.load('somescript.js');
     *
     * // Load a script using a unique instance of the script loader
     * const scriptLoader = new tinymce.dom.ScriptLoader();
     *
     * scriptLoader.load('somescript.js');
     *
     * // Load multiple scripts
     * scriptLoader.add('somescript1.js');
     * scriptLoader.add('somescript2.js');
     * scriptLoader.add('somescript3.js');
     *
     * scriptLoader.loadQueue().then(() => {
     *   alert('All scripts are now loaded.');
     * });
     */
    const DOM$e = DOMUtils.DOM;
    const QUEUED = 0;
    const LOADING = 1;
    const LOADED = 2;
    const FAILED = 3;
    class ScriptLoader {
        static ScriptLoader = new ScriptLoader();
        settings;
        states = {};
        queue = [];
        scriptLoadedCallbacks = {};
        queueLoadedCallbacks = [];
        loading = false;
        constructor(settings = {}) {
            this.settings = settings;
        }
        _setReferrerPolicy(referrerPolicy) {
            this.settings.referrerPolicy = referrerPolicy;
        }
        _setCrossOrigin(crossOrigin) {
            this.settings.crossOrigin = crossOrigin;
        }
        /**
         * Loads a specific script directly without adding it to the load queue.
         *
         * @method loadScript
         * @param {String} url Absolute URL to script to add.
         * @return {Promise} A promise that will resolve when the script loaded successfully or reject if it failed to load.
         */
        loadScript(url) {
            return new Promise((resolve, reject) => {
                const dom = DOM$e;
                const doc = document;
                let elm;
                const cleanup = () => {
                    dom.remove(id);
                    if (elm) {
                        elm.onerror = elm.onload = elm = null;
                    }
                };
                // Execute callback when script is loaded
                const done = () => {
                    cleanup();
                    resolve();
                };
                const error = () => {
                    // We can't mark it as done if there is a load error since
                    // A) We don't want to produce 404 errors on the server and
                    // B) the onerror event won't fire on all browsers.
                    cleanup();
                    reject('Failed to load script: ' + url);
                };
                const id = dom.uniqueId();
                // Create new script element
                elm = doc.createElement('script');
                elm.id = id;
                elm.type = 'text/javascript';
                elm.src = Tools._addCacheSuffix(url);
                if (this.settings.referrerPolicy) {
                    // Note: Don't use elm.referrerPolicy = ... here as it doesn't work on Safari
                    dom.setAttrib(elm, 'referrerpolicy', this.settings.referrerPolicy);
                }
                const crossOrigin = this.settings.crossOrigin;
                if (isFunction(crossOrigin)) {
                    const resultCrossOrigin = crossOrigin(url);
                    if (resultCrossOrigin !== undefined) {
                        dom.setAttrib(elm, 'crossorigin', resultCrossOrigin);
                    }
                }
                elm.onload = done;
                // Add onerror event will get fired on some browsers but not all of them
                elm.onerror = error;
                // Add script to document
                (doc.head || doc.body).appendChild(elm);
            });
        }
        /**
         * Returns true/false if a script has been loaded or not.
         *
         * @method isDone
         * @param {String} url URL to check for.
         * @return {Boolean} true/false if the URL is loaded.
         */
        isDone(url) {
            return this.states[url] === LOADED;
        }
        /**
         * Marks a specific script to be loaded. This can be useful if a script got loaded outside
         * the script loader or to skip it from loading some script.
         *
         * @method markDone
         * @param {String} url Absolute URL to the script to mark as loaded.
         */
        markDone(url) {
            this.states[url] = LOADED;
        }
        /**
         * Adds a specific script to the load queue of the script loader.
         *
         * @method add
         * @param {String} url Absolute URL to script to add.
         * @return {Promise} A promise that will resolve when the script loaded successfully or reject if it failed to load.
         */
        add(url) {
            const self = this;
            self.queue.push(url);
            // Add url to load queue
            const state = self.states[url];
            if (state === undefined) {
                self.states[url] = QUEUED;
            }
            return new Promise((resolve, reject) => {
                // Store away callback for later execution
                if (!self.scriptLoadedCallbacks[url]) {
                    self.scriptLoadedCallbacks[url] = [];
                }
                self.scriptLoadedCallbacks[url].push({
                    resolve,
                    reject
                });
            });
        }
        load(url) {
            return this.add(url);
        }
        remove(url) {
            delete this.states[url];
            delete this.scriptLoadedCallbacks[url];
        }
        /**
         * Starts the loading of the queue.
         *
         * @method loadQueue
         * @return {Promise} A promise that is resolved when all queued items are loaded or rejected with the script urls that failed to load.
         */
        loadQueue() {
            const queue = this.queue;
            this.queue = [];
            return this.loadScripts(queue);
        }
        /**
         * Loads the specified queue of files and executes the callback ones they are loaded.
         * This method is generally not used outside this class but it might be useful in some scenarios.
         *
         * @method loadScripts
         * @param {Array} scripts Array of queue items to load.
         * @return {Promise} A promise that is resolved when all scripts are loaded or rejected with the script urls that failed to load.
         */
        loadScripts(scripts) {
            const self = this;
            const execCallbacks = (name, url) => {
                // Execute URL callback functions
                get$a(self.scriptLoadedCallbacks, url).each((callbacks) => {
                    each$e(callbacks, (callback) => callback[name](url));
                });
                delete self.scriptLoadedCallbacks[url];
            };
            const processResults = (results) => {
                const failures = filter$5(results, (result) => result.status === 'rejected');
                if (failures.length > 0) {
                    return Promise.reject(bind$3(failures, ({ reason }) => isArray$1(reason) ? reason : [reason]));
                }
                else {
                    return Promise.resolve();
                }
            };
            const load = (urls) => Promise.allSettled(map$3(urls, (url) => {
                // Script is already loaded then execute script callbacks directly
                if (self.states[url] === LOADED) {
                    execCallbacks('resolve', url);
                    return Promise.resolve();
                }
                else if (self.states[url] === FAILED) {
                    execCallbacks('reject', url);
                    return Promise.reject(url);
                }
                else {
                    // Script is not already loaded, so load it
                    self.states[url] = LOADING;
                    return self.loadScript(url).then(() => {
                        self.states[url] = LOADED;
                        execCallbacks('resolve', url);
                        // Immediately load additional scripts if any were added to the queue while loading this script
                        const queue = self.queue;
                        if (queue.length > 0) {
                            self.queue = [];
                            return load(queue).then(processResults);
                        }
                        else {
                            return Promise.resolve();
                        }
                    }, () => {
                        self.states[url] = FAILED;
                        execCallbacks('reject', url);
                        return Promise.reject(url);
                    });
                }
            }));
            const processQueue = (urls) => {
                self.loading = true;
                return load(urls).then((results) => {
                    self.loading = false;
                    // Start loading the next queued item
                    const nextQueuedItem = self.queueLoadedCallbacks.shift();
                    Optional.from(nextQueuedItem).each(call);
                    return processResults(results);
                });
            };
            // Wait for any other scripts to finish loading first, otherwise load immediately
            const uniqueScripts = stringArray(scripts);
            if (self.loading) {
                return new Promise((resolve, reject) => {
                    self.queueLoadedCallbacks.push(() => {
                        processQueue(uniqueScripts).then(resolve, reject);
                    });
                });
            }
            else {
                return processQueue(uniqueScripts);
            }
        }
        /**
         * Returns the attributes that should be added to a script tag when loading the specified URL.
         *
         * @method getScriptAttributes
         * @param {String} url Url to get attributes for.
         * @return {Object} Object with attributes to add to the script tag.
         */
        getScriptAttributes(url) {
            const attrs = {};
            if (this.settings.referrerPolicy) {
                attrs.referrerpolicy = this.settings.referrerPolicy;
            }
            const crossOrigin = this.settings.crossOrigin;
            if (isFunction(crossOrigin)) {
                const resultCrossOrigin = crossOrigin(url);
                if (isString(resultCrossOrigin)) {
                    attrs.crossorigin = resultCrossOrigin;
                }
            }
            return attrs;
        }
    }

    const isDuplicated = (items, item) => {
        const firstIndex = items.indexOf(item);
        return firstIndex !== -1 && items.indexOf(item, firstIndex + 1) > firstIndex;
    };
    const isRaw = (str) => isObject(str) && has$2(str, 'raw');
    const isTokenised = (str) => isArray$1(str) && str.length > 1;
    const data = {};
    const currentCode = Cell('en');
    const getLanguageData = () => get$a(data, currentCode.get());
    const getData$1 = () => map$2(data, (value) => ({ ...value }));
    /**
     * Sets the current language code.
     *
     * @method setCode
     * @param {String} newCode Current language code.
     */
    const setCode = (newCode) => {
        if (newCode) {
            currentCode.set(newCode);
        }
    };
    /**
     * Returns the current language code.
     *
     * @method getCode
     * @return {String} Current language code.
     */
    const getCode = () => currentCode.get();
    /**
     * Adds translations for a specific language code.
     * Translation keys are set to be case insensitive.
     *
     * @method add
     * @param {String} code Language code like sv_SE.
     * @param {Object} items Name/value object where key is english and value is the translation.
     */
    const add = (code, items) => {
        let langData = data[code];
        if (!langData) {
            data[code] = langData = {};
        }
        const lcNames = map$3(keys(items), (name) => name.toLowerCase());
        each$d(items, (translation, name) => {
            const lcName = name.toLowerCase();
            if (lcName !== name && isDuplicated(lcNames, lcName)) {
                if (!has$2(items, lcName)) {
                    langData[lcName] = translation;
                }
                langData[name] = translation;
            }
            else {
                langData[lcName] = translation;
            }
        });
    };
    /**
     * Translates the specified text.
     *
     * It has a few formats:
     * I18n.translate("Text");
     * I18n.translate(["Text {0}/{1}", 0, 1]);
     * I18n.translate({raw: "Raw string"});
     *
     * @method translate
     * @param {String/Object/Array} text Text to translate.
     * @return {String} String that got translated.
     */
    const translate = (text) => {
        const langData = getLanguageData().getOr({});
        /*
         * number - string
         * null, undefined and empty string - empty string
         * array - comma-delimited string
         * object - in [object Object]
         * function - in [object Function]
         */
        const toString = (obj) => {
            if (isFunction(obj)) {
                return Object.prototype.toString.call(obj);
            }
            return !isEmpty(obj) ? '' + obj : '';
        };
        const isEmpty = (text) => text === '' || text === null || text === undefined;
        const getLangData = (text) => {
            // make sure we work on a string and return a string
            const textStr = toString(text);
            return has$2(langData, textStr)
                ? toString(langData[textStr])
                : get$a(langData, textStr.toLowerCase()).map(toString).getOr(textStr);
        };
        const removeContext = (str) => str.replace(/{context:\w+}$/, '');
        const replaceWithEllipsisChar = (text) => text.replaceAll('...', ellipsis);
        // empty strings
        if (isEmpty(text)) {
            return '';
        }
        // Raw, already translated
        if (isRaw(text)) {
            return replaceWithEllipsisChar(toString(text.raw));
        }
        // Tokenised {translations}
        if (isTokenised(text)) {
            const values = text.slice(1);
            const substitued = getLangData(text[0]).replace(/\{([0-9]+)\}/g, ($1, $2) => has$2(values, $2) ? toString(values[$2]) : $1);
            return replaceWithEllipsisChar(removeContext(substitued));
        }
        // straight forward translation mapping
        return replaceWithEllipsisChar(removeContext(getLangData(text)));
    };
    /**
     * Returns true/false if the currently active language pack is rtl or not.
     *
     * @method isRtl
     * @return {Boolean} True if the current language pack is rtl.
     */
    const isRtl$1 = () => getLanguageData()
        .bind((items) => get$a(items, '_dir'))
        .exists((dir) => dir === 'rtl');
    /**
     * Returns true/false if specified language pack exists.
     *
     * @method hasCode
     * @param {String} code Code to check for.
     * @return {Boolean} True if the current language pack for the specified code exists.
     */
    const hasCode = (code) => has$2(data, code);
    const I18n = {
        getData: getData$1,
        setCode,
        getCode,
        add,
        translate,
        isRtl: isRtl$1,
        hasCode
    };

    const AddOnManager = () => {
        const items = [];
        const urls = {};
        const lookup = {};
        const _listeners = [];
        const runListeners = (name, state) => {
            const matchedListeners = filter$5(_listeners, (listener) => listener.name === name && listener.state === state);
            each$e(matchedListeners, (listener) => listener.resolve());
        };
        const isLoaded = (name) => has$2(urls, name);
        const isAdded = (name) => has$2(lookup, name);
        const get = (name) => {
            if (lookup[name]) {
                return lookup[name].instance;
            }
            return undefined;
        };
        const loadLanguagePack = (name, languages) => {
            const language = I18n.getCode();
            const wrappedLanguages = ',' + (languages || '') + ',';
            if (!language || languages && wrappedLanguages.indexOf(',' + language + ',') === -1) {
                return;
            }
            // eslint-disable-next-line @typescript-eslint/no-floating-promises
            ScriptLoader.ScriptLoader.add(urls[name] + '/langs/' + language + '.js');
        };
        const requireLangPack = (name, languages) => {
            if (AddOnManager.languageLoad !== false) {
                if (isLoaded(name)) {
                    loadLanguagePack(name, languages);
                }
                else {
                    // eslint-disable-next-line @typescript-eslint/no-floating-promises
                    waitFor(name, 'loaded').then(() => loadLanguagePack(name, languages));
                }
            }
        };
        const add = (id, addOn) => {
            items.push(addOn);
            lookup[id] = { instance: addOn };
            runListeners(id, 'added');
            return addOn;
        };
        const remove = (name) => {
            delete urls[name];
            delete lookup[name];
        };
        const createUrl = (baseUrl, dep) => {
            if (isString(dep)) {
                return isString(baseUrl) ?
                    { prefix: '', resource: dep, suffix: '' } :
                    { prefix: baseUrl.prefix, resource: dep, suffix: baseUrl.suffix };
            }
            else {
                return dep;
            }
        };
        const load = (name, addOnUrl) => {
            if (urls[name]) {
                return Promise.resolve();
            }
            let urlString = isString(addOnUrl) ? addOnUrl : addOnUrl.prefix + addOnUrl.resource + addOnUrl.suffix;
            if (urlString.indexOf('/') !== 0 && urlString.indexOf('://') === -1) {
                urlString = AddOnManager.baseURL + '/' + urlString;
            }
            urls[name] = urlString.substring(0, urlString.lastIndexOf('/'));
            const done = () => {
                runListeners(name, 'loaded');
                return Promise.resolve();
            };
            if (lookup[name]) {
                return done();
            }
            else {
                return ScriptLoader.ScriptLoader.add(urlString).then(done);
            }
        };
        const waitFor = (name, state = 'added') => {
            if (state === 'added' && isAdded(name)) {
                return Promise.resolve();
            }
            else if (state === 'loaded' && isLoaded(name)) {
                return Promise.resolve();
            }
            else {
                return new Promise((resolve) => {
                    _listeners.push({ name, state, resolve });
                });
            }
        };
        return {
            items,
            urls,
            lookup,
            /**
             * Returns the specified add on by the short name.
             *
             * @method get
             * @param {String} name Add-on to look for.
             * @return {tinymce.Theme/tinymce.Plugin} Theme or plugin add-on instance or undefined.
             */
            get,
            /**
             * Loads a language pack for the specified add-on.
             *
             * @method requireLangPack
             * @param {String} name Short name of the add-on.
             * @param {String} languages Optional comma or space separated list of languages to check if it matches the name.
             */
            requireLangPack,
            /**
             * Adds a instance of the add-on by it's short name.
             *
             * @method add
             * @param {String} id Short name/id for the add-on.
             * @param {tinymce.Theme/tinymce.Plugin} addOn Theme or plugin to add.
             * @return {tinymce.Theme/tinymce.Plugin} The same theme or plugin instance that got passed in.
             * @example
             * // Create a simple plugin
             * const TestPlugin = (ed, url) => {
             *   ed.on('click', (e) => {
             *     ed.windowManager.alert('Hello World!');
             *   });
             * };
             *
             * // Register plugin using the add method
             * tinymce.PluginManager.add('test', TestPlugin);
             *
             * // Initialize TinyMCE
             * tinymce.init({
             *   ...
             *   plugins: '-test' // Init the plugin but don't try to load it
             * });
             */
            add,
            remove,
            createUrl,
            /**
             * Loads an add-on from a specific url.
             *
             * @method load
             * @param {String} name Short name of the add-on that gets loaded.
             * @param {String} addOnUrl URL to the add-on that will get loaded.
             * @return {Promise} A promise that will resolve when the add-on is loaded successfully or reject if it failed to load.
             * @example
             * // Loads a plugin from an external URL
             * tinymce.PluginManager.load('myplugin', '/some/dir/someplugin/plugin.js');
             *
             * // Initialize TinyMCE
             * tinymce.init({
             *   ...
             *   plugins: '-myplugin' // Don't try to load it again
             * });
             */
            load,
            waitFor
        };
    };
    AddOnManager.languageLoad = true;
    AddOnManager.baseURL = '';
    AddOnManager.PluginManager = AddOnManager();
    AddOnManager.ThemeManager = AddOnManager();
    AddOnManager.ModelManager = AddOnManager();

    const annotation = constant('mce-annotation');
    const dataAnnotation = constant('data-mce-annotation');
    const dataAnnotationId = constant('data-mce-annotation-uid');
    const dataAnnotationActive = constant('data-mce-annotation-active');
    const dataAnnotationClasses = constant('data-mce-annotation-classes');
    const dataAnnotationAttributes = constant('data-mce-annotation-attrs');

    const isRoot$1 = (root) => (node) => eq(node, root);
    // Given the current editor selection, identify the uid of any current
    // annotation
    const identify = (editor, annotationName) => {
        const rng = editor.selection.getRng();
        const start = SugarElement.fromDom(rng.startContainer);
        const root = SugarElement.fromDom(editor.getBody());
        const selector = annotationName.fold(() => '.' + annotation(), (an) => `[${dataAnnotation()}="${an}"]`);
        const newStart = child$1(start, rng.startOffset).getOr(start);
        const closest = closest$4(newStart, selector, isRoot$1(root));
        return closest.bind((c) => getOpt(c, `${dataAnnotationId()}`).bind((uid) => getOpt(c, `${dataAnnotation()}`).map((name) => {
            const elements = findMarkers(editor, uid);
            return {
                uid,
                name,
                elements
            };
        })));
    };
    const isAnnotation = (elem) => isElement$8(elem) && has(elem, annotation());
    const isBogusElement = (elem, root) => has$1(elem, 'data-mce-bogus') || ancestor$1(elem, '[data-mce-bogus="all"]', isRoot$1(root));
    const findMarkers = (editor, uid) => {
        const body = SugarElement.fromDom(editor.getBody());
        const descendants$1 = descendants(body, `[${dataAnnotationId()}="${uid}"]`);
        return filter$5(descendants$1, (descendant) => !isBogusElement(descendant, body));
    };
    const findAll = (editor, name) => {
        const body = SugarElement.fromDom(editor.getBody());
        const markers = descendants(body, `[${dataAnnotation()}="${name}"]`);
        const directory = {};
        each$e(markers, (m) => {
            if (!isBogusElement(m, body)) {
                const uid = get$9(m, dataAnnotationId());
                const nodesAlready = get$a(directory, uid).getOr([]);
                directory[uid] = nodesAlready.concat([m]);
            }
        });
        return directory;
    };

    const setup$E = (editor, registry) => {
        const changeCallbacks = Cell({});
        const initData = () => ({
            listeners: [],
            previous: value$1()
        });
        const withCallbacks = (name, f) => {
            updateCallbacks(name, (data) => {
                f(data);
                return data;
            });
        };
        const updateCallbacks = (name, f) => {
            const callbackMap = changeCallbacks.get();
            const data = get$a(callbackMap, name).getOrThunk(initData);
            const outputData = f(data);
            callbackMap[name] = outputData;
            changeCallbacks.set(callbackMap);
        };
        const fireCallbacks = (name, uid, elements) => {
            withCallbacks(name, (data) => {
                each$e(data.listeners, (f) => f(true, name, {
                    uid,
                    nodes: map$3(elements, (elem) => elem.dom)
                }));
            });
        };
        const fireNoAnnotation = (name) => {
            withCallbacks(name, (data) => {
                each$e(data.listeners, (f) => f(false, name));
            });
        };
        const toggleActiveAttr = (uid, state) => {
            each$e(findMarkers(editor, uid), (elem) => {
                if (state) {
                    set$4(elem, dataAnnotationActive(), 'true');
                }
                else {
                    remove$9(elem, dataAnnotationActive());
                }
            });
        };
        // NOTE: Runs in alphabetical order.
        const onNodeChange = last$1(() => {
            const annotations = sort(registry.getNames());
            each$e(annotations, (name) => {
                updateCallbacks(name, (data) => {
                    const prev = data.previous.get();
                    identify(editor, Optional.some(name)).fold(() => {
                        prev.each((uid) => {
                            // Changed from something to nothing.
                            fireNoAnnotation(name);
                            data.previous.clear();
                            toggleActiveAttr(uid, false);
                        });
                    }, ({ uid, name, elements }) => {
                        // Changed from a different annotation (or nothing)
                        if (!is$4(prev, uid)) {
                            prev.each((uid) => toggleActiveAttr(uid, false));
                            fireCallbacks(name, uid, elements);
                            data.previous.set(uid);
                            toggleActiveAttr(uid, true);
                        }
                    });
                    return {
                        previous: data.previous,
                        listeners: data.listeners
                    };
                });
            });
        }, 30);
        editor.on('remove', () => {
            onNodeChange.cancel();
        });
        editor.on('NodeChange', () => {
            onNodeChange.throttle();
        });
        const addListener = (name, f) => {
            updateCallbacks(name, (data) => ({
                previous: data.previous,
                listeners: data.listeners.concat([f])
            }));
        };
        return {
            addListener
        };
    };

    const setup$D = (editor, registry) => {
        const dataAnnotation$1 = dataAnnotation();
        const identifyParserNode = (node) => Optional.from(node.attr(dataAnnotation$1)).bind(registry.lookup);
        const removeDirectAnnotation = (node) => {
            node.attr(dataAnnotationId(), null);
            node.attr(dataAnnotation(), null);
            node.attr(dataAnnotationActive(), null);
            const customAttrNames = Optional.from(node.attr(dataAnnotationAttributes())).map((names) => names.split(',')).getOr([]);
            const customClasses = Optional.from(node.attr(dataAnnotationClasses())).map((names) => names.split(',')).getOr([]);
            each$e(customAttrNames, (name) => node.attr(name, null));
            const classList = node.attr('class')?.split(' ') ?? [];
            const newClassList = difference(classList, [annotation()].concat(customClasses));
            node.attr('class', newClassList.length > 0 ? newClassList.join(' ') : null);
            node.attr(dataAnnotationClasses(), null);
            node.attr(dataAnnotationAttributes(), null);
        };
        editor.serializer.addTempAttr(dataAnnotationActive());
        editor.serializer.addAttributeFilter(dataAnnotation$1, (nodes) => {
            for (const node of nodes) {
                identifyParserNode(node).each((settings) => {
                    if (settings.persistent === false) {
                        if (node.name === 'span') {
                            node.unwrap();
                        }
                        else {
                            removeDirectAnnotation(node);
                        }
                    }
                });
            }
        });
    };

    const create$a = () => {
        const annotations = {};
        const register = (name, settings) => {
            annotations[name] = {
                name,
                settings
            };
        };
        const lookup = (name) => get$a(annotations, name).map((a) => a.settings);
        const getNames = () => keys(annotations);
        return {
            register,
            lookup,
            getNames
        };
    };

    const TextWalker = (startNode, rootNode, isBoundary = never) => {
        const walker = new DomTreeWalker(startNode, rootNode);
        const walk = (direction) => {
            let next;
            do {
                next = walker[direction]();
            } while (next && !isText$b(next) && !isBoundary(next));
            return Optional.from(next).filter(isText$b);
        };
        return {
            current: () => Optional.from(walker.current()).filter(isText$b),
            next: () => walk('next'),
            prev: () => walk('prev'),
            prev2: () => walk('prev2')
        };
    };

    /**
     * The TextSeeker class enables you to seek for a specific point in text across the DOM.
     *
     * @class tinymce.dom.TextSeeker
     * @example
     * const seeker = tinymce.dom.TextSeeker(editor.dom);
     * const startOfWord = seeker.backwards(startNode, startOffset, (textNode, offset, text) => {
     *   const lastSpaceCharIndex = text.lastIndexOf(' ');
     *   if (lastSpaceCharIndex !== -1) {
     *     return lastSpaceCharIndex + 1;
     *   } else {
     *     // No space found so continue searching
     *     return -1;
     *   }
     * });
     */
    /**
     * Constructs a new TextSeeker instance.
     *
     * @constructor
     * @param {tinymce.dom.DOMUtils} dom DOMUtils object reference.
     * @param {Function} isBoundary Optional function to determine if the seeker should continue to walk past the node provided. The default is to search until a block or <code>br</code> element is found.
     */
    const TextSeeker = (dom, isBoundary) => {
        const isBlockBoundary = isBoundary ? isBoundary : (node) => dom.isBlock(node) || isBr$7(node) || isContentEditableFalse$a(node);
        const walk = (node, offset, walker, process) => {
            if (isText$b(node)) {
                const newOffset = process(node, offset, node.data);
                if (newOffset !== -1) {
                    return Optional.some({ container: node, offset: newOffset });
                }
            }
            return walker().bind((next) => walk(next.container, next.offset, walker, process));
        };
        /**
         * Search backwards through text nodes until a match, boundary, or root node has been found.
         *
         * @method backwards
         * @param {Node} node The node to start searching from.
         * @param {Number} offset The offset of the node to start searching from.
         * @param {Function} process A function that's passed the current text node, the current offset and the text content of the node. It should return the offset of the match or -1 to continue searching.
         * @param {Node} root An optional root node to constrain the search to.
         * @return {Object} An object containing the matched text node and offset. If no match is found, null will be returned.
         */
        const backwards = (node, offset, process, root) => {
            const walker = TextWalker(node, root ?? dom.getRoot(), isBlockBoundary);
            return walk(node, offset, () => walker.prev().map((prev) => ({ container: prev, offset: prev.length })), process).getOrNull();
        };
        /**
         * Search forwards through text nodes until a match, boundary, or root node has been found.
         *
         * @method forwards
         * @param {Node} node The node to start searching from.
         * @param {Number} offset The offset of the node to start searching from.
         * @param {Function} process A function that's passed the current text node, the current offset and the text content of the node. It should return the offset of the match or -1 to continue searching.
         * @param {Node} root An optional root node to constrain the search to.
         * @return {Object} An object containing the matched text node and offset. If no match is found, null will be returned.
         */
        const forwards = (node, offset, process, root) => {
            const walker = TextWalker(node, root ?? dom.getRoot(), isBlockBoundary);
            return walk(node, offset, () => walker.next().map((next) => ({ container: next, offset: 0 })), process).getOrNull();
        };
        return {
            backwards,
            forwards
        };
    };

    const tableCells = ['td', 'th'];
    const tableSections = ['thead', 'tbody', 'tfoot'];
    const textBlocks = [
        'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'div', 'address', 'pre', 'form',
        'blockquote', 'center', 'dir', 'fieldset', 'header', 'footer', 'article',
        'section', 'hgroup', 'aside', 'nav', 'figure'
    ];
    const listItems$1 = ['li', 'dd', 'dt'];
    const lists = ['ul', 'ol', 'dl'];
    const wsElements = ['pre', 'script', 'textarea', 'style'];
    const lazyLookup = (items) => {
        let lookup;
        return (node) => {
            lookup = lookup ? lookup : mapToObject(items, always);
            return has$2(lookup, name(node));
        };
    };
    // WARNING: don't add anything to this file, the intention is to move these checks into the Schema
    const isTable$1 = (node) => name(node) === 'table';
    const isBr$6 = (node) => isElement$8(node) && name(node) === 'br';
    const isTextBlock$3 = lazyLookup(textBlocks);
    const isList$1 = lazyLookup(lists);
    const isListItem$2 = lazyLookup(listItems$1);
    const isTableSection = lazyLookup(tableSections);
    const isTableCell$2 = lazyLookup(tableCells);
    const isWsPreserveElement = lazyLookup(wsElements);

    const getLastChildren$1 = (elm) => {
        const children = [];
        let rawNode = elm.dom;
        while (rawNode) {
            children.push(SugarElement.fromDom(rawNode));
            rawNode = rawNode.lastChild;
        }
        return children;
    };
    const removeTrailingBr = (elm) => {
        const allBrs = descendants(elm, 'br');
        const brs = filter$5(getLastChildren$1(elm).slice(-1), isBr$6);
        if (allBrs.length === brs.length) {
            each$e(brs, remove$8);
        }
    };
    const createPaddingBr = () => {
        const br = SugarElement.fromTag('br');
        set$4(br, 'data-mce-bogus', '1');
        return br;
    };
    const fillWithPaddingBr = (elm) => {
        empty(elm);
        append$1(elm, createPaddingBr());
    };
    const trimBlockTrailingBr = (elm, schema) => {
        lastChild(elm).each((lastChild) => {
            prevSibling(lastChild).each((lastChildPrevSibling) => {
                if (schema.isBlock(name(elm)) && isBr$6(lastChild) && schema.isBlock(name(lastChildPrevSibling))) {
                    remove$8(lastChild);
                }
            });
        });
    };

    /**
     * Utility functions for working with zero width space
     * characters used as character containers etc.
     *
     * @private
     * @class tinymce.text.Zwsp
     * @example
     * const isZwsp = Zwsp.isZwsp('\uFEFF');
     * const abc = Zwsp.trim('a\uFEFFc');
     */
    // This is technically not a ZWSP but a ZWNBSP or a BYTE ORDER MARK it used to be a ZWSP
    const ZWSP$1 = zeroWidth;
    const isZwsp = isZwsp$2;
    const trim$2 = removeZwsp;
    const insert$5 = (editor) => editor.insertContent(ZWSP$1, { preserve_zwsp: true });

    /**
     * This module handles caret containers. A caret container is a node that
     * holds the caret for positional purposes.
     *
     * @private
     * @class tinymce.caret.CaretContainer
     */
    const isElement$6 = isElement$7;
    const isText$9 = isText$b;
    const isCaretContainerBlock$1 = (node) => {
        if (isText$9(node)) {
            node = node.parentNode;
        }
        return isElement$6(node) && node.hasAttribute('data-mce-caret');
    };
    const isCaretContainerInline = (node) => isText$9(node) && isZwsp(node.data);
    const isCaretContainer$2 = (node) => isCaretContainerBlock$1(node) || isCaretContainerInline(node);
    const hasContent = (node) => node.firstChild !== node.lastChild || !isBr$7(node.firstChild);
    const insertInline$1 = (node, before) => {
        const doc = node.ownerDocument ?? document;
        const textNode = doc.createTextNode(ZWSP$1);
        const parentNode = node.parentNode;
        if (!before) {
            const sibling = node.nextSibling;
            if (isText$9(sibling)) {
                if (isCaretContainer$2(sibling)) {
                    return sibling;
                }
                if (startsWithCaretContainer$1(sibling)) {
                    sibling.splitText(1);
                    return sibling;
                }
            }
            if (node.nextSibling) {
                parentNode?.insertBefore(textNode, node.nextSibling);
            }
            else {
                parentNode?.appendChild(textNode);
            }
        }
        else {
            const sibling = node.previousSibling;
            if (isText$9(sibling)) {
                if (isCaretContainer$2(sibling)) {
                    return sibling;
                }
                if (endsWithCaretContainer$1(sibling)) {
                    return sibling.splitText(sibling.data.length - 1);
                }
            }
            parentNode?.insertBefore(textNode, node);
        }
        return textNode;
    };
    const isBeforeInline = (pos) => {
        const container = pos.container();
        if (!isText$b(container)) {
            return false;
        }
        // The text nodes may not be normalized, so check the current node and the previous one
        return container.data.charAt(pos.offset()) === ZWSP$1 || pos.isAtStart() && isCaretContainerInline(container.previousSibling);
    };
    const isAfterInline = (pos) => {
        const container = pos.container();
        if (!isText$b(container)) {
            return false;
        }
        // The text nodes may not be normalized, so check the current node and the next one
        return container.data.charAt(pos.offset() - 1) === ZWSP$1 || pos.isAtEnd() && isCaretContainerInline(container.nextSibling);
    };
    const insertBlock = (blockName, node, before) => {
        const doc = node.ownerDocument ?? document;
        const blockNode = doc.createElement(blockName);
        blockNode.setAttribute('data-mce-caret', before ? 'before' : 'after');
        blockNode.setAttribute('data-mce-bogus', 'all');
        blockNode.appendChild(createPaddingBr().dom);
        const parentNode = node.parentNode;
        if (!before) {
            if (node.nextSibling) {
                parentNode?.insertBefore(blockNode, node.nextSibling);
            }
            else {
                parentNode?.appendChild(blockNode);
            }
        }
        else {
            parentNode?.insertBefore(blockNode, node);
        }
        return blockNode;
    };
    const startsWithCaretContainer$1 = (node) => isText$9(node) && node.data[0] === ZWSP$1;
    const endsWithCaretContainer$1 = (node) => isText$9(node) && node.data[node.data.length - 1] === ZWSP$1;
    const trimBogusBr = (elm) => {
        const brs = elm.getElementsByTagName('br');
        const lastBr = brs[brs.length - 1];
        if (isBogus$1(lastBr)) {
            lastBr.parentNode?.removeChild(lastBr);
        }
    };
    const showCaretContainerBlock = (caretContainer) => {
        if (caretContainer && caretContainer.hasAttribute('data-mce-caret')) {
            trimBogusBr(caretContainer);
            caretContainer.removeAttribute('data-mce-caret');
            caretContainer.removeAttribute('data-mce-bogus');
            caretContainer.removeAttribute('style');
            caretContainer.removeAttribute('data-mce-style');
            caretContainer.removeAttribute('_moz_abspos');
            return caretContainer;
        }
        return null;
    };
    const isRangeInCaretContainerBlock = (range) => isCaretContainerBlock$1(range.startContainer);

    const round$2 = Math.round;
    const clone$1 = (rect) => {
        if (!rect) {
            return { left: 0, top: 0, bottom: 0, right: 0, width: 0, height: 0 };
        }
        return {
            left: round$2(rect.left),
            top: round$2(rect.top),
            bottom: round$2(rect.bottom),
            right: round$2(rect.right),
            width: round$2(rect.width),
            height: round$2(rect.height)
        };
    };
    const collapse = (rect, toStart) => {
        rect = clone$1(rect);
        if (toStart) {
            rect.right = rect.left;
        }
        else {
            rect.left = rect.left + rect.width;
            rect.right = rect.left;
        }
        rect.width = 0;
        return rect;
    };
    const isEqual = (rect1, rect2) => (rect1.left === rect2.left &&
        rect1.top === rect2.top &&
        rect1.bottom === rect2.bottom &&
        rect1.right === rect2.right);
    const isValidOverflow = (overflowY, rect1, rect2) => overflowY >= 0 && overflowY <= Math.min(rect1.height, rect2.height) / 2;
    const isAbove$1 = (rect1, rect2) => {
        const halfHeight = Math.min(rect2.height / 2, rect1.height / 2);
        if ((rect1.bottom - halfHeight) < rect2.top) {
            return true;
        }
        if (rect1.top > rect2.bottom) {
            return false;
        }
        return isValidOverflow(rect2.top - rect1.bottom, rect1, rect2);
    };
    const isBelow$1 = (rect1, rect2) => {
        if (rect1.top > rect2.bottom) {
            return true;
        }
        if (rect1.bottom < rect2.top) {
            return false;
        }
        return isValidOverflow(rect2.bottom - rect1.top, rect1, rect2);
    };
    const containsXY = (rect, clientX, clientY) => (clientX >= rect.left &&
        clientX <= rect.right &&
        clientY >= rect.top &&
        clientY <= rect.bottom);
    const boundingClientRectFromRects = (rects) => {
        return foldl(rects, (acc, rect) => {
            return acc.fold(() => Optional.some(rect), (prevRect) => {
                const left = Math.min(rect.left, prevRect.left);
                const top = Math.min(rect.top, prevRect.top);
                const right = Math.max(rect.right, prevRect.right);
                const bottom = Math.max(rect.bottom, prevRect.bottom);
                return Optional.some({
                    top,
                    right,
                    bottom,
                    left,
                    width: right - left,
                    height: bottom - top
                });
            });
        }, Optional.none());
    };
    const distanceToRectEdgeFromXY = (rect, x, y) => {
        const cx = Math.max(Math.min(x, rect.left + rect.width), rect.left);
        const cy = Math.max(Math.min(y, rect.top + rect.height), rect.top);
        return Math.sqrt((x - cx) * (x - cx) + (y - cy) * (y - cy));
    };
    const overlapY = (r1, r2) => Math.max(0, Math.min(r1.bottom, r2.bottom) - Math.max(r1.top, r2.top));

    const getSelectedNode = (range) => {
        const startContainer = range.startContainer, startOffset = range.startOffset;
        if (startContainer === range.endContainer && startContainer.hasChildNodes() && range.endOffset === startOffset + 1) {
            return startContainer.childNodes[startOffset];
        }
        return null;
    };
    const getNode$1 = (container, offset) => {
        if (isElement$7(container) && container.hasChildNodes()) {
            const childNodes = container.childNodes;
            const safeOffset = clamp$2(offset, 0, childNodes.length - 1);
            return childNodes[safeOffset];
        }
        else {
            return container;
        }
    };
    /** @deprecated Use getNode instead */
    const getNodeUnsafe = (container, offset) => {
        // If a negative offset is used on an element then `undefined` should be returned
        if (offset < 0 && isElement$7(container) && container.hasChildNodes()) {
            return undefined;
        }
        else {
            return getNode$1(container, offset);
        }
    };

    /**
     * This class contains logic for detecting extending characters.
     *
     * @private
     * @class tinymce.text.ExtendingChar
     * @example
     * const isExtending = ExtendingChar.isExtendingChar('a');
     */
    // Generated from: http://www.unicode.org/Public/UNIDATA/DerivedCoreProperties.txt
    // Only includes the characters in that fit into UCS-2 16 bit
    const extendingChars = new RegExp('[\u0300-\u036F\u0483-\u0487\u0488-\u0489\u0591-\u05BD\u05BF\u05C1-\u05C2\u05C4-\u05C5\u05C7\u0610-\u061A' +
        '\u064B-\u065F\u0670\u06D6-\u06DC\u06DF-\u06E4\u06E7-\u06E8\u06EA-\u06ED\u0711\u0730-\u074A\u07A6-\u07B0' +
        '\u07EB-\u07F3\u0816-\u0819\u081B-\u0823\u0825-\u0827\u0829-\u082D\u0859-\u085B\u08E3-\u0902\u093A\u093C' +
        '\u0941-\u0948\u094D\u0951-\u0957\u0962-\u0963\u0981\u09BC\u09BE\u09C1-\u09C4\u09CD\u09D7\u09E2-\u09E3' +
        '\u0A01-\u0A02\u0A3C\u0A41-\u0A42\u0A47-\u0A48\u0A4B-\u0A4D\u0A51\u0A70-\u0A71\u0A75\u0A81-\u0A82\u0ABC' +
        '\u0AC1-\u0AC5\u0AC7-\u0AC8\u0ACD\u0AE2-\u0AE3\u0B01\u0B3C\u0B3E\u0B3F\u0B41-\u0B44\u0B4D\u0B56\u0B57' +
        '\u0B62-\u0B63\u0B82\u0BBE\u0BC0\u0BCD\u0BD7\u0C00\u0C3E-\u0C40\u0C46-\u0C48\u0C4A-\u0C4D\u0C55-\u0C56' +
        '\u0C62-\u0C63\u0C81\u0CBC\u0CBF\u0CC2\u0CC6\u0CCC-\u0CCD\u0CD5-\u0CD6\u0CE2-\u0CE3\u0D01\u0D3E\u0D41-\u0D44' +
        '\u0D4D\u0D57\u0D62-\u0D63\u0DCA\u0DCF\u0DD2-\u0DD4\u0DD6\u0DDF\u0E31\u0E34-\u0E3A\u0E47-\u0E4E\u0EB1\u0EB4-\u0EB9' +
        '\u0EBB-\u0EBC\u0EC8-\u0ECD\u0F18-\u0F19\u0F35\u0F37\u0F39\u0F71-\u0F7E\u0F80-\u0F84\u0F86-\u0F87\u0F8D-\u0F97' +
        '\u0F99-\u0FBC\u0FC6\u102D-\u1030\u1032-\u1037\u1039-\u103A\u103D-\u103E\u1058-\u1059\u105E-\u1060\u1071-\u1074' +
        '\u1082\u1085-\u1086\u108D\u109D\u135D-\u135F\u1712-\u1714\u1732-\u1734\u1752-\u1753\u1772-\u1773\u17B4-\u17B5' +
        '\u17B7-\u17BD\u17C6\u17C9-\u17D3\u17DD\u180B-\u180D\u18A9\u1920-\u1922\u1927-\u1928\u1932\u1939-\u193B\u1A17-\u1A18' +
        '\u1A1B\u1A56\u1A58-\u1A5E\u1A60\u1A62\u1A65-\u1A6C\u1A73-\u1A7C\u1A7F\u1AB0-\u1ABD\u1ABE\u1B00-\u1B03\u1B34' +
        '\u1B36-\u1B3A\u1B3C\u1B42\u1B6B-\u1B73\u1B80-\u1B81\u1BA2-\u1BA5\u1BA8-\u1BA9\u1BAB-\u1BAD\u1BE6\u1BE8-\u1BE9' +
        '\u1BED\u1BEF-\u1BF1\u1C2C-\u1C33\u1C36-\u1C37\u1CD0-\u1CD2\u1CD4-\u1CE0\u1CE2-\u1CE8\u1CED\u1CF4\u1CF8-\u1CF9' +
        '\u1DC0-\u1DF5\u1DFC-\u1DFF\u200C-\u200D\u20D0-\u20DC\u20DD-\u20E0\u20E1\u20E2-\u20E4\u20E5-\u20F0\u2CEF-\u2CF1' +
        '\u2D7F\u2DE0-\u2DFF\u302A-\u302D\u302E-\u302F\u3099-\u309A\uA66F\uA670-\uA672\uA674-\uA67D\uA69E-\uA69F\uA6F0-\uA6F1' +
        '\uA802\uA806\uA80B\uA825-\uA826\uA8C4\uA8E0-\uA8F1\uA926-\uA92D\uA947-\uA951\uA980-\uA982\uA9B3\uA9B6-\uA9B9\uA9BC' +
        '\uA9E5\uAA29-\uAA2E\uAA31-\uAA32\uAA35-\uAA36\uAA43\uAA4C\uAA7C\uAAB0\uAAB2-\uAAB4\uAAB7-\uAAB8\uAABE-\uAABF\uAAC1' +
        '\uAAEC-\uAAED\uAAF6\uABE5\uABE8\uABED\uFB1E\uFE00-\uFE0F\uFE20-\uFE2F\uFF9E-\uFF9F]');
    const isExtendingChar = (ch) => isString(ch) && ch.charCodeAt(0) >= 768 && extendingChars.test(ch);

    const or = (...args) => {
        return (x) => {
            for (let i = 0; i < args.length; i++) {
                if (args[i](x)) {
                    return true;
                }
            }
            return false;
        };
    };
    const and = (...args) => {
        return (x) => {
            for (let i = 0; i < args.length; i++) {
                if (!args[i](x)) {
                    return false;
                }
            }
            return true;
        };
    };

    /**
     * This module contains logic for handling caret candidates. A caret candidate is
     * for example text nodes, images, input elements, cE=false elements etc.
     *
     * @private
     * @class tinymce.caret.CaretCandidate
     */
    const isContentEditableTrue$2 = isContentEditableTrue$3;
    const isContentEditableFalse$9 = isContentEditableFalse$a;
    const isBr$5 = isBr$7;
    const isText$8 = isText$b;
    const isInvalidTextElement = matchNodeNames$1(['script', 'style', 'textarea']);
    const isAtomicInline = matchNodeNames$1(['img', 'input', 'textarea', 'hr', 'iframe', 'video', 'audio', 'object', 'embed']);
    const isTable = matchNodeNames$1(['table']);
    const isCaretContainer$1 = isCaretContainer$2;
    const isCaretCandidate$3 = (node) => {
        if (isCaretContainer$1(node)) {
            return false;
        }
        if (isText$8(node)) {
            return !isInvalidTextElement(node.parentNode);
        }
        return isAtomicInline(node) || isBr$5(node) || isTable(node) || isNonUiContentEditableFalse(node);
    };
    // UI components on IE is marked with contenteditable=false and unselectable=true so lets not handle those as real content editables
    const isUnselectable = (node) => isElement$7(node) && node.getAttribute('unselectable') === 'true';
    const isNonUiContentEditableFalse = (node) => !isUnselectable(node) && isContentEditableFalse$9(node);
    const isInEditable = (node, root) => {
        for (let tempNode = node.parentNode; tempNode && tempNode !== root; tempNode = tempNode.parentNode) {
            if (isNonUiContentEditableFalse(tempNode)) {
                return false;
            }
            if (isContentEditableTrue$2(tempNode)) {
                return true;
            }
        }
        return true;
    };
    const isAtomicContentEditableFalse = (node) => {
        if (!isNonUiContentEditableFalse(node)) {
            return false;
        }
        return !foldl(from(node.getElementsByTagName('*')), (result, elm) => {
            return result || isContentEditableTrue$2(elm);
        }, false);
    };
    const isAtomic$1 = (node) => isAtomicInline(node) || isAtomicContentEditableFalse(node);
    const isEditableCaretCandidate$1 = (node, root) => isCaretCandidate$3(node) && isInEditable(node, root);

    /**
     * This module contains logic for creating caret positions within a document a caretposition
     * is similar to a DOMRange object but it doesn't have two endpoints and is also more lightweight
     * since it's now updated live when the DOM changes.
     *
     * @private
     * @class tinymce.caret.CaretPosition
     * @example
     * const caretPos1 = CaretPosition(container, offset);
     * const caretPos2 = CaretPosition.fromRangeStart(someRange);
     */
    const isElement$5 = isElement$7;
    const isCaretCandidate$2 = isCaretCandidate$3;
    const isBlock$3 = matchStyleValues('display', 'block table');
    const isFloated = matchStyleValues('float', 'left right');
    const isValidElementCaretCandidate = and(isElement$5, isCaretCandidate$2, not(isFloated));
    const isNotPre = not(matchStyleValues('white-space', 'pre pre-line pre-wrap'));
    const isText$7 = isText$b;
    const isBr$4 = isBr$7;
    const nodeIndex$1 = DOMUtils.nodeIndex;
    const resolveIndex$1 = getNodeUnsafe;
    const createRange$1 = (doc) => doc ? doc.createRange() : DOMUtils.DOM.createRng();
    const isWhiteSpace$1 = (chr) => isString(chr) && /[\r\n\t ]/.test(chr);
    const isRange = (rng) => !!rng.setStart && !!rng.setEnd;
    const isHiddenWhiteSpaceRange = (range) => {
        const container = range.startContainer;
        const offset = range.startOffset;
        if (isWhiteSpace$1(range.toString()) && isNotPre(container.parentNode) && isText$b(container)) {
            const text = container.data;
            if (isWhiteSpace$1(text[offset - 1]) || isWhiteSpace$1(text[offset + 1])) {
                return true;
            }
        }
        return false;
    };
    // Hack for older WebKit versions that doesn't
    // support getBoundingClientRect on BR elements
    const getBrClientRect = (brNode) => {
        const doc = brNode.ownerDocument;
        const rng = createRange$1(doc);
        const nbsp$1 = doc.createTextNode(nbsp);
        const parentNode = brNode.parentNode;
        parentNode.insertBefore(nbsp$1, brNode);
        rng.setStart(nbsp$1, 0);
        rng.setEnd(nbsp$1, 1);
        const clientRect = clone$1(rng.getBoundingClientRect());
        parentNode.removeChild(nbsp$1);
        return clientRect;
    };
    // Safari will not return a rect for <p>a<br>|b</p> for some odd reason
    const getBoundingClientRectWebKitText = (rng) => {
        const sc = rng.startContainer;
        const ec = rng.endContainer;
        const so = rng.startOffset;
        const eo = rng.endOffset;
        if (sc === ec && isText$b(ec) && so === 0 && eo === 1) {
            const newRng = rng.cloneRange();
            newRng.setEndAfter(ec);
            return getBoundingClientRect$1(newRng);
        }
        else {
            return null;
        }
    };
    const isZeroRect = (r) => r.left === 0 && r.right === 0 && r.top === 0 && r.bottom === 0;
    const getBoundingClientRect$1 = (item) => {
        let clientRect;
        const clientRects = item.getClientRects();
        if (clientRects.length > 0) {
            clientRect = clone$1(clientRects[0]);
        }
        else {
            clientRect = clone$1(item.getBoundingClientRect());
        }
        if (!isRange(item) && isBr$4(item) && isZeroRect(clientRect)) {
            return getBrClientRect(item);
        }
        if (isZeroRect(clientRect) && isRange(item)) {
            return getBoundingClientRectWebKitText(item) ?? clientRect;
        }
        return clientRect;
    };
    const collapseAndInflateWidth = (clientRect, toStart) => {
        const newClientRect = collapse(clientRect, toStart);
        newClientRect.width = 1;
        newClientRect.right = newClientRect.left + 1;
        return newClientRect;
    };
    const getCaretPositionClientRects = (caretPosition) => {
        const clientRects = [];
        const addUniqueAndValidRect = (clientRect) => {
            if (clientRect.height === 0) {
                return;
            }
            if (clientRects.length > 0) {
                if (isEqual(clientRect, clientRects[clientRects.length - 1])) {
                    return;
                }
            }
            clientRects.push(clientRect);
        };
        const addCharacterOffset = (container, offset) => {
            const range = createRange$1(container.ownerDocument);
            if (offset < container.data.length) {
                if (isExtendingChar(container.data[offset])) {
                    return;
                }
                // WebKit returns two client rects for a position after an extending
                // character a\uxxx|b so expand on "b" and collapse to start of "b" box
                if (isExtendingChar(container.data[offset - 1])) {
                    range.setStart(container, offset);
                    range.setEnd(container, offset + 1);
                    if (!isHiddenWhiteSpaceRange(range)) {
                        addUniqueAndValidRect(collapseAndInflateWidth(getBoundingClientRect$1(range), false));
                        return;
                    }
                }
            }
            if (offset > 0) {
                range.setStart(container, offset - 1);
                range.setEnd(container, offset);
                if (!isHiddenWhiteSpaceRange(range)) {
                    addUniqueAndValidRect(collapseAndInflateWidth(getBoundingClientRect$1(range), false));
                }
            }
            if (offset < container.data.length) {
                range.setStart(container, offset);
                range.setEnd(container, offset + 1);
                if (!isHiddenWhiteSpaceRange(range)) {
                    addUniqueAndValidRect(collapseAndInflateWidth(getBoundingClientRect$1(range), true));
                }
            }
        };
        const container = caretPosition.container();
        const offset = caretPosition.offset();
        if (isText$7(container)) {
            addCharacterOffset(container, offset);
            return clientRects;
        }
        if (isElement$5(container)) {
            if (caretPosition.isAtEnd()) {
                const node = resolveIndex$1(container, offset);
                if (isText$7(node)) {
                    addCharacterOffset(node, node.data.length);
                }
                if (isValidElementCaretCandidate(node) && !isBr$4(node)) {
                    addUniqueAndValidRect(collapseAndInflateWidth(getBoundingClientRect$1(node), false));
                }
            }
            else {
                const node = resolveIndex$1(container, offset);
                if (isText$7(node)) {
                    addCharacterOffset(node, 0);
                }
                if (isValidElementCaretCandidate(node) && caretPosition.isAtEnd()) {
                    addUniqueAndValidRect(collapseAndInflateWidth(getBoundingClientRect$1(node), false));
                    return clientRects;
                }
                const beforeNode = resolveIndex$1(caretPosition.container(), caretPosition.offset() - 1);
                if (isValidElementCaretCandidate(beforeNode) && !isBr$4(beforeNode)) {
                    if (isBlock$3(beforeNode) || isBlock$3(node) || !isValidElementCaretCandidate(node)) {
                        addUniqueAndValidRect(collapseAndInflateWidth(getBoundingClientRect$1(beforeNode), false));
                    }
                }
                if (isValidElementCaretCandidate(node)) {
                    addUniqueAndValidRect(collapseAndInflateWidth(getBoundingClientRect$1(node), true));
                }
            }
        }
        return clientRects;
    };
    /**
     * Represents a location within the document by a container and an offset.
     *
     * @constructor
     * @param {Node} container Container node.
     * @param {Number} offset Offset within that container node.
     * @param {Array} clientRects Optional client rects array for the position.
     */
    const CaretPosition = (container, offset, clientRects) => {
        const isAtStart = () => {
            if (isText$7(container)) {
                return offset === 0;
            }
            return offset === 0;
        };
        const isAtEnd = () => {
            if (isText$7(container)) {
                return offset >= container.data.length;
            }
            return offset >= container.childNodes.length;
        };
        const toRange = () => {
            const range = createRange$1(container.ownerDocument);
            range.setStart(container, offset);
            range.setEnd(container, offset);
            return range;
        };
        const getClientRects = () => {
            if (!clientRects) {
                clientRects = getCaretPositionClientRects(CaretPosition(container, offset));
            }
            return clientRects;
        };
        const isVisible = () => getClientRects().length > 0;
        const isEqual = (caretPosition) => caretPosition && container === caretPosition.container() && offset === caretPosition.offset();
        const getNode = (before) => resolveIndex$1(container, before ? offset - 1 : offset);
        return {
            /**
             * Returns the container node.
             *
             * @method container
             * @return {Node} Container node.
             */
            container: constant(container),
            /**
             * Returns the offset within the container node.
             *
             * @method offset
             * @return {Number} Offset within the container node.
             */
            offset: constant(offset),
            /**
             * Returns a range out of a the caret position.
             *
             * @method toRange
             * @return {DOMRange} range for the caret position.
             */
            toRange,
            /**
             * Returns the client rects for the caret position. Might be multiple rects between
             * block elements.
             *
             * @method getClientRects
             * @return {Array} Array of client rects.
             */
            getClientRects,
            /**
             * Returns true if the caret location is visible/displayed on screen.
             *
             * @method isVisible
             * @return {Boolean} true/false if the position is visible or not.
             */
            isVisible,
            /**
             * Returns true if the caret location is at the beginning of text node or container.
             *
             * @method isVisible
             * @return {Boolean} true/false if the position is at the beginning.
             */
            isAtStart,
            /**
             * Returns true if the caret location is at the end of text node or container.
             *
             * @method isVisible
             * @return {Boolean} true/false if the position is at the end.
             */
            isAtEnd,
            /**
             * Compares the caret position to another caret position. This will only compare the
             * container and offset not it's visual position.
             *
             * @method isEqual
             * @param {tinymce.caret.CaretPosition} caretPosition Caret position to compare with.
             * @return {Boolean} true if the caret positions are equal.
             */
            isEqual,
            /**
             * Returns the closest resolved node from a node index. That means if you have an offset after the
             * last node in a container it will return that last node.
             *
             * @method getNode
             * @return {Node} Node that is closest to the index.
             */
            getNode
        };
    };
    /**
     * Creates a caret position from the start of a range.
     *
     * @method fromRangeStart
     * @param {DOMRange} range DOM Range to create caret position from.
     * @return {tinymce.caret.CaretPosition} Caret position from the start of DOM range.
     */
    CaretPosition.fromRangeStart = (range) => CaretPosition(range.startContainer, range.startOffset);
    /**
     * Creates a caret position from the end of a range.
     *
     * @method fromRangeEnd
     * @param {DOMRange} range DOM Range to create caret position from.
     * @return {tinymce.caret.CaretPosition} Caret position from the end of DOM range.
     */
    CaretPosition.fromRangeEnd = (range) => CaretPosition(range.endContainer, range.endOffset);
    /**
     * Creates a caret position from a node and places the offset after it.
     *
     * @method after
     * @param {Node} node Node to get caret position from.
     * @return {tinymce.caret.CaretPosition} Caret position from the node.
     */
    // TODO: TINY-8865 - This may not be safe to cast as Node and alternative solutions need to be looked into
    CaretPosition.after = (node) => CaretPosition(node.parentNode, nodeIndex$1(node) + 1);
    /**
     * Creates a caret position from a node and places the offset before it.
     *
     * @method before
     * @param {Node} node Node to get caret position from.
     * @return {tinymce.caret.CaretPosition} Caret position from the node.
     */
    // TODO: TINY-8865 - This may not be safe to cast as Node and alternative solutions need to be looked into
    CaretPosition.before = (node) => CaretPosition(node.parentNode, nodeIndex$1(node));
    CaretPosition.isAbove = (pos1, pos2) => lift2(head(pos2.getClientRects()), last$2(pos1.getClientRects()), isAbove$1).getOr(false);
    CaretPosition.isBelow = (pos1, pos2) => lift2(last$2(pos2.getClientRects()), head(pos1.getClientRects()), isBelow$1).getOr(false);
    CaretPosition.isAtStart = (pos) => pos ? pos.isAtStart() : false;
    CaretPosition.isAtEnd = (pos) => pos ? pos.isAtEnd() : false;
    CaretPosition.isTextPosition = (pos) => pos ? isText$b(pos.container()) : false;
    CaretPosition.isElementPosition = (pos) => !CaretPosition.isTextPosition(pos);

    const trimEmptyTextNode$1 = (dom, node) => {
        if (isText$b(node) && node.data.length === 0) {
            dom.remove(node);
        }
    };
    const insertNode = (dom, rng, node) => {
        rng.insertNode(node);
        trimEmptyTextNode$1(dom, node.previousSibling);
        trimEmptyTextNode$1(dom, node.nextSibling);
    };
    const insertFragment = (dom, rng, frag) => {
        const firstChild = Optional.from(frag.firstChild);
        const lastChild = Optional.from(frag.lastChild);
        rng.insertNode(frag);
        firstChild.each((child) => trimEmptyTextNode$1(dom, child.previousSibling));
        lastChild.each((child) => trimEmptyTextNode$1(dom, child.nextSibling));
    };
    // Wrapper to Range.insertNode which removes any empty text nodes created in the process.
    // Doesn't merge adjacent text nodes - this is according to the DOM spec.
    const rangeInsertNode = (dom, rng, node) => {
        if (isDocumentFragment(node)) {
            insertFragment(dom, rng, node);
        }
        else {
            insertNode(dom, rng, node);
        }
    };

    /**
     * This module creates or resolves xpath like string representation of a CaretPositions.
     *
     * The format is a / separated list of chunks with:
     * <element|text()>[index|after|before]
     *
     * For example:
     *  p[0]/b[0]/text()[0],1 = <p><b>a|c</b></p>
     *  p[0]/img[0],before = <p>|<img></p>
     *  p[0]/img[0],after = <p><img>|</p>
     *
     * @private
     * @static
     * @class tinymce.caret.CaretBookmark
     * @example
     * const bookmark = CaretBookmark.create(rootElm, CaretPosition.before(rootElm.firstChild));
     * const caretPosition = CaretBookmark.resolve(bookmark);
     */
    const isText$6 = isText$b;
    const isBogus = isBogus$1;
    const nodeIndex = DOMUtils.nodeIndex;
    const normalizedParent = (node) => {
        const parentNode = node.parentNode;
        if (isBogus(parentNode)) {
            return normalizedParent(parentNode);
        }
        return parentNode;
    };
    const getChildNodes = (node) => {
        if (!node) {
            return [];
        }
        return reduce(node.childNodes, (result, node) => {
            if (isBogus(node) && node.nodeName !== 'BR') {
                result = result.concat(getChildNodes(node));
            }
            else {
                result.push(node);
            }
            return result;
        }, []);
    };
    const normalizedTextOffset = (node, offset) => {
        let tempNode = node;
        while ((tempNode = tempNode.previousSibling)) {
            if (!isText$6(tempNode)) {
                break;
            }
            offset += tempNode.data.length;
        }
        return offset;
    };
    const equal = (a) => (b) => a === b;
    const normalizedNodeIndex = (node) => {
        let nodes, index;
        nodes = getChildNodes(normalizedParent(node));
        index = findIndex$1(nodes, equal(node), node);
        nodes = nodes.slice(0, index + 1);
        const numTextFragments = reduce(nodes, (result, node, i) => {
            if (isText$6(node) && isText$6(nodes[i - 1])) {
                result++;
            }
            return result;
        }, 0);
        nodes = filter$3(nodes, matchNodeNames$1([node.nodeName]));
        index = findIndex$1(nodes, equal(node), node);
        return index - numTextFragments;
    };
    const createPathItem = (node) => {
        const name = isText$6(node) ? 'text()' : node.nodeName.toLowerCase();
        return name + '[' + normalizedNodeIndex(node) + ']';
    };
    const parentsUntil$1 = (root, node, predicate) => {
        const parents = [];
        for (let tempNode = node.parentNode; tempNode && tempNode !== root; tempNode = tempNode.parentNode) {
            if (predicate && predicate(tempNode)) {
                break;
            }
            parents.push(tempNode);
        }
        return parents;
    };
    const create$9 = (root, caretPosition) => {
        let path = [];
        let container = caretPosition.container();
        let offset = caretPosition.offset();
        let outputOffset;
        if (isText$6(container)) {
            outputOffset = normalizedTextOffset(container, offset);
        }
        else {
            const childNodes = container.childNodes;
            if (offset >= childNodes.length) {
                outputOffset = 'after';
                offset = childNodes.length - 1;
            }
            else {
                outputOffset = 'before';
            }
            container = childNodes[offset];
        }
        path.push(createPathItem(container));
        let parents = parentsUntil$1(root, container);
        parents = filter$3(parents, not(isBogus$1));
        path = path.concat(map$1(parents, (node) => {
            return createPathItem(node);
        }));
        return path.reverse().join('/') + ',' + outputOffset;
    };
    const resolvePathItem = (node, name, index) => {
        let nodes = getChildNodes(node);
        nodes = filter$3(nodes, (node, index) => {
            return !isText$6(node) || !isText$6(nodes[index - 1]);
        });
        nodes = filter$3(nodes, matchNodeNames$1([name]));
        return nodes[index];
    };
    const findTextPosition = (container, offset) => {
        let node = container;
        let targetOffset = 0;
        while (isText$6(node)) {
            const dataLen = node.data.length;
            if (offset >= targetOffset && offset <= targetOffset + dataLen) {
                container = node;
                offset = offset - targetOffset;
                break;
            }
            if (!isText$6(node.nextSibling)) {
                container = node;
                offset = dataLen;
                break;
            }
            targetOffset += dataLen;
            node = node.nextSibling;
        }
        if (isText$6(container) && offset > container.data.length) {
            offset = container.data.length;
        }
        return CaretPosition(container, offset);
    };
    const resolve$1 = (root, path) => {
        if (!path) {
            return null;
        }
        const parts = path.split(',');
        const paths = parts[0].split('/');
        const offset = parts.length > 1 ? parts[1] : 'before';
        const container = reduce(paths, (result, value) => {
            const match = /([\w\-\(\)]+)\[([0-9]+)\]/.exec(value);
            if (!match) {
                return null;
            }
            if (match[1] === 'text()') {
                match[1] = '#text';
            }
            return resolvePathItem(result, match[1], parseInt(match[2], 10));
        }, root);
        if (!container) {
            return null;
        }
        if (!isText$6(container) && container.parentNode) {
            let nodeOffset;
            if (offset === 'after') {
                nodeOffset = nodeIndex(container) + 1;
            }
            else {
                nodeOffset = nodeIndex(container);
            }
            return CaretPosition(container.parentNode, nodeOffset);
        }
        return findTextPosition(container, parseInt(offset, 10));
    };

    const isContentEditableFalse$8 = isContentEditableFalse$a;
    const getNormalizedTextOffset$1 = (trim, container, offset) => {
        let trimmedOffset = trim(container.data.slice(0, offset)).length;
        for (let node = container.previousSibling; node && isText$b(node); node = node.previousSibling) {
            trimmedOffset += trim(node.data).length;
        }
        return trimmedOffset;
    };
    const getPoint = (dom, trim, normalized, rng, start) => {
        const container = start ? rng.startContainer : rng.endContainer;
        let offset = start ? rng.startOffset : rng.endOffset;
        const point = [];
        const root = dom.getRoot();
        if (isText$b(container)) {
            point.push(normalized ? getNormalizedTextOffset$1(trim, container, offset) : offset);
        }
        else {
            let after = 0;
            const childNodes = container.childNodes;
            if (offset >= childNodes.length && childNodes.length) {
                after = 1;
                offset = Math.max(0, childNodes.length - 1);
            }
            point.push(dom.nodeIndex(childNodes[offset], normalized) + after);
        }
        for (let node = container; node && node !== root; node = node.parentNode) {
            point.push(dom.nodeIndex(node, normalized));
        }
        return point;
    };
    const getLocation = (trim, selection, normalized, rng) => {
        const dom = selection.dom;
        const start = getPoint(dom, trim, normalized, rng, true);
        const forward = selection.isForward();
        const fakeCaret = isRangeInCaretContainerBlock(rng) ? { isFakeCaret: true } : {};
        if (!selection.isCollapsed()) {
            const end = getPoint(dom, trim, normalized, rng, false);
            return { start, end, forward, ...fakeCaret };
        }
        else {
            return { start, forward, ...fakeCaret };
        }
    };
    const findIndex = (dom, name, element) => {
        let count = 0;
        Tools.each(dom.select(name), (node) => {
            if (node.getAttribute('data-mce-bogus') === 'all') {
                return;
            }
            else if (node === element) {
                return false;
            }
            else {
                count++;
                return;
            }
        });
        return count;
    };
    const moveEndPoint$1 = (rng, start) => {
        let container = start ? rng.startContainer : rng.endContainer;
        let offset = start ? rng.startOffset : rng.endOffset;
        // normalize Table Cell selection
        if (isElement$7(container) && container.nodeName === 'TR') {
            const childNodes = container.childNodes;
            container = childNodes[Math.min(start ? offset : offset - 1, childNodes.length - 1)];
            if (container) {
                offset = start ? 0 : container.childNodes.length;
                if (start) {
                    rng.setStart(container, offset);
                }
                else {
                    rng.setEnd(container, offset);
                }
            }
        }
    };
    const normalizeTableCellSelection = (rng) => {
        moveEndPoint$1(rng, true);
        moveEndPoint$1(rng, false);
        return rng;
    };
    const findSibling = (node, offset) => {
        if (isElement$7(node)) {
            node = getNode$1(node, offset);
            if (isContentEditableFalse$8(node)) {
                return node;
            }
        }
        if (isCaretContainer$2(node)) {
            if (isText$b(node) && isCaretContainerBlock$1(node)) {
                node = node.parentNode;
            }
            let sibling = node.previousSibling;
            if (isContentEditableFalse$8(sibling)) {
                return sibling;
            }
            sibling = node.nextSibling;
            if (isContentEditableFalse$8(sibling)) {
                return sibling;
            }
        }
        return undefined;
    };
    const findAdjacentContentEditableFalseElm = (rng) => {
        return findSibling(rng.startContainer, rng.startOffset) || findSibling(rng.endContainer, rng.endOffset);
    };
    const getOffsetBookmark = (trim, normalized, selection) => {
        const element = selection.getNode();
        const rng = selection.getRng();
        if (element.nodeName === 'IMG' || isContentEditableFalse$8(element)) {
            const name = element.nodeName;
            return { name, index: findIndex(selection.dom, name, element) };
        }
        const sibling = findAdjacentContentEditableFalseElm(rng);
        if (sibling) {
            const name = sibling.tagName;
            return { name, index: findIndex(selection.dom, name, sibling) };
        }
        return getLocation(trim, selection, normalized, rng);
    };
    const getCaretBookmark = (selection) => {
        const rng = selection.getRng();
        return {
            start: create$9(selection.dom.getRoot(), CaretPosition.fromRangeStart(rng)),
            end: create$9(selection.dom.getRoot(), CaretPosition.fromRangeEnd(rng)),
            forward: selection.isForward()
        };
    };
    const getRangeBookmark = (selection) => {
        return { rng: selection.getRng(), forward: selection.isForward() };
    };
    const createBookmarkSpan = (dom, id, filled) => {
        const args = { 'data-mce-type': 'bookmark', id, 'style': 'overflow:hidden;line-height:0px' };
        return filled ? dom.create('span', args, '&#xFEFF;') : dom.create('span', args);
    };
    const getPersistentBookmark = (selection, filled) => {
        const dom = selection.dom;
        let rng = selection.getRng();
        const id = dom.uniqueId();
        const collapsed = selection.isCollapsed();
        const element = selection.getNode();
        const name = element.nodeName;
        const forward = selection.isForward();
        if (name === 'IMG') {
            return { name, index: findIndex(dom, name, element) };
        }
        // W3C method
        const rng2 = normalizeTableCellSelection(rng.cloneRange());
        // Insert end marker
        if (!collapsed) {
            rng2.collapse(false);
            const endBookmarkNode = createBookmarkSpan(dom, id + '_end', filled);
            rangeInsertNode(dom, rng2, endBookmarkNode);
        }
        rng = normalizeTableCellSelection(rng);
        rng.collapse(true);
        const startBookmarkNode = createBookmarkSpan(dom, id + '_start', filled);
        rangeInsertNode(dom, rng, startBookmarkNode);
        selection.moveToBookmark({ id, keep: true, forward });
        return { id, forward };
    };
    const getBookmark$2 = (selection, type, normalized = false) => {
        if (type === 2) {
            return getOffsetBookmark(trim$2, normalized, selection);
        }
        else if (type === 3) {
            return getCaretBookmark(selection);
        }
        else if (type) {
            return getRangeBookmark(selection);
        }
        else {
            return getPersistentBookmark(selection, false);
        }
    };
    const getUndoBookmark = curry(getOffsetBookmark, identity, true);

    /**
     * Checks if the direction is forwards.
     */
    const isForwards = (direction) => direction === 1 /* HDirection.Forwards */;
    /**
     * Checks if the direction is backwards.
     */
    const isBackwards = (direction) => direction === -1 /* HDirection.Backwards */;

    var SimpleResultType;
    (function (SimpleResultType) {
        SimpleResultType[SimpleResultType["Error"] = 0] = "Error";
        SimpleResultType[SimpleResultType["Value"] = 1] = "Value";
    })(SimpleResultType || (SimpleResultType = {}));
    const fold$1 = (res, onError, onValue) => res.stype === SimpleResultType.Error ? onError(res.serror) : onValue(res.svalue);
    const partition = (results) => {
        const values = [];
        const errors = [];
        each$e(results, (obj) => {
            fold$1(obj, (err) => errors.push(err), (val) => values.push(val));
        });
        return { values, errors };
    };
    const mapError = (res, f) => {
        if (res.stype === SimpleResultType.Error) {
            return { stype: SimpleResultType.Error, serror: f(res.serror) };
        }
        else {
            return res;
        }
    };
    const map = (res, f) => {
        if (res.stype === SimpleResultType.Value) {
            return { stype: SimpleResultType.Value, svalue: f(res.svalue) };
        }
        else {
            return res;
        }
    };
    const bind = (res, f) => {
        if (res.stype === SimpleResultType.Value) {
            return f(res.svalue);
        }
        else {
            return res;
        }
    };
    const bindError = (res, f) => {
        if (res.stype === SimpleResultType.Error) {
            return f(res.serror);
        }
        else {
            return res;
        }
    };
    const svalue = (v) => ({ stype: SimpleResultType.Value, svalue: v });
    const serror = (e) => ({ stype: SimpleResultType.Error, serror: e });
    const toResult = (res) => fold$1(res, Result.error, Result.value);
    const fromResult = (res) => res.fold(serror, svalue);
    const SimpleResult = {
        fromResult,
        toResult,
        svalue,
        partition,
        serror,
        bind,
        bindError,
        map,
        mapError,
        fold: fold$1
    };

    const formatObj = (input) => {
        return isObject(input) && keys(input).length > 100 ? ' removed due to size' : JSON.stringify(input, null, 2);
    };
    const formatErrors = (errors) => {
        const es = errors.length > 10 ? errors.slice(0, 10).concat([
            {
                path: [],
                getErrorInfo: constant('... (only showing first ten failures)')
            }
        ]) : errors;
        // TODO: Work out a better split between PrettyPrinter and SchemaError
        return map$3(es, (e) => {
            return 'Failed path: (' + e.path.join(' > ') + ')\n' + e.getErrorInfo();
        });
    };

    const nu = (path, getErrorInfo) => {
        return SimpleResult.serror([{
                path,
                // This is lazy so that it isn't calculated unnecessarily
                getErrorInfo
            }]);
    };
    const missingRequired = (path, key, obj) => nu(path, () => 'Could not find valid *required* value for "' + key + '" in ' + formatObj(obj));
    const custom = (path, err) => nu(path, constant(err));

    const value = (validator) => {
        const extract = (path, val) => {
            return SimpleResult.bindError(validator(val), (err) => custom(path, err));
        };
        const toString = constant('val');
        return {
            extract,
            toString
        };
    };
    const anyValue$1 = value(SimpleResult.svalue);

    const anyValue = constant(anyValue$1);
    const typedValue = (validator, expectedType) => value((a) => {
        const actualType = typeof a;
        return validator(a) ? SimpleResult.svalue(a) : SimpleResult.serror(`Expected type: ${expectedType} but got: ${actualType}`);
    });
    const number = typedValue(isNumber, 'number');
    const string = typedValue(isString, 'string');
    const functionProcessor = typedValue(isFunction, 'function');

    const required$1 = () => ({ tag: "required" /* FieldPresenceTag.Required */, process: {} });
    const defaultedThunk = (fallbackThunk) => ({ tag: "defaultedThunk" /* FieldPresenceTag.DefaultedThunk */, process: fallbackThunk });
    const defaulted$1 = (fallback) => defaultedThunk(constant(fallback));
    const asOption = () => ({ tag: "option" /* FieldPresenceTag.Option */, process: {} });

    const field$1 = (key, newKey, presence, prop) => ({ tag: "field" /* FieldTag.Field */, key, newKey, presence, prop });
    const fold = (value, ifField, ifCustom) => {
        switch (value.tag) {
            case "field" /* FieldTag.Field */:
                return ifField(value.key, value.newKey, value.presence, value.prop);
            case "custom" /* FieldTag.CustomField */:
                return ifCustom(value.newKey, value.instantiator);
        }
    };

    const mergeValues = (values, base) => {
        return SimpleResult.svalue(deepMerge(base, merge$1.apply(undefined, values)));
    };
    const mergeErrors = (errors) => compose(SimpleResult.serror, flatten$1)(errors);
    const consolidateObj = (objects, base) => {
        const partition = SimpleResult.partition(objects);
        return partition.errors.length > 0 ? mergeErrors(partition.errors) : mergeValues(partition.values, base);
    };
    const consolidateArr = (objects) => {
        const partitions = SimpleResult.partition(objects);
        return partitions.errors.length > 0 ? mergeErrors(partitions.errors) : SimpleResult.svalue(partitions.values);
    };
    const ResultCombine = {
        consolidateObj,
        consolidateArr
    };

    const requiredAccess = (path, obj, key, bundle) => 
    // In required mode, if it is undefined, it is an error.
    get$a(obj, key).fold(() => missingRequired(path, key, obj), bundle);
    const fallbackAccess = (obj, key, fallback, bundle) => {
        const v = get$a(obj, key).getOrThunk(() => fallback(obj));
        return bundle(v);
    };
    const optionAccess = (obj, key, bundle) => bundle(get$a(obj, key));
    const optionDefaultedAccess = (obj, key, fallback, bundle) => {
        const opt = get$a(obj, key).map((val) => val === true ? fallback(obj) : val);
        return bundle(opt);
    };
    const extractField = (field, path, obj, key, prop) => {
        const bundle = (av) => prop.extract(path.concat([key]), av);
        const bundleAsOption = (optValue) => optValue.fold(() => SimpleResult.svalue(Optional.none()), (ov) => {
            const result = prop.extract(path.concat([key]), ov);
            return SimpleResult.map(result, Optional.some);
        });
        switch (field.tag) {
            case "required" /* FieldPresenceTag.Required */:
                return requiredAccess(path, obj, key, bundle);
            case "defaultedThunk" /* FieldPresenceTag.DefaultedThunk */:
                return fallbackAccess(obj, key, field.process, bundle);
            case "option" /* FieldPresenceTag.Option */:
                return optionAccess(obj, key, bundleAsOption);
            case "defaultedOptionThunk" /* FieldPresenceTag.DefaultedOptionThunk */:
                return optionDefaultedAccess(obj, key, field.process, bundleAsOption);
            case "mergeWithThunk" /* FieldPresenceTag.MergeWithThunk */: {
                return fallbackAccess(obj, key, constant({}), (v) => {
                    const result = deepMerge(field.process(obj), v);
                    return bundle(result);
                });
            }
        }
    };
    const extractFields = (path, obj, fields) => {
        const success = {};
        const errors = [];
        // PERFORMANCE: We use a for loop here instead of Arr.each as this is a hot code path
        for (const field of fields) {
            fold(field, (key, newKey, presence, prop) => {
                const result = extractField(presence, path, obj, key, prop);
                SimpleResult.fold(result, (err) => {
                    errors.push(...err);
                }, (res) => {
                    success[newKey] = res;
                });
            }, (newKey, instantiator) => {
                success[newKey] = instantiator(obj);
            });
        }
        return errors.length > 0 ? SimpleResult.serror(errors) : SimpleResult.svalue(success);
    };
    const objOf = (values) => {
        const extract = (path, o) => extractFields(path, o, values);
        const toString = () => {
            const fieldStrings = map$3(values, (value) => fold(value, (key, _okey, _presence, prop) => key + ' -> ' + prop.toString(), (newKey, _instantiator) => 'state(' + newKey + ')'));
            return 'obj{\n' + fieldStrings.join('\n') + '}';
        };
        return {
            extract,
            toString
        };
    };
    const arrOf = (prop) => {
        const extract = (path, array) => {
            const results = map$3(array, (a, i) => prop.extract(path.concat(['[' + i + ']']), a));
            return ResultCombine.consolidateArr(results);
        };
        const toString = () => 'array(' + prop.toString() + ')';
        return {
            extract,
            toString
        };
    };
    const arrOfObj = compose(arrOf, objOf);

    const valueOf = (validator) => value((v) => validator(v).fold(SimpleResult.serror, SimpleResult.svalue));
    const extractValue = (label, prop, obj) => {
        const res = prop.extract([label], obj);
        return SimpleResult.mapError(res, (errs) => ({ input: obj, errors: errs }));
    };
    const asRaw = (label, prop, obj) => SimpleResult.toResult(extractValue(label, prop, obj));
    const formatError = (errInfo) => {
        return 'Errors: \n' + formatErrors(errInfo.errors).join('\n') +
            '\n\nInput object: ' + formatObj(errInfo.input);
    };

    const field = field$1;
    const required = (key) => field(key, key, required$1(), anyValue());
    const requiredOf = (key, schema) => field(key, key, required$1(), schema);
    const requiredString = (key) => requiredOf(key, string);
    const requiredFunction = (key) => requiredOf(key, functionProcessor);
    const requiredArrayOf = (key, schema) => field(key, key, required$1(), arrOf(schema));
    const option$1 = (key) => field(key, key, asOption(), anyValue());
    const optionOf = (key, schema) => field(key, key, asOption(), schema);
    const optionString = (key) => optionOf(key, string);
    const optionFunction = (key) => optionOf(key, functionProcessor);
    const defaulted = (key, fallback) => field(key, key, defaulted$1(fallback), anyValue());
    const defaultedOf = (key, fallback, schema) => field(key, key, defaulted$1(fallback), schema);
    const defaultedNumber = (key, fallback) => defaultedOf(key, fallback, number);
    const defaultedArrayOf = (key, fallback, schema) => defaultedOf(key, fallback, arrOf(schema));

    const isInlinePattern = (pattern) => pattern.type === 'inline-command' || pattern.type === 'inline-format';
    const isBlockPattern = (pattern) => pattern.type === 'block-command' || pattern.type === 'block-format';
    const hasBlockTrigger = (pattern, trigger) => (pattern.type === 'block-command' || pattern.type === 'block-format') && pattern.trigger === trigger;
    const normalizePattern = (pattern) => {
        const err = (message) => Result.error({ message, pattern });
        const formatOrCmd = (name, onFormat, onCommand) => {
            if (pattern.format !== undefined) {
                let formats;
                if (isArray$1(pattern.format)) {
                    if (!forall(pattern.format, isString)) {
                        return err(name + ' pattern has non-string items in the `format` array');
                    }
                    formats = pattern.format;
                }
                else if (isString(pattern.format)) {
                    formats = [pattern.format];
                }
                else {
                    return err(name + ' pattern has non-string `format` parameter');
                }
                return Result.value(onFormat(formats));
            }
            else if (pattern.cmd !== undefined) {
                if (!isString(pattern.cmd)) {
                    return err(name + ' pattern has non-string `cmd` parameter');
                }
                return Result.value(onCommand(pattern.cmd, pattern.value));
            }
            else {
                return err(name + ' pattern is missing both `format` and `cmd` parameters');
            }
        };
        if (!isObject(pattern)) {
            return err('Raw pattern is not an object');
        }
        if (!isString(pattern.start)) {
            return err('Raw pattern is missing `start` parameter');
        }
        if (pattern.end !== undefined) {
            // inline pattern
            if (!isString(pattern.end)) {
                return err('Inline pattern has non-string `end` parameter');
            }
            if (pattern.start.length === 0 && pattern.end.length === 0) {
                return err('Inline pattern has empty `start` and `end` parameters');
            }
            let start = pattern.start;
            let end = pattern.end;
            // when the end is empty swap with start as it is more efficient
            if (end.length === 0) {
                end = start;
                start = '';
            }
            return formatOrCmd('Inline', (format) => ({ type: 'inline-format', start, end, format }), (cmd, value) => ({ type: 'inline-command', start, end, cmd, value }));
        }
        else if (pattern.replacement !== undefined) {
            // replacement pattern
            if (!isString(pattern.replacement)) {
                return err('Replacement pattern has non-string `replacement` parameter');
            }
            if (pattern.start.length === 0) {
                return err('Replacement pattern has empty `start` parameter');
            }
            return Result.value({
                type: 'inline-command',
                start: '',
                end: pattern.start,
                cmd: 'mceInsertContent',
                value: pattern.replacement
            });
        }
        else {
            // block pattern
            const trigger = pattern.trigger ?? 'space';
            if (pattern.start.length === 0) {
                return err('Block pattern has empty `start` parameter');
            }
            return formatOrCmd('Block', (formats) => ({
                type: 'block-format',
                start: pattern.start,
                format: formats[0],
                trigger
            }), (command, commandValue) => ({
                type: 'block-command',
                start: pattern.start,
                cmd: command,
                value: commandValue,
                trigger
            }));
        }
    };
    const getBlockPatterns = (patterns) => filter$5(patterns, isBlockPattern);
    const getInlinePatterns = (patterns) => filter$5(patterns, isInlinePattern);
    const createPatternSet = (patterns, dynamicPatternsLookup) => ({
        inlinePatterns: getInlinePatterns(patterns),
        blockPatterns: getBlockPatterns(patterns),
        dynamicPatternsLookup
    });
    const filterByTrigger = (patterns, trigger) => {
        return {
            ...patterns,
            blockPatterns: filter$5(patterns.blockPatterns, (pattern) => hasBlockTrigger(pattern, trigger))
        };
    };
    const fromRawPatterns = (patterns) => {
        const normalized = partition$1(map$3(patterns, normalizePattern));
        // eslint-disable-next-line no-console
        each$e(normalized.errors, (err) => console.error(err.message, err.pattern));
        return normalized.values;
    };
    const fromRawPatternsLookup = (lookupFn) => {
        return (ctx) => {
            const rawPatterns = lookupFn(ctx);
            return fromRawPatterns(rawPatterns);
        };
    };

    const firePreProcess = (editor, args) => editor.dispatch('PreProcess', args);
    const firePostProcess = (editor, args) => editor.dispatch('PostProcess', args);
    const fireRemove = (editor) => {
        editor.dispatch('remove');
    };
    const fireDetach = (editor) => {
        editor.dispatch('detach');
    };
    const fireSwitchMode = (editor, mode) => {
        editor.dispatch('SwitchMode', { mode });
    };
    const fireObjectResizeStart = (editor, target, width, height, origin) => {
        editor.dispatch('ObjectResizeStart', { target, width, height, origin });
    };
    const fireObjectResized = (editor, target, width, height, origin) => {
        editor.dispatch('ObjectResized', { target, width, height, origin });
    };
    const firePreInit = (editor) => {
        editor.dispatch('PreInit');
    };
    const firePostRender = (editor) => {
        editor.dispatch('PostRender');
    };
    const fireInit = (editor) => {
        editor.dispatch('Init');
    };
    const firePlaceholderToggle = (editor, state) => {
        editor.dispatch('PlaceholderToggle', { state });
    };
    const fireError = (editor, errorType, error) => {
        editor.dispatch(errorType, error);
    };
    const fireFormatApply = (editor, format, node, vars) => {
        editor.dispatch('FormatApply', { format, node, vars });
    };
    const fireFormatRemove = (editor, format, node, vars) => {
        editor.dispatch('FormatRemove', { format, node, vars });
    };
    const fireBeforeSetContent = (editor, args) => editor.dispatch('BeforeSetContent', args);
    const fireSetContent = (editor, args) => editor.dispatch('SetContent', args);
    const fireBeforeGetContent = (editor, args) => editor.dispatch('BeforeGetContent', args);
    const fireGetContent = (editor, args) => editor.dispatch('GetContent', args);
    const fireAutocompleterStart = (editor, args) => {
        editor.dispatch('AutocompleterStart', args);
    };
    const fireAutocompleterUpdate = (editor, args) => {
        editor.dispatch('AutocompleterUpdate', args);
    };
    const fireAutocompleterUpdateActiveRange = (editor, args) => {
        editor.dispatch('AutocompleterUpdateActiveRange', args);
    };
    const fireAutocompleterEnd = (editor) => {
        editor.dispatch('AutocompleterEnd');
    };
    const firePastePreProcess = (editor, html, internal) => editor.dispatch('PastePreProcess', { content: html, internal });
    const firePastePostProcess = (editor, node, internal) => editor.dispatch('PastePostProcess', { node, internal });
    const firePastePlainTextToggle = (editor, state) => editor.dispatch('PastePlainTextToggle', { state });
    const fireEditableRootStateChange = (editor, state) => editor.dispatch('EditableRootStateChange', { state });
    const fireDisabledStateChange = (editor, state) => editor.dispatch('DisabledStateChange', { state });
    const fireCloseTooltips = (editor) => editor.dispatch('CloseActiveTooltips');

    const deviceDetection$1 = detect$1().deviceType;
    const isTouch = deviceDetection$1.isTouch();
    const DOM$d = DOMUtils.DOM;
    const getHash = (value) => {
        const items = value.indexOf('=') > 0 ? value.split(/[;,](?![^=;,]*(?:[;,]|$))/) : value.split(',');
        return foldl(items, (output, item) => {
            const arr = item.split('=');
            const key = arr[0];
            const val = arr.length > 1 ? arr[1] : key;
            output[trim$4(key)] = trim$4(val);
            return output;
        }, {});
    };
    const isRegExp = (x) => is$5(x, RegExp);
    const option = (name) => (editor) => editor.options.get(name);
    const stringOrObjectProcessor = (value) => isString(value) || isObject(value);
    const bodyOptionProcessor = (editor, defaultValue = '') => (value) => {
        const valid = isString(value);
        if (valid) {
            if (value.indexOf('=') !== -1) {
                const bodyObj = getHash(value);
                return { value: get$a(bodyObj, editor.id).getOr(defaultValue), valid };
            }
            else {
                return { value, valid };
            }
        }
        else {
            return { valid: false, message: 'Must be a string.' };
        }
    };
    const register$7 = (editor) => {
        const registerOption = editor.options.register;
        registerOption('id', {
            processor: 'string',
            default: editor.id
        });
        registerOption('selector', {
            processor: 'string'
        });
        registerOption('target', {
            processor: 'object'
        });
        registerOption('suffix', {
            processor: 'string'
        });
        registerOption('cache_suffix', {
            processor: 'string'
        });
        registerOption('base_url', {
            processor: 'string'
        });
        registerOption('referrer_policy', {
            processor: 'string',
            default: ''
        });
        registerOption('crossorigin', {
            processor: 'function',
            default: constant(undefined)
        });
        registerOption('language_load', {
            processor: 'boolean',
            default: true
        });
        registerOption('inline', {
            processor: 'boolean',
            default: false
        });
        registerOption('iframe_attrs', {
            processor: 'object',
            default: {}
        });
        registerOption('doctype', {
            processor: 'string',
            default: '<!DOCTYPE html>'
        });
        registerOption('document_base_url', {
            processor: 'string',
            default: editor.editorManager.documentBaseURL
        });
        registerOption('body_id', {
            processor: bodyOptionProcessor(editor, 'tinymce'),
            default: 'tinymce'
        });
        registerOption('body_class', {
            processor: bodyOptionProcessor(editor),
            default: ''
        });
        registerOption('content_security_policy', {
            processor: 'string',
            default: ''
        });
        registerOption('br_in_pre', {
            processor: 'boolean',
            default: true
        });
        registerOption('forced_root_block', {
            processor: (value) => {
                const valid = isString(value) && isNotEmpty(value);
                if (valid) {
                    return { value, valid };
                }
                else {
                    return { valid: false, message: 'Must be a non-empty string.' };
                }
            },
            default: 'p'
        });
        registerOption('forced_root_block_attrs', {
            processor: 'object',
            default: {}
        });
        registerOption('newline_behavior', {
            processor: (value) => {
                const valid = contains$2(['block', 'linebreak', 'invert', 'default'], value);
                return valid ? { value, valid } : { valid: false, message: 'Must be one of: block, linebreak, invert or default.' };
            },
            default: 'default'
        });
        registerOption('br_newline_selector', {
            processor: 'string',
            default: '.mce-toc h2,figcaption,caption'
        });
        registerOption('no_newline_selector', {
            processor: 'string',
            default: ''
        });
        registerOption('keep_styles', {
            processor: 'boolean',
            default: true
        });
        registerOption('end_container_on_empty_block', {
            processor: (value) => {
                if (isBoolean(value)) {
                    return { valid: true, value };
                }
                else if (isString(value)) {
                    return { valid: true, value };
                }
                else {
                    return { valid: false, message: 'Must be boolean or a string' };
                }
            },
            default: 'blockquote'
        });
        registerOption('font_size_style_values', {
            processor: 'string',
            default: 'xx-small,x-small,small,medium,large,x-large,xx-large'
        });
        registerOption('font_size_legacy_values', {
            processor: 'string',
            // See: http://www.w3.org/TR/CSS2/fonts.html#propdef-font-size
            default: 'xx-small,small,medium,large,x-large,xx-large,300%'
        });
        registerOption('font_size_classes', {
            processor: 'string',
            default: ''
        });
        registerOption('automatic_uploads', {
            processor: 'boolean',
            default: true
        });
        registerOption('images_reuse_filename', {
            processor: 'boolean',
            default: false
        });
        registerOption('images_replace_blob_uris', {
            processor: 'boolean',
            default: true
        });
        registerOption('icons', {
            processor: 'string',
            default: ''
        });
        registerOption('icons_url', {
            processor: 'string',
            default: ''
        });
        registerOption('images_upload_url', {
            processor: 'string',
            default: ''
        });
        registerOption('images_upload_base_path', {
            processor: 'string',
            default: ''
        });
        registerOption('images_upload_credentials', {
            processor: 'boolean',
            default: false
        });
        registerOption('images_upload_handler', {
            processor: 'function'
        });
        registerOption('language', {
            processor: 'string',
            default: 'en'
        });
        registerOption('language_url', {
            processor: 'string',
            default: ''
        });
        registerOption('entity_encoding', {
            processor: 'string',
            default: 'named'
        });
        registerOption('indent', {
            processor: 'boolean',
            default: true
        });
        registerOption('indent_before', {
            processor: 'string',
            default: 'p,h1,h2,h3,h4,h5,h6,blockquote,div,title,style,pre,script,td,th,ul,ol,li,dl,dt,dd,area,table,thead,' +
                'tfoot,tbody,tr,section,details,summary,article,hgroup,aside,figure,figcaption,option,optgroup,datalist'
        });
        registerOption('indent_after', {
            processor: 'string',
            default: 'p,h1,h2,h3,h4,h5,h6,blockquote,div,title,style,pre,script,td,th,ul,ol,li,dl,dt,dd,area,table,thead,' +
                'tfoot,tbody,tr,section,details,summary,article,hgroup,aside,figure,figcaption,option,optgroup,datalist'
        });
        registerOption('indent_use_margin', {
            processor: 'boolean',
            default: false
        });
        registerOption('indentation', {
            processor: 'string',
            default: '40px'
        });
        registerOption('content_css', {
            processor: (value) => {
                const valid = value === false || isString(value) || isArrayOf(value, isString);
                if (valid) {
                    if (isString(value)) {
                        return { value: map$3(value.split(','), trim$4), valid };
                    }
                    else if (isArray$1(value)) {
                        return { value, valid };
                    }
                    else if (value === false) {
                        return { value: [], valid };
                    }
                    else {
                        return { value, valid };
                    }
                }
                else {
                    return { valid: false, message: 'Must be false, a string or an array of strings.' };
                }
            },
            default: isInline$2(editor) ? [] : ['default']
        });
        registerOption('content_style', {
            processor: 'string'
        });
        registerOption('content_css_cors', {
            processor: 'boolean',
            default: false
        });
        registerOption('font_css', {
            processor: (value) => {
                const valid = isString(value) || isArrayOf(value, isString);
                if (valid) {
                    const newValue = isArray$1(value) ? value : map$3(value.split(','), trim$4);
                    return { value: newValue, valid };
                }
                else {
                    return { valid: false, message: 'Must be a string or an array of strings.' };
                }
            },
            default: []
        });
        registerOption('extended_mathml_attributes', {
            processor: 'string[]'
        });
        registerOption('extended_mathml_elements', {
            processor: 'string[]'
        });
        registerOption('inline_boundaries', {
            processor: 'boolean',
            default: true
        });
        registerOption('inline_boundaries_selector', {
            processor: 'string',
            default: 'a[href],code,span.mce-annotation'
        });
        registerOption('object_resizing', {
            processor: (value) => {
                const valid = isBoolean(value) || isString(value);
                if (valid) {
                    if (value === false || deviceDetection$1.isiPhone() || deviceDetection$1.isiPad()) {
                        return { value: '', valid };
                    }
                    else {
                        return { value: value === true ? 'table,img,figure.image,div,video,iframe' : value, valid };
                    }
                }
                else {
                    return { valid: false, message: 'Must be boolean or a string' };
                }
            },
            // No nice way to do object resizing on touch devices at this stage
            default: !isTouch
        });
        registerOption('resize_img_proportional', {
            processor: 'boolean',
            default: true
        });
        registerOption('event_root', {
            processor: 'string'
        });
        registerOption('service_message', {
            processor: 'string'
        });
        registerOption('onboarding', {
            processor: 'boolean',
            default: true
        });
        registerOption('tiny_cloud_entry_url', {
            processor: 'string'
        });
        registerOption('theme', {
            processor: (value) => value === false || isString(value) || isFunction(value),
            default: 'silver'
        });
        registerOption('theme_url', {
            processor: 'string'
        });
        registerOption('formats', {
            processor: 'object'
        });
        registerOption('format_empty_lines', {
            processor: 'boolean',
            default: false
        });
        registerOption('format_noneditable_selector', {
            processor: 'string',
            default: ''
        });
        registerOption('preview_styles', {
            processor: (value) => {
                const valid = value === false || isString(value);
                if (valid) {
                    return { value: value === false ? '' : value, valid };
                }
                else {
                    return { valid: false, message: 'Must be false or a string' };
                }
            },
            default: 'font-family font-size font-weight font-style text-decoration text-transform color background-color border border-radius outline text-shadow'
        });
        registerOption('custom_ui_selector', {
            processor: 'string',
            default: ''
        });
        registerOption('hidden_input', {
            processor: 'boolean',
            default: true
        });
        registerOption('submit_patch', {
            processor: 'boolean',
            default: true
        });
        registerOption('encoding', {
            processor: 'string'
        });
        registerOption('add_form_submit_trigger', {
            processor: 'boolean',
            default: true
        });
        registerOption('add_unload_trigger', {
            processor: 'boolean',
            default: true
        });
        registerOption('custom_undo_redo_levels', {
            processor: 'number',
            default: 0
        });
        registerOption('disable_nodechange', {
            processor: 'boolean',
            default: false
        });
        registerOption('disabled', {
            processor: (value) => {
                if (isBoolean(value)) {
                    if (editor.initialized && isDisabled$1(editor) !== value) {
                        // Schedules the callback to run in the next microtask queue once the option is updated
                        // TODO: TINY-11586 - Implement `onChange` callback when the value of an option changes
                        // eslint-disable-next-line @typescript-eslint/no-floating-promises
                        Promise.resolve().then(() => {
                            fireDisabledStateChange(editor, value);
                        });
                    }
                    return { valid: true, value };
                }
                return { valid: false, message: 'The value must be a boolean.' };
            },
            default: false
        });
        registerOption('readonly', {
            processor: 'boolean',
            default: false
        });
        registerOption('editable_root', {
            processor: 'boolean',
            default: true
        });
        registerOption('plugins', {
            processor: 'string[]',
            default: []
        });
        registerOption('external_plugins', {
            processor: 'object'
        });
        registerOption('forced_plugins', {
            processor: 'string[]'
        });
        registerOption('model', {
            processor: 'string',
            default: editor.hasPlugin('rtc') ? 'plugin' : 'dom'
        });
        registerOption('model_url', {
            processor: 'string'
        });
        registerOption('block_unsupported_drop', {
            processor: 'boolean',
            default: true
        });
        registerOption('visual', {
            processor: 'boolean',
            default: true
        });
        registerOption('visual_table_class', {
            processor: 'string',
            default: 'mce-item-table'
        });
        registerOption('visual_anchor_class', {
            processor: 'string',
            default: 'mce-item-anchor'
        });
        registerOption('iframe_aria_text', {
            processor: 'string',
            default: 'Rich Text Area'.concat(editor.hasPlugin('help') ? '. Press ALT-0 for help.' : '')
        });
        registerOption('setup', {
            processor: 'function'
        });
        registerOption('init_instance_callback', {
            processor: 'function'
        });
        registerOption('url_converter', {
            processor: 'function',
            // Note: Don't bind here, as the binding is handled via the `url_converter_scope`
            // eslint-disable-next-line @typescript-eslint/unbound-method
            default: editor.convertURL
        });
        registerOption('url_converter_scope', {
            processor: 'object',
            default: editor
        });
        registerOption('urlconverter_callback', {
            processor: 'function'
        });
        registerOption('allow_conditional_comments', {
            processor: 'boolean',
            default: false
        });
        registerOption('allow_html_data_urls', {
            processor: 'boolean',
            default: false
        });
        registerOption('allow_svg_data_urls', {
            processor: 'boolean'
        });
        registerOption('allow_html_in_named_anchor', {
            processor: 'boolean',
            default: false
        });
        registerOption('allow_html_in_comments', {
            processor: 'boolean',
            default: false
        });
        registerOption('allow_script_urls', {
            processor: 'boolean',
            default: false
        });
        registerOption('allow_unsafe_link_target', {
            processor: 'boolean',
            default: false
        });
        registerOption('allow_mathml_annotation_encodings', {
            processor: (value) => {
                const valid = isArrayOf(value, isString);
                return valid ? { value, valid } : { valid: false, message: 'Must be an array of strings.' };
            },
            default: []
        });
        registerOption('convert_fonts_to_spans', {
            processor: 'boolean',
            default: true,
            deprecated: true
        });
        registerOption('fix_list_elements', {
            processor: 'boolean',
            default: false
        });
        registerOption('preserve_cdata', {
            processor: 'boolean',
            default: false
        });
        registerOption('remove_trailing_brs', {
            processor: 'boolean',
            default: true
        });
        registerOption('pad_empty_with_br', {
            processor: 'boolean',
            default: false,
        });
        registerOption('inline_styles', {
            processor: 'boolean',
            default: true,
            deprecated: true
        });
        registerOption('element_format', {
            processor: 'string',
            default: 'html'
        });
        registerOption('entities', {
            processor: 'string'
        });
        registerOption('schema', {
            processor: 'string',
            default: 'html5'
        });
        registerOption('convert_urls', {
            processor: 'boolean',
            default: true
        });
        registerOption('relative_urls', {
            processor: 'boolean',
            default: true
        });
        registerOption('remove_script_host', {
            processor: 'boolean',
            default: true
        });
        registerOption('custom_elements', {
            processor: stringOrObjectProcessor
        });
        registerOption('extended_valid_elements', {
            processor: 'string'
        });
        registerOption('invalid_elements', {
            processor: 'string'
        });
        registerOption('invalid_styles', {
            processor: stringOrObjectProcessor
        });
        registerOption('valid_children', {
            processor: 'string'
        });
        registerOption('valid_classes', {
            processor: stringOrObjectProcessor
        });
        registerOption('valid_elements', {
            processor: 'string'
        });
        registerOption('valid_styles', {
            processor: stringOrObjectProcessor
        });
        registerOption('verify_html', {
            processor: 'boolean',
            default: true
        });
        registerOption('auto_focus', {
            processor: (value) => isString(value) || value === true
        });
        registerOption('browser_spellcheck', {
            processor: 'boolean',
            default: false
        });
        registerOption('protect', {
            processor: 'array'
        });
        registerOption('images_file_types', {
            processor: 'string',
            default: 'jpeg,jpg,jpe,jfi,jif,jfif,png,gif,bmp,webp'
        });
        registerOption('deprecation_warnings', {
            processor: 'boolean',
            default: true
        });
        registerOption('a11y_advanced_options', {
            processor: 'boolean',
            default: false
        });
        registerOption('api_key', {
            processor: 'string'
        });
        registerOption('license_key', {
            processor: 'string'
        });
        registerOption('paste_block_drop', {
            processor: 'boolean',
            default: false
        });
        registerOption('paste_data_images', {
            processor: 'boolean',
            default: true
        });
        registerOption('paste_preprocess', {
            processor: 'function'
        });
        registerOption('paste_postprocess', {
            processor: 'function'
        });
        registerOption('paste_webkit_styles', {
            processor: 'string',
            default: 'none'
        });
        registerOption('paste_remove_styles_if_webkit', {
            processor: 'boolean',
            default: true
        });
        registerOption('paste_merge_formats', {
            processor: 'boolean',
            default: true
        });
        registerOption('smart_paste', {
            processor: 'boolean',
            default: true
        });
        registerOption('paste_as_text', {
            processor: 'boolean',
            default: false
        });
        registerOption('paste_tab_spaces', {
            processor: 'number',
            default: 4
        });
        registerOption('text_patterns', {
            processor: (value) => {
                if (isArrayOf(value, isObject) || value === false) {
                    const patterns = value === false ? [] : value;
                    return { value: fromRawPatterns(patterns), valid: true };
                }
                else {
                    return { valid: false, message: 'Must be an array of objects or false.' };
                }
            },
            default: [
                { start: '*', end: '*', format: 'italic' },
                { start: '**', end: '**', format: 'bold' },
                { start: '#', format: 'h1', trigger: 'space' },
                { start: '##', format: 'h2', trigger: 'space' },
                { start: '###', format: 'h3', trigger: 'space' },
                { start: '####', format: 'h4', trigger: 'space' },
                { start: '#####', format: 'h5', trigger: 'space' },
                { start: '######', format: 'h6', trigger: 'space' },
                { start: '1.', cmd: 'InsertOrderedList', trigger: 'space' },
                { start: '*', cmd: 'InsertUnorderedList', trigger: 'space' },
                { start: '-', cmd: 'InsertUnorderedList', trigger: 'space' },
                { start: '>', cmd: 'mceBlockQuote', trigger: 'space' },
                { start: '---', cmd: 'InsertHorizontalRule', trigger: 'space' },
            ]
        });
        registerOption('text_patterns_lookup', {
            processor: (value) => {
                if (isFunction(value)) {
                    return {
                        value: fromRawPatternsLookup(value),
                        valid: true,
                    };
                }
                else {
                    return { valid: false, message: 'Must be a single function' };
                }
            },
            default: (_ctx) => []
        });
        registerOption('noneditable_class', {
            processor: 'string',
            default: 'mceNonEditable'
        });
        registerOption('editable_class', {
            processor: 'string',
            default: 'mceEditable'
        });
        registerOption('noneditable_regexp', {
            processor: (value) => {
                if (isArrayOf(value, isRegExp)) {
                    return { value, valid: true };
                }
                else if (isRegExp(value)) {
                    return { value: [value], valid: true };
                }
                else {
                    return { valid: false, message: 'Must be a RegExp or an array of RegExp.' };
                }
            },
            default: []
        });
        registerOption('table_tab_navigation', {
            processor: 'boolean',
            default: true
        });
        registerOption('highlight_on_focus', {
            processor: 'boolean',
            default: true
        });
        registerOption('xss_sanitization', {
            processor: 'boolean',
            default: true
        });
        registerOption('details_initial_state', {
            processor: (value) => {
                const valid = contains$2(['inherited', 'collapsed', 'expanded'], value);
                return valid ? { value, valid } : { valid: false, message: 'Must be one of: inherited, collapsed, or expanded.' };
            },
            default: 'inherited'
        });
        registerOption('details_serialized_state', {
            processor: (value) => {
                const valid = contains$2(['inherited', 'collapsed', 'expanded'], value);
                return valid ? { value, valid } : { valid: false, message: 'Must be one of: inherited, collapsed, or expanded.' };
            },
            default: 'inherited'
        });
        registerOption('init_content_sync', {
            processor: 'boolean',
            default: false
        });
        registerOption('newdocument_content', {
            processor: 'string',
            default: ''
        });
        registerOption('sandbox_iframes', {
            processor: 'boolean',
            default: true
        });
        registerOption('sandbox_iframes_exclusions', {
            processor: 'string[]',
            default: [
                'youtube.com',
                'youtu.be',
                'vimeo.com',
                'player.vimeo.com',
                'dailymotion.com',
                'embed.music.apple.com',
                'open.spotify.com',
                'giphy.com',
                'dai.ly',
                'codepen.io',
            ]
        });
        registerOption('convert_unsafe_embeds', {
            processor: 'boolean',
            default: true
        });
        registerOption('user_id', {
            processor: 'string',
            default: 'Anonymous'
        });
        registerOption('fetch_users', {
            processor: (value) => {
                if (value === undefined) {
                    return { valid: true, value: undefined };
                }
                if (isFunction(value)) {
                    return { valid: true, value };
                }
                return {
                    valid: false,
                    message: 'fetch_users must be a function that returns a Promise<ExpectedUser[]>'
                };
            }
        });
        const documentsFileTypesOptionsSchema = arrOfObj([
            requiredString('mimeType'),
            requiredArrayOf('extensions', valueOf((ext) => {
                if (isString(ext)) {
                    return Result.value(ext);
                }
                else {
                    return Result.error('Extensions must be an array of strings');
                }
            })),
        ]);
        registerOption('documents_file_types', {
            processor: (value) => asRaw('documents_file_types', documentsFileTypesOptionsSchema, value).fold((_err) => ({
                valid: false,
                message: 'Must be a non-empty array of objects matching the configuration schema: https://www.tiny.cloud/docs/tinymce/latest/uploadcare-documents/#documents-file-types'
            }), (val) => ({ valid: true, value: val }))
        });
        // These options must be registered later in the init sequence due to their default values
        editor.on('ScriptsLoaded', () => {
            registerOption('directionality', {
                processor: 'string',
                default: I18n.isRtl() ? 'rtl' : undefined
            });
            registerOption('placeholder', {
                processor: 'string',
                // Fallback to the original elements placeholder if not set in the settings
                default: DOM$d.getAttrib(editor.getElement(), 'placeholder')
            });
        });
        registerOption('lists_indent_on_tab', {
            processor: 'boolean',
            default: true
        });
        registerOption('list_max_depth', {
            processor: (value) => {
                const valid = isNumber(value);
                if (valid) {
                    if (value < 0) {
                        throw new Error('list_max_depth cannot be set to lower than 0');
                    }
                    return { value, valid };
                }
                else {
                    return { valid: false, message: 'Must be a number' };
                }
            },
        });
    };
    const getIframeAttrs = option('iframe_attrs');
    const getDocType = option('doctype');
    const getDocumentBaseUrl = option('document_base_url');
    const getBodyId = option('body_id');
    const getBodyClass = option('body_class');
    const getContentSecurityPolicy = option('content_security_policy');
    const shouldPutBrInPre$1 = option('br_in_pre');
    const getForcedRootBlock = option('forced_root_block');
    const getForcedRootBlockAttrs = option('forced_root_block_attrs');
    const getNewlineBehavior = option('newline_behavior');
    const getBrNewLineSelector = option('br_newline_selector');
    const getNoNewLineSelector = option('no_newline_selector');
    const shouldKeepStyles = option('keep_styles');
    const shouldEndContainerOnEmptyBlock = option('end_container_on_empty_block');
    const isAutomaticUploadsEnabled = option('automatic_uploads');
    const shouldReuseFileName = option('images_reuse_filename');
    const shouldReplaceBlobUris = option('images_replace_blob_uris');
    const getIconPackName = option('icons');
    const getIconsUrl = option('icons_url');
    const getImageUploadUrl = option('images_upload_url');
    const getImageUploadBasePath = option('images_upload_base_path');
    const getImagesUploadCredentials = option('images_upload_credentials');
    const getImagesUploadHandler = option('images_upload_handler');
    const shouldUseContentCssCors = option('content_css_cors');
    const getReferrerPolicy = option('referrer_policy');
    const getCrossOrigin = option('crossorigin');
    const getLanguageCode = option('language');
    const getLanguageUrl = option('language_url');
    const shouldIndentUseMargin = option('indent_use_margin');
    const getIndentation = option('indentation');
    const getContentCss = option('content_css');
    const getContentStyle = option('content_style');
    const getFontCss = option('font_css');
    const getDirectionality = option('directionality');
    const getInlineBoundarySelector = option('inline_boundaries_selector');
    const getObjectResizing = option('object_resizing');
    const getResizeImgProportional = option('resize_img_proportional');
    const getPlaceholder = option('placeholder');
    const getEventRoot = option('event_root');
    const getServiceMessage = option('service_message');
    const getTheme = option('theme');
    const getThemeUrl = option('theme_url');
    const getModel = option('model');
    const getModelUrl = option('model_url');
    const isInlineBoundariesEnabled = option('inline_boundaries');
    const getFormats = option('formats');
    const getPreviewStyles = option('preview_styles');
    const canFormatEmptyLines = option('format_empty_lines');
    const getFormatNoneditableSelector = option('format_noneditable_selector');
    const getCustomUiSelector = option('custom_ui_selector');
    const isInline$2 = option('inline');
    const hasHiddenInput = option('hidden_input');
    const shouldPatchSubmit = option('submit_patch');
    const shouldAddFormSubmitTrigger = option('add_form_submit_trigger');
    const shouldAddUnloadTrigger = option('add_unload_trigger');
    const getCustomUndoRedoLevels = option('custom_undo_redo_levels');
    const shouldDisableNodeChange = option('disable_nodechange');
    const isReadOnly$1 = option('readonly');
    const hasEditableRoot$1 = option('editable_root');
    const hasContentCssCors = option('content_css_cors');
    const getPlugins = option('plugins');
    const getExternalPlugins$1 = option('external_plugins');
    const shouldBlockUnsupportedDrop = option('block_unsupported_drop');
    const isVisualAidsEnabled = option('visual');
    const getVisualAidsTableClass = option('visual_table_class');
    const getVisualAidsAnchorClass = option('visual_anchor_class');
    const getIframeAriaText = option('iframe_aria_text');
    const getSetupCallback = option('setup');
    const getInitInstanceCallback = option('init_instance_callback');
    const getUrlConverterCallback = option('urlconverter_callback');
    const getAutoFocus = option('auto_focus');
    const shouldBrowserSpellcheck = option('browser_spellcheck');
    const getProtect = option('protect');
    const shouldPasteBlockDrop = option('paste_block_drop');
    const shouldPasteDataImages = option('paste_data_images');
    const getPastePreProcess = option('paste_preprocess');
    const getPastePostProcess = option('paste_postprocess');
    const getNewDocumentContent = option('newdocument_content');
    const getPasteWebkitStyles = option('paste_webkit_styles');
    const shouldPasteRemoveWebKitStyles = option('paste_remove_styles_if_webkit');
    const shouldPasteMergeFormats = option('paste_merge_formats');
    const isSmartPasteEnabled = option('smart_paste');
    const isPasteAsTextEnabled = option('paste_as_text');
    const getPasteTabSpaces = option('paste_tab_spaces');
    const shouldAllowHtmlDataUrls = option('allow_html_data_urls');
    const getTextPatterns = option('text_patterns');
    const getTextPatternsLookup = option('text_patterns_lookup');
    const getNonEditableClass = option('noneditable_class');
    const getEditableClass = option('editable_class');
    const getNonEditableRegExps = option('noneditable_regexp');
    const shouldPreserveCData = option('preserve_cdata');
    const shouldHighlightOnFocus = option('highlight_on_focus');
    const shouldSanitizeXss = option('xss_sanitization');
    const shouldUseDocumentWrite = option('init_content_sync');
    const hasTextPatternsLookup = (editor) => editor.options.isSet('text_patterns_lookup');
    const getFontStyleValues = (editor) => Tools.explode(editor.options.get('font_size_style_values'));
    const getFontSizeClasses = (editor) => Tools.explode(editor.options.get('font_size_classes'));
    const isEncodingXml = (editor) => editor.options.get('encoding') === 'xml';
    const getAllowedImageFileTypes = (editor) => Tools.explode(editor.options.get('images_file_types'));
    const hasTableTabNavigation = option('table_tab_navigation');
    const getDetailsInitialState = option('details_initial_state');
    const getDetailsSerializedState = option('details_serialized_state');
    const shouldSandboxIframes = option('sandbox_iframes');
    const getSandboxIframesExclusions = (editor) => editor.options.get('sandbox_iframes_exclusions');
    const shouldConvertUnsafeEmbeds = option('convert_unsafe_embeds');
    const getLicenseKey = option('license_key');
    const getApiKey = option('api_key');
    const isDisabled$1 = option('disabled');
    const getUserId = option('user_id');
    const getFetchUsers = option('fetch_users');
    const shouldIndentOnTab = option('lists_indent_on_tab');
    const getListMaxDepth = (editor) => Optional.from(editor.options.get('list_max_depth'));

    const isElement$4 = isElement$7;
    const isText$5 = isText$b;
    const removeNode$1 = (node) => {
        const parentNode = node.parentNode;
        if (parentNode) {
            parentNode.removeChild(node);
        }
    };
    const trimCount = (text) => {
        const trimmedText = trim$2(text);
        return {
            count: text.length - trimmedText.length,
            text: trimmedText
        };
    };
    const deleteZwspChars = (caretContainer) => {
        // We use the Text.deleteData API here so as to preserve selection offsets
        let idx;
        while ((idx = caretContainer.data.lastIndexOf(ZWSP$1)) !== -1) {
            caretContainer.deleteData(idx, 1);
        }
    };
    const removeUnchanged = (caretContainer, pos) => {
        remove$2(caretContainer);
        return pos;
    };
    const removeTextAndReposition = (caretContainer, pos) => {
        const before = trimCount(caretContainer.data.substr(0, pos.offset()));
        const after = trimCount(caretContainer.data.substr(pos.offset()));
        const text = before.text + after.text;
        if (text.length > 0) {
            deleteZwspChars(caretContainer);
            return CaretPosition(caretContainer, pos.offset() - before.count);
        }
        else {
            return pos;
        }
    };
    const removeElementAndReposition = (caretContainer, pos) => {
        const parentNode = pos.container();
        const newPosition = indexOf$1(from(parentNode.childNodes), caretContainer).map((index) => {
            return index < pos.offset() ? CaretPosition(parentNode, pos.offset() - 1) : pos;
        }).getOr(pos);
        remove$2(caretContainer);
        return newPosition;
    };
    const removeTextCaretContainer = (caretContainer, pos) => isText$5(caretContainer) && pos.container() === caretContainer ? removeTextAndReposition(caretContainer, pos) : removeUnchanged(caretContainer, pos);
    const removeElementCaretContainer = (caretContainer, pos) => pos.container() === caretContainer.parentNode ? removeElementAndReposition(caretContainer, pos) : removeUnchanged(caretContainer, pos);
    const removeAndReposition = (container, pos) => CaretPosition.isTextPosition(pos) ? removeTextCaretContainer(container, pos) : removeElementCaretContainer(container, pos);
    const remove$2 = (caretContainerNode) => {
        if (isElement$4(caretContainerNode) && isCaretContainer$2(caretContainerNode)) {
            if (hasContent(caretContainerNode)) {
                caretContainerNode.removeAttribute('data-mce-caret');
            }
            else {
                removeNode$1(caretContainerNode);
            }
        }
        if (isText$5(caretContainerNode)) {
            deleteZwspChars(caretContainerNode);
            if (caretContainerNode.data.length === 0) {
                removeNode$1(caretContainerNode);
            }
        }
    };

    const isContentEditableFalse$7 = isContentEditableFalse$a;
    const isMedia$1 = isMedia$2;
    const isTableCell$1 = isTableCell$3;
    const inlineFakeCaretSelector = '*[contentEditable=false],video,audio,embed,object';
    const getAbsoluteClientRect = (root, element, before) => {
        const clientRect = collapse(element.getBoundingClientRect(), before);
        let scrollX;
        let scrollY;
        if (root.tagName === 'BODY') {
            const docElm = root.ownerDocument.documentElement;
            scrollX = root.scrollLeft || docElm.scrollLeft;
            scrollY = root.scrollTop || docElm.scrollTop;
        }
        else {
            const rootRect = root.getBoundingClientRect();
            scrollX = root.scrollLeft - rootRect.left;
            scrollY = root.scrollTop - rootRect.top;
        }
        clientRect.left += scrollX;
        clientRect.right += scrollX;
        clientRect.top += scrollY;
        clientRect.bottom += scrollY;
        clientRect.width = 1;
        let margin = element.offsetWidth - element.clientWidth;
        if (margin > 0) {
            if (before) {
                margin *= -1;
            }
            clientRect.left += margin;
            clientRect.right += margin;
        }
        return clientRect;
    };
    const trimInlineCaretContainers = (root) => {
        const fakeCaretTargetNodes = descendants(SugarElement.fromDom(root), inlineFakeCaretSelector);
        for (let i = 0; i < fakeCaretTargetNodes.length; i++) {
            const node = fakeCaretTargetNodes[i].dom;
            let sibling = node.previousSibling;
            if (endsWithCaretContainer$1(sibling)) {
                const data = sibling.data;
                if (data.length === 1) {
                    sibling.parentNode?.removeChild(sibling);
                }
                else {
                    sibling.deleteData(data.length - 1, 1);
                }
            }
            sibling = node.nextSibling;
            if (startsWithCaretContainer$1(sibling)) {
                const data = sibling.data;
                if (data.length === 1) {
                    sibling.parentNode?.removeChild(sibling);
                }
                else {
                    sibling.deleteData(0, 1);
                }
            }
        }
    };
    const FakeCaret = (editor, root, isBlock, hasFocus) => {
        const lastVisualCaret = value$1();
        let cursorInterval;
        let caretContainerNode;
        const caretBlock = getForcedRootBlock(editor);
        const dom = editor.dom;
        const show = (before, element) => {
            let rng;
            hide();
            if (isTableCell$1(element)) {
                return null;
            }
            if (isBlock(element)) {
                const caretContainer = insertBlock(caretBlock, element, before);
                const clientRect = getAbsoluteClientRect(root, element, before);
                dom.setStyle(caretContainer, 'top', clientRect.top);
                dom.setStyle(caretContainer, 'caret-color', 'transparent');
                caretContainerNode = caretContainer;
                const caret = dom.create('div', { 'class': 'mce-visual-caret', 'data-mce-bogus': 'all' });
                dom.setStyles(caret, { ...clientRect });
                dom.add(root, caret);
                lastVisualCaret.set({ caret, element, before });
                if (before) {
                    dom.addClass(caret, 'mce-visual-caret-before');
                }
                startBlink();
                rng = element.ownerDocument.createRange();
                rng.setStart(caretContainer, 0);
                rng.setEnd(caretContainer, 0);
            }
            else {
                caretContainerNode = insertInline$1(element, before);
                rng = element.ownerDocument.createRange();
                if (isInlineFakeCaretTarget(caretContainerNode.nextSibling)) {
                    rng.setStart(caretContainerNode, 0);
                    rng.setEnd(caretContainerNode, 0);
                }
                else {
                    rng.setStart(caretContainerNode, 1);
                    rng.setEnd(caretContainerNode, 1);
                }
                return rng;
            }
            return rng;
        };
        const hide = () => {
            // TODO: TINY-6015 - Ensure cleaning up the fake caret preserves the selection, as currently
            //  the CaretContainerRemove.remove below will change the selection in some cases
            trimInlineCaretContainers(root);
            if (caretContainerNode) {
                remove$2(caretContainerNode);
                caretContainerNode = null;
            }
            lastVisualCaret.on((caretState) => {
                dom.remove(caretState.caret);
                lastVisualCaret.clear();
            });
            if (cursorInterval) {
                clearInterval(cursorInterval);
                cursorInterval = undefined;
            }
        };
        const startBlink = () => {
            cursorInterval = window.setInterval(() => {
                lastVisualCaret.on((caretState) => {
                    if (hasFocus()) {
                        dom.toggleClass(caretState.caret, 'mce-visual-caret-hidden');
                    }
                    else {
                        dom.addClass(caretState.caret, 'mce-visual-caret-hidden');
                    }
                });
            }, 500);
        };
        const reposition = () => {
            lastVisualCaret.on((caretState) => {
                const clientRect = getAbsoluteClientRect(root, caretState.element, caretState.before);
                dom.setStyles(caretState.caret, { ...clientRect });
            });
        };
        const destroy = () => clearInterval(cursorInterval);
        const getCss = () => ('.mce-visual-caret {' +
            'position: absolute;' +
            'background-color: black;' +
            'background-color: currentcolor;' +
            // 'background-color: red;' +
            '}' +
            '.mce-visual-caret-hidden {' +
            'display: none;' +
            '}' +
            '*[data-mce-caret] {' +
            'position: absolute;' +
            'left: -1000px;' +
            'right: auto;' +
            'top: 0;' +
            'margin: 0;' +
            'padding: 0;' +
            '}');
        return {
            isShowing: lastVisualCaret.isSet,
            show,
            hide,
            getCss,
            reposition,
            destroy
        };
    };
    const isFakeCaretTableBrowser = () => Env.browser.isFirefox();
    const isInlineFakeCaretTarget = (node) => isContentEditableFalse$7(node) || isMedia$1(node);
    const isFakeCaretTarget = (node) => {
        const isTarget = isInlineFakeCaretTarget(node) || (isTable$2(node) && isFakeCaretTableBrowser());
        return isTarget && parentElement(SugarElement.fromDom(node)).exists(isEditable$2);
    };

    const isContentEditableTrue$1 = isContentEditableTrue$3;
    const isContentEditableFalse$6 = isContentEditableFalse$a;
    const isMedia = isMedia$2;
    const isBlockLike = matchStyleValues('display', 'block table table-cell table-row table-caption list-item');
    const isCaretContainer = isCaretContainer$2;
    const isCaretContainerBlock = isCaretContainerBlock$1;
    const isElement$3 = isElement$7;
    const isText$4 = isText$b;
    const isCaretCandidate$1 = isCaretCandidate$3;
    const skipCaretContainers = (walk, shallow) => {
        let node;
        while ((node = walk(shallow))) {
            if (!isCaretContainerBlock(node)) {
                return node;
            }
        }
        return null;
    };
    const findNode = (node, direction, predicateFn, rootNode, shallow) => {
        const walker = new DomTreeWalker(node, rootNode);
        const isCefOrCaretContainer = isContentEditableFalse$6(node) || isCaretContainerBlock(node);
        let tempNode;
        if (isBackwards(direction)) {
            if (isCefOrCaretContainer) {
                tempNode = skipCaretContainers(walker.prev.bind(walker), true);
                if (predicateFn(tempNode)) {
                    return tempNode;
                }
            }
            while ((tempNode = skipCaretContainers(walker.prev.bind(walker), shallow))) {
                if (predicateFn(tempNode)) {
                    return tempNode;
                }
            }
        }
        if (isForwards(direction)) {
            if (isCefOrCaretContainer) {
                tempNode = skipCaretContainers(walker.next.bind(walker), true);
                if (predicateFn(tempNode)) {
                    return tempNode;
                }
            }
            while ((tempNode = skipCaretContainers(walker.next.bind(walker), shallow))) {
                if (predicateFn(tempNode)) {
                    return tempNode;
                }
            }
        }
        return null;
    };
    const getEditingHost = (node, rootNode) => {
        const isCETrue = (node) => isContentEditableTrue$1(node.dom);
        const isRoot = (node) => node.dom === rootNode;
        return ancestor$5(SugarElement.fromDom(node), isCETrue, isRoot)
            .map((elm) => elm.dom)
            .getOr(rootNode);
    };
    const isAbsPositionedElement = (node) => isElement$7(node) && get$7(SugarElement.fromDom(node), 'position') === 'absolute';
    const isInlineBlock = (node, rootNode) => node.parentNode !== rootNode;
    const isInlineAbsPositionedCEF = (node, rootNode) => isContentEditableFalse$6(node) && isAbsPositionedElement(node) && isInlineBlock(node, rootNode);
    const getParentBlock$3 = (node, rootNode) => {
        while (node && node !== rootNode) {
            // Exclude inline absolutely positioned CEF elements since they have 'display: block'
            // Created TINY-12922 to improve handling non CEF elements
            if (isBlockLike(node) && !isInlineAbsPositionedCEF(node, rootNode)) {
                return node;
            }
            node = node.parentNode;
        }
        return null;
    };
    const isInSameBlock = (caretPosition1, caretPosition2, rootNode) => getParentBlock$3(caretPosition1.container(), rootNode) === getParentBlock$3(caretPosition2.container(), rootNode);
    const getChildNodeAtRelativeOffset = (relativeOffset, caretPosition) => {
        if (!caretPosition) {
            return Optional.none();
        }
        const container = caretPosition.container();
        const offset = caretPosition.offset();
        if (!isElement$3(container)) {
            return Optional.none();
        }
        return Optional.from(container.childNodes[offset + relativeOffset]);
    };
    const beforeAfter = (before, node) => {
        const doc = node.ownerDocument ?? document;
        const range = doc.createRange();
        if (before) {
            range.setStartBefore(node);
            range.setEndBefore(node);
        }
        else {
            range.setStartAfter(node);
            range.setEndAfter(node);
        }
        return range;
    };
    const isNodesInSameBlock = (root, node1, node2) => getParentBlock$3(node1, root) === getParentBlock$3(node2, root);
    const lean = (left, root, node) => {
        const siblingName = left ? 'previousSibling' : 'nextSibling';
        let tempNode = node;
        while (tempNode && tempNode !== root) {
            let sibling = tempNode[siblingName];
            if (sibling && isCaretContainer(sibling)) {
                sibling = sibling[siblingName];
            }
            if (isContentEditableFalse$6(sibling) || isMedia(sibling)) {
                if (isNodesInSameBlock(root, sibling, tempNode)) {
                    return sibling;
                }
                break;
            }
            if (isCaretCandidate$1(sibling)) {
                break;
            }
            tempNode = tempNode.parentNode;
        }
        return null;
    };
    const before$1 = curry(beforeAfter, true);
    const after$1 = curry(beforeAfter, false);
    const normalizeRange$2 = (direction, root, range) => {
        let node;
        const leanLeft = curry(lean, true, root);
        const leanRight = curry(lean, false, root);
        const container = range.startContainer;
        const offset = range.startOffset;
        if (isCaretContainerBlock$1(container)) {
            const block = isText$4(container) ? container.parentNode : container;
            const location = block.getAttribute('data-mce-caret');
            if (location === 'before') {
                node = block.nextSibling;
                if (isFakeCaretTarget(node)) {
                    return before$1(node);
                }
            }
            if (location === 'after') {
                node = block.previousSibling;
                if (isFakeCaretTarget(node)) {
                    return after$1(node);
                }
            }
        }
        if (!range.collapsed) {
            return range;
        }
        if (isText$b(container)) {
            if (isCaretContainer(container)) {
                if (direction === 1) {
                    node = leanRight(container);
                    if (node) {
                        return before$1(node);
                    }
                    node = leanLeft(container);
                    if (node) {
                        return after$1(node);
                    }
                }
                if (direction === -1) {
                    node = leanLeft(container);
                    if (node) {
                        return after$1(node);
                    }
                    node = leanRight(container);
                    if (node) {
                        return before$1(node);
                    }
                }
                return range;
            }
            if (endsWithCaretContainer$1(container) && offset >= container.data.length - 1) {
                if (direction === 1) {
                    node = leanRight(container);
                    if (node) {
                        return before$1(node);
                    }
                }
                return range;
            }
            if (startsWithCaretContainer$1(container) && offset <= 1) {
                if (direction === -1) {
                    node = leanLeft(container);
                    if (node) {
                        return after$1(node);
                    }
                }
                return range;
            }
            if (offset === container.data.length) {
                node = leanRight(container);
                if (node) {
                    return before$1(node);
                }
                return range;
            }
            if (offset === 0) {
                node = leanLeft(container);
                if (node) {
                    return after$1(node);
                }
                return range;
            }
        }
        return range;
    };
    const getRelativeCefElm = (forward, caretPosition) => getChildNodeAtRelativeOffset(forward ? 0 : -1, caretPosition).filter(isContentEditableFalse$6);
    const getNormalizedRangeEndPoint = (direction, root, range) => {
        const normalizedRange = normalizeRange$2(direction, root, range);
        return direction === -1 ? CaretPosition.fromRangeStart(normalizedRange) : CaretPosition.fromRangeEnd(normalizedRange);
    };
    const getElementFromPosition = (pos) => Optional.from(pos.getNode()).map(SugarElement.fromDom);
    const getElementFromPrevPosition = (pos) => Optional.from(pos.getNode(true)).map(SugarElement.fromDom);
    const getVisualCaretPosition = (walkFn, caretPosition) => {
        let pos = caretPosition;
        while ((pos = walkFn(pos))) {
            if (pos.isVisible()) {
                return pos;
            }
        }
        return pos;
    };
    const isMoveInsideSameBlock = (from, to) => {
        const inSameBlock = isInSameBlock(from, to);
        // Handle bogus BR <p>abc|<br></p>
        if (!inSameBlock && isBr$7(from.getNode())) {
            return true;
        }
        return inSameBlock;
    };

    /**
     * This module contains logic for moving around a virtual caret in logical order within a DOM element.
     *
     * It ignores the most obvious invalid caret locations such as within a script element or within a
     * contentEditable=false element but it will return locations that isn't possible to render visually.
     *
     * @private
     * @class tinymce.caret.CaretWalker
     * @example
     * const caretWalker = CaretWalker(rootElm);
     *
     * const prevLogicalCaretPosition = caretWalker.prev(CaretPosition.fromRangeStart(range));
     * const nextLogicalCaretPosition = caretWalker.next(CaretPosition.fromRangeEnd(range));
     */
    const isContentEditableFalse$5 = isContentEditableFalse$a;
    const isText$3 = isText$b;
    const isElement$2 = isElement$7;
    const isBr$3 = isBr$7;
    const isCaretCandidate = isCaretCandidate$3;
    const isAtomic = isAtomic$1;
    const isEditableCaretCandidate = isEditableCaretCandidate$1;
    const getParents$3 = (node, root) => {
        const parents = [];
        let tempNode = node;
        while (tempNode && tempNode !== root) {
            parents.push(tempNode);
            tempNode = tempNode.parentNode;
        }
        return parents;
    };
    const nodeAtIndex = (container, offset) => {
        if (container.hasChildNodes() && offset < container.childNodes.length) {
            return container.childNodes[offset];
        }
        return null;
    };
    const getCaretCandidatePosition = (direction, node) => {
        if (isForwards(direction)) {
            if (isCaretCandidate(node.previousSibling) && !isText$3(node.previousSibling)) {
                return CaretPosition.before(node);
            }
            if (isText$3(node)) {
                return CaretPosition(node, 0);
            }
        }
        if (isBackwards(direction)) {
            if (isCaretCandidate(node.nextSibling) && !isText$3(node.nextSibling)) {
                return CaretPosition.after(node);
            }
            if (isText$3(node)) {
                return CaretPosition(node, node.data.length);
            }
        }
        if (isBackwards(direction)) {
            if (isBr$3(node)) {
                return CaretPosition.before(node);
            }
            return CaretPosition.after(node);
        }
        return CaretPosition.before(node);
    };
    const moveForwardFromBr = (root, nextNode) => {
        const nextSibling = nextNode.nextSibling;
        if (nextSibling && isCaretCandidate(nextSibling)) {
            if (isText$3(nextSibling)) {
                return CaretPosition(nextSibling, 0);
            }
            else {
                return CaretPosition.before(nextSibling);
            }
        }
        else {
            return findCaretPosition$1(1 /* HDirection.Forwards */, CaretPosition.after(nextNode), root);
        }
    };
    const findCaretPosition$1 = (direction, startPos, root) => {
        let node;
        let nextNode;
        let innerNode;
        let caretPosition;
        if (!isElement$2(root) || !startPos) {
            return null;
        }
        if (startPos.isEqual(CaretPosition.after(root)) && root.lastChild) {
            caretPosition = CaretPosition.after(root.lastChild);
            if (isBackwards(direction) && isCaretCandidate(root.lastChild) && isElement$2(root.lastChild)) {
                return isBr$3(root.lastChild) ? CaretPosition.before(root.lastChild) : caretPosition;
            }
        }
        else {
            caretPosition = startPos;
        }
        const container = caretPosition.container();
        let offset = caretPosition.offset();
        if (isText$3(container)) {
            if (isBackwards(direction) && offset > 0) {
                return CaretPosition(container, --offset);
            }
            if (isForwards(direction) && offset < container.length) {
                return CaretPosition(container, ++offset);
            }
            node = container;
        }
        else {
            if (isBackwards(direction) && offset > 0) {
                nextNode = nodeAtIndex(container, offset - 1);
                if (isCaretCandidate(nextNode)) {
                    if (!isAtomic(nextNode)) {
                        innerNode = findNode(nextNode, direction, isEditableCaretCandidate, nextNode);
                        if (innerNode) {
                            if (isText$3(innerNode)) {
                                return CaretPosition(innerNode, innerNode.data.length);
                            }
                            return CaretPosition.after(innerNode);
                        }
                    }
                    if (isText$3(nextNode)) {
                        return CaretPosition(nextNode, nextNode.data.length);
                    }
                    return CaretPosition.before(nextNode);
                }
            }
            if (isForwards(direction) && offset < container.childNodes.length) {
                nextNode = nodeAtIndex(container, offset);
                if (isCaretCandidate(nextNode)) {
                    if (isBr$3(nextNode)) {
                        return moveForwardFromBr(root, nextNode);
                    }
                    if (!isAtomic(nextNode)) {
                        innerNode = findNode(nextNode, direction, isEditableCaretCandidate, nextNode);
                        if (innerNode) {
                            if (isText$3(innerNode)) {
                                return CaretPosition(innerNode, 0);
                            }
                            return CaretPosition.before(innerNode);
                        }
                    }
                    if (isText$3(nextNode)) {
                        return CaretPosition(nextNode, 0);
                    }
                    return CaretPosition.after(nextNode);
                }
            }
            node = nextNode ? nextNode : caretPosition.getNode();
        }
        if (node && ((isForwards(direction) && caretPosition.isAtEnd()) || (isBackwards(direction) && caretPosition.isAtStart()))) {
            node = findNode(node, direction, always, root, true);
            if (isEditableCaretCandidate(node, root)) {
                return getCaretCandidatePosition(direction, node);
            }
        }
        nextNode = node ? findNode(node, direction, isEditableCaretCandidate, root) : node;
        const rootContentEditableFalseElm = last(filter$5(getParents$3(container, root), isContentEditableFalse$5));
        if (rootContentEditableFalseElm && (!nextNode || !rootContentEditableFalseElm.contains(nextNode))) {
            if (isForwards(direction)) {
                caretPosition = CaretPosition.after(rootContentEditableFalseElm);
            }
            else {
                caretPosition = CaretPosition.before(rootContentEditableFalseElm);
            }
            return caretPosition;
        }
        if (nextNode) {
            return getCaretCandidatePosition(direction, nextNode);
        }
        return null;
    };
    const CaretWalker = (root) => ({
        /**
           * Returns the next logical caret position from the specified input
           * caretPosition or null if there isn't any more positions left for example
           * at the end specified root element.
           *
           * @method next
           * @param {tinymce.caret.CaretPosition} caretPosition Caret position to start from.
           * @return {tinymce.caret.CaretPosition} CaretPosition or null if no position was found.
           */
        next: (caretPosition) => {
            return findCaretPosition$1(1 /* HDirection.Forwards */, caretPosition, root);
        },
        /**
           * Returns the previous logical caret position from the specified input
           * caretPosition or null if there isn't any more positions left for example
           * at the end specified root element.
           *
           * @method prev
           * @param {tinymce.caret.CaretPosition} caretPosition Caret position to start from.
           * @return {tinymce.caret.CaretPosition} CaretPosition or null if no position was found.
           */
        prev: (caretPosition) => {
            return findCaretPosition$1(-1 /* HDirection.Backwards */, caretPosition, root);
        }
    });

    const walkToPositionIn = (forward, root, start) => {
        const position = forward ? CaretPosition.before(start) : CaretPosition.after(start);
        return fromPosition(forward, root, position);
    };
    const afterElement = (node) => isBr$7(node) ? CaretPosition.before(node) : CaretPosition.after(node);
    const isBeforeOrStart = (position) => {
        if (CaretPosition.isTextPosition(position)) {
            return position.offset() === 0;
        }
        else {
            return isCaretCandidate$3(position.getNode());
        }
    };
    const isAfterOrEnd = (position) => {
        if (CaretPosition.isTextPosition(position)) {
            const container = position.container();
            return position.offset() === container.data.length;
        }
        else {
            return isCaretCandidate$3(position.getNode(true));
        }
    };
    const isBeforeAfterSameElement = (from, to) => !CaretPosition.isTextPosition(from) && !CaretPosition.isTextPosition(to) && from.getNode() === to.getNode(true);
    const isAtBr = (position) => !CaretPosition.isTextPosition(position) && isBr$7(position.getNode());
    const shouldSkipPosition = (forward, from, to) => {
        if (forward) {
            return !isBeforeAfterSameElement(from, to) && !isAtBr(from) && isAfterOrEnd(from) && isBeforeOrStart(to);
        }
        else {
            return !isBeforeAfterSameElement(to, from) && isBeforeOrStart(from) && isAfterOrEnd(to);
        }
    };
    // Finds: <p>a|<b>b</b></p> -> <p>a<b>|b</b></p>
    const fromPosition = (forward, root, pos) => {
        const walker = CaretWalker(root);
        return Optional.from(forward ? walker.next(pos) : walker.prev(pos));
    };
    // Finds: <p>a|<b>b</b></p> -> <p>a<b>b|</b></p>
    const navigate = (forward, root, from) => fromPosition(forward, root, from).bind((to) => {
        if (isInSameBlock(from, to, root) && shouldSkipPosition(forward, from, to)) {
            return fromPosition(forward, root, to);
        }
        else {
            return Optional.some(to);
        }
    });
    const navigateIgnore = (forward, root, from, ignoreFilter) => navigate(forward, root, from)
        .bind((pos) => ignoreFilter(pos) ? navigateIgnore(forward, root, pos, ignoreFilter) : Optional.some(pos));
    const positionIn = (forward, element) => {
        const startNode = forward ? element.firstChild : element.lastChild;
        if (isText$b(startNode)) {
            return Optional.some(CaretPosition(startNode, forward ? 0 : startNode.data.length));
        }
        else if (startNode) {
            if (isCaretCandidate$3(startNode)) {
                return Optional.some(forward ? CaretPosition.before(startNode) : afterElement(startNode));
            }
            else {
                return walkToPositionIn(forward, element, startNode);
            }
        }
        else {
            return Optional.none();
        }
    };
    const nextPosition = curry(fromPosition, true);
    const prevPosition = curry(fromPosition, false);
    const firstPositionIn = curry(positionIn, true);
    const lastPositionIn = curry(positionIn, false);

    const CARET_ID = '_mce_caret';
    const isCaretNode = (node) => isElement$7(node) && node.id === CARET_ID;
    const getParentCaretContainer = (body, node) => {
        let currentNode = node;
        while (currentNode && currentNode !== body) {
            if (isCaretNode(currentNode)) {
                return currentNode;
            }
            currentNode = currentNode.parentNode;
        }
        return null;
    };

    const isStringPathBookmark = (bookmark) => isString(bookmark.start);
    const isRangeBookmark = (bookmark) => has$2(bookmark, 'rng');
    const isIdBookmark = (bookmark) => has$2(bookmark, 'id');
    const isIndexBookmark = (bookmark) => has$2(bookmark, 'name');
    const isPathBookmark = (bookmark) => Tools.isArray(bookmark.start);

    const isForwardBookmark = (bookmark) => !isIndexBookmark(bookmark) && isBoolean(bookmark.forward) ? bookmark.forward : true;
    const addBogus = (dom, node) => {
        // Adds a bogus BR element for empty block elements
        if (isElement$7(node) && dom.isBlock(node) && !node.innerHTML) {
            node.innerHTML = '<br data-mce-bogus="1" />';
        }
        return node;
    };
    const resolveCaretPositionBookmark = (dom, bookmark) => {
        const startPos = Optional.from(resolve$1(dom.getRoot(), bookmark.start));
        const endPos = Optional.from(resolve$1(dom.getRoot(), bookmark.end));
        return lift2(startPos, endPos, (start, end) => {
            const range = dom.createRng();
            range.setStart(start.container(), start.offset());
            range.setEnd(end.container(), end.offset());
            return { range, forward: isForwardBookmark(bookmark) };
        });
    };
    const insertZwsp = (node, rng) => {
        const doc = node.ownerDocument ?? document;
        const textNode = doc.createTextNode(ZWSP$1);
        node.appendChild(textNode);
        rng.setStart(textNode, 0);
        rng.setEnd(textNode, 0);
    };
    const isEmpty$3 = (node) => !node.hasChildNodes();
    const tryFindRangePosition = (node, rng) => lastPositionIn(node).fold(never, (pos) => {
        rng.setStart(pos.container(), pos.offset());
        rng.setEnd(pos.container(), pos.offset());
        return true;
    });
    // Since we trim zwsp from undo levels the caret format containers
    // may be empty if so pad them with a zwsp and move caret there
    const padEmptyCaretContainer = (root, node, rng) => {
        if (isEmpty$3(node) && getParentCaretContainer(root, node)) {
            insertZwsp(node, rng);
            return true;
        }
        else {
            return false;
        }
    };
    const setEndPoint = (dom, start, bookmark, rng) => {
        const point = bookmark[start ? 'start' : 'end'];
        const root = dom.getRoot();
        if (point) {
            let node = root;
            let offset = point[0];
            // Find container node
            for (let i = point.length - 1; node && i >= 1; i--) {
                const children = node.childNodes;
                if (padEmptyCaretContainer(root, node, rng)) {
                    return true;
                }
                if (point[i] > children.length - 1) {
                    if (padEmptyCaretContainer(root, node, rng)) {
                        return true;
                    }
                    return tryFindRangePosition(node, rng);
                }
                node = children[point[i]];
            }
            // Move text offset to best suitable location
            if (isText$b(node)) {
                offset = Math.min(point[0], node.data.length);
            }
            // Move element offset to best suitable location
            if (isElement$7(node)) {
                offset = Math.min(point[0], node.childNodes.length);
            }
            // Set offset within container node
            if (start) {
                rng.setStart(node, offset);
            }
            else {
                rng.setEnd(node, offset);
            }
        }
        return true;
    };
    const isValidTextNode = (node) => isText$b(node) && node.data.length > 0;
    const restoreEndPoint$1 = (dom, suffix, bookmark) => {
        const marker = dom.get(bookmark.id + '_' + suffix);
        const markerParent = marker?.parentNode;
        const keep = bookmark.keep;
        if (marker && markerParent) {
            let container;
            let offset;
            if (suffix === 'start') {
                if (!keep) {
                    container = markerParent;
                    offset = dom.nodeIndex(marker);
                }
                else {
                    if (marker.hasChildNodes()) {
                        container = marker.firstChild;
                        offset = 1;
                    }
                    else if (isValidTextNode(marker.nextSibling)) {
                        container = marker.nextSibling;
                        offset = 0;
                    }
                    else if (isValidTextNode(marker.previousSibling)) {
                        container = marker.previousSibling;
                        offset = marker.previousSibling.data.length;
                    }
                    else {
                        container = markerParent;
                        offset = dom.nodeIndex(marker) + 1;
                    }
                }
            }
            else {
                if (!keep) {
                    container = markerParent;
                    offset = dom.nodeIndex(marker);
                }
                else {
                    if (marker.hasChildNodes()) {
                        container = marker.firstChild;
                        offset = 1;
                    }
                    else if (isValidTextNode(marker.previousSibling)) {
                        container = marker.previousSibling;
                        offset = marker.previousSibling.data.length;
                    }
                    else {
                        container = markerParent;
                        offset = dom.nodeIndex(marker);
                    }
                }
            }
            if (!keep) {
                const prev = marker.previousSibling;
                const next = marker.nextSibling;
                // Remove all marker text nodes
                Tools.each(Tools.grep(marker.childNodes), (node) => {
                    if (isText$b(node)) {
                        node.data = node.data.replace(/\uFEFF/g, '');
                    }
                });
                // Remove marker but keep children if for example contents where inserted into the marker
                // Also remove duplicated instances of the marker for example by a
                // split operation or by WebKit auto split on paste feature
                let otherMarker;
                while ((otherMarker = dom.get(bookmark.id + '_' + suffix))) {
                    dom.remove(otherMarker, true);
                }
                // If siblings are text nodes then merge them unless it's Opera since it some how removes the node
                // and we are sniffing since adding a lot of detection code for a browser with 3% of the market
                // isn't worth the effort. Sorry, Opera but it's just a fact
                if (isText$b(next) && isText$b(prev) && !Env.browser.isOpera()) {
                    const idx = prev.data.length;
                    prev.appendData(next.data);
                    dom.remove(next);
                    container = prev;
                    offset = idx;
                }
            }
            return Optional.some(CaretPosition(container, offset));
        }
        else {
            return Optional.none();
        }
    };
    const resolvePaths = (dom, bookmark) => {
        const range = dom.createRng();
        if (setEndPoint(dom, true, bookmark, range) && setEndPoint(dom, false, bookmark, range)) {
            return Optional.some({ range, forward: isForwardBookmark(bookmark) });
        }
        else {
            return Optional.none();
        }
    };
    const resolveId = (dom, bookmark) => {
        const startPos = restoreEndPoint$1(dom, 'start', bookmark);
        const endPos = restoreEndPoint$1(dom, 'end', bookmark);
        return lift2(startPos, endPos.or(startPos), (spos, epos) => {
            const range = dom.createRng();
            range.setStart(addBogus(dom, spos.container()), spos.offset());
            range.setEnd(addBogus(dom, epos.container()), epos.offset());
            return { range, forward: isForwardBookmark(bookmark) };
        });
    };
    const resolveIndex = (dom, bookmark) => Optional.from(dom.select(bookmark.name)[bookmark.index]).map((elm) => {
        const range = dom.createRng();
        range.selectNode(elm);
        return { range, forward: true };
    });
    const resolve = (selection, bookmark) => {
        const dom = selection.dom;
        if (bookmark) {
            if (isPathBookmark(bookmark)) {
                return resolvePaths(dom, bookmark);
            }
            else if (isStringPathBookmark(bookmark)) {
                return resolveCaretPositionBookmark(dom, bookmark);
            }
            else if (isIdBookmark(bookmark)) {
                return resolveId(dom, bookmark);
            }
            else if (isIndexBookmark(bookmark)) {
                return resolveIndex(dom, bookmark);
            }
            else if (isRangeBookmark(bookmark)) {
                return Optional.some({ range: bookmark.rng, forward: isForwardBookmark(bookmark) });
            }
        }
        return Optional.none();
    };

    const getBookmark$1 = (selection, type, normalized) => {
        return getBookmark$2(selection, type, normalized);
    };
    const moveToBookmark = (selection, bookmark) => {
        resolve(selection, bookmark).each(({ range, forward }) => {
            selection.setRng(range, forward);
        });
    };
    const isBookmarkNode$1 = (node) => {
        return isElement$7(node) && node.tagName === 'SPAN' && node.getAttribute('data-mce-type') === 'bookmark';
    };

    const is = (expected) => (actual) => expected === actual;
    const isNbsp = is(nbsp);
    const isWhiteSpace = (chr) => chr !== '' && ' \f\n\r\t\v'.indexOf(chr) !== -1;
    const isContent = (chr) => !isWhiteSpace(chr) && !isNbsp(chr) && !isZwsp$2(chr);

    const getRanges = (selection) => {
        const ranges = [];
        if (selection) {
            for (let i = 0; i < selection.rangeCount; i++) {
                ranges.push(selection.getRangeAt(i));
            }
        }
        return ranges;
    };
    const getSelectedNodes = (ranges) => {
        return bind$3(ranges, (range) => {
            const node = getSelectedNode(range);
            return node ? [SugarElement.fromDom(node)] : [];
        });
    };
    const hasMultipleRanges = (selection) => {
        return getRanges(selection).length > 1;
    };

    const getCellsFromRanges = (ranges) => filter$5(getSelectedNodes(ranges), isTableCell$2);
    const getCellsFromElement = (elm) => descendants(elm, 'td[data-mce-selected],th[data-mce-selected]');
    const getCellsFromElementOrRanges = (ranges, element) => {
        const selectedCells = getCellsFromElement(element);
        return selectedCells.length > 0 ? selectedCells : getCellsFromRanges(ranges);
    };
    const getCellsFromEditor = (editor) => getCellsFromElementOrRanges(getRanges(editor.selection.getSel()), SugarElement.fromDom(editor.getBody()));
    const getClosestTable = (cell, isRoot) => ancestor$4(cell, 'table', isRoot);

    const getStartNode = (rng) => {
        const sc = rng.startContainer, so = rng.startOffset;
        if (isText$b(sc)) {
            return so === 0 ? Optional.some(SugarElement.fromDom(sc)) : Optional.none();
        }
        else {
            return Optional.from(sc.childNodes[so]).map(SugarElement.fromDom);
        }
    };
    const getEndNode = (rng) => {
        const ec = rng.endContainer, eo = rng.endOffset;
        if (isText$b(ec)) {
            return eo === ec.data.length ? Optional.some(SugarElement.fromDom(ec)) : Optional.none();
        }
        else {
            return Optional.from(ec.childNodes[eo - 1]).map(SugarElement.fromDom);
        }
    };
    const getFirstChildren = (node) => {
        return firstChild(node).fold(constant([node]), (child) => {
            return [node].concat(getFirstChildren(child));
        });
    };
    const getLastChildren = (node) => {
        return lastChild(node).fold(constant([node]), (child) => {
            if (name(child) === 'br') {
                return prevSibling(child).map((sibling) => {
                    return [node].concat(getLastChildren(sibling));
                }).getOr([]);
            }
            else {
                return [node].concat(getLastChildren(child));
            }
        });
    };
    const hasAllContentsSelected = (elm, rng) => {
        return lift2(getStartNode(rng), getEndNode(rng), (startNode, endNode) => {
            const start = find$2(getFirstChildren(elm), curry(eq, startNode));
            const end = find$2(getLastChildren(elm), curry(eq, endNode));
            return start.isSome() && end.isSome();
        }).getOr(false);
    };
    const moveEndPoint = (dom, rng, node, start) => {
        const root = node;
        const walker = new DomTreeWalker(node, root);
        const moveCaretBeforeOnEnterElementsMap = filter$4(dom.schema.getMoveCaretBeforeOnEnterElements(), (_, name) => !contains$2(['td', 'th', 'table'], name.toLowerCase()));
        let currentNode = node;
        do {
            if (isText$b(currentNode) && Tools.trim(currentNode.data).length !== 0) {
                if (start) {
                    rng.setStart(currentNode, 0);
                }
                else {
                    rng.setEnd(currentNode, currentNode.data.length);
                }
                return;
            }
            // BR/IMG/INPUT elements but not table cells
            if (moveCaretBeforeOnEnterElementsMap[currentNode.nodeName]) {
                if (start) {
                    rng.setStartBefore(currentNode);
                }
                else {
                    if (currentNode.nodeName === 'BR') {
                        rng.setEndBefore(currentNode);
                    }
                    else {
                        rng.setEndAfter(currentNode);
                    }
                }
                return;
            }
        } while ((currentNode = (start ? walker.next() : walker.prev())));
        // Failed to find any text node or other suitable location then move to the root of body
        if (root.nodeName === 'BODY') {
            if (start) {
                rng.setStart(root, 0);
            }
            else {
                rng.setEnd(root, root.childNodes.length);
            }
        }
    };
    const hasAnyRanges = (editor) => {
        const sel = editor.selection.getSel();
        return isNonNullable(sel) && sel.rangeCount > 0;
    };
    const runOnRanges = (editor, executor) => {
        // Check to see if a fake selection is active. If so then we are simulating a multi range
        // selection so we should return a range for each selected node.
        // Note: Currently tables are the only thing supported for fake selections.
        const fakeSelectionNodes = getCellsFromEditor(editor);
        if (fakeSelectionNodes.length > 0) {
            each$e(fakeSelectionNodes, (elem) => {
                const node = elem.dom;
                const fakeNodeRng = editor.dom.createRng();
                fakeNodeRng.setStartBefore(node);
                fakeNodeRng.setEndAfter(node);
                executor(fakeNodeRng, true);
            });
        }
        else {
            executor(editor.selection.getRng(), false);
        }
    };
    const preserve = (selection, fillBookmark, executor) => {
        const bookmark = getPersistentBookmark(selection, fillBookmark);
        executor(bookmark);
        selection.moveToBookmark(bookmark);
    };
    const isSelectionOverWholeNode = (range, nodeTypePredicate) => range.startContainer === range.endContainer
        && range.endOffset - range.startOffset === 1
        && nodeTypePredicate(range.startContainer.childNodes[range.startOffset]);
    const isSelectionOverWholeHTMLElement = (range) => isSelectionOverWholeNode(range, isHTMLElement);
    const isSelectionOverWholeTextNode = (range) => isSelectionOverWholeNode(range, isText$b);
    const isSelectionOverWholeAnchor = (range) => isSelectionOverWholeNode(range, isAnchor);

    const isNode = (node) => isNumber(node?.nodeType);
    const isElementNode$1 = (node) => isElement$7(node) && !isBookmarkNode$1(node) && !isCaretNode(node) && !isBogus$1(node);
    // In TinyMCE, directly selected elements are indicated with the data-mce-selected attribute
    // Elements that can be directly selected include control elements such as img, media elements, noneditable elements and others
    const isElementDirectlySelected = (dom, node) => {
        // Table cells are a special case and are separately handled from native editor selection
        if (isElementNode$1(node) && !/^(TD|TH)$/.test(node.nodeName)) {
            const selectedAttr = dom.getAttrib(node, 'data-mce-selected');
            const value = parseInt(selectedAttr, 10);
            // Avoid cases where data-mce-selected is not a positive number e.g. inline-boundary
            return !isNaN(value) && value > 0;
        }
        else {
            return false;
        }
    };
    // TODO: TINY-9130 Look at making SelectionUtils.preserve maintain the noneditable selection instead
    const preserveSelection = (editor, action, shouldMoveStart) => {
        const { selection, dom } = editor;
        const selectedNodeBeforeAction = selection.getNode();
        const isSelectedBeforeNodeNoneditable = isContentEditableFalse$a(selectedNodeBeforeAction);
        preserve(selection, true, () => {
            action();
        });
        // Check previous selected node before the action still exists in the DOM
        // and is still noneditable
        const isBeforeNodeStillNoneditable = isSelectedBeforeNodeNoneditable && isContentEditableFalse$a(selectedNodeBeforeAction);
        if (isBeforeNodeStillNoneditable && dom.isChildOf(selectedNodeBeforeAction, editor.getBody())) {
            editor.selection.select(selectedNodeBeforeAction);
        }
        else if (shouldMoveStart(selection.getStart())) {
            moveStartToNearestText(dom, selection);
        }
    };
    // Note: The reason why we only care about moving the start is because MatchFormat and its function use the start of the selection to determine if a selection has a given format or not
    const moveStartToNearestText = (dom, selection) => {
        const rng = selection.getRng();
        const { startContainer, startOffset } = rng;
        const selectedNode = selection.getNode();
        if (isElementDirectlySelected(dom, selectedNode)) {
            return;
        }
        // Try move startContainer/startOffset to a suitable text node
        if (isElement$7(startContainer)) {
            const nodes = startContainer.childNodes;
            const root = dom.getRoot();
            let walker;
            if (startOffset < nodes.length) {
                const startNode = nodes[startOffset];
                walker = new DomTreeWalker(startNode, dom.getParent(startNode, dom.isBlock) ?? root);
            }
            else {
                const startNode = nodes[nodes.length - 1];
                walker = new DomTreeWalker(startNode, dom.getParent(startNode, dom.isBlock) ?? root);
                walker.next(true);
            }
            for (let node = walker.current(); node; node = walker.next()) {
                // If we have found a noneditable element before we have found any text
                // then we cannot move forward any further as otherwise the start could be put inside
                // the non-editable element which is not valid
                if (dom.getContentEditable(node) === 'false') {
                    return;
                }
                else if (isText$b(node) && !isWhiteSpaceNode$1(node)) {
                    rng.setStart(node, 0);
                    selection.setRng(rng);
                    return;
                }
            }
        }
    };
    /**
     * Returns the next/previous non whitespace node.
     *
     * @private
     * @param {Node} node Node to start at.
     * @param {Boolean} next (Optional) Include next or previous node defaults to previous.
     * @param {Boolean} inc (Optional) Include the current node in checking. Defaults to false.
     * @return {Node} Next or previous node or undefined if it wasn't found.
     */
    const getNonWhiteSpaceSibling = (node, next, inc) => {
        if (node) {
            const nextName = next ? 'nextSibling' : 'previousSibling';
            for (node = inc ? node : node[nextName]; node; node = node[nextName]) {
                if (isElement$7(node) || !isWhiteSpaceNode$1(node)) {
                    return node;
                }
            }
        }
        return undefined;
    };
    const isTextBlock$2 = (schema, node) => !!schema.getTextBlockElements()[node.nodeName.toLowerCase()] || isTransparentBlock(schema, node);
    const isValid = (ed, parent, child) => {
        return ed.schema.isValidChild(parent, child);
    };
    const isWhiteSpaceNode$1 = (node, allowSpaces = false) => {
        if (isNonNullable(node) && isText$b(node)) {
            // If spaces are allowed, treat them as a non-breaking space
            const data = allowSpaces ? node.data.replace(/ /g, '\u00a0') : node.data;
            return isWhitespaceText(data);
        }
        else {
            return false;
        }
    };
    const isEmptyTextNode$1 = (node) => {
        return isNonNullable(node) && isText$b(node) && node.length === 0;
    };
    const isWrapNoneditableTarget = (editor, node) => {
        const baseDataSelector = '[data-mce-cef-wrappable]';
        const formatNoneditableSelector = getFormatNoneditableSelector(editor);
        const selector = isEmpty$5(formatNoneditableSelector) ? baseDataSelector : `${baseDataSelector},${formatNoneditableSelector}`;
        return is$2(SugarElement.fromDom(node), selector);
    };
    // A noneditable element is wrappable if it:
    // - is valid target (has data-mce-cef-wrappable attribute or matches selector from option)
    // - has no editable descendants - removing formats in the editable region can result in the wrapped noneditable being split which is undesirable
    const isWrappableNoneditable = (editor, node) => {
        const dom = editor.dom;
        return (isElementNode$1(node) &&
            dom.getContentEditable(node) === 'false' &&
            isWrapNoneditableTarget(editor, node) &&
            dom.select('[contenteditable="true"]', node).length === 0);
    };
    /**
     * Replaces variables in the value. The variable format is %var.
     *
     * @private
     * @param {String} value Value to replace variables in.
     * @param {Object} vars Name/value array with variables to replace.
     * @return {String} New value with replaced variables.
     */
    const replaceVars = (value, vars) => {
        if (isFunction(value)) {
            return value(vars);
        }
        else if (isNonNullable(vars)) {
            value = value.replace(/%(\w+)/g, (str, name) => {
                return vars[name] || str;
            });
        }
        return value;
    };
    /**
     * Compares two string/nodes regardless of their case.
     *
     * @private
     * @param {String/Node} str1 Node or string to compare.
     * @param {String/Node} str2 Node or string to compare.
     * @return {Boolean} True/false if they match.
     */
    const isEq$5 = (str1, str2) => {
        str1 = str1 || '';
        str2 = str2 || '';
        str1 = '' + (str1.nodeName || str1);
        str2 = '' + (str2.nodeName || str2);
        return str1.toLowerCase() === str2.toLowerCase();
    };
    const normalizeStyleValue = (value, name) => {
        if (isNullable(value)) {
            return null;
        }
        else {
            let strValue = String(value);
            // Force the format to hex
            if (name === 'color' || name === 'backgroundColor') {
                strValue = rgbaToHexString(strValue);
            }
            // Opera will return bold as 700
            if (name === 'fontWeight' && value === 700) {
                strValue = 'bold';
            }
            // Normalize fontFamily so "'Font name', Font" becomes: "Font name,Font"
            if (name === 'fontFamily') {
                strValue = strValue.replace(/[\'\"]/g, '').replace(/,\s+/g, ',');
            }
            return strValue;
        }
    };
    const getStyle = (dom, node, name) => {
        const style = dom.getStyle(node, name);
        return normalizeStyleValue(style, name);
    };
    const getTextDecoration = (dom, node) => {
        let decoration;
        dom.getParent(node, (n) => {
            if (isElement$7(n)) {
                decoration = dom.getStyle(n, 'text-decoration');
                return !!decoration && decoration !== 'none';
            }
            else {
                return false;
            }
        });
        return decoration;
    };
    const getParents$2 = (dom, node, selector) => {
        return dom.getParents(node, selector, dom.getRoot());
    };
    const isFormatPredicate = (editor, formatName, predicate) => {
        const formats = editor.formatter.get(formatName);
        return isNonNullable(formats) && exists(formats, predicate);
    };
    const isVariableFormatName = (editor, formatName) => {
        const hasVariableValues = (format) => {
            const isVariableValue = (val) => isFunction(val) || val.length > 1 && val.charAt(0) === '%';
            return exists(['styles', 'attributes'], (key) => get$a(format, key).exists((field) => {
                const fieldValues = isArray$1(field) ? field : values(field);
                return exists(fieldValues, isVariableValue);
            }));
        };
        return isFormatPredicate(editor, formatName, hasVariableValues);
    };
    /**
     * Checks if the two formats are similar based on the format type, attributes, styles and classes
     */
    const areSimilarFormats = (editor, formatName, otherFormatName) => {
        // Note: MatchFormat.matchNode() uses these parameters to check if a format matches a node
        // Therefore, these are ideal to check if two formats are similar
        const validKeys = ['inline', 'block', 'selector', 'attributes', 'styles', 'classes'];
        const filterObj = (format) => filter$4(format, (_, key) => exists(validKeys, (validKey) => validKey === key));
        return isFormatPredicate(editor, formatName, (fmt1) => {
            const filteredFmt1 = filterObj(fmt1);
            return isFormatPredicate(editor, otherFormatName, (fmt2) => {
                const filteredFmt2 = filterObj(fmt2);
                return equal$1(filteredFmt1, filteredFmt2);
            });
        });
    };
    const isBlockFormat = (format) => hasNonNullableKey(format, 'block');
    const isWrappingBlockFormat = (format) => isBlockFormat(format) && format.wrapper === true;
    const isNonWrappingBlockFormat = (format) => isBlockFormat(format) && format.wrapper !== true;
    const isSelectorFormat = (format) => hasNonNullableKey(format, 'selector');
    const isInlineFormat = (format) => hasNonNullableKey(format, 'inline');
    const isMixedFormat = (format) => isSelectorFormat(format) && isInlineFormat(format) && is$4(get$a(format, 'mixed'), true);
    const shouldExpandToSelector = (format) => isSelectorFormat(format) && format.expand !== false && !isInlineFormat(format);
    const getEmptyCaretContainers = (node) => {
        const nodes = [];
        let tempNode = node;
        while (tempNode) {
            if ((isText$b(tempNode) && tempNode.data !== ZWSP$1) || tempNode.childNodes.length > 1) {
                return [];
            }
            // Collect nodes
            if (isElement$7(tempNode)) {
                nodes.push(tempNode);
            }
            tempNode = tempNode.firstChild;
        }
        return nodes;
    };
    const isCaretContainerEmpty = (node) => {
        return getEmptyCaretContainers(node).length > 0;
    };
    const isEmptyCaretFormatElement = (element) => {
        return isCaretNode(element.dom) && isCaretContainerEmpty(element.dom);
    };

    const isBookmarkNode = isBookmarkNode$1;
    const getParents$1 = getParents$2;
    const isWhiteSpaceNode = isWhiteSpaceNode$1;
    const isTextBlock$1 = isTextBlock$2;
    const isBogusBr$1 = (node) => {
        return isBr$7(node) && node.getAttribute('data-mce-bogus') && !node.nextSibling;
    };
    // Expands the node to the closes contentEditable false element if it exists
    const findParentContentEditable = (dom, node) => {
        let parent = node;
        while (parent) {
            if (isElement$7(parent) && dom.getContentEditable(parent)) {
                return dom.getContentEditable(parent) === 'false' ? parent : node;
            }
            parent = parent.parentNode;
        }
        return node;
    };
    const walkText = (start, node, offset, predicate) => {
        const str = node.data;
        if (start) {
            for (let i = offset; i > 0; i--) {
                if (predicate(str.charAt(i - 1))) {
                    return i;
                }
            }
        }
        else {
            for (let i = offset; i < str.length; i++) {
                if (predicate(str.charAt(i))) {
                    return i;
                }
            }
        }
        return -1;
    };
    const findSpace = (start, node, offset) => walkText(start, node, offset, (c) => isNbsp(c) || isWhiteSpace(c));
    const findContent = (start, node, offset) => walkText(start, node, offset, isContent);
    const findWordEndPoint = (dom, body, container, offset, start, includeTrailingSpaces) => {
        let lastTextNode;
        const closestRoot = dom.getParent(container, (node) => isEditingHost(node) || dom.isBlock(node));
        const rootNode = isNonNullable(closestRoot) ? closestRoot : body;
        const walk = (container, offset, pred) => {
            const textSeeker = TextSeeker(dom);
            const walker = start ? textSeeker.backwards : textSeeker.forwards;
            return Optional.from(walker(container, offset, (text, textOffset) => {
                if (isBookmarkNode(text.parentNode)) {
                    return -1;
                }
                else {
                    lastTextNode = text;
                    return pred(start, text, textOffset);
                }
            }, rootNode));
        };
        const spaceResult = walk(container, offset, findSpace);
        return spaceResult.bind((result) => includeTrailingSpaces ?
            walk(result.container, result.offset + (start ? -1 : 0), findContent) :
            Optional.some(result)).orThunk(() => lastTextNode ?
            Optional.some({ container: lastTextNode, offset: start ? 0 : lastTextNode.length }) :
            Optional.none());
    };
    const findSelectorEndPoint = (dom, formatList, rng, container, siblingName) => {
        const sibling = container[siblingName];
        if (isText$b(container) && isEmpty$5(container.data) && sibling) {
            container = sibling;
        }
        const parents = getParents$1(dom, container);
        for (let i = 0; i < parents.length; i++) {
            for (let y = 0; y < formatList.length; y++) {
                const curFormat = formatList[y];
                // If collapsed state is set then skip formats that doesn't match that
                if (isNonNullable(curFormat.collapsed) && curFormat.collapsed !== rng.collapsed) {
                    continue;
                }
                if (isSelectorFormat(curFormat) && dom.is(parents[i], curFormat.selector)) {
                    return parents[i];
                }
            }
        }
        return container;
    };
    const findBlockEndPoint = (dom, formatList, container, siblingName) => {
        let node = container;
        const root = dom.getRoot();
        const format = formatList[0];
        // Expand to block of similar type
        if (isBlockFormat(format)) {
            node = format.wrapper ? null : dom.getParent(container, format.block, root);
        }
        // Expand to first wrappable block element or any block element
        if (!node) {
            const scopeRoot = dom.getParent(container, 'LI,TD,TH,SUMMARY') ?? root;
            node = dom.getParent(isText$b(container) ? container.parentNode : container, 
            // Fixes #6183 where it would expand to editable parent element in inline mode
            (node) => node !== root && isTextBlock$1(dom.schema, node), scopeRoot);
        }
        // Exclude inner lists from wrapping
        if (node && isBlockFormat(format) && format.wrapper) {
            node = getParents$1(dom, node, 'ul,ol').reverse()[0] || node;
        }
        // Didn't find a block element look for first/last wrappable element
        if (!node) {
            node = container;
            while (node && node[siblingName] && !dom.isBlock(node[siblingName])) {
                node = node[siblingName];
                // Break on BR but include it will be removed later on
                // we can't remove it now since we need to check if it can be wrapped
                if (isEq$5(node, 'br')) {
                    break;
                }
            }
        }
        return node || container;
    };
    // We're at the edge if the parent is a block and there's no next sibling. Alternatively,
    // if we reach the root or can't walk further we also consider it to be a boundary.
    const isAtBlockBoundary$1 = (dom, root, container, siblingName) => {
        const parent = container.parentNode;
        if (isNonNullable(container[siblingName])) {
            return false;
        }
        else if (parent === root || isNullable(parent) || dom.isBlock(parent)) {
            return true;
        }
        else {
            return isAtBlockBoundary$1(dom, root, parent, siblingName);
        }
    };
    // This function walks up the tree if there is no siblings before/after the node.
    // If a sibling is found then the container is returned
    const findParentContainer = (dom, formatList, container, offset, start, expandToBlock) => {
        let parent = container;
        const siblingName = start ? 'previousSibling' : 'nextSibling';
        const root = dom.getRoot();
        // If it's a text node and the offset is inside the text
        if (isText$b(container) && !isWhiteSpaceNode(container)) {
            if (start ? offset > 0 : offset < container.data.length) {
                return container;
            }
        }
        while (parent) {
            if (isEditingHost(parent)) {
                return container;
            }
            // Stop expanding on block elements
            if (!formatList[0].block_expand && dom.isBlock(parent)) {
                return expandToBlock ? parent : container;
            }
            // Walk left/right
            for (let sibling = parent[siblingName]; sibling; sibling = sibling[siblingName]) {
                // Allow spaces if not at the edge of a block element, as the spaces won't have been collapsed
                const allowSpaces = isText$b(sibling) && !isAtBlockBoundary$1(dom, root, sibling, siblingName);
                if (!isBookmarkNode(sibling) && !isBogusBr$1(sibling) && !isWhiteSpaceNode(sibling, allowSpaces)) {
                    return parent;
                }
            }
            // Check if we can move up are we at root level or body level
            if (parent === root || parent.parentNode === root) {
                container = parent;
                break;
            }
            parent = parent.parentNode;
        }
        return container;
    };
    const isSelfOrParentBookmark = (container) => isBookmarkNode(container.parentNode) || isBookmarkNode(container);
    const expandRng = (dom, rng, formatList, expandOptions = {}) => {
        const { includeTrailingSpace = false, expandToBlock = true } = expandOptions;
        const editableHost = dom.getParent(rng.commonAncestorContainer, (node) => isEditingHost(node));
        const root = isNonNullable(editableHost) ? editableHost : dom.getRoot();
        let { startContainer, startOffset, endContainer, endOffset } = rng;
        const format = formatList[0];
        // If index based start position then resolve it
        if (isElement$7(startContainer) && startContainer.hasChildNodes()) {
            startContainer = getNode$1(startContainer, startOffset);
            if (isText$b(startContainer)) {
                startOffset = 0;
            }
        }
        // If index based end position then resolve it
        if (isElement$7(endContainer) && endContainer.hasChildNodes()) {
            endContainer = getNode$1(endContainer, rng.collapsed ? endOffset : endOffset - 1);
            if (isText$b(endContainer)) {
                endOffset = endContainer.data.length;
            }
        }
        // Expand to closest contentEditable element
        startContainer = findParentContentEditable(dom, startContainer);
        endContainer = findParentContentEditable(dom, endContainer);
        // Exclude bookmark nodes if possible
        if (isSelfOrParentBookmark(startContainer)) {
            startContainer = isBookmarkNode(startContainer) ? startContainer : startContainer.parentNode;
            if (rng.collapsed) {
                startContainer = startContainer.previousSibling || startContainer;
            }
            else {
                startContainer = startContainer.nextSibling || startContainer;
            }
            if (isText$b(startContainer)) {
                startOffset = rng.collapsed ? startContainer.length : 0;
            }
        }
        if (isSelfOrParentBookmark(endContainer)) {
            endContainer = isBookmarkNode(endContainer) ? endContainer : endContainer.parentNode;
            if (rng.collapsed) {
                endContainer = endContainer.nextSibling || endContainer;
            }
            else {
                endContainer = endContainer.previousSibling || endContainer;
            }
            if (isText$b(endContainer)) {
                endOffset = rng.collapsed ? 0 : endContainer.length;
            }
        }
        if (rng.collapsed) {
            // Expand left to closest word boundary
            const startPoint = findWordEndPoint(dom, root, startContainer, startOffset, true, includeTrailingSpace);
            startPoint.each(({ container, offset }) => {
                startContainer = container;
                startOffset = offset;
            });
            // Expand right to closest word boundary
            const endPoint = findWordEndPoint(dom, root, endContainer, endOffset, false, includeTrailingSpace);
            endPoint.each(({ container, offset }) => {
                endContainer = container;
                endOffset = offset;
            });
        }
        // Move start/end point up the tree if the leaves are sharp and if we are in different containers
        // Example * becomes !: !<p><b><i>*text</i><i>text*</i></b></p>!
        // This will reduce the number of wrapper elements that needs to be created
        // Move start point up the tree
        if (isInlineFormat(format) || format.block_expand) {
            if (!isInlineFormat(format) || (!isText$b(startContainer) || startOffset === 0)) {
                startContainer = findParentContainer(dom, formatList, startContainer, startOffset, true, expandToBlock);
            }
            if (!isInlineFormat(format) || (!isText$b(endContainer) || endOffset === endContainer.data.length)) {
                endContainer = findParentContainer(dom, formatList, endContainer, endOffset, false, expandToBlock);
            }
        }
        // Expand start/end container to matching selector
        if (shouldExpandToSelector(format)) {
            // Find new startContainer/endContainer if there is better one
            startContainer = findSelectorEndPoint(dom, formatList, rng, startContainer, 'previousSibling');
            endContainer = findSelectorEndPoint(dom, formatList, rng, endContainer, 'nextSibling');
        }
        // Expand start/end container to matching block element or text node
        if (isBlockFormat(format) || isSelectorFormat(format)) {
            // Find new startContainer/endContainer if there is better one
            startContainer = findBlockEndPoint(dom, formatList, startContainer, 'previousSibling');
            endContainer = findBlockEndPoint(dom, formatList, endContainer, 'nextSibling');
            // Non block element then try to expand up the leaf
            if (isBlockFormat(format)) {
                if (!dom.isBlock(startContainer)) {
                    startContainer = findParentContainer(dom, formatList, startContainer, startOffset, true, expandToBlock);
                    if (isText$b(startContainer)) {
                        startOffset = 0;
                    }
                }
                if (!dom.isBlock(endContainer)) {
                    endContainer = findParentContainer(dom, formatList, endContainer, endOffset, false, expandToBlock);
                    if (isText$b(endContainer)) {
                        endOffset = endContainer.data.length;
                    }
                }
            }
        }
        // Setup index for startContainer
        if (isElement$7(startContainer) && startContainer.parentNode) {
            startOffset = dom.nodeIndex(startContainer);
            startContainer = startContainer.parentNode;
        }
        // Setup index for endContainer
        if (isElement$7(endContainer) && endContainer.parentNode) {
            endOffset = dom.nodeIndex(endContainer) + 1;
            endContainer = endContainer.parentNode;
        }
        // Return new range like object
        return {
            startContainer,
            startOffset,
            endContainer,
            endOffset
        };
    };

    const walk$3 = (dom, rng, callback) => {
        const startOffset = rng.startOffset;
        const startContainer = getNode$1(rng.startContainer, startOffset);
        const endOffset = rng.endOffset;
        const endContainer = getNode$1(rng.endContainer, endOffset - 1);
        /**
         * Excludes start/end text node if they are out side the range
         *
         * @private
         * @param {Array} nodes Nodes to exclude items from.
         * @return {Array} Array with nodes excluding the start/end container if needed.
         */
        const exclude = (nodes) => {
            // First node is excluded
            const firstNode = nodes[0];
            if (isText$b(firstNode) && firstNode === startContainer && startOffset >= firstNode.data.length) {
                nodes.splice(0, 1);
            }
            // Last node is excluded
            const lastNode = nodes[nodes.length - 1];
            if (endOffset === 0 && nodes.length > 0 && lastNode === endContainer && isText$b(lastNode)) {
                nodes.splice(nodes.length - 1, 1);
            }
            return nodes;
        };
        const collectSiblings = (node, name, endNode) => {
            const siblings = [];
            for (; node && node !== endNode; node = node[name]) {
                siblings.push(node);
            }
            return siblings;
        };
        const findEndPoint = (node, root) => dom.getParent(node, (node) => node.parentNode === root, root);
        const walkBoundary = (startNode, endNode, next) => {
            const siblingName = next ? 'nextSibling' : 'previousSibling';
            for (let node = startNode, parent = node.parentNode; node && node !== endNode; node = parent) {
                parent = node.parentNode;
                const siblings = collectSiblings(node === startNode ? node : node[siblingName], siblingName);
                if (siblings.length) {
                    if (!next) {
                        siblings.reverse();
                    }
                    callback(exclude(siblings));
                }
            }
        };
        // Same container
        if (startContainer === endContainer) {
            return callback(exclude([startContainer]));
        }
        // Find common ancestor and end points
        const ancestor = dom.findCommonAncestor(startContainer, endContainer) ?? dom.getRoot();
        // Process left side
        if (dom.isChildOf(startContainer, endContainer)) {
            return walkBoundary(startContainer, ancestor, true);
        }
        // Process right side
        if (dom.isChildOf(endContainer, startContainer)) {
            return walkBoundary(endContainer, ancestor);
        }
        // Find start/end point
        const startPoint = findEndPoint(startContainer, ancestor) || startContainer;
        const endPoint = findEndPoint(endContainer, ancestor) || endContainer;
        // Walk left leaf
        walkBoundary(startContainer, startPoint, true);
        // Walk the middle from start to end point
        const siblings = collectSiblings(startPoint === startContainer ? startPoint : startPoint.nextSibling, 'nextSibling', endPoint === endContainer ? endPoint.nextSibling : endPoint);
        if (siblings.length) {
            callback(exclude(siblings));
        }
        // Walk right leaf
        walkBoundary(endContainer, endPoint);
    };

    const validBlocks = [
        // Codesample plugin
        'pre[class*=language-][contenteditable="false"]',
        // Image plugin - captioned image
        'figure.image',
        // Mediaembed plugin
        'div[data-ephox-embed-iri]',
        // Pageembed plugin
        'div.tiny-pageembed',
        // Tableofcontents plugin
        'div.mce-toc',
        'div[data-mce-toc]',
        // Footnootes plugin
        'div.mce-footnotes'
    ];
    const isZeroWidth = (elem) => isText$c(elem) && get$4(elem) === ZWSP$1;
    const context = (editor, elem, wrapName, nodeName) => parentElement(elem).fold(() => "skipping" /* ChildContext.Skipping */, (parent) => {
        // We used to skip these, but given that they might be representing empty paragraphs, it probably
        // makes sense to treat them just like text nodes
        if (nodeName === 'br' || isZeroWidth(elem)) {
            return "valid" /* ChildContext.Valid */;
        }
        else if (isAnnotation(elem)) {
            return "existing" /* ChildContext.Existing */;
        }
        else if (isCaretNode(elem.dom)) {
            return "caret" /* ChildContext.Caret */;
        }
        else if (exists(validBlocks, (selector) => is$2(elem, selector))) {
            return "valid-block" /* ChildContext.ValidBlock */;
        }
        else if (!isValid(editor, wrapName, nodeName) || !isValid(editor, name(parent), wrapName)) {
            return "invalid-child" /* ChildContext.InvalidChild */;
        }
        else {
            return "valid" /* ChildContext.Valid */;
        }
    });

    const applyWordGrab = (editor, rng) => {
        const r = expandRng(editor.dom, rng, [{ inline: 'span' }]);
        rng.setStart(r.startContainer, r.startOffset);
        rng.setEnd(r.endContainer, r.endOffset);
        editor.selection.setRng(rng);
    };
    const applyAnnotation = (elem, masterUId, data, annotationName, decorate, directAnnotation) => {
        const { uid = masterUId, ...otherData } = data;
        add$2(elem, annotation());
        set$4(elem, `${dataAnnotationId()}`, uid);
        set$4(elem, `${dataAnnotation()}`, annotationName);
        const { attributes = {}, classes = [] } = decorate(uid, otherData);
        setAll$1(elem, attributes);
        add$1(elem, classes);
        if (directAnnotation) {
            if (classes.length > 0) {
                set$4(elem, `${dataAnnotationClasses()}`, classes.join(','));
            }
            const attributeNames = keys(attributes);
            if (attributeNames.length > 0) {
                set$4(elem, `${dataAnnotationAttributes()}`, attributeNames.join(','));
            }
        }
    };
    const removeDirectAnnotation = (elem) => {
        remove$4(elem, annotation());
        remove$9(elem, `${dataAnnotationId()}`);
        remove$9(elem, `${dataAnnotation()}`);
        remove$9(elem, `${dataAnnotationActive()}`);
        const customAttrNames = getOpt(elem, `${dataAnnotationAttributes()}`).map((names) => names.split(',')).getOr([]);
        const customClasses = getOpt(elem, `${dataAnnotationClasses()}`).map((names) => names.split(',')).getOr([]);
        each$e(customAttrNames, (name) => remove$9(elem, name));
        remove$3(elem, customClasses);
        remove$9(elem, `${dataAnnotationClasses()}`);
        remove$9(elem, `${dataAnnotationAttributes()}`);
    };
    const makeAnnotation = (eDoc, uid, data, annotationName, decorate) => {
        const master = SugarElement.fromTag('span', eDoc);
        applyAnnotation(master, uid, data, annotationName, decorate, false);
        return master;
    };
    const annotate = (editor, rng, uid, annotationName, decorate, data) => {
        // Setup all the wrappers that are going to be used.
        const newWrappers = [];
        // Setup the spans for the comments
        const master = makeAnnotation(editor.getDoc(), uid, data, annotationName, decorate);
        // Set the current wrapping element
        const wrapper = value$1();
        // Clear the current wrapping element, so that subsequent calls to
        // getOrOpenWrapper spawns a new one.
        const finishWrapper = () => {
            wrapper.clear();
        };
        // Get the existing wrapper, or spawn a new one.
        const getOrOpenWrapper = () => wrapper.get().getOrThunk(() => {
            const nu = shallow(master);
            newWrappers.push(nu);
            wrapper.set(nu);
            return nu;
        });
        const processElements = (elems) => {
            each$e(elems, processElement);
        };
        const processElement = (elem) => {
            const ctx = context(editor, elem, 'span', name(elem));
            switch (ctx) {
                case "invalid-child" /* ChildContext.InvalidChild */: {
                    finishWrapper();
                    const children = children$1(elem);
                    processElements(children);
                    finishWrapper();
                    break;
                }
                case "valid-block" /* ChildContext.ValidBlock */: {
                    finishWrapper();
                    applyAnnotation(elem, uid, data, annotationName, decorate, true);
                    break;
                }
                case "valid" /* ChildContext.Valid */: {
                    const w = getOrOpenWrapper();
                    wrap$2(elem, w);
                    break;
                }
            }
        };
        const processNodes = (nodes) => {
            const elems = map$3(nodes, SugarElement.fromDom);
            processElements(elems);
        };
        walk$3(editor.dom, rng, (nodes) => {
            finishWrapper();
            processNodes(nodes);
        });
        return newWrappers;
    };
    const annotateWithBookmark = (editor, name, settings, data) => {
        editor.undoManager.transact(() => {
            const selection = editor.selection;
            const initialRng = selection.getRng();
            const hasFakeSelection = getCellsFromEditor(editor).length > 0;
            const masterUid = generate('mce-annotation');
            if (initialRng.collapsed && !hasFakeSelection) {
                applyWordGrab(editor, initialRng);
            }
            // Even after applying word grab, we could not find a selection. Therefore,
            // just make a wrapper and insert it at the current cursor
            if (selection.getRng().collapsed && !hasFakeSelection) {
                const wrapper = makeAnnotation(editor.getDoc(), masterUid, data, name, settings.decorate);
                // Put something visible in the marker
                set$3(wrapper, nbsp);
                selection.getRng().insertNode(wrapper.dom);
                selection.select(wrapper.dom);
            }
            else {
                // The bookmark is responsible for splitting the nodes beforehand at the selection points
                // The "false" here means a zero width cursor is NOT put in the bookmark. It seems to be required
                // to stop an empty paragraph splitting into two paragraphs. Probably a better way exists.
                preserve(selection, false, () => {
                    runOnRanges(editor, (selectionRng) => {
                        annotate(editor, selectionRng, masterUid, name, settings.decorate, data);
                    });
                });
            }
        });
    };

    const Annotator = (editor) => {
        const registry = create$a();
        setup$D(editor, registry);
        const changes = setup$E(editor, registry);
        const isSpan = isTag('span');
        const removeAnnotations = (elements) => {
            each$e(elements, (element) => {
                if (isSpan(element)) {
                    unwrap(element);
                }
                else {
                    removeDirectAnnotation(element);
                }
            });
        };
        return {
            /**
             * Registers a specific annotator by name
             *
             * @method register
             * @param {String} name the name of the annotation
             * @param {Object} settings settings for the annotation (e.g. decorate)
             */
            register: (name, settings) => {
                registry.register(name, settings);
            },
            /**
             * Applies the annotation at the current selection using data
             *
             * @method annotate
             * @param {String} name the name of the annotation to apply
             * @param {Object} data information to pass through to this particular
             * annotation
             */
            annotate: (name, data) => {
                registry.lookup(name).each((settings) => {
                    annotateWithBookmark(editor, name, settings, data);
                });
            },
            /**
             * Executes the specified callback when the current selection matches the annotation or not.
             *
             * @method annotationChanged
             * @param {String} name Name of annotation to listen for
             * @param {Function} callback Callback with (state, name, and data) fired when the annotation
             * at the cursor changes. If state if false, data will not be provided.
             */
            annotationChanged: (name, callback) => {
                changes.addListener(name, callback);
            },
            /**
             * Removes any annotations from the current selection that match
             * the name
             *
             * @method remove
             * @param {String} name the name of the annotation to remove
             */
            remove: (name) => {
                identify(editor, Optional.some(name)).each(({ elements }) => {
                    /**
                     * TINY-9399: It is important to keep the bookmarking in the callback
                     * because it adjusts selection in a way that `identify` function
                     * cannot retain the selected word.
                     */
                    const bookmark = editor.selection.getBookmark();
                    removeAnnotations(elements);
                    editor.selection.moveToBookmark(bookmark);
                });
            },
            /**
             * Removes all annotations that match the specified name from the entire document.
             *
             * @method removeAll
             * @param {String} name the name of the annotation to remove
             */
            removeAll: (name) => {
                const bookmark = editor.selection.getBookmark();
                each$d(findAll(editor, name), (elements, _) => {
                    removeAnnotations(elements);
                });
                editor.selection.moveToBookmark(bookmark);
            },
            /**
             * Retrieve all the annotations for a given name
             *
             * @method getAll
             * @param {String} name the name of the annotations to retrieve
             * @return {Object} an index of annotations from uid => DOM nodes
             */
            getAll: (name) => {
                const directory = findAll(editor, name);
                return map$2(directory, (elems) => map$3(elems, (elem) => elem.dom));
            }
        };
    };

    /**
     * Constructs a new BookmarkManager instance for a specific selection instance.
     *
     * @constructor
     * @method BookmarkManager
     * @param {tinymce.dom.Selection} selection Selection instance to handle bookmarks for.
     */
    const BookmarkManager = (selection) => {
        return {
            /**
             * Returns a bookmark location for the current selection. This bookmark object
             * can then be used to restore the selection after some content modification to the document.
             *
             * @method getBookmark
             * @param {Number} type Optional state if the bookmark should be simple or not. Default is complex.
             * @param {Boolean} normalized Optional state that enables you to get a position that it would be after normalization.
             * @return {Object} Bookmark object, use moveToBookmark with this object to restore the selection.
             * @example
             * // Stores a bookmark of the current selection
             * const bm = tinymce.activeEditor.selection.getBookmark();
             *
             * tinymce.activeEditor.setContent(tinymce.activeEditor.getContent() + 'Some new content');
             *
             * // Restore the selection bookmark
             * tinymce.activeEditor.selection.moveToBookmark(bm);
             */
            getBookmark: curry(getBookmark$1, selection),
            /**
             * Restores the selection to the specified bookmark.
             *
             * @method moveToBookmark
             * @param {Object} bookmark Bookmark to restore selection from.
             * @example
             * // Stores a bookmark of the current selection
             * const bm = tinymce.activeEditor.selection.getBookmark();
             *
             * tinymce.activeEditor.setContent(tinymce.activeEditor.getContent() + 'Some new content');
             *
             * // Restore the selection bookmark
             * tinymce.activeEditor.selection.moveToBookmark(bm);
             */
            moveToBookmark: curry(moveToBookmark, selection)
        };
    };
    /**
     * Returns true/false if the specified node is a bookmark node or not.
     *
     * @static
     * @method isBookmarkNode
     * @param {DOMNode} node DOM Node to check if it's a bookmark node or not.
     * @return {Boolean} true/false if the node is a bookmark node or not.
     */
    BookmarkManager.isBookmarkNode = isBookmarkNode$1;

    const isXYWithinRange = (clientX, clientY, range) => {
        if (range.collapsed) {
            return false;
        }
        else {
            return exists(range.getClientRects(), (rect) => containsXY(rect, clientX, clientY));
        }
    };

    const clamp$1 = (offset, element) => {
        const max = isText$c(element) ? get$4(element).length : children$1(element).length + 1;
        if (offset > max) {
            return max;
        }
        else if (offset < 0) {
            return 0;
        }
        return offset;
    };
    const normalizeRng = (rng) => SimSelection.range(rng.start, clamp$1(rng.soffset, rng.start), rng.finish, clamp$1(rng.foffset, rng.finish));
    const isOrContains = (root, elm) => !isRestrictedNode(elm.dom) && (contains(root, elm) || eq(root, elm));
    const isRngInRoot = (root) => (rng) => isOrContains(root, rng.start) && isOrContains(root, rng.finish);
    // TINY-9259: We need to store the selection on Firefox since if the editor is hidden the selection.getRng() api will not work as expected.
    const shouldStore = (editor) => editor.inline || Env.browser.isFirefox();
    const nativeRangeToSelectionRange = (r) => SimSelection.range(SugarElement.fromDom(r.startContainer), r.startOffset, SugarElement.fromDom(r.endContainer), r.endOffset);
    const readRange = (win) => {
        const selection = win.getSelection();
        const rng = !selection || selection.rangeCount === 0 ? Optional.none() : Optional.from(selection.getRangeAt(0));
        return rng.map(nativeRangeToSelectionRange);
    };
    const getBookmark = (root) => {
        const win = defaultView(root);
        return readRange(win.dom)
            .filter(isRngInRoot(root));
    };
    const validate = (root, bookmark) => Optional.from(bookmark)
        .filter(isRngInRoot(root))
        .map(normalizeRng);
    const bookmarkToNativeRng = (bookmark) => {
        const rng = document.createRange();
        try {
            // Might throw IndexSizeError
            rng.setStart(bookmark.start.dom, bookmark.soffset);
            rng.setEnd(bookmark.finish.dom, bookmark.foffset);
            return Optional.some(rng);
        }
        catch {
            return Optional.none();
        }
    };
    const store = (editor) => {
        const newBookmark = shouldStore(editor) ? getBookmark(SugarElement.fromDom(editor.getBody())) : Optional.none();
        editor.bookmark = newBookmark.isSome() ? newBookmark : editor.bookmark;
    };
    const getRng = (editor) => {
        const bookmark = editor.bookmark ? editor.bookmark : Optional.none();
        return bookmark
            .bind((x) => validate(SugarElement.fromDom(editor.getBody()), x))
            .bind(bookmarkToNativeRng);
    };
    const restore = (editor) => {
        getRng(editor).each((rng) => editor.selection.setRng(rng));
    };

    /**
     * This class manages the focus/blur state of the editor. This class is needed since some
     * browsers fire false focus/blur states when the selection is moved to a UI dialog or similar.
     *
     * This class will fire two events focus and blur on the editor instances that got affected.
     * It will also handle the restore of selection when the focus is lost and returned.
     *
     * @class tinymce.FocusManager
     * @private
     */
    /**
     * Returns true if the specified element is part of the UI for example an button or text input.
     *
     * @static
     * @method isEditorUIElement
     * @param  {Element} elm Element to check if it's part of the UI or not.
     * @return {Boolean} True/false state if the element is part of the UI or not.
     */
    const isEditorUIElement$1 = (elm) => {
        // Needs to be converted to string since svg can have focus: #6776
        const className = elm.className.toString();
        return className.indexOf('tox-') !== -1 || className.indexOf('mce-') !== -1;
    };
    const FocusManager = {
        isEditorUIElement: isEditorUIElement$1
    };

    /**
     * Utility class for working with delayed actions like setTimeout.
     *
     * @class tinymce.util.Delay
     */
    const wrappedSetTimeout = (callback, time) => {
        if (!isNumber(time)) {
            time = 0;
        }
        return window.setTimeout(callback, time);
    };
    const wrappedSetInterval = (callback, time) => {
        if (!isNumber(time)) {
            time = 0;
        }
        return window.setInterval(callback, time);
    };
    const Delay = {
        /**
         * Sets a timeout that's similar to the native browser <a href="https://developer.mozilla.org/en-US/docs/Web/API/setTimeout">setTimeout</a>
         * API, except that it checks if the editor instance is still alive when the callback gets executed.
         *
         * @method setEditorTimeout
         * @param {tinymce.Editor} editor Editor instance to check the removed state on.
         * @param {Function} callback Callback to execute when timer runs out.
         * @param {Number} time Optional time to wait before the callback is executed, defaults to 0.
         * @return {Number} Timeout id number.
         */
        setEditorTimeout: (editor, callback, time) => {
            return wrappedSetTimeout(() => {
                if (!editor.removed) {
                    callback();
                }
            }, time);
        },
        /**
         * Sets an interval timer that's similar to native browser <a href="https://developer.mozilla.org/en-US/docs/Web/API/setInterval">setInterval</a>
         * API, except that it checks if the editor instance is still alive when the callback gets executed.
         *
         * @method setEditorInterval
         * @param {Function} callback Callback to execute when interval time runs out.
         * @param {Number} time Optional time to wait before the callback is executed, defaults to 0.
         * @return {Number} Timeout id number.
         */
        setEditorInterval: (editor, callback, time) => {
            const timer = wrappedSetInterval(() => {
                if (!editor.removed) {
                    callback();
                }
                else {
                    window.clearInterval(timer);
                }
            }, time);
            return timer;
        }
    };

    const isManualNodeChange = (e) => {
        return e.type === 'nodechange' && e.selectionChange;
    };
    const registerPageMouseUp = (editor, throttledStore) => {
        const mouseUpPage = () => {
            throttledStore.throttle();
        };
        DOMUtils.DOM.bind(document, 'mouseup', mouseUpPage);
        editor.on('remove', () => {
            DOMUtils.DOM.unbind(document, 'mouseup', mouseUpPage);
        });
    };
    const registerMouseUp = (editor, throttledStore) => {
        editor.on('mouseup touchend', (_e) => {
            throttledStore.throttle();
        });
    };
    const registerEditorEvents = (editor, throttledStore) => {
        registerMouseUp(editor, throttledStore);
        editor.on('keyup NodeChange AfterSetSelectionRange', (e) => {
            if (!isManualNodeChange(e)) {
                store(editor);
            }
        });
    };
    const register$6 = (editor) => {
        const throttledStore = first$1(() => {
            store(editor);
        }, 0);
        editor.on('init', () => {
            if (editor.inline) {
                registerPageMouseUp(editor, throttledStore);
            }
            registerEditorEvents(editor, throttledStore);
        });
        editor.on('remove', () => {
            throttledStore.cancel();
        });
    };

    let documentFocusInHandler;
    const DOM$c = DOMUtils.DOM;
    const isEditorUIElement = (elm) => {
        // Since this can be overridden by third party we need to use the API reference here
        return isElement$7(elm) && FocusManager.isEditorUIElement(elm);
    };
    const isEditorContentAreaElement = (elm) => {
        const classList = elm.classList;
        if (classList !== undefined) {
            // tox-edit-area__iframe === iframe container element
            // mce-content-body === inline body element
            return classList.contains('tox-edit-area') || classList.contains('tox-edit-area__iframe') || classList.contains('mce-content-body');
        }
        else {
            return false;
        }
    };
    const isUIElement = (editor, elm) => {
        const customSelector = getCustomUiSelector(editor);
        const parent = DOM$c.getParent(elm, (elm) => {
            return (isEditorUIElement(elm) ||
                (customSelector ? editor.dom.is(elm, customSelector) : false));
        });
        return parent !== null;
    };
    const getActiveElement = (editor) => {
        try {
            const root = getRootNode(SugarElement.fromDom(editor.getElement()));
            return active(root).fold(() => document.body, (x) => x.dom);
        }
        catch {
            // IE sometimes fails to get the activeElement when resizing table
            // TODO: Investigate this
            return document.body;
        }
    };
    const registerEvents$1 = (editorManager, e) => {
        const editor = e.editor;
        register$6(editor);
        const toggleContentAreaOnFocus = (editor, fn) => {
            // Inline editors have a different approach to highlight the content area on focus
            if (shouldHighlightOnFocus(editor) && editor.inline !== true) {
                const contentArea = SugarElement.fromDom(editor.getContainer());
                fn(contentArea, 'tox-edit-focus');
            }
        };
        editor.on('focusin', () => {
            const focusedEditor = editorManager.focusedEditor;
            if (isEditorContentAreaElement(getActiveElement(editor))) {
                toggleContentAreaOnFocus(editor, add$2);
            }
            if (focusedEditor !== editor) {
                if (focusedEditor) {
                    focusedEditor.dispatch('blur', { focusedEditor: editor });
                }
                editorManager.setActive(editor);
                editorManager.focusedEditor = editor;
                editor.dispatch('focus', { blurredEditor: focusedEditor });
                editor.focus(true);
            }
        });
        editor.on('focusout', () => {
            Delay.setEditorTimeout(editor, () => {
                const focusedEditor = editorManager.focusedEditor;
                // Remove focus highlight when the content area is no longer the active editor element, or if the highlighted editor is not the current focused editor
                if (!isEditorContentAreaElement(getActiveElement(editor)) || focusedEditor !== editor) {
                    toggleContentAreaOnFocus(editor, remove$4);
                }
                // Still the same editor the blur was outside any editor UI
                if (!isUIElement(editor, getActiveElement(editor)) && focusedEditor === editor) {
                    editor.dispatch('blur', { focusedEditor: null });
                    editorManager.focusedEditor = null;
                }
            });
        });
        // Check if focus is moved to an element outside the active editor by checking if the target node
        // isn't within the body of the activeEditor nor a UI element such as a dialog child control
        if (!documentFocusInHandler) {
            documentFocusInHandler = (e) => {
                const activeEditor = editorManager.activeEditor;
                if (activeEditor) {
                    getOriginalEventTarget(e).each((target) => {
                        const elem = target;
                        if (elem.ownerDocument === document) {
                            // Fire a blur event if the element isn't a UI element
                            if (elem !== document.body && !isUIElement(activeEditor, elem) && editorManager.focusedEditor === activeEditor) {
                                activeEditor.dispatch('blur', { focusedEditor: null });
                                editorManager.focusedEditor = null;
                            }
                        }
                    });
                }
            };
            DOM$c.bind(document, 'focusin', documentFocusInHandler);
        }
    };
    const unregisterDocumentEvents = (editorManager, e) => {
        if (editorManager.focusedEditor === e.editor) {
            editorManager.focusedEditor = null;
        }
        if (!editorManager.activeEditor && documentFocusInHandler) {
            DOM$c.unbind(document, 'focusin', documentFocusInHandler);
            documentFocusInHandler = null;
        }
    };
    const setup$C = (editorManager) => {
        editorManager.on('AddEditor', curry(registerEvents$1, editorManager));
        editorManager.on('RemoveEditor', curry(unregisterDocumentEvents, editorManager));
    };

    const getContentEditableHost = (editor, node) => editor.dom.getParent(node, (node) => editor.dom.getContentEditable(node) === 'true');
    const hasContentEditableFalseParent$1 = (editor, node) => editor.dom.getParent(node, (node) => editor.dom.getContentEditable(node) === 'false') !== null;
    const getCollapsedNode = (rng) => rng.collapsed ? Optional.from(getNode$1(rng.startContainer, rng.startOffset)).map(SugarElement.fromDom) : Optional.none();
    const getFocusInElement = (root, rng) => getCollapsedNode(rng).bind((node) => {
        if (isTableSection(node)) {
            return Optional.some(node);
        }
        else if (!contains(root, node)) {
            return Optional.some(root);
        }
        else {
            return Optional.none();
        }
    });
    const normalizeSelection = (editor, rng) => {
        getFocusInElement(SugarElement.fromDom(editor.getBody()), rng).bind((elm) => {
            return firstPositionIn(elm.dom);
        }).fold(() => {
            editor.selection.normalize();
        }, (caretPos) => editor.selection.setRng(caretPos.toRange()));
    };
    const focusBody = (body) => {
        if (body.setActive) {
            // IE 11 sometimes throws "Invalid function" then fallback to focus
            // setActive is better since it doesn't scroll to the element being focused
            try {
                body.setActive();
            }
            catch {
                body.focus();
            }
        }
        else {
            body.focus();
        }
    };
    const hasElementFocus = (elm) => hasFocus$1(elm) || search(elm).isSome();
    const hasIframeFocus = (editor) => isNonNullable(editor.iframeElement) && hasFocus$1(SugarElement.fromDom(editor.iframeElement));
    const hasInlineFocus = (editor) => {
        const rawBody = editor.getBody();
        return rawBody && hasElementFocus(SugarElement.fromDom(rawBody));
    };
    const hasUiFocus = (editor) => {
        const dos = getRootNode(SugarElement.fromDom(editor.getElement()));
        // Editor container is the obvious one (Menubar, Toolbar, Status bar, Sidebar) and dialogs and menus are in an auxiliary element (silver theme specific)
        // This can't use Focus.search() because only the theme has this element reference
        return active(dos)
            .filter((elem) => !isEditorContentAreaElement(elem.dom) && isUIElement(editor, elem.dom))
            .isSome();
    };
    const hasFocus = (editor) => editor.inline ? hasInlineFocus(editor) : hasIframeFocus(editor);
    const hasEditorOrUiFocus = (editor) => hasFocus(editor) || hasUiFocus(editor);
    const focusEditor = (editor) => {
        const selection = editor.selection;
        const body = editor.getBody();
        let rng = selection.getRng();
        editor.quirks.refreshContentEditable();
        const restoreBookmark = (editor) => {
            getRng(editor).each((bookmarkRng) => {
                editor.selection.setRng(bookmarkRng);
                rng = bookmarkRng;
            });
        };
        if (!hasFocus(editor) && editor.hasEditableRoot()) {
            restoreBookmark(editor);
        }
        // Move focus to contentEditable=true child if needed
        const contentEditableHost = getContentEditableHost(editor, selection.getNode());
        if (contentEditableHost && editor.dom.isChildOf(contentEditableHost, body)) {
            if (!hasContentEditableFalseParent$1(editor, contentEditableHost)) {
                focusBody(body);
            }
            focusBody(contentEditableHost);
            if (!editor.hasEditableRoot()) {
                restoreBookmark(editor);
            }
            normalizeSelection(editor, rng);
            activateEditor(editor);
            return;
        }
        // Focus the window iframe
        if (!editor.inline) {
            // WebKit needs this call to fire focusin event properly see #5948
            // But Opera pre Blink engine will produce an empty selection so skip Opera
            if (!Env.browser.isOpera()) {
                focusBody(body);
            }
            editor.getWin().focus();
        }
        // Focus the body as well since it's contentEditable
        if (Env.browser.isFirefox() || editor.inline) {
            focusBody(body);
            normalizeSelection(editor, rng);
        }
        activateEditor(editor);
    };
    const activateEditor = (editor) => editor.editorManager.setActive(editor);
    const focus = (editor, skipFocus) => {
        if (editor.removed) {
            return;
        }
        if (skipFocus) {
            activateEditor(editor);
        }
        else {
            focusEditor(editor);
        }
    };

    /**
     * This file exposes a set of the common KeyCodes for use. Please grow it as needed.
     */
    const VK = {
        BACKSPACE: 8,
        DELETE: 46,
        DOWN: 40,
        ENTER: 13,
        ESC: 27,
        LEFT: 37,
        RIGHT: 39,
        SPACEBAR: 32,
        TAB: 9,
        UP: 38,
        PAGE_UP: 33,
        PAGE_DOWN: 34,
        END: 35,
        HOME: 36,
        modifierPressed: (e) => {
            return e.shiftKey || e.ctrlKey || e.altKey || VK.metaKeyPressed(e);
        },
        metaKeyPressed: (e) => {
            // Check if ctrl or meta key is pressed. Edge case for AltGr on Windows where it produces ctrlKey+altKey states
            return Env.os.isMacOS() || Env.os.isiOS() ? e.metaKey : e.ctrlKey && !e.altKey;
        }
    };

    const elementSelectionAttr = 'data-mce-selected';
    const controlElmSelector = `table,img,figure.image,hr,video,span.mce-preview-object,details,${ucVideoNodeName}`;
    const abs = Math.abs;
    const round$1 = Math.round;
    // Details about each resize handle how to scale etc
    const resizeHandles = {
        // Name: x multiplier, y multiplier, delta size x, delta size y
        nw: [0, 0, -1, -1],
        ne: [1, 0, 1, -1],
        se: [1, 1, 1, 1],
        sw: [0, 1, -1, 1]
    };
    const isTouchEvent = (evt) => evt.type === 'longpress' || evt.type.indexOf('touch') === 0;
    /**
     * This class handles control selection of elements. Controls are elements
     * that can be resized and needs to be selected as a whole. It adds custom resize handles
     * to all browser engines that support properly disabling the built in resize logic.
     *
     * @private
     * @class tinymce.dom.ControlSelection
     */
    const ControlSelection = (selection, editor) => {
        const dom = editor.dom;
        const editableDoc = editor.getDoc();
        const rootDocument = document;
        const rootElement = editor.getBody();
        let selectedElm, selectedElmGhost, resizeHelper, selectedHandle, resizeBackdrop;
        let startX, startY, startW, startH, ratio, resizeStarted;
        let width;
        let height;
        let startScrollWidth;
        let startScrollHeight;
        const isImage = (elm) => isNonNullable(elm) && (isImg(elm) || dom.is(elm, 'figure.image'));
        const isMedia = (elm) => isMedia$2(elm) || dom.hasClass(elm, 'mce-preview-object');
        const isEventOnImageOutsideRange = (evt, range) => {
            if (isTouchEvent(evt)) {
                const touch = evt.touches[0];
                return isImage(evt.target) && !isXYWithinRange(touch.clientX, touch.clientY, range);
            }
            else {
                return isImage(evt.target) && !isXYWithinRange(evt.clientX, evt.clientY, range);
            }
        };
        const contextMenuSelectImage = (evt) => {
            const target = evt.target;
            if (isEventOnImageOutsideRange(evt, editor.selection.getRng()) && !evt.isDefaultPrevented()) {
                editor.selection.select(target);
            }
        };
        const getResizeTargets = (elm) => {
            if (dom.hasClass(elm, 'mce-preview-object') && isNonNullable(elm.firstElementChild)) {
                // When resizing a preview object we need to resize both the original element and the wrapper span
                return [elm, elm.firstElementChild];
            }
            else if (dom.is(elm, 'figure.image')) {
                return [elm.querySelector('img')];
            }
            else {
                return [elm];
            }
        };
        const isResizable = (elm) => {
            const selector = getObjectResizing(editor);
            if (!selector || editor.mode.isReadOnly()) {
                return false;
            }
            if (elm.getAttribute('data-mce-resize') === 'false') {
                return false;
            }
            if (elm === editor.getBody()) {
                return false;
            }
            if (dom.hasClass(elm, 'mce-preview-object') && isNonNullable(elm.firstElementChild)) {
                return is$2(SugarElement.fromDom(elm.firstElementChild), selector);
            }
            else {
                return is$2(SugarElement.fromDom(elm), selector);
            }
        };
        const createGhostElement = (dom, elm) => {
            if (isMedia(elm)) {
                return dom.create('img', { src: Env.transparentSrc });
            }
            else if (isTable$2(elm)) {
                const isNorth = startsWith(selectedHandle.name, 'n');
                const rowSelect = isNorth ? head : last$2;
                const tableElm = elm.cloneNode(true);
                // Get row, remove all height styles
                rowSelect(dom.select('tr', tableElm)).each((tr) => {
                    const cells = dom.select('td,th', tr);
                    dom.setStyle(tr, 'height', null);
                    each$e(cells, (cell) => dom.setStyle(cell, 'height', null));
                });
                return tableElm;
            }
            else {
                return elm.cloneNode(true);
            }
        };
        const setUcVideoSizeProp = (element, name, value) => {
            // this is needed because otherwise the ghost for `uc-video` is not correctly rendered
            element[name] = value;
            const minimumWidth = 400;
            if (element.width > minimumWidth && !(name === 'width' && value < minimumWidth)) {
                element[name] = value;
                dom.setStyle(element, name, value);
            }
            else {
                const valueConsideringMinWidth = name === 'height' ? minimumWidth * (ratio ?? 1) : minimumWidth;
                element[name] = valueConsideringMinWidth;
                dom.setStyle(element, name, valueConsideringMinWidth);
            }
        };
        const setSizeProp = (element, name, value) => {
            if (isNonNullable(value)) {
                // Resize by using style or attribute
                const targets = getResizeTargets(element);
                each$e(targets, (target) => {
                    if (isUcVideo(target)) {
                        setUcVideoSizeProp(target, name, value);
                    }
                    else {
                        if (target.style[name] || !editor.schema.isValid(target.nodeName.toLowerCase(), name)) {
                            dom.setStyle(target, name, value);
                        }
                        else {
                            dom.setAttrib(target, name, '' + value);
                        }
                    }
                });
            }
        };
        const setGhostElmSize = (ghostElm, width, height) => {
            setSizeProp(ghostElm, 'width', width);
            setSizeProp(ghostElm, 'height', height);
        };
        const resizeGhostElement = (e) => {
            let deltaX, deltaY, proportional;
            let resizeHelperX, resizeHelperY;
            // Calc new width/height
            deltaX = e.screenX - startX;
            deltaY = e.screenY - startY;
            // Calc new size
            width = deltaX * selectedHandle[2] + startW;
            height = deltaY * selectedHandle[3] + startH;
            // Never scale down lower than 5 pixels
            width = width < 5 ? 5 : width;
            height = height < 5 ? 5 : height;
            if ((isImage(selectedElm) || isMedia(selectedElm) || isUcVideo(selectedElm)) && getResizeImgProportional(editor) !== false) {
                proportional = !VK.modifierPressed(e);
            }
            else {
                proportional = VK.modifierPressed(e);
            }
            // Constrain proportions
            if (proportional) {
                if (abs(deltaX) > abs(deltaY)) {
                    height = round$1(width * ratio);
                    width = round$1(height / ratio);
                }
                else {
                    width = round$1(height / ratio);
                    height = round$1(width * ratio);
                }
            }
            // Update ghost size
            setGhostElmSize(selectedElmGhost, width, height);
            // Update resize helper position
            resizeHelperX = selectedHandle.startPos.x + deltaX;
            resizeHelperY = selectedHandle.startPos.y + deltaY;
            resizeHelperX = resizeHelperX > 0 ? resizeHelperX : 0;
            resizeHelperY = resizeHelperY > 0 ? resizeHelperY : 0;
            dom.setStyles(resizeHelper, {
                left: resizeHelperX,
                top: resizeHelperY,
                display: 'block'
            });
            resizeHelper.innerHTML = width + ' &times; ' + height;
            /* TODO: TINY-11702 dom.setStyle() has no effect because the value is NaN
              // Update ghost X position if needed
              if (selectedHandle[2] < 0 && selectedElmGhost.clientWidth <= width) {
                dom.setStyle(selectedElmGhost, 'left', selectedElmX + (startW - width));
              }
        
              // Update ghost Y position if needed
              if (selectedHandle[3] < 0 && selectedElmGhost.clientHeight <= height) {
                dom.setStyle(selectedElmGhost, 'top', selectedElmY + (startH - height));
              }
            */
            // Calculate how must overflow we got
            deltaX = rootElement.scrollWidth - startScrollWidth;
            deltaY = rootElement.scrollHeight - startScrollHeight;
            // Re-position the resize helper based on the overflow
            if (deltaX + deltaY !== 0) {
                dom.setStyles(resizeHelper, {
                    left: resizeHelperX - deltaX,
                    top: resizeHelperY - deltaY
                });
            }
            if (!resizeStarted) {
                fireObjectResizeStart(editor, selectedElm, startW, startH, 'corner-' + selectedHandle.name);
                resizeStarted = true;
            }
        };
        const endGhostResize = () => {
            const wasResizeStarted = resizeStarted;
            resizeStarted = false;
            // Set width/height properties
            if (wasResizeStarted) {
                setSizeProp(selectedElm, 'width', width);
                setSizeProp(selectedElm, 'height', height);
            }
            dom.unbind(editableDoc, 'mousemove', resizeGhostElement);
            dom.unbind(editableDoc, 'mouseup', endGhostResize);
            if (rootDocument !== editableDoc) {
                dom.unbind(rootDocument, 'mousemove', resizeGhostElement);
                dom.unbind(rootDocument, 'mouseup', endGhostResize);
            }
            // Remove ghost/helper and update resize handle positions
            dom.remove(selectedElmGhost);
            dom.remove(resizeHelper);
            dom.remove(resizeBackdrop);
            showResizeRect(selectedElm);
            if (wasResizeStarted) {
                fireObjectResized(editor, selectedElm, width, height, 'corner-' + selectedHandle.name);
                dom.setAttrib(selectedElm, 'style', dom.getAttrib(selectedElm, 'style'));
            }
            editor.nodeChanged();
        };
        const showResizeRect = (targetElm) => {
            unbindResizeHandleEvents();
            // Get position and size of target
            const position = dom.getPos(targetElm, rootElement);
            const selectedElmX = position.x;
            const selectedElmY = position.y;
            const rect = targetElm.getBoundingClientRect(); // Fix for Gecko offsetHeight for table with caption
            const targetWidth = rect.width || (rect.right - rect.left);
            const targetHeight = rect.height || (rect.bottom - rect.top);
            // Reset width/height if user selects a new image/table
            if (selectedElm !== targetElm) {
                hideResizeRect();
                selectedElm = targetElm;
                width = height = 0;
            }
            // Makes it possible to disable resizing
            const e = editor.dispatch('ObjectSelected', { target: targetElm });
            if (isResizable(targetElm) && !e.isDefaultPrevented()) {
                each$d(resizeHandles, (handle, name) => {
                    const startDrag = (e) => {
                        // Note: We're guaranteed to have at least one target here
                        const target = getResizeTargets(selectedElm)[0];
                        startX = e.screenX;
                        startY = e.screenY;
                        startW = target.clientWidth;
                        startH = target.clientHeight;
                        ratio = startH / startW;
                        selectedHandle = handle;
                        selectedHandle.name = name;
                        selectedHandle.startPos = {
                            x: targetWidth * handle[0] + selectedElmX,
                            y: targetHeight * handle[1] + selectedElmY
                        };
                        startScrollWidth = rootElement.scrollWidth;
                        startScrollHeight = rootElement.scrollHeight;
                        resizeBackdrop = dom.add(rootElement, 'div', {
                            'class': 'mce-resize-backdrop',
                            'data-mce-bogus': 'all'
                        });
                        dom.setStyles(resizeBackdrop, {
                            position: 'fixed',
                            left: '0',
                            top: '0',
                            width: '100%',
                            height: '100%'
                        });
                        selectedElmGhost = createGhostElement(dom, selectedElm);
                        dom.addClass(selectedElmGhost, 'mce-clonedresizable');
                        dom.setAttrib(selectedElmGhost, 'data-mce-bogus', 'all');
                        selectedElmGhost.contentEditable = 'false'; // Hides IE move layer cursor
                        dom.setStyles(selectedElmGhost, {
                            left: selectedElmX,
                            top: selectedElmY,
                            margin: 0
                        });
                        // Set initial ghost size
                        setGhostElmSize(selectedElmGhost, targetWidth, targetHeight);
                        selectedElmGhost.removeAttribute(elementSelectionAttr);
                        rootElement.appendChild(selectedElmGhost);
                        dom.bind(editableDoc, 'mousemove', resizeGhostElement);
                        dom.bind(editableDoc, 'mouseup', endGhostResize);
                        if (rootDocument !== editableDoc) {
                            dom.bind(rootDocument, 'mousemove', resizeGhostElement);
                            dom.bind(rootDocument, 'mouseup', endGhostResize);
                        }
                        resizeHelper = dom.add(rootElement, 'div', {
                            'class': 'mce-resize-helper',
                            'data-mce-bogus': 'all'
                        }, startW + ' &times; ' + startH);
                    };
                    // Get existing or render resize handle
                    let handleElm = dom.get('mceResizeHandle' + name);
                    if (handleElm) {
                        dom.remove(handleElm);
                    }
                    handleElm = dom.add(rootElement, 'div', {
                        'id': 'mceResizeHandle' + name,
                        'data-mce-bogus': 'all',
                        'class': 'mce-resizehandle',
                        'unselectable': true,
                        'style': 'cursor:' + name + '-resize; margin:0; padding:0'
                    });
                    dom.bind(handleElm, 'mousedown', (e) => {
                        e.stopImmediatePropagation();
                        e.preventDefault();
                        startDrag(e);
                    });
                    handle.elm = handleElm;
                    // Position element
                    dom.setStyles(handleElm, {
                        left: (targetWidth * handle[0] + selectedElmX) - (handleElm.offsetWidth / 2),
                        top: (targetHeight * handle[1] + selectedElmY) - (handleElm.offsetHeight / 2)
                    });
                });
            }
            else {
                hideResizeRect(false);
            }
        };
        const throttledShowResizeRect = first$1(showResizeRect, 0);
        const hideResizeRect = (removeSelected = true) => {
            throttledShowResizeRect.cancel();
            unbindResizeHandleEvents();
            if (selectedElm && removeSelected) {
                selectedElm.removeAttribute(elementSelectionAttr);
            }
            each$d(resizeHandles, (value, name) => {
                const handleElm = dom.get('mceResizeHandle' + name);
                if (handleElm) {
                    dom.unbind(handleElm);
                    dom.remove(handleElm);
                }
            });
        };
        const isChildOrEqual = (node, parent) => dom.isChildOf(node, parent);
        const updateResizeRect = (e) => {
            // Ignore all events while resizing, if the editor instance is composing or the editor was removed
            if (resizeStarted || editor.removed || editor.composing) {
                return;
            }
            const targetElm = e.type === 'mousedown' ? e.target : selection.getNode();
            const controlElm = closest$4(SugarElement.fromDom(targetElm), controlElmSelector)
                .map((e) => e.dom)
                .filter((e) => dom.isEditable(e.parentElement) || (e.nodeName === 'IMG' && dom.isEditable(e)))
                .getOrUndefined();
            // Store the original data-mce-selected value or fallback to '1' if not set
            const selectedValue = isNonNullable(controlElm) ? dom.getAttrib(controlElm, elementSelectionAttr, '1') : '1';
            // Remove data-mce-selected from all elements since they might have been copied using Ctrl+c/v
            each$e(dom.select(`img[${elementSelectionAttr}],hr[${elementSelectionAttr}]`), (img) => {
                img.removeAttribute(elementSelectionAttr);
            });
            if (isNonNullable(controlElm) && isChildOrEqual(controlElm, rootElement) && hasEditorOrUiFocus(editor)) {
                disableGeckoResize();
                const startElm = selection.getStart(true);
                if (isChildOrEqual(startElm, controlElm) && isChildOrEqual(selection.getEnd(true), controlElm)) {
                    // Note: We must ensure the selected attribute is added first before showing the rect so that we don't get any selection flickering
                    dom.setAttrib(controlElm, elementSelectionAttr, selectedValue);
                    throttledShowResizeRect.throttle(controlElm);
                    return;
                }
            }
            hideResizeRect();
        };
        const unbindResizeHandleEvents = () => {
            each$d(resizeHandles, (handle) => {
                if (handle.elm) {
                    dom.unbind(handle.elm);
                    // eslint-disable-next-line @typescript-eslint/no-array-delete
                    delete handle.elm;
                }
            });
        };
        const disableGeckoResize = () => {
            try {
                // Disable object resizing on Gecko
                editor.getDoc().execCommand('enableObjectResizing', false, 'false');
            }
            catch {
                // Ignore
            }
        };
        editor.on('init', () => {
            disableGeckoResize();
            editor.on('NodeChange ResizeEditor ResizeWindow ResizeContent drop', updateResizeRect);
            // Update resize rect while typing in a table
            editor.on('keyup compositionend', (e) => {
                // Don't update the resize rect while composing since it blows away the IME see: #2710
                if (selectedElm && selectedElm.nodeName === 'TABLE') {
                    updateResizeRect(e);
                }
            });
            editor.on('hide blur', hideResizeRect);
            editor.on('contextmenu longpress', contextMenuSelectImage, true);
            // Hide rect on focusout since it would float on top of windows otherwise
            // editor.on('focusout', hideResizeRect);
        });
        editor.on('remove', unbindResizeHandleEvents);
        const destroy = () => {
            throttledShowResizeRect.cancel();
            selectedElm = selectedElmGhost = resizeBackdrop = null;
        };
        return {
            isResizable,
            showResizeRect,
            hideResizeRect,
            updateResizeRect,
            destroy
        };
    };

    const fromPoint = (clientX, clientY, doc) => {
        const win = defaultView(SugarElement.fromDom(doc));
        return getAtPoint(win.dom, clientX, clientY).map((simRange) => {
            const rng = doc.createRange();
            rng.setStart(simRange.start.dom, simRange.soffset);
            rng.setEnd(simRange.finish.dom, simRange.foffset);
            return rng;
        }).getOrUndefined();
    };

    const isEq$4 = (rng1, rng2) => {
        return isNonNullable(rng1) && isNonNullable(rng2) &&
            (rng1.startContainer === rng2.startContainer && rng1.startOffset === rng2.startOffset) &&
            (rng1.endContainer === rng2.endContainer && rng1.endOffset === rng2.endOffset);
    };

    const findParent = (node, rootNode, predicate) => {
        let currentNode = node;
        while (currentNode && currentNode !== rootNode) {
            if (predicate(currentNode)) {
                return currentNode;
            }
            currentNode = currentNode.parentNode;
        }
        return null;
    };
    const hasParent$1 = (node, rootNode, predicate) => findParent(node, rootNode, predicate) !== null;
    const hasParentWithName = (node, rootNode, name) => hasParent$1(node, rootNode, (node) => node.nodeName === name);
    const isCeFalseCaretContainer = (node, rootNode) => isCaretContainer$2(node) && !hasParent$1(node, rootNode, isCaretNode);
    const hasBrBeforeAfter = (dom, node, left) => {
        const parentNode = node.parentNode;
        if (parentNode) {
            const walker = new DomTreeWalker(node, dom.getParent(parentNode, dom.isBlock) || dom.getRoot());
            let currentNode;
            while ((currentNode = walker[left ? 'prev' : 'next']())) {
                if (isBr$7(currentNode)) {
                    return true;
                }
            }
        }
        return false;
    };
    const isPrevNode = (node, name) => node.previousSibling?.nodeName === name;
    const hasContentEditableFalseParent = (root, node) => {
        let currentNode = node;
        while (currentNode && currentNode !== root) {
            if (isContentEditableFalse$a(currentNode)) {
                return true;
            }
            currentNode = currentNode.parentNode;
        }
        return false;
    };
    // Walks the dom left/right to find a suitable text node to move the endpoint into
    // It will only walk within the current parent block or body and will stop if it hits a block or a BR/IMG
    const findTextNodeRelative = (dom, isAfterNode, collapsed, left, startNode) => {
        const body = dom.getRoot();
        const nonEmptyElementsMap = dom.schema.getNonEmptyElements();
        const parentNode = startNode.parentNode;
        let lastInlineElement;
        let node;
        if (!parentNode) {
            return Optional.none();
        }
        const parentBlockContainer = dom.getParent(parentNode, dom.isBlock) || body;
        // Lean left before the BR element if it's the only BR within a block element. Gecko bug: #6680
        // This: <p><br>|</p> becomes <p>|<br></p>
        if (left && isBr$7(startNode) && isAfterNode && dom.isEmpty(parentBlockContainer)) {
            return Optional.some(CaretPosition(parentNode, dom.nodeIndex(startNode)));
        }
        // Walk left until we hit a text node we can move to or a block/br/img
        const walker = new DomTreeWalker(startNode, parentBlockContainer);
        while ((node = walker[left ? 'prev' : 'next']())) {
            // Break if we hit a non content editable node
            if (dom.getContentEditableParent(node) === 'false' || isCeFalseCaretContainer(node, body)) {
                return Optional.none();
            }
            // Found text node that has a length
            if (isText$b(node) && node.data.length > 0) {
                if (!hasParentWithName(node, body, 'A')) {
                    return Optional.some(CaretPosition(node, left ? node.data.length : 0));
                }
                return Optional.none();
            }
            // Break if we find a block or a BR/IMG/INPUT etc
            if (dom.isBlock(node) || nonEmptyElementsMap[node.nodeName.toLowerCase()]) {
                return Optional.none();
            }
            lastInlineElement = node;
        }
        if (isComment(lastInlineElement)) {
            return Optional.none();
        }
        // Only fetch the last inline element when in caret mode for now
        if (collapsed && lastInlineElement) {
            return Optional.some(CaretPosition(lastInlineElement, 0));
        }
        return Optional.none();
    };
    const normalizeEndPoint = (dom, collapsed, start, rng) => {
        const body = dom.getRoot();
        let node;
        let normalized = false;
        let container = start ? rng.startContainer : rng.endContainer;
        let offset = start ? rng.startOffset : rng.endOffset;
        const isAfterNode = isElement$7(container) && offset === container.childNodes.length;
        const nonEmptyElementsMap = dom.schema.getNonEmptyElements();
        let directionLeft = start;
        if (isCaretContainer$2(container)) {
            return Optional.none();
        }
        if (isElement$7(container) && offset > container.childNodes.length - 1) {
            directionLeft = false;
        }
        // If the container is a document move it to the body element
        if (isDocument$1(container)) {
            container = body;
            offset = 0;
        }
        // If the container is body try move it into the closest text node or position
        if (container === body) {
            // If start is before/after a image, table etc
            if (directionLeft) {
                node = container.childNodes[offset > 0 ? offset - 1 : 0];
                if (node) {
                    if (isCaretContainer$2(node)) {
                        return Optional.none();
                    }
                    if (nonEmptyElementsMap[node.nodeName] || isTable$2(node)) {
                        return Optional.none();
                    }
                }
            }
            // Resolve the index
            if (container.hasChildNodes()) {
                offset = Math.min(!directionLeft && offset > 0 ? offset - 1 : offset, container.childNodes.length - 1);
                container = container.childNodes[offset];
                offset = isText$b(container) && isAfterNode ? container.data.length : 0;
                // Don't normalize non collapsed selections like <p>[a</p><table></table>]
                if (!collapsed && container === body.lastChild && isTable$2(container)) {
                    return Optional.none();
                }
                if (hasContentEditableFalseParent(body, container) || isCaretContainer$2(container)) {
                    return Optional.none();
                }
                if (isDetails(container)) {
                    return Optional.none();
                }
                // Don't walk into elements that doesn't have any child nodes like a IMG
                if (container.hasChildNodes() && !isTable$2(container)) {
                    // Walk the DOM to find a text node to place the caret at or a BR
                    node = container;
                    const walker = new DomTreeWalker(container, body);
                    do {
                        if (isContentEditableFalse$a(node) || isCaretContainer$2(node)) {
                            normalized = false;
                            break;
                        }
                        // Found a text node use that position
                        if (isText$b(node) && node.data.length > 0) {
                            offset = directionLeft ? 0 : node.data.length;
                            container = node;
                            normalized = true;
                            break;
                        }
                        // Found a BR/IMG/PRE element that we can place the caret before
                        if (nonEmptyElementsMap[node.nodeName.toLowerCase()] && !isTableCellOrCaption(node)) {
                            offset = dom.nodeIndex(node);
                            container = node.parentNode;
                            // Put caret after image and pre tag when moving the end point
                            if (!directionLeft) {
                                offset++;
                            }
                            normalized = true;
                            break;
                        }
                    } while ((node = (directionLeft ? walker.next() : walker.prev())));
                }
            }
        }
        // Lean the caret to the left if possible
        if (collapsed) {
            // So this: <b>x</b><i>|x</i>
            // Becomes: <b>x|</b><i>x</i>
            // Seems that only gecko has issues with this
            if (isText$b(container) && offset === 0) {
                findTextNodeRelative(dom, isAfterNode, collapsed, true, container).each((pos) => {
                    container = pos.container();
                    offset = pos.offset();
                    normalized = true;
                });
            }
            // Lean left into empty inline elements when the caret is before a BR
            // So this: <i><b></b><i>|<br></i>
            // Becomes: <i><b>|</b><i><br></i>
            // Seems that only gecko has issues with this.
            // Special edge case for <p><a>x</a>|<br></p> since we don't want <p><a>x|</a><br></p>
            if (isElement$7(container)) {
                node = container.childNodes[offset];
                // Offset is after the containers last child
                // then use the previous child for normalization
                if (!node) {
                    node = container.childNodes[offset - 1];
                }
                if (node && isBr$7(node) && !isPrevNode(node, 'A') &&
                    !hasBrBeforeAfter(dom, node, false) && !hasBrBeforeAfter(dom, node, true)) {
                    findTextNodeRelative(dom, isAfterNode, collapsed, true, node).each((pos) => {
                        container = pos.container();
                        offset = pos.offset();
                        normalized = true;
                    });
                }
            }
        }
        // Lean the start of the selection right if possible
        // So this: x[<b>x]</b>
        // Becomes: x<b>[x]</b>
        if (directionLeft && !collapsed && isText$b(container) && offset === container.data.length) {
            findTextNodeRelative(dom, isAfterNode, collapsed, false, container).each((pos) => {
                container = pos.container();
                offset = pos.offset();
                normalized = true;
            });
        }
        return normalized && container ? Optional.some(CaretPosition(container, offset)) : Optional.none();
    };
    const normalize$2 = (dom, rng) => {
        const collapsed = rng.collapsed, normRng = rng.cloneRange();
        const startPos = CaretPosition.fromRangeStart(rng);
        normalizeEndPoint(dom, collapsed, true, normRng).each((pos) => {
            // #TINY-1595: Do not move the caret to previous line
            if (!collapsed || !CaretPosition.isAbove(startPos, pos)) {
                normRng.setStart(pos.container(), pos.offset());
            }
        });
        if (!collapsed) {
            normalizeEndPoint(dom, collapsed, false, normRng).each((pos) => {
                normRng.setEnd(pos.container(), pos.offset());
            });
        }
        // If it was collapsed then make sure it still is
        if (collapsed) {
            normRng.collapse(true);
        }
        return isEq$4(rng, normRng) ? Optional.none() : Optional.some(normRng);
    };

    const splitText = (node, offset) => {
        return node.splitText(offset);
    };
    const split = (rng) => {
        let startContainer = rng.startContainer, startOffset = rng.startOffset, endContainer = rng.endContainer, endOffset = rng.endOffset;
        // Handle single text node
        if (startContainer === endContainer && isText$b(startContainer)) {
            if (startOffset > 0 && startOffset < startContainer.data.length) {
                endContainer = splitText(startContainer, startOffset);
                startContainer = endContainer.previousSibling;
                if (endOffset > startOffset) {
                    endOffset = endOffset - startOffset;
                    const newContainer = splitText(endContainer, endOffset).previousSibling;
                    startContainer = endContainer = newContainer;
                    endOffset = newContainer.data.length;
                    startOffset = 0;
                }
                else {
                    endOffset = 0;
                }
            }
        }
        else {
            // Split startContainer text node if needed
            if (isText$b(startContainer) && startOffset > 0 && startOffset < startContainer.data.length) {
                startContainer = splitText(startContainer, startOffset);
                startOffset = 0;
            }
            // Split endContainer text node if needed
            if (isText$b(endContainer) && endOffset > 0 && endOffset < endContainer.data.length) {
                const newContainer = splitText(endContainer, endOffset).previousSibling;
                endContainer = newContainer;
                endOffset = newContainer.data.length;
            }
        }
        return {
            startContainer,
            startOffset,
            endContainer,
            endOffset
        };
    };

    /**
     * This class contains a few utility methods for ranges.
     *
     * @class tinymce.dom.RangeUtils
     */
    const RangeUtils = (dom) => {
        /**
         * Walks the specified range like object and executes the callback for each sibling collection it finds.
         *
         * @private
         * @method walk
         * @param {RangeObject} rng Range like object.
         * @param {Function} callback Callback function to execute for each sibling collection.
         */
        const walk = (rng, callback) => {
            return walk$3(dom, rng, callback);
        };
        /**
         * Splits the specified range at it's start/end points.
         *
         * @private
         * @param {RangeObject} rng Range to split.
         * @return {RangeObject} Range position object.
         */
        const split$1 = split;
        /**
         * Normalizes the specified range by finding the closest best suitable caret location.
         *
         * @private
         * @param {Range} rng Range to normalize.
         * @return {Boolean} True or false if the specified range was normalized or not.
         */
        const normalize = (rng) => {
            return normalize$2(dom, rng).fold(never, (normalizedRng) => {
                rng.setStart(normalizedRng.startContainer, normalizedRng.startOffset);
                rng.setEnd(normalizedRng.endContainer, normalizedRng.endOffset);
                return true;
            });
        };
        /**
         * Returns a range expanded around the entire word the provided selection was collapsed within.
         *
         * @method expand
         * @param {Range} rng The initial range to work from.
         * @param {Object} options Optional options provided to the expansion. Defaults to { type: 'word' }
         * @return {Range} Returns the expanded range.
         */
        const expand = (rng, options = { type: 'word' }) => {
            if (options.type === 'word') {
                const rangeLike = expandRng(dom, rng, [{ inline: 'span' }], { includeTrailingSpace: false, expandToBlock: false });
                const newRange = dom.createRng();
                newRange.setStart(rangeLike.startContainer, rangeLike.startOffset);
                newRange.setEnd(rangeLike.endContainer, rangeLike.endOffset);
                return newRange;
            }
            return rng;
        };
        return {
            walk,
            split: split$1,
            expand,
            normalize
        };
    };
    /**
     * Compares two ranges and checks if they are equal.
     *
     * @static
     * @method compareRanges
     * @param {RangeObject} rng1 First range to compare.
     * @param {RangeObject} rng2 First range to compare.
     * @return {Boolean} True or false if the ranges are equal.
     */
    RangeUtils.compareRanges = isEq$4;
    /**
     * Gets the caret range for the given x/y location.
     *
     * @static
     * @method getCaretRangeFromPoint
     * @param {Number} clientX X coordinate for range
     * @param {Number} clientY Y coordinate for range
     * @param {Document} doc Document that the x and y coordinates are relative to
     * @returns {Range} Caret range
     */
    RangeUtils.getCaretRangeFromPoint = fromPoint;
    RangeUtils.getSelectedNode = getSelectedNode;
    RangeUtils.getNode = getNode$1;

    const walkUp = (navigation, doc) => {
        const frame = navigation.view(doc);
        return frame.fold(constant([]), (f) => {
            const parent = navigation.owner(f);
            const rest = walkUp(navigation, parent);
            return [f].concat(rest);
        });
    };
    const pathTo = (element, navigation) => {
        const d = navigation.owner(element);
        return walkUp(navigation, d);
    };

    const view = (doc) => {
        // Only walk up to the document this script is defined in.
        // This prevents walking up to the parent window when the editor is in an iframe.
        const element = doc.dom === document ? Optional.none() : Optional.from(doc.dom.defaultView?.frameElement);
        return element.map(SugarElement.fromDom);
    };
    const owner = (element) => documentOrOwner(element);

    var Navigation = /*#__PURE__*/Object.freeze({
        __proto__: null,
        view: view,
        owner: owner
    });

    const find = (element) => {
        const doc = getDocument();
        const scroll = get$5(doc);
        const frames = pathTo(element, Navigation);
        const offset = viewport(element);
        const r = foldr(frames, (b, a) => {
            const loc = viewport(a);
            return {
                left: b.left + loc.left,
                top: b.top + loc.top
            };
        }, { left: 0, top: 0 });
        return SugarPosition(r.left + offset.left + scroll.left, r.top + offset.top + scroll.top);
    };

    const excludeFromDescend = (element) => name(element) === 'textarea';
    const fireScrollIntoViewEvent = (editor, data) => {
        const scrollEvent = editor.dispatch('ScrollIntoView', data);
        return scrollEvent.isDefaultPrevented();
    };
    const fireAfterScrollIntoViewEvent = (editor, data) => {
        editor.dispatch('AfterScrollIntoView', data);
    };
    const descend = (element, offset) => {
        const children = children$1(element);
        if (children.length === 0 || excludeFromDescend(element)) {
            return { element, offset };
        }
        else if (offset < children.length && !excludeFromDescend(children[offset])) {
            return { element: children[offset], offset: 0 };
        }
        else {
            const last = children[children.length - 1];
            if (excludeFromDescend(last)) {
                return { element, offset };
            }
            else {
                if (name(last) === 'img') {
                    return { element: last, offset: 1 };
                }
                else if (isText$c(last)) {
                    return { element: last, offset: get$4(last).length };
                }
                else {
                    return { element: last, offset: children$1(last).length };
                }
            }
        }
    };
    const markerInfo = (element, cleanupFun) => {
        const pos = absolute(element);
        const height = get$6(element);
        return {
            element,
            bottom: pos.top + height,
            height,
            pos,
            cleanup: cleanupFun
        };
    };
    const createMarker$1 = (element, offset) => {
        const startPoint = descend(element, offset);
        const span = SugarElement.fromHtml('<span data-mce-bogus="all" style="display: inline-block;">' + ZWSP$1 + '</span>');
        before$4(startPoint.element, span);
        return markerInfo(span, () => remove$8(span));
    };
    const elementMarker = (element) => markerInfo(SugarElement.fromDom(element), noop);
    const withMarker = (editor, f, rng, alignToTop) => {
        preserveWith(editor, (_s, _e) => applyWithMarker(editor, f, rng, alignToTop), rng);
    };
    const withScrollEvents = (editor, doc, f, marker, alignToTop) => {
        const data = { elm: marker.element.dom, alignToTop };
        if (fireScrollIntoViewEvent(editor, data)) {
            return;
        }
        const scrollTop = get$5(doc).top;
        f(editor, doc, scrollTop, marker, alignToTop);
        fireAfterScrollIntoViewEvent(editor, data);
    };
    const applyWithMarker = (editor, f, rng, alignToTop) => {
        const body = SugarElement.fromDom(editor.getBody());
        const doc = SugarElement.fromDom(editor.getDoc());
        reflow(body);
        const marker = createMarker$1(SugarElement.fromDom(rng.startContainer), rng.startOffset);
        withScrollEvents(editor, doc, f, marker, alignToTop);
        marker.cleanup();
    };
    const withElement = (editor, element, f, alignToTop) => {
        const doc = SugarElement.fromDom(editor.getDoc());
        withScrollEvents(editor, doc, f, elementMarker(element), alignToTop);
    };
    const preserveWith = (editor, f, rng) => {
        const startElement = rng.startContainer;
        const startOffset = rng.startOffset;
        const endElement = rng.endContainer;
        const endOffset = rng.endOffset;
        f(SugarElement.fromDom(startElement), SugarElement.fromDom(endElement));
        const newRng = editor.dom.createRng();
        newRng.setStart(startElement, startOffset);
        newRng.setEnd(endElement, endOffset);
        editor.selection.setRng(rng);
    };
    const scrollToMarker = (editor, marker, viewHeight, alignToTop, doc) => {
        const pos = marker.pos;
        // with default font size 16px font and 1.3 line height (~21px per line),
        // adding roughly 50% extra space gives about 30px of breathing room ensuring comfortable spacing.
        const scrollMargin = 30;
        if (alignToTop) {
            // When scrolling to top, add margin to the top position
            to(pos.left, Math.max(0, pos.top - scrollMargin), doc);
        }
        else {
            // The position we want to scroll to is the...
            // (absolute position of the marker, minus the view height) plus (the height of the marker)
            // When scrolling to bottom, add margin to ensure content isn't at the very bottom
            const y = (pos.top - viewHeight) + marker.height + scrollMargin;
            to(-editor.getBody().getBoundingClientRect().left, y, doc);
        }
    };
    const intoWindowIfNeeded = (editor, doc, scrollTop, viewHeight, marker, alignToTop) => {
        const viewportBottom = viewHeight + scrollTop;
        const markerTop = marker.pos.top;
        const markerBottom = marker.bottom;
        const largerThanViewport = markerBottom - markerTop >= viewHeight;
        // above the screen, scroll to top by default
        if (markerTop < scrollTop) {
            scrollToMarker(editor, marker, viewHeight, alignToTop !== false, doc);
            // completely below the screen. Default scroll to the top if element height is larger
            // than the viewport, otherwise default to scrolling to the bottom
        }
        else if (markerTop > viewportBottom) {
            const align = largerThanViewport ? alignToTop !== false : alignToTop === true;
            scrollToMarker(editor, marker, viewHeight, align, doc);
            // partially below the bottom, only scroll if element height is less than viewport
        }
        else if (markerBottom > viewportBottom && !largerThanViewport) {
            scrollToMarker(editor, marker, viewHeight, alignToTop === true, doc);
        }
    };
    const intoWindow = (editor, doc, scrollTop, marker, alignToTop) => {
        const viewHeight = defaultView(doc).dom.innerHeight;
        intoWindowIfNeeded(editor, doc, scrollTop, viewHeight, marker, alignToTop);
    };
    const intoFrame = (editor, doc, scrollTop, marker, alignToTop) => {
        const frameViewHeight = defaultView(doc).dom.innerHeight; // height of iframe container
        // If the position is outside the iframe viewport, scroll to it
        intoWindowIfNeeded(editor, doc, scrollTop, frameViewHeight, marker, alignToTop);
        // If the new position is outside the window viewport, scroll to it
        const op = find(marker.element);
        const viewportBounds = getBounds(window);
        if (op.top < viewportBounds.y) {
            intoView(marker.element, alignToTop !== false);
        }
        else if (op.top > viewportBounds.bottom) {
            intoView(marker.element, alignToTop === true);
        }
    };
    const rangeIntoWindow = (editor, rng, alignToTop) => withMarker(editor, intoWindow, rng, alignToTop);
    const elementIntoWindow = (editor, element, alignToTop) => withElement(editor, element, intoWindow, alignToTop);
    const rangeIntoFrame = (editor, rng, alignToTop) => withMarker(editor, intoFrame, rng, alignToTop);
    const elementIntoFrame = (editor, element, alignToTop) => withElement(editor, element, intoFrame, alignToTop);
    const scrollElementIntoView = (editor, element, alignToTop) => {
        const scroller = editor.inline ? elementIntoWindow : elementIntoFrame;
        scroller(editor, element, alignToTop);
    };
    // This method is made to deal with the user pressing enter, it is not useful
    // if we want for example scroll in content after a paste event.
    const scrollRangeIntoView = (editor, rng, alignToTop) => {
        const scroller = editor.inline ? rangeIntoWindow : rangeIntoFrame;
        scroller(editor, rng, alignToTop);
    };

    const isEditableRange = (dom, rng) => {
        if (rng.collapsed) {
            return dom.isEditable(rng.startContainer);
        }
        else {
            return dom.isEditable(rng.startContainer) && dom.isEditable(rng.endContainer);
        }
    };

    const getEndpointElement = (root, rng, start, real, resolve) => {
        const container = start ? rng.startContainer : rng.endContainer;
        const offset = start ? rng.startOffset : rng.endOffset;
        return Optional.from(container)
            .map(SugarElement.fromDom)
            .map((elm) => !real || !rng.collapsed ? child$1(elm, resolve(elm, offset)).getOr(elm) : elm)
            .bind((elm) => isElement$8(elm) ? Optional.some(elm) : parent(elm).filter(isElement$8))
            .map((elm) => elm.dom)
            .getOr(root);
    };
    const getStart = (root, rng, real = false) => getEndpointElement(root, rng, true, real, (elm, offset) => Math.min(childNodesCount(elm), offset));
    const getEnd = (root, rng, real = false) => getEndpointElement(root, rng, false, real, (elm, offset) => offset > 0 ? offset - 1 : offset);
    const skipEmptyTextNodes = (node, forwards) => {
        const orig = node;
        while (node && isText$b(node) && node.length === 0) {
            node = forwards ? node.nextSibling : node.previousSibling;
        }
        return node || orig;
    };
    const getNode = (root, rng) => {
        // Range maybe lost after the editor is made visible again
        if (!rng) {
            return root;
        }
        let startContainer = rng.startContainer;
        let endContainer = rng.endContainer;
        const startOffset = rng.startOffset;
        const endOffset = rng.endOffset;
        let node = rng.commonAncestorContainer;
        // Handle selection a image or other control like element such as anchors
        if (!rng.collapsed) {
            if (startContainer === endContainer) {
                if (endOffset - startOffset < 2) {
                    if (startContainer.hasChildNodes()) {
                        node = startContainer.childNodes[startOffset];
                    }
                }
            }
            // If the anchor node is a element instead of a text node then return this element
            // if (tinymce.isWebKit && sel.anchorNode && sel.anchorNode.nodeType == 1)
            // return sel.anchorNode.childNodes[sel.anchorOffset];
            // Handle cases where the selection is immediately wrapped around a node and return that node instead of it's parent.
            // This happens when you double click an underlined word in FireFox.
            if (isText$b(startContainer) && isText$b(endContainer)) {
                if (startContainer.length === startOffset) {
                    startContainer = skipEmptyTextNodes(startContainer.nextSibling, true);
                }
                else {
                    startContainer = startContainer.parentNode;
                }
                if (endOffset === 0) {
                    endContainer = skipEmptyTextNodes(endContainer.previousSibling, false);
                }
                else {
                    endContainer = endContainer.parentNode;
                }
                if (startContainer && startContainer === endContainer) {
                    node = startContainer;
                }
            }
        }
        const elm = isText$b(node) ? node.parentNode : node;
        return isHTMLElement(elm) ? elm : root;
    };
    const getSelectedBlocks = (dom, rng, startElm, endElm) => {
        const selectedBlocks = [];
        const root = dom.getRoot();
        const start = dom.getParent(startElm || getStart(root, rng, rng.collapsed), dom.isBlock);
        const end = dom.getParent(endElm || getEnd(root, rng, rng.collapsed), dom.isBlock);
        if (start && start !== root) {
            selectedBlocks.push(start);
        }
        if (start && end && start !== end) {
            let node;
            const walker = new DomTreeWalker(start, root);
            while ((node = walker.next()) && node !== end) {
                if (dom.isBlock(node)) {
                    selectedBlocks.push(node);
                }
            }
        }
        if (end && start !== end && end !== root) {
            selectedBlocks.push(end);
        }
        return selectedBlocks;
    };
    const select = (dom, node, content) => Optional.from(node).bind((node) => Optional.from(node.parentNode).map((parent) => {
        const idx = dom.nodeIndex(node);
        const rng = dom.createRng();
        rng.setStart(parent, idx);
        rng.setEnd(parent, idx + 1);
        // Find first/last text node or BR element
        if (content) {
            moveEndPoint(dom, rng, node, true);
            moveEndPoint(dom, rng, node, false);
        }
        return rng;
    }));

    const processRanges = (editor, ranges) => map$3(ranges, (range) => {
        const evt = editor.dispatch('GetSelectionRange', { range });
        return evt.range !== range ? evt.range : range;
    });

    const typeLookup = {
        '#text': 3,
        '#comment': 8,
        '#cdata': 4,
        '#pi': 7,
        '#doctype': 10,
        '#document-fragment': 11
    };
    // Walks the tree left/right
    const walk$2 = (node, root, prev) => {
        const startName = prev ? 'lastChild' : 'firstChild';
        const siblingName = prev ? 'prev' : 'next';
        // Walk into nodes if it has a start
        if (node[startName]) {
            return node[startName];
        }
        // Return the sibling if it has one
        if (node !== root) {
            let sibling = node[siblingName];
            if (sibling) {
                return sibling;
            }
            // Walk up the parents to look for siblings
            for (let parent = node.parent; parent && parent !== root; parent = parent.parent) {
                sibling = parent[siblingName];
                if (sibling) {
                    return sibling;
                }
            }
        }
        return undefined;
    };
    const isEmptyTextNode = (node) => {
        const text = node.value ?? '';
        // Non whitespace content
        if (!isWhitespaceText(text)) {
            return false;
        }
        // Parent is not a span and only spaces or is a span but has styles
        const parentNode = node.parent;
        if (parentNode && (parentNode.name !== 'span' || parentNode.attr('style')) && /^[ ]+$/.test(text)) {
            return false;
        }
        return true;
    };
    // Check if node contains data-bookmark attribute, name attribute, id attribute or is a named anchor
    const isNonEmptyElement = (node) => {
        const isNamedAnchor = node.name === 'a' && !node.attr('href') && node.attr('id');
        return (node.attr('name') || (node.attr('id') && !node.firstChild) || node.attr('data-mce-bookmark') || isNamedAnchor);
    };
    /**
     * This class is a minimalistic implementation of a DOM like node used by the DomParser class.
     *
     * @class tinymce.html.Node
     * @version 3.4
     * @example
     * const node = new tinymce.html.Node('strong', 1);
     * someRoot.append(node);
     */
    class AstNode {
        /**
         * Creates a node of a specific type.
         *
         * @static
         * @method create
         * @param {String} name Name of the node type to create for example "b" or "#text".
         * @param {Object} attrs Name/value collection of attributes that will be applied to elements.
         */
        static create(name, attrs) {
            // Create node
            const node = new AstNode(name, typeLookup[name] || 1);
            // Add attributes if needed
            if (attrs) {
                each$d(attrs, (value, attrName) => {
                    node.attr(attrName, value);
                });
            }
            return node;
        }
        name;
        type;
        attributes;
        value;
        parent;
        firstChild;
        lastChild;
        next;
        prev;
        raw;
        /**
         * Constructs a new Node instance.
         *
         * @constructor
         * @method Node
         * @param {String} name Name of the node type.
         * @param {Number} type Numeric type representing the node.
         */
        constructor(name, type) {
            this.name = name;
            this.type = type;
            if (type === 1) {
                this.attributes = [];
                this.attributes.map = {}; // Should be considered internal
            }
        }
        /**
         * Replaces the current node with the specified one.
         *
         * @method replace
         * @param {tinymce.html.Node} node Node to replace the current node with.
         * @return {tinymce.html.Node} The old node that got replaced.
         * @example
         * someNode.replace(someNewNode);
         */
        replace(node) {
            const self = this;
            if (node.parent) {
                node.remove();
            }
            self.insert(node, self);
            self.remove();
            return self;
        }
        attr(name, value) {
            const self = this;
            if (!isString(name)) {
                if (isNonNullable(name)) {
                    each$d(name, (value, key) => {
                        self.attr(key, value);
                    });
                }
                return self;
            }
            const attrs = self.attributes;
            if (attrs) {
                if (value !== undefined) {
                    // Remove attribute
                    if (value === null) {
                        if (name in attrs.map) {
                            delete attrs.map[name];
                            let i = attrs.length;
                            while (i--) {
                                if (attrs[i].name === name) {
                                    attrs.splice(i, 1);
                                    return self;
                                }
                            }
                        }
                        return self;
                    }
                    // Set attribute
                    if (name in attrs.map) {
                        // Set attribute
                        let i = attrs.length;
                        while (i--) {
                            if (attrs[i].name === name) {
                                attrs[i].value = value;
                                break;
                            }
                        }
                    }
                    else {
                        attrs.push({ name, value });
                    }
                    attrs.map[name] = value;
                    return self;
                }
                return attrs.map[name];
            }
            return undefined;
        }
        /**
         * Does a shallow clones the node into a new node. It will also exclude id attributes since
         * there should only be one id per document.
         *
         * @method clone
         * @return {tinymce.html.Node} New copy of the original node.
         * @example
         * const clonedNode = node.clone();
         */
        clone() {
            const self = this;
            const clone = new AstNode(self.name, self.type);
            const selfAttrs = self.attributes;
            // Clone element attributes
            if (selfAttrs) {
                const cloneAttrs = [];
                cloneAttrs.map = {};
                for (let i = 0, l = selfAttrs.length; i < l; i++) {
                    const selfAttr = selfAttrs[i];
                    // Clone everything except id
                    if (selfAttr.name !== 'id') {
                        cloneAttrs[cloneAttrs.length] = { name: selfAttr.name, value: selfAttr.value };
                        cloneAttrs.map[selfAttr.name] = selfAttr.value;
                    }
                }
                clone.attributes = cloneAttrs;
            }
            clone.value = self.value;
            return clone;
        }
        /**
         * Wraps the node in in another node.
         *
         * @method wrap
         * @example
         * node.wrap(wrapperNode);
         */
        wrap(wrapper) {
            const self = this;
            if (self.parent) {
                self.parent.insert(wrapper, self);
                wrapper.append(self);
            }
            return self;
        }
        /**
         * Unwraps the node in other words it removes the node but keeps the children.
         *
         * @method unwrap
         * @example
         * node.unwrap();
         */
        unwrap() {
            const self = this;
            for (let node = self.firstChild; node;) {
                const next = node.next;
                self.insert(node, self, true);
                node = next;
            }
            self.remove();
        }
        /**
         * Removes the node from it's parent.
         *
         * @method remove
         * @return {tinymce.html.Node} Current node that got removed.
         * @example
         * node.remove();
         */
        remove() {
            const self = this, parent = self.parent, next = self.next, prev = self.prev;
            if (parent) {
                if (parent.firstChild === self) {
                    parent.firstChild = next;
                    if (next) {
                        next.prev = null;
                    }
                }
                else if (prev) {
                    prev.next = next;
                }
                if (parent.lastChild === self) {
                    parent.lastChild = prev;
                    if (prev) {
                        prev.next = null;
                    }
                }
                else if (next) {
                    next.prev = prev;
                }
                self.parent = self.next = self.prev = null;
            }
            return self;
        }
        /**
         * Appends a new node as a child of the current node.
         *
         * @method append
         * @param {tinymce.html.Node} node Node to append as a child of the current one.
         * @return {tinymce.html.Node} The node that got appended.
         * @example
         * node.append(someNode);
         */
        append(node) {
            const self = this;
            if (node.parent) {
                node.remove();
            }
            const last = self.lastChild;
            if (last) {
                last.next = node;
                node.prev = last;
                self.lastChild = node;
            }
            else {
                self.lastChild = self.firstChild = node;
            }
            node.parent = self;
            return node;
        }
        /**
         * Inserts a node at a specific position as a child of this node.
         *
         * @method insert
         * @param {tinymce.html.Node} node Node to insert as a child of this node.
         * @param {tinymce.html.Node} refNode Reference node to set node before/after.
         * @param {Boolean} before Optional state to insert the node before the reference node.
         * @return {tinymce.html.Node} The node that got inserted.
         * @example
         * parentNode.insert(newChildNode, oldChildNode);
         */
        insert(node, refNode, before) {
            if (node.parent) {
                node.remove();
            }
            const parent = refNode.parent || this;
            if (before) {
                if (refNode === parent.firstChild) {
                    parent.firstChild = node;
                }
                else if (refNode.prev) {
                    refNode.prev.next = node;
                }
                node.prev = refNode.prev;
                node.next = refNode;
                refNode.prev = node;
            }
            else {
                if (refNode === parent.lastChild) {
                    parent.lastChild = node;
                }
                else if (refNode.next) {
                    refNode.next.prev = node;
                }
                node.next = refNode.next;
                node.prev = refNode;
                refNode.next = node;
            }
            node.parent = parent;
            return node;
        }
        /**
         * Get all descendants by name.
         *
         * @method getAll
         * @param {String} name Name of the descendant nodes to collect.
         * @return {Array} Array with descendant nodes matching the specified name.
         */
        getAll(name) {
            const self = this;
            const collection = [];
            for (let node = self.firstChild; node; node = walk$2(node, self)) {
                if (node.name === name) {
                    collection.push(node);
                }
            }
            return collection;
        }
        /**
         * Get all children of this node.
         *
         * @method children
         * @return {Array} Array containing child nodes.
         */
        children() {
            const self = this;
            const collection = [];
            for (let node = self.firstChild; node; node = node.next) {
                collection.push(node);
            }
            return collection;
        }
        /**
         * Removes all children of the current node.
         *
         * @method empty
         * @return {tinymce.html.Node} The current node that got cleared.
         */
        empty() {
            const self = this;
            // Remove all children
            if (self.firstChild) {
                const nodes = [];
                // Collect the children
                for (let node = self.firstChild; node; node = walk$2(node, self)) {
                    nodes.push(node);
                }
                // Remove the children
                let i = nodes.length;
                while (i--) {
                    const node = nodes[i];
                    node.parent = node.firstChild = node.lastChild = node.next = node.prev = null;
                }
            }
            self.firstChild = self.lastChild = null;
            return self;
        }
        /**
         * Returns true/false if the node is to be considered empty or not.
         *
         * @method isEmpty
         * @param {Object} elements Name/value object with elements that are automatically treated as non empty elements.
         * @param {Object} whitespace Name/value object with elements that are automatically treated whitespace preservables.
         * @param {Function} predicate Optional predicate that gets called after the other rules determine that the node is empty. Should return true if the node is a content node.
         * @return {Boolean} true/false if the node is empty or not.
         * @example
         * node.isEmpty({ img: true });
         */
        isEmpty(elements, whitespace = {}, predicate) {
            const self = this;
            let node = self.firstChild;
            if (isNonEmptyElement(self)) {
                return false;
            }
            if (node) {
                do {
                    if (node.type === 1) {
                        // Ignore bogus elements
                        if (node.attr('data-mce-bogus')) {
                            continue;
                        }
                        // Keep empty elements like <img />
                        if (elements[node.name]) {
                            return false;
                        }
                        if (isNonEmptyElement(node)) {
                            return false;
                        }
                    }
                    // Keep comments
                    if (node.type === 8) {
                        return false;
                    }
                    // Keep non whitespace text nodes
                    if (node.type === 3 && !isEmptyTextNode(node)) {
                        return false;
                    }
                    // Keep whitespace preserve elements
                    if (node.type === 3 && node.parent && whitespace[node.parent.name] && isWhitespaceText(node.value ?? '')) {
                        return false;
                    }
                    // Predicate tells that the node is contents
                    if (predicate && predicate(node)) {
                        return false;
                    }
                } while ((node = walk$2(node, self)));
            }
            return true;
        }
        /**
         * Walks to the next or previous node and returns that node or null if it wasn't found.
         *
         * @method walk
         * @param {Boolean} prev Optional previous node state defaults to false.
         * @return {tinymce.html.Node} Node that is next to or previous of the current node.
         */
        walk(prev) {
            return walk$2(this, null, prev);
        }
    }

    // TINY-10305: Map over array for faster lookup.
    const unescapedTextParents = Tools.makeMap('NOSCRIPT STYLE SCRIPT XMP IFRAME NOEMBED NOFRAMES PLAINTEXT', ' ');
    const containsZwsp = (node) => isString(node.nodeValue) && node.nodeValue.includes(ZWSP$1);
    const getTemporaryNodeSelector = (tempAttrs) => `${tempAttrs.length === 0 ? '' : `${map$3(tempAttrs, (attr) => `[${attr}]`).join(',')},`}[data-mce-bogus="all"]`;
    const getTemporaryNodes = (tempAttrs, body) => body.querySelectorAll(getTemporaryNodeSelector(tempAttrs));
    const createZwspCommentWalker = (body) => document.createTreeWalker(body, NodeFilter.SHOW_COMMENT, (node) => containsZwsp(node) ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_SKIP);
    const createUnescapedZwspTextWalker = (body) => document.createTreeWalker(body, NodeFilter.SHOW_TEXT, (node) => {
        if (containsZwsp(node)) {
            const parent = node.parentNode;
            return parent && has$2(unescapedTextParents, parent.nodeName) ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_SKIP;
        }
        else {
            return NodeFilter.FILTER_SKIP;
        }
    });
    const hasZwspComment = (body) => createZwspCommentWalker(body).nextNode() !== null;
    const hasUnescapedZwspText = (body) => createUnescapedZwspTextWalker(body).nextNode() !== null;
    const hasTemporaryNode = (tempAttrs, body) => body.querySelector(getTemporaryNodeSelector(tempAttrs)) !== null;
    const trimTemporaryNodes = (tempAttrs, body) => {
        each$e(getTemporaryNodes(tempAttrs, body), (elm) => {
            const element = SugarElement.fromDom(elm);
            if (get$9(element, 'data-mce-bogus') === 'all') {
                remove$8(element);
            }
            else {
                each$e(tempAttrs, (attr) => {
                    if (has$1(element, attr)) {
                        remove$9(element, attr);
                    }
                });
            }
        });
    };
    const emptyAllNodeValuesInWalker = (walker) => {
        let curr = walker.nextNode();
        while (curr !== null) {
            curr.nodeValue = null;
            curr = walker.nextNode();
        }
    };
    const emptyZwspComments = compose(emptyAllNodeValuesInWalker, createZwspCommentWalker);
    const emptyUnescapedZwspTexts = compose(emptyAllNodeValuesInWalker, createUnescapedZwspTextWalker);
    const trim$1 = (body, tempAttrs) => {
        const conditionalTrims = [
            {
                condition: curry(hasTemporaryNode, tempAttrs),
                action: curry(trimTemporaryNodes, tempAttrs)
            },
            {
                condition: hasZwspComment,
                action: emptyZwspComments
            },
            {
                condition: hasUnescapedZwspText,
                action: emptyUnescapedZwspTexts
            }
        ];
        let trimmed = body;
        let cloned = false;
        each$e(conditionalTrims, ({ condition, action }) => {
            if (condition(trimmed)) {
                if (!cloned) {
                    trimmed = body.cloneNode(true);
                    cloned = true;
                }
                action(trimmed);
            }
        });
        return trimmed;
    };

    const cleanupBogusElements = (parent) => {
        const bogusElements = descendants(parent, '[data-mce-bogus]');
        each$e(bogusElements, (elem) => {
            const bogusValue = get$9(elem, 'data-mce-bogus');
            if (bogusValue === 'all') {
                remove$8(elem);
            }
            else if (isBr$6(elem)) {
                // Need to keep bogus padding brs represented as a zero-width space so that they aren't collapsed by the browser
                before$4(elem, SugarElement.fromText(zeroWidth));
                remove$8(elem);
            }
            else {
                unwrap(elem);
            }
        });
    };
    const cleanupInputNames = (parent) => {
        const inputs = descendants(parent, 'input');
        each$e(inputs, (input) => {
            remove$9(input, 'name');
        });
    };

    const trimEmptyContents = (editor, html) => {
        const blockName = getForcedRootBlock(editor);
        const emptyRegExp = new RegExp(`^(<${blockName}[^>]*>(&nbsp;|&#160;|\\s|\u00a0|<br \\/>|)<\\/${blockName}>[\r\n]*|<br \\/>[\r\n]*)$`);
        return html.replace(emptyRegExp, '');
    };
    const getPlainTextContent = (editor, body) => {
        const doc = editor.getDoc();
        const dos = getRootNode(SugarElement.fromDom(editor.getBody()));
        const offscreenDiv = SugarElement.fromTag('div', doc);
        set$4(offscreenDiv, 'data-mce-bogus', 'all');
        setAll(offscreenDiv, {
            position: 'fixed',
            left: '-9999999px',
            top: '0'
        });
        set$3(offscreenDiv, body.innerHTML);
        cleanupBogusElements(offscreenDiv);
        cleanupInputNames(offscreenDiv);
        // Append the wrapper element so that the browser will evaluate styles when getting the `innerText`
        const root = getContentContainer(dos);
        append$1(root, offscreenDiv);
        const content = trim$2(offscreenDiv.dom.innerText);
        remove$8(offscreenDiv);
        return content;
    };
    const getContentFromBody = (editor, args, body) => {
        let content;
        if (args.format === 'raw') {
            content = Tools.trim(trim$2(trim$1(body, editor.serializer.getTempAttrs()).innerHTML));
        }
        else if (args.format === 'text') {
            content = getPlainTextContent(editor, body);
        }
        else if (args.format === 'tree') {
            content = editor.serializer.serialize(body, args);
        }
        else {
            content = trimEmptyContents(editor, editor.serializer.serialize(body, args));
        }
        // Trim if not using a whitespace preserve format/element
        const shouldTrim = args.format !== 'text' && !isWsPreserveElement(SugarElement.fromDom(body));
        return shouldTrim && isString(content) ? Tools.trim(content) : content;
    };
    const getContentInternal = (editor, args) => Optional.from(editor.getBody())
        .fold(constant(args.format === 'tree' ? new AstNode('body', 11) : ''), (body) => getContentFromBody(editor, args, body));

    /**
     * This class is used to write HTML tags out it can be used with the Serializer.
     *
     * @class tinymce.html.Writer
     * @version 3.4
     * @example
     * const writer = tinymce.html.Writer({ indent: true });
     * writer.start('node', { attr: 'value' });
     * writer.end('node');
     * console.log(writer.getContent());
     */
    const makeMap$1 = Tools.makeMap;
    const Writer = (settings) => {
        const html = [];
        settings = settings || {};
        const indent = settings.indent;
        const indentBefore = makeMap$1(settings.indent_before || '');
        const indentAfter = makeMap$1(settings.indent_after || '');
        const encode = Entities.getEncodeFunc(settings.entity_encoding || 'raw', settings.entities);
        const htmlOutput = settings.element_format !== 'xhtml';
        return {
            /**
             * Writes a start element, such as `<p id="a">`.
             *
             * @method start
             * @param {String} name Name of the element.
             * @param {Array} attrs Optional array of objects containing an attribute name and value, or undefined if the element has no attributes.
             * @param {Boolean} empty Optional empty state if the tag should serialize as a void element. For example: `<img />`
             */
            start: (name, attrs, empty) => {
                if (indent && indentBefore[name] && html.length > 0) {
                    const value = html[html.length - 1];
                    if (value.length > 0 && value !== '\n') {
                        html.push('\n');
                    }
                }
                html.push('<', name);
                if (attrs) {
                    for (let i = 0, l = attrs.length; i < l; i++) {
                        const attr = attrs[i];
                        html.push(' ', attr.name, '="', encode(attr.value, true), '"');
                    }
                }
                if (!empty || htmlOutput) {
                    html[html.length] = '>';
                }
                else {
                    html[html.length] = ' />';
                }
                if (empty && indent && indentAfter[name] && html.length > 0) {
                    const value = html[html.length - 1];
                    if (value.length > 0 && value !== '\n') {
                        html.push('\n');
                    }
                }
            },
            /**
             * Writes an end element, such as `</p>`.
             *
             * @method end
             * @param {String} name Name of the element.
             */
            end: (name) => {
                let value;
                /* if (indent && indentBefore[name] && html.length > 0) {
                  value = html[html.length - 1];
          
                  if (value.length > 0 && value !== '\n')
                    html.push('\n');
                }*/
                html.push('</', name, '>');
                if (indent && indentAfter[name] && html.length > 0) {
                    value = html[html.length - 1];
                    if (value.length > 0 && value !== '\n') {
                        html.push('\n');
                    }
                }
            },
            /**
             * Writes a text node.
             *
             * @method text
             * @param {String} text String to write out.
             * @param {Boolean} raw Optional raw state. If true, the contents won't get encoded.
             */
            text: (text, raw) => {
                if (text.length > 0) {
                    html[html.length] = raw ? text : encode(text);
                }
            },
            /**
             * Writes a cdata node, such as `<![CDATA[data]]>`.
             *
             * @method cdata
             * @param {String} text String to write out inside the cdata.
             */
            cdata: (text) => {
                html.push('<![CDATA[', text, ']]>');
            },
            /**
             * Writes a comment node, such as `<!-- Comment -->`.
             *
             * @method comment
             * @param {String} text String to write out inside the comment.
             */
            comment: (text) => {
                html.push('<!--', text, '-->');
            },
            /**
             * Writes a processing instruction (PI) node, such as `<?xml attr="value" ?>`.
             *
             * @method pi
             * @param {String} name Name of the pi.
             * @param {String} text String to write out inside the pi.
             */
            pi: (name, text) => {
                if (text) {
                    html.push('<?', name, ' ', encode(text), '?>');
                }
                else {
                    html.push('<?', name, '?>');
                }
                if (indent) {
                    html.push('\n');
                }
            },
            /**
             * Writes a doctype node, such as `<!DOCTYPE data>`.
             *
             * @method doctype
             * @param {String} text String to write out inside the doctype.
             */
            doctype: (text) => {
                html.push('<!DOCTYPE', text, '>', indent ? '\n' : '');
            },
            /**
             * Resets the internal buffer. For example, if one wants to reuse the writer.
             *
             * @method reset
             */
            reset: () => {
                html.length = 0;
            },
            /**
             * Returns the contents that was serialized.
             *
             * @method getContent
             * @return {String} HTML contents that got written down.
             */
            getContent: () => {
                return html.join('').replace(/\n$/, '');
            }
        };
    };

    /**
     * This class is used to serialize down the DOM tree into a string using a Writer instance.
     *
     * @class tinymce.html.Serializer
     * @version 3.4
     * @example
     * tinymce.html.Serializer().serialize(tinymce.html.DomParser().parse('<p>text</p>'));
     */
    const HtmlSerializer = (settings = {}, schema = Schema()) => {
        const writer = Writer(settings);
        settings.validate = 'validate' in settings ? settings.validate : true;
        /**
         * Serializes the specified node into a string.
         *
         * @method serialize
         * @param {tinymce.html.Node} node Node instance to serialize.
         * @return {String} String with HTML based on the DOM tree.
         * @example
         * tinymce.html.Serializer().serialize(tinymce.html.DomParser().parse('<p>text</p>'));
         */
        const serialize = (node) => {
            const validate = settings.validate;
            const handlers = {
                // #text
                3: (node) => {
                    writer.text(node.value ?? '', node.raw);
                },
                // #comment
                8: (node) => {
                    writer.comment(node.value ?? '');
                },
                // Processing instruction
                7: (node) => {
                    writer.pi(node.name, node.value);
                },
                // Doctype
                10: (node) => {
                    writer.doctype(node.value ?? '');
                },
                // CDATA
                4: (node) => {
                    writer.cdata(node.value ?? '');
                },
                // Document fragment
                11: (node) => {
                    let tempNode = node;
                    if ((tempNode = tempNode.firstChild)) {
                        do {
                            walk(tempNode);
                        } while ((tempNode = tempNode.next));
                    }
                }
            };
            writer.reset();
            const walk = (node) => {
                const handler = handlers[node.type];
                if (!handler) {
                    const name = node.name;
                    const isEmpty = name in schema.getVoidElements();
                    let attrs = node.attributes;
                    // Sort attributes
                    if (validate && attrs && attrs.length > 1) {
                        const sortedAttrs = [];
                        sortedAttrs.map = {};
                        const elementRule = schema.getElementRule(node.name);
                        if (elementRule) {
                            for (let i = 0, l = elementRule.attributesOrder.length; i < l; i++) {
                                const attrName = elementRule.attributesOrder[i];
                                if (attrName in attrs.map) {
                                    const attrValue = attrs.map[attrName];
                                    sortedAttrs.map[attrName] = attrValue;
                                    sortedAttrs.push({ name: attrName, value: attrValue });
                                }
                            }
                            for (let i = 0, l = attrs.length; i < l; i++) {
                                const attrName = attrs[i].name;
                                if (!(attrName in sortedAttrs.map)) {
                                    const attrValue = attrs.map[attrName];
                                    sortedAttrs.map[attrName] = attrValue;
                                    sortedAttrs.push({ name: attrName, value: attrValue });
                                }
                            }
                            attrs = sortedAttrs;
                        }
                    }
                    writer.start(name, attrs, isEmpty);
                    if (isNonHtmlElementRootName(name)) {
                        if (isString(node.value)) {
                            writer.text(node.value, true);
                        }
                        writer.end(name);
                    }
                    else {
                        if (!isEmpty) {
                            let child = node.firstChild;
                            if (child) {
                                // Pre and textarea elements treat the first newline character as optional and will omit it. As such, if the content starts
                                // with a newline we need to add in an additional newline to prevent the current newline in the value being treated as optional
                                // See https://html.spec.whatwg.org/multipage/syntax.html#element-restrictions
                                if ((name === 'pre' || name === 'textarea') && child.type === 3 && child.value?.[0] === '\n') {
                                    writer.text('\n', true);
                                }
                                do {
                                    walk(child);
                                } while ((child = child.next));
                            }
                            writer.end(name);
                        }
                    }
                }
                else {
                    handler(node);
                }
            };
            // Serialize element or text nodes and treat all other nodes as fragments
            if (node.type === 1 && !settings.inner) {
                walk(node);
            }
            else if (node.type === 3) {
                handlers[3](node);
            }
            else {
                handlers[11](node);
            }
            return writer.getContent();
        };
        return {
            serialize
        };
    };

    const nonInheritableStyles = new Set();
    (() => {
        // TODO: TINY-7326 Figure out what else should go in the nonInheritableStyles list
        const nonInheritableStylesArr = [
            'margin', 'margin-left', 'margin-right', 'margin-top', 'margin-bottom',
            'padding', 'padding-left', 'padding-right', 'padding-top', 'padding-bottom',
            'border', 'border-width', 'border-style', 'border-color',
            'background', 'background-attachment', 'background-clip',
            'background-image', 'background-origin', 'background-position', 'background-repeat', 'background-size',
            'float', 'position', 'left', 'right', 'top', 'bottom',
            'z-index', 'display', 'transform',
            'width', 'max-width', 'min-width', 'height', 'max-height', 'min-height',
            'overflow', 'overflow-x', 'overflow-y', 'text-overflow', 'vertical-align',
            'transition', 'transition-delay', 'transition-duration', 'transition-property', 'transition-timing-function'
        ];
        each$e(nonInheritableStylesArr, (style) => {
            nonInheritableStyles.add(style);
        });
    })();
    const conditionalNonInheritableStyles = new Set();
    (() => {
        // These styles are only noninheritable when applied to an element with a noninheritable style
        // For example, background-color is visible on an element with padding, even when children have background-color;
        // however, when the element has no padding, background-color is either visible or overridden by children
        const conditionalNonInheritableStylesArr = [
            'background-color'
        ];
        each$e(conditionalNonInheritableStylesArr, (style) => {
            conditionalNonInheritableStyles.add(style);
        });
    })();
    // TODO: TINY-7326 Figure out what else should be added to the shorthandStyleProps list
    // Does not include non-inherited shorthand style properties
    const shorthandStyleProps = ['font', 'text-decoration', 'text-emphasis'];
    const getStyles$1 = (dom, node) => dom.parseStyle(dom.getAttrib(node, 'style'));
    const getStyleProps = (dom, node) => keys(getStyles$1(dom, node));
    const isNonInheritableStyle = (style) => nonInheritableStyles.has(style);
    const isConditionalNonInheritableStyle = (style) => conditionalNonInheritableStyles.has(style);
    const hasNonInheritableStyles = (dom, node) => exists(getStyleProps(dom, node), (style) => isNonInheritableStyle(style));
    const hasConditionalNonInheritableStyles = (dom, node) => hasNonInheritableStyles(dom, node) &&
        exists(getStyleProps(dom, node), (style) => isConditionalNonInheritableStyle(style));
    const getLonghandStyleProps = (styles) => filter$5(styles, (style) => exists(shorthandStyleProps, (prop) => startsWith(style, prop)));
    const hasStyleConflict = (dom, node, parentNode) => {
        const nodeStyleProps = getStyleProps(dom, node);
        const parentNodeStyleProps = getStyleProps(dom, parentNode);
        const valueMismatch = (prop) => {
            const nodeValue = dom.getStyle(node, prop) ?? '';
            const parentValue = dom.getStyle(parentNode, prop) ?? '';
            return isNotEmpty(nodeValue) && isNotEmpty(parentValue) && nodeValue !== parentValue;
        };
        return exists(nodeStyleProps, (nodeStyleProp) => {
            const propExists = (props) => exists(props, (prop) => prop === nodeStyleProp);
            // If parent has a longhand property e.g. margin-left but the child (node) style is margin, need to get the margin-left value of node to be able to do a proper comparison
            // This is because getting the style using the key of 'margin' on a 'margin-left' parent would give a string of space separated values or empty string depending on the browser
            if (!propExists(parentNodeStyleProps) && propExists(shorthandStyleProps)) {
                const longhandProps = getLonghandStyleProps(parentNodeStyleProps);
                return exists(longhandProps, valueMismatch);
            }
            else {
                return valueMismatch(nodeStyleProp);
            }
        });
    };

    const isChar = (forward, predicate, pos) => Optional.from(pos.container()).filter(isText$b).exists((text) => {
        const delta = forward ? 0 : -1;
        return predicate(text.data.charAt(pos.offset() + delta));
    });
    const isBeforeSpace = curry(isChar, true, isWhiteSpace);
    const isAfterSpace = curry(isChar, false, isWhiteSpace);
    const isEmptyText = (pos) => {
        const container = pos.container();
        return isText$b(container) && (container.data.length === 0 || isZwsp(container.data) && BookmarkManager.isBookmarkNode(container.parentNode));
    };
    const matchesElementPosition = (before, predicate) => (pos) => getChildNodeAtRelativeOffset(before ? 0 : -1, pos).filter(predicate).isSome();
    const isImageBlock = (node) => isImg(node) && get$7(SugarElement.fromDom(node), 'display') === 'block';
    const isCefNode = (node) => isContentEditableFalse$a(node) && !isBogusAll(node);
    const isBeforeImageBlock = matchesElementPosition(true, isImageBlock);
    const isAfterImageBlock = matchesElementPosition(false, isImageBlock);
    const isBeforeMedia = matchesElementPosition(true, isMedia$2);
    const isAfterMedia = matchesElementPosition(false, isMedia$2);
    const isBeforeTable = matchesElementPosition(true, isTable$2);
    const isAfterTable = matchesElementPosition(false, isTable$2);
    const isBeforeContentEditableFalse = matchesElementPosition(true, isCefNode);
    const isAfterContentEditableFalse = matchesElementPosition(false, isCefNode);

    const dropLast = (xs) => xs.slice(0, -1);
    const parentsUntil = (start, root, predicate) => {
        if (contains(root, start)) {
            return dropLast(parents$1(start, (elm) => {
                return predicate(elm) || eq(elm, root);
            }));
        }
        else {
            return [];
        }
    };
    const parents = (start, root) => parentsUntil(start, root, never);
    const parentsAndSelf = (start, root) => [start].concat(parents(start, root));

    const navigateIgnoreEmptyTextNodes = (forward, root, from) => navigateIgnore(forward, root, from, isEmptyText);
    const isBlock$2 = (schema) => (el) => schema.isBlock(name(el));
    const getClosestBlock$1 = (root, pos, schema) => find$2(parentsAndSelf(SugarElement.fromDom(pos.container()), root), isBlock$2(schema));
    const isAtBeforeAfterBlockBoundary = (forward, root, pos, schema) => navigateIgnoreEmptyTextNodes(forward, root.dom, pos)
        .forall((newPos) => getClosestBlock$1(root, pos, schema).fold(() => !isInSameBlock(newPos, pos, root.dom), (fromBlock) => !isInSameBlock(newPos, pos, root.dom) && contains(fromBlock, SugarElement.fromDom(newPos.container()))));
    const isAtBlockBoundary = (forward, root, pos, schema) => getClosestBlock$1(root, pos, schema).fold(() => navigateIgnoreEmptyTextNodes(forward, root.dom, pos).forall((newPos) => !isInSameBlock(newPos, pos, root.dom)), (parent) => navigateIgnoreEmptyTextNodes(forward, parent.dom, pos).isNone());
    const isAtStartOfBlock = curry(isAtBlockBoundary, false);
    const isAtEndOfBlock = curry(isAtBlockBoundary, true);
    const isBeforeBlock = curry(isAtBeforeAfterBlockBoundary, false);
    const isAfterBlock = curry(isAtBeforeAfterBlockBoundary, true);

    const isBr$2 = (pos) => getElementFromPosition(pos).exists(isBr$6);
    const findBr = (forward, root, pos, schema) => {
        const parentBlocks = filter$5(parentsAndSelf(SugarElement.fromDom(pos.container()), root), (el) => schema.isBlock(name(el)));
        const scope = head(parentBlocks).getOr(root);
        return fromPosition(forward, scope.dom, pos).filter(isBr$2);
    };
    const isBeforeBr$1 = (root, pos, schema) => getElementFromPosition(pos).exists(isBr$6) || findBr(true, root, pos, schema).isSome();
    const isAfterBr = (root, pos, schema) => getElementFromPrevPosition(pos).exists(isBr$6) || findBr(false, root, pos, schema).isSome();
    const findPreviousBr = curry(findBr, false);
    const findNextBr = curry(findBr, true);

    const isInMiddleOfText = (pos) => CaretPosition.isTextPosition(pos) && !pos.isAtStart() && !pos.isAtEnd();
    const getClosestBlock = (root, pos, schema) => {
        const parentBlocks = filter$5(parentsAndSelf(SugarElement.fromDom(pos.container()), root), (el) => schema.isBlock(name(el)));
        return head(parentBlocks).getOr(root);
    };
    const hasSpaceBefore = (root, pos, schema) => {
        if (isInMiddleOfText(pos)) {
            return isAfterSpace(pos);
        }
        else {
            return isAfterSpace(pos) || prevPosition(getClosestBlock(root, pos, schema).dom, pos).exists(isAfterSpace);
        }
    };
    const hasSpaceAfter = (root, pos, schema) => {
        if (isInMiddleOfText(pos)) {
            return isBeforeSpace(pos);
        }
        else {
            return isBeforeSpace(pos) || nextPosition(getClosestBlock(root, pos, schema).dom, pos).exists(isBeforeSpace);
        }
    };
    const isPreValue = (value) => contains$2(['pre', 'pre-wrap'], value);
    const isInPre = (pos) => getElementFromPosition(pos)
        .bind((elm) => closest$5(elm, isElement$8))
        .exists((elm) => isPreValue(get$7(elm, 'white-space')));
    const isAtBeginningOfBody = (root, pos) => prevPosition(root.dom, pos).isNone();
    const isAtEndOfBody = (root, pos) => nextPosition(root.dom, pos).isNone();
    const isAtLineBoundary = (root, pos, schema) => (isAtBeginningOfBody(root, pos) ||
        isAtEndOfBody(root, pos) ||
        isAtStartOfBlock(root, pos, schema) ||
        isAtEndOfBlock(root, pos, schema) ||
        isAfterBr(root, pos, schema) ||
        isBeforeBr$1(root, pos, schema));
    const isCefBlock = (node) => isNonNullable(node) && isContentEditableFalse$a(node) && isBlockLike(node);
    // Check the next/previous element in case it is a cef and the next/previous caret position then would skip it, then check
    // the next next/previous caret position ( for example in case the next element is a strong, containing a cef ).
    const isSiblingCefBlock = (root, direction) => (container) => {
        return isCefBlock(new DomTreeWalker(container, root)[direction]());
    };
    const isBeforeCefBlock = (root, pos) => {
        const nextPos = nextPosition(root.dom, pos).getOr(pos);
        const isNextCefBlock = isSiblingCefBlock(root.dom, 'next');
        return pos.isAtEnd() && (isNextCefBlock(pos.container()) || isNextCefBlock(nextPos.container()));
    };
    const isAfterCefBlock = (root, pos) => {
        const prevPos = prevPosition(root.dom, pos).getOr(pos);
        const isPrevCefBlock = isSiblingCefBlock(root.dom, 'prev');
        return pos.isAtStart() && (isPrevCefBlock(pos.container()) || isPrevCefBlock(prevPos.container()));
    };
    const needsToHaveNbsp = (root, pos, schema) => {
        if (isInPre(pos)) {
            return false;
        }
        else {
            return isAtLineBoundary(root, pos, schema) || hasSpaceBefore(root, pos, schema) || hasSpaceAfter(root, pos, schema);
        }
    };
    const needsToBeNbspLeft = (root, pos, schema) => {
        if (isInPre(pos)) {
            return false;
        }
        else {
            return isAtStartOfBlock(root, pos, schema) || isBeforeBlock(root, pos, schema) || isAfterBr(root, pos, schema) || hasSpaceBefore(root, pos, schema) || isAfterCefBlock(root, pos);
        }
    };
    const leanRight = (pos) => {
        const container = pos.container();
        const offset = pos.offset();
        if (isText$b(container) && offset < container.data.length) {
            return CaretPosition(container, offset + 1);
        }
        else {
            return pos;
        }
    };
    const needsToBeNbspRight = (root, pos, schema) => {
        if (isInPre(pos)) {
            return false;
        }
        else {
            return isAtEndOfBlock(root, pos, schema) || isAfterBlock(root, pos, schema) || isBeforeBr$1(root, pos, schema) || hasSpaceAfter(root, pos, schema) || isBeforeCefBlock(root, pos);
        }
    };
    const needsToBeNbsp = (root, pos, schema) => needsToBeNbspLeft(root, pos, schema) || needsToBeNbspRight(root, leanRight(pos), schema);
    const isNbspAt = (text, offset) => isNbsp(text.charAt(offset));
    const isWhiteSpaceAt = (text, offset) => isWhiteSpace(text.charAt(offset));
    const hasNbsp = (pos) => {
        const container = pos.container();
        return isText$b(container) && contains$1(container.data, nbsp);
    };
    const normalizeNbspMiddle = (text) => {
        const chars = text.split('');
        return map$3(chars, (chr, i) => {
            if (isNbsp(chr) && i > 0 && i < chars.length - 1 && isContent(chars[i - 1]) && isContent(chars[i + 1])) {
                return ' ';
            }
            else {
                return chr;
            }
        }).join('');
    };
    const normalizeNbspAtStart = (root, node, makeNbsp, schema) => {
        const text = node.data;
        const firstPos = CaretPosition(node, 0);
        if (!makeNbsp && isNbspAt(text, 0) && !needsToBeNbsp(root, firstPos, schema)) {
            node.data = ' ' + text.slice(1);
            return true;
        }
        else if (makeNbsp && isWhiteSpaceAt(text, 0) && needsToBeNbspLeft(root, firstPos, schema)) {
            node.data = nbsp + text.slice(1);
            return true;
        }
        else {
            return false;
        }
    };
    const normalizeNbspInMiddleOfTextNode = (node) => {
        const text = node.data;
        const newText = normalizeNbspMiddle(text);
        if (newText !== text) {
            node.data = newText;
            return true;
        }
        else {
            return false;
        }
    };
    const normalizeNbspAtEnd = (root, node, makeNbsp, schema) => {
        const text = node.data;
        const lastPos = CaretPosition(node, text.length - 1);
        if (!makeNbsp && isNbspAt(text, text.length - 1) && !needsToBeNbsp(root, lastPos, schema)) {
            node.data = text.slice(0, -1) + ' ';
            return true;
        }
        else if (makeNbsp && isWhiteSpaceAt(text, text.length - 1) && needsToBeNbspRight(root, lastPos, schema)) {
            node.data = text.slice(0, -1) + nbsp;
            return true;
        }
        else {
            return false;
        }
    };
    const normalizeNbsps$1 = (root, pos, schema) => {
        const container = pos.container();
        if (!isText$b(container)) {
            return Optional.none();
        }
        if (hasNbsp(pos)) {
            const normalized = normalizeNbspAtStart(root, container, false, schema) || normalizeNbspInMiddleOfTextNode(container) || normalizeNbspAtEnd(root, container, false, schema);
            return someIf(normalized, pos);
        }
        else if (needsToBeNbsp(root, pos, schema)) {
            const normalized = normalizeNbspAtStart(root, container, true, schema) || normalizeNbspAtEnd(root, container, true, schema);
            return someIf(normalized, pos);
        }
        else {
            return Optional.none();
        }
    };
    const normalizeNbspsInEditor = (editor) => {
        const root = SugarElement.fromDom(editor.getBody());
        if (editor.selection.isCollapsed()) {
            normalizeNbsps$1(root, CaretPosition.fromRangeStart(editor.selection.getRng()), editor.schema).each((pos) => {
                editor.selection.setRng(pos.toRange());
            });
        }
    };

    const normalize$1 = (node, offset, count, schema) => {
        if (count === 0) {
            return;
        }
        const elm = SugarElement.fromDom(node);
        const root = ancestor$5(elm, (el) => schema.isBlock(name(el))).getOr(elm);
        // Get the whitespace
        const whitespace = node.data.slice(offset, offset + count);
        // Determine if we're at the end or start of the content
        const isEndOfContent = offset + count >= node.data.length && needsToBeNbspRight(root, CaretPosition(node, node.data.length), schema);
        const isStartOfContent = offset === 0 && needsToBeNbspLeft(root, CaretPosition(node, 0), schema);
        // Replace the original whitespace with the normalized whitespace content
        node.replaceData(offset, count, normalize$4(whitespace, 4, isStartOfContent, isEndOfContent));
    };
    const normalizeWhitespaceAfter = (node, offset, schema) => {
        const content = node.data.slice(offset);
        const whitespaceCount = content.length - lTrim(content).length;
        normalize$1(node, offset, whitespaceCount, schema);
    };
    const normalizeWhitespaceBefore = (node, offset, schema) => {
        const content = node.data.slice(0, offset);
        const whitespaceCount = content.length - rTrim(content).length;
        normalize$1(node, offset - whitespaceCount, whitespaceCount, schema);
    };
    const mergeTextNodes = (prevNode, nextNode, schema, normalizeWhitespace, mergeToPrev = true) => {
        const whitespaceOffset = rTrim(prevNode.data).length;
        const newNode = mergeToPrev ? prevNode : nextNode;
        const removeNode = mergeToPrev ? nextNode : prevNode;
        // Merge the elements
        if (mergeToPrev) {
            newNode.appendData(removeNode.data);
        }
        else {
            newNode.insertData(0, removeNode.data);
        }
        remove$8(SugarElement.fromDom(removeNode));
        // Normalize the whitespace around the merged elements, to ensure it doesn't get lost
        if (normalizeWhitespace) {
            normalizeWhitespaceAfter(newNode, whitespaceOffset, schema);
        }
        return newNode;
    };

    const needsReposition = (pos, elm) => {
        const container = pos.container();
        const offset = pos.offset();
        return !CaretPosition.isTextPosition(pos) && container === elm.parentNode && offset > CaretPosition.before(elm).offset();
    };
    const reposition = (elm, pos) => needsReposition(pos, elm) ? CaretPosition(pos.container(), pos.offset() - 1) : pos;
    const beforeOrStartOf = (node) => isText$b(node) ? CaretPosition(node, 0) : CaretPosition.before(node);
    const afterOrEndOf = (node) => isText$b(node) ? CaretPosition(node, node.data.length) : CaretPosition.after(node);
    const getPreviousSiblingCaretPosition = (elm) => {
        if (isCaretCandidate$3(elm.previousSibling)) {
            return Optional.some(afterOrEndOf(elm.previousSibling));
        }
        else {
            return elm.previousSibling ? lastPositionIn(elm.previousSibling) : Optional.none();
        }
    };
    const getNextSiblingCaretPosition = (elm) => {
        if (isCaretCandidate$3(elm.nextSibling)) {
            return Optional.some(beforeOrStartOf(elm.nextSibling));
        }
        else {
            return elm.nextSibling ? firstPositionIn(elm.nextSibling) : Optional.none();
        }
    };
    const findCaretPositionBackwardsFromElm = (rootElement, elm) => {
        return Optional.from(elm.previousSibling ? elm.previousSibling : elm.parentNode)
            .bind((node) => prevPosition(rootElement, CaretPosition.before(node)))
            .orThunk(() => nextPosition(rootElement, CaretPosition.after(elm)));
    };
    const findCaretPositionForwardsFromElm = (rootElement, elm) => nextPosition(rootElement, CaretPosition.after(elm)).orThunk(() => prevPosition(rootElement, CaretPosition.before(elm)));
    const findCaretPositionBackwards = (rootElement, elm) => getPreviousSiblingCaretPosition(elm).orThunk(() => getNextSiblingCaretPosition(elm))
        .orThunk(() => findCaretPositionBackwardsFromElm(rootElement, elm));
    const findCaretPositionForward = (rootElement, elm) => getNextSiblingCaretPosition(elm)
        .orThunk(() => getPreviousSiblingCaretPosition(elm))
        .orThunk(() => findCaretPositionForwardsFromElm(rootElement, elm));
    const findCaretPosition = (forward, rootElement, elm) => forward ? findCaretPositionForward(rootElement, elm) : findCaretPositionBackwards(rootElement, elm);
    const findCaretPosOutsideElmAfterDelete = (forward, rootElement, elm) => findCaretPosition(forward, rootElement, elm).map(curry(reposition, elm));
    const setSelection$1 = (editor, forward, pos) => {
        pos.fold(() => {
            editor.focus();
        }, (pos) => {
            editor.selection.setRng(pos.toRange(), forward);
        });
    };
    const eqRawNode = (rawNode) => (elm) => elm.dom === rawNode;
    const isBlock$1 = (editor, elm) => elm && has$2(editor.schema.getBlockElements(), name(elm));
    const paddEmptyBlock = (schema, elm, preserveEmptyCaret) => {
        if (isEmpty$4(schema, elm)) {
            const br = SugarElement.fromHtml('<br data-mce-bogus="1">');
            // Remove all bogus elements except caret
            if (preserveEmptyCaret) {
                each$e(children$1(elm), (node) => {
                    if (!isEmptyCaretFormatElement(node)) {
                        remove$8(node);
                    }
                });
            }
            else {
                empty(elm);
            }
            append$1(elm, br);
            return Optional.some(CaretPosition.before(br.dom));
        }
        else {
            return Optional.none();
        }
    };
    const deleteNormalized = (elm, afterDeletePosOpt, schema, normalizeWhitespace) => {
        const prevTextOpt = prevSibling(elm).filter(isText$c);
        const nextTextOpt = nextSibling(elm).filter(isText$c);
        // Delete the element
        remove$8(elm);
        // Merge and normalize any prev/next text nodes, so that they are merged and don't lose meaningful whitespace
        // eg. <p>a <span></span> b</p> -> <p>a &nsbp;b</p> or <p><span></span> a</p> -> <p>&nbsp;a</a>
        return lift3(prevTextOpt, nextTextOpt, afterDeletePosOpt, (prev, next, pos) => {
            const prevNode = prev.dom, nextNode = next.dom;
            const offset = prevNode.data.length;
            mergeTextNodes(prevNode, nextNode, schema, normalizeWhitespace);
            // Update the cursor position if required
            return pos.container() === nextNode ? CaretPosition(prevNode, offset) : pos;
        }).orThunk(() => {
            if (normalizeWhitespace) {
                prevTextOpt.each((elm) => normalizeWhitespaceBefore(elm.dom, elm.dom.length, schema));
                nextTextOpt.each((elm) => normalizeWhitespaceAfter(elm.dom, 0, schema));
            }
            return afterDeletePosOpt;
        });
    };
    const isInlineElement = (editor, element) => has$2(editor.schema.getTextInlineElements(), name(element));
    const deleteElement$2 = (editor, forward, elm, moveCaret = true, preserveEmptyCaret = false) => {
        // Existing delete logic
        const afterDeletePos = findCaretPosOutsideElmAfterDelete(forward, editor.getBody(), elm.dom);
        const parentBlock = ancestor$5(elm, curry(isBlock$1, editor), eqRawNode(editor.getBody()));
        const normalizedAfterDeletePos = deleteNormalized(elm, afterDeletePos, editor.schema, isInlineElement(editor, elm));
        if (editor.dom.isEmpty(editor.getBody())) {
            editor.setContent('');
            editor.selection.setCursorLocation();
        }
        else {
            parentBlock.bind((elm) => paddEmptyBlock(editor.schema, elm, preserveEmptyCaret)).fold(() => {
                if (moveCaret) {
                    setSelection$1(editor, forward, normalizedAfterDeletePos);
                }
            }, (paddPos) => {
                if (moveCaret) {
                    setSelection$1(editor, forward, Optional.some(paddPos));
                }
            });
        }
    };

    const strongRtl = /[\u0591-\u07FF\uFB1D-\uFDFF\uFE70-\uFEFC]/;
    const hasStrongRtl = (text) => strongRtl.test(text);

    const isInlineTarget = (editor, elm) => is$2(SugarElement.fromDom(elm), getInlineBoundarySelector(editor))
        && !isTransparentBlock(editor.schema, elm)
        && editor.dom.isEditable(elm);
    const isRtl = (element) => DOMUtils.DOM.getStyle(element, 'direction', true) === 'rtl' || hasStrongRtl(element.textContent ?? '');
    const findInlineParents = (isInlineTarget, rootNode, pos) => filter$5(DOMUtils.DOM.getParents(pos.container(), '*', rootNode), isInlineTarget);
    const findRootInline = (isInlineTarget, rootNode, pos) => {
        const parents = findInlineParents(isInlineTarget, rootNode, pos);
        return Optional.from(parents[parents.length - 1]);
    };
    const hasSameParentBlock = (rootNode, node1, node2) => {
        const block1 = getParentBlock$3(node1, rootNode);
        const block2 = getParentBlock$3(node2, rootNode);
        return isNonNullable(block1) && block1 === block2;
    };
    const isAtZwsp = (pos) => isBeforeInline(pos) || isAfterInline(pos);
    const normalizePosition = (forward, pos) => {
        const container = pos.container(), offset = pos.offset();
        if (forward) {
            if (isCaretContainerInline(container)) {
                if (isText$b(container.nextSibling)) {
                    return CaretPosition(container.nextSibling, 0);
                }
                else {
                    return CaretPosition.after(container);
                }
            }
            else {
                return isBeforeInline(pos) ? CaretPosition(container, offset + 1) : pos;
            }
        }
        else {
            if (isCaretContainerInline(container)) {
                if (isText$b(container.previousSibling)) {
                    return CaretPosition(container.previousSibling, container.previousSibling.data.length);
                }
                else {
                    return CaretPosition.before(container);
                }
            }
            else {
                return isAfterInline(pos) ? CaretPosition(container, offset - 1) : pos;
            }
        }
    };
    const normalizeForwards = curry(normalizePosition, true);
    const normalizeBackwards = curry(normalizePosition, false);

    const execCommandIgnoreInputEvents = (editor, command) => {
        // We need to prevent the input events from being fired by execCommand when delete is used internally
        const inputBlocker = (e) => e.stopImmediatePropagation();
        editor.on('beforeinput input', inputBlocker, true);
        editor.getDoc().execCommand(command);
        editor.off('beforeinput input', inputBlocker);
    };
    // ASSUMPTION: The editor command 'delete' doesn't have any `beforeinput` and `input` trapping
    // because those events are only triggered by native contenteditable behaviour.
    const execEditorDeleteCommand = (editor) => {
        editor.execCommand('delete');
    };
    const execNativeDeleteCommand = (editor) => execCommandIgnoreInputEvents(editor, 'Delete');
    const execNativeForwardDeleteCommand = (editor) => execCommandIgnoreInputEvents(editor, 'ForwardDelete');
    const isBeforeRoot = (rootNode) => (elm) => is$4(parent(elm), rootNode, eq);
    const isTextBlockOrListItem = (element) => isTextBlock$3(element) || isListItem$2(element);
    const getParentBlock$2 = (rootNode, elm) => {
        if (contains(rootNode, elm)) {
            return closest$5(elm, isTextBlockOrListItem, isBeforeRoot(rootNode));
        }
        else {
            return Optional.none();
        }
    };
    const paddEmptyBody = (editor, moveSelection = true) => {
        if (editor.dom.isEmpty(editor.getBody())) {
            editor.setContent('', { no_selection: !moveSelection });
        }
    };
    const willDeleteLastPositionInElement = (forward, fromPos, elm) => lift2(firstPositionIn(elm), lastPositionIn(elm), (firstPos, lastPos) => {
        const normalizedFirstPos = normalizePosition(true, firstPos);
        const normalizedLastPos = normalizePosition(false, lastPos);
        const normalizedFromPos = normalizePosition(false, fromPos);
        if (forward) {
            return nextPosition(elm, normalizedFromPos).exists((nextPos) => nextPos.isEqual(normalizedLastPos) && fromPos.isEqual(normalizedFirstPos));
        }
        else {
            return prevPosition(elm, normalizedFromPos).exists((prevPos) => prevPos.isEqual(normalizedFirstPos) && fromPos.isEqual(normalizedLastPos));
        }
    }).getOr(true);
    const freefallRtl = (root) => {
        const child = isComment$1(root) ? prevSibling(root) : lastChild(root);
        return child.bind(freefallRtl).orThunk(() => Optional.some(root));
    };
    const deleteRangeContents = (editor, rng, root, moveSelection = true) => {
        rng.deleteContents();
        // Pad the last block node
        const lastNode = freefallRtl(root).getOr(root);
        const lastBlock = SugarElement.fromDom(editor.dom.getParent(lastNode.dom, editor.dom.isBlock) ?? root.dom);
        // If the block is the editor body then we need to insert the root block as well
        if (lastBlock.dom === editor.getBody()) {
            paddEmptyBody(editor, moveSelection);
        }
        else if (isEmpty$4(editor.schema, lastBlock, { checkRootAsContent: false })) {
            fillWithPaddingBr(lastBlock);
            if (moveSelection) {
                editor.selection.setCursorLocation(lastBlock.dom, 0);
            }
        }
        // Clean up any additional leftover nodes. If the last block wasn't a direct child, then we also need to clean up siblings
        if (!eq(root, lastBlock)) {
            const additionalCleanupNodes = is$4(parent(lastBlock), root) ? [] : siblings(lastBlock);
            each$e(additionalCleanupNodes.concat(children$1(root)), (node) => {
                if (!eq(node, lastBlock) && !contains(node, lastBlock) && isEmpty$4(editor.schema, node)) {
                    remove$8(node);
                }
            });
        }
    };

    const isRootFromElement = (root) => (cur) => eq(root, cur);
    const getTableCells = (table) => descendants(table, 'td,th');
    const getTable$1 = (node, isRoot) => getClosestTable(SugarElement.fromDom(node), isRoot);
    const selectionInTableWithNestedTable = (details) => {
        return lift2(details.startTable, details.endTable, (startTable, endTable) => {
            const isStartTableParentOfEndTable = descendant(startTable, (t) => eq(t, endTable));
            const isEndTableParentOfStartTable = descendant(endTable, (t) => eq(t, startTable));
            return !isStartTableParentOfEndTable && !isEndTableParentOfStartTable ? details : {
                ...details,
                startTable: isStartTableParentOfEndTable ? Optional.none() : details.startTable,
                endTable: isEndTableParentOfStartTable ? Optional.none() : details.endTable,
                isSameTable: false,
                isMultiTable: false
            };
        }).getOr(details);
    };
    const adjustQuirksInDetails = (details) => {
        return selectionInTableWithNestedTable(details);
    };
    const getTableDetailsFromRange = (rng, isRoot) => {
        const startTable = getTable$1(rng.startContainer, isRoot);
        const endTable = getTable$1(rng.endContainer, isRoot);
        const isStartInTable = startTable.isSome();
        const isEndInTable = endTable.isSome();
        // Partial selection - selection is not within the same table
        const isSameTable = lift2(startTable, endTable, eq).getOr(false);
        const isMultiTable = !isSameTable && isStartInTable && isEndInTable;
        return adjustQuirksInDetails({
            startTable,
            endTable,
            isStartInTable,
            isEndInTable,
            isSameTable,
            isMultiTable
        });
    };

    const tableCellRng = (start, end) => ({
        start,
        end,
    });
    const tableSelection = (rng, table, cells) => ({
        rng,
        table,
        cells
    });
    const deleteAction = Adt.generate([
        { singleCellTable: ['rng', 'cell'] },
        { fullTable: ['table'] },
        { partialTable: ['cells', 'outsideDetails'] },
        { multiTable: ['startTableCells', 'endTableCells', 'betweenRng'] },
    ]);
    const getClosestCell$1 = (container, isRoot) => closest$4(SugarElement.fromDom(container), 'td,th', isRoot);
    const isExpandedCellRng = (cellRng) => !eq(cellRng.start, cellRng.end);
    const getTableFromCellRng = (cellRng, isRoot) => getClosestTable(cellRng.start, isRoot)
        .bind((startParentTable) => getClosestTable(cellRng.end, isRoot)
        .bind((endParentTable) => someIf(eq(startParentTable, endParentTable), startParentTable)));
    const isSingleCellTable = (cellRng, isRoot) => !isExpandedCellRng(cellRng) &&
        getTableFromCellRng(cellRng, isRoot).exists((table) => {
            const rows = table.dom.rows;
            return rows.length === 1 && rows[0].cells.length === 1;
        });
    const getCellRng = (rng, isRoot) => {
        const startCell = getClosestCell$1(rng.startContainer, isRoot);
        const endCell = getClosestCell$1(rng.endContainer, isRoot);
        return lift2(startCell, endCell, tableCellRng);
    };
    const getCellRangeFromStartTable = (isRoot) => (startCell) => getClosestTable(startCell, isRoot).bind((table) => last$2(getTableCells(table)).map((endCell) => tableCellRng(startCell, endCell)));
    const getCellRangeFromEndTable = (isRoot) => (endCell) => getClosestTable(endCell, isRoot).bind((table) => head(getTableCells(table)).map((startCell) => tableCellRng(startCell, endCell)));
    const getTableSelectionFromCellRng = (isRoot) => (cellRng) => getTableFromCellRng(cellRng, isRoot).map((table) => tableSelection(cellRng, table, getTableCells(table)));
    const getTableSelections = (cellRng, selectionDetails, rng, isRoot) => {
        if (rng.collapsed || !cellRng.forall(isExpandedCellRng)) {
            return Optional.none();
        }
        else if (selectionDetails.isSameTable) {
            const sameTableSelection = cellRng.bind(getTableSelectionFromCellRng(isRoot));
            return Optional.some({
                start: sameTableSelection,
                end: sameTableSelection
            });
        }
        else {
            // Covers partial table selection (either start or end will have a tableSelection) and multitable selection (both start and end will have a tableSelection)
            const startCell = getClosestCell$1(rng.startContainer, isRoot);
            const endCell = getClosestCell$1(rng.endContainer, isRoot);
            const startTableSelection = startCell
                .bind(getCellRangeFromStartTable(isRoot))
                .bind(getTableSelectionFromCellRng(isRoot));
            const endTableSelection = endCell
                .bind(getCellRangeFromEndTable(isRoot))
                .bind(getTableSelectionFromCellRng(isRoot));
            return Optional.some({
                start: startTableSelection,
                end: endTableSelection
            });
        }
    };
    const getCellIndex = (cells, cell) => findIndex$2(cells, (x) => eq(x, cell));
    const getSelectedCells = (tableSelection) => lift2(getCellIndex(tableSelection.cells, tableSelection.rng.start), getCellIndex(tableSelection.cells, tableSelection.rng.end), (startIndex, endIndex) => tableSelection.cells.slice(startIndex, endIndex + 1));
    const isSingleCellTableContentSelected = (optCellRng, rng, isRoot) => optCellRng.exists((cellRng) => isSingleCellTable(cellRng, isRoot) && hasAllContentsSelected(cellRng.start, rng));
    const unselectCells = (rng, selectionDetails) => {
        const { startTable, endTable } = selectionDetails;
        const otherContentRng = rng.cloneRange();
        // If the table is some, it should be unselected (works for single table and multitable cases)
        startTable.each((table) => otherContentRng.setStartAfter(table.dom));
        endTable.each((table) => otherContentRng.setEndBefore(table.dom));
        return otherContentRng;
    };
    const handleSingleTable = (cellRng, selectionDetails, rng, isRoot) => getTableSelections(cellRng, selectionDetails, rng, isRoot)
        .bind(({ start, end }) => start.or(end))
        .bind((tableSelection) => {
        const { isSameTable } = selectionDetails;
        const selectedCells = getSelectedCells(tableSelection).getOr([]);
        if (isSameTable && tableSelection.cells.length === selectedCells.length) {
            return Optional.some(deleteAction.fullTable(tableSelection.table));
        }
        else if (selectedCells.length > 0) {
            if (isSameTable) {
                return Optional.some(deleteAction.partialTable(selectedCells, Optional.none()));
            }
            else {
                const otherContentRng = unselectCells(rng, selectionDetails);
                return Optional.some(deleteAction.partialTable(selectedCells, Optional.some({
                    ...selectionDetails,
                    rng: otherContentRng
                })));
            }
        }
        else {
            return Optional.none();
        }
    });
    const handleMultiTable = (cellRng, selectionDetails, rng, isRoot) => getTableSelections(cellRng, selectionDetails, rng, isRoot)
        .bind(({ start, end }) => {
        const startTableSelectedCells = start.bind(getSelectedCells).getOr([]);
        const endTableSelectedCells = end.bind(getSelectedCells).getOr([]);
        if (startTableSelectedCells.length > 0 && endTableSelectedCells.length > 0) {
            const otherContentRng = unselectCells(rng, selectionDetails);
            return Optional.some(deleteAction.multiTable(startTableSelectedCells, endTableSelectedCells, otherContentRng));
        }
        else {
            return Optional.none();
        }
    });
    const getActionFromRange = (root, rng) => {
        const isRoot = isRootFromElement(root);
        const optCellRng = getCellRng(rng, isRoot);
        const selectionDetails = getTableDetailsFromRange(rng, isRoot);
        if (isSingleCellTableContentSelected(optCellRng, rng, isRoot)) {
            // SingleCellTable
            return optCellRng.map((cellRng) => deleteAction.singleCellTable(rng, cellRng.start));
        }
        else if (selectionDetails.isMultiTable) {
            // MultiTable
            return handleMultiTable(optCellRng, selectionDetails, rng, isRoot);
        }
        else {
            // FullTable, PartialTable with no rng or PartialTable with outside rng
            return handleSingleTable(optCellRng, selectionDetails, rng, isRoot);
        }
    };

    // Reset the contenteditable state and fill the content with a padding br
    const cleanCells = (cells) => each$e(cells, (cell) => {
        remove$9(cell, 'contenteditable');
        fillWithPaddingBr(cell);
    });
    const getOutsideBlock = (editor, container) => Optional.from(editor.dom.getParent(container, editor.dom.isBlock)).map(SugarElement.fromDom);
    const handleEmptyBlock = (editor, startInTable, emptyBlock) => {
        emptyBlock.each((block) => {
            if (startInTable) {
                // Note that we don't need to set the selection as it'll be within the table
                remove$8(block);
            }
            else {
                // Set the cursor location as it'll move when filling with padding
                fillWithPaddingBr(block);
                editor.selection.setCursorLocation(block.dom, 0);
            }
        });
    };
    const deleteContentInsideCell = (editor, cell, rng, isFirstCellInSelection) => {
        const insideTableRng = rng.cloneRange();
        if (isFirstCellInSelection) {
            insideTableRng.setStart(rng.startContainer, rng.startOffset);
            insideTableRng.setEndAfter(cell.dom.lastChild);
        }
        else {
            insideTableRng.setStartBefore(cell.dom.firstChild);
            insideTableRng.setEnd(rng.endContainer, rng.endOffset);
        }
        deleteCellContents(editor, insideTableRng, cell, false).each((action) => action());
    };
    const collapseAndRestoreCellSelection = (editor) => {
        const selectedCells = getCellsFromEditor(editor);
        const selectedNode = SugarElement.fromDom(editor.selection.getNode());
        if (isTableCell$3(selectedNode.dom) && isEmpty$4(editor.schema, selectedNode)) {
            editor.selection.setCursorLocation(selectedNode.dom, 0);
        }
        else {
            editor.selection.collapse(true);
        }
        // Restore the data-mce-selected attribute if multiple cells were selected, as if it was a cef element
        // then selection overrides would remove it as it was using an offscreen selection clone.
        if (selectedCells.length > 1 && exists(selectedCells, (cell) => eq(cell, selectedNode))) {
            set$4(selectedNode, 'data-mce-selected', '1');
        }
    };
    /*
     * Runs when
     * - the start and end of the selection is contained within the same table (called directly from deleteRange)
     * - part of a table and content outside is selected
     */
    const emptySingleTableCells = (editor, cells, outsideDetails) => Optional.some(() => {
        const editorRng = editor.selection.getRng();
        const cellsToClean = outsideDetails.bind(({ rng, isStartInTable }) => {
            /*
             * Delete all content outside of the table that is in the selection
             * - Get the outside block before deleting the contents
             * - Delete the contents outside
             * - Handle the block outside the table if it is empty since rng.deleteContents leaves it
             */
            const outsideBlock = getOutsideBlock(editor, isStartInTable ? rng.endContainer : rng.startContainer);
            rng.deleteContents();
            handleEmptyBlock(editor, isStartInTable, outsideBlock.filter(curry(isEmpty$4, editor.schema)));
            /*
             * The only time we can have only part of the cell contents selected is when part of the selection
             * is outside the table (otherwise we use the Darwin fake selection, which always selects entire cells),
             * in which case we need to delete the contents inside and check if the entire contents of the cell have been deleted.
             *
             * Note: The endPointCell is the only cell which may have only part of its contents selected.
             */
            const endPointCell = isStartInTable ? cells[0] : cells[cells.length - 1];
            deleteContentInsideCell(editor, endPointCell, editorRng, isStartInTable);
            if (!isEmpty$4(editor.schema, endPointCell)) {
                return Optional.some(isStartInTable ? cells.slice(1) : cells.slice(0, -1));
            }
            else {
                return Optional.none();
            }
        }).getOr(cells);
        // Remove content from cells we need to clean
        cleanCells(cellsToClean);
        // Collapse the original selection after deleting everything
        collapseAndRestoreCellSelection(editor);
    });
    /*
     * Runs when the start of the selection is in a table and the end of the selection is in another table
     */
    const emptyMultiTableCells = (editor, startTableCells, endTableCells, betweenRng) => Optional.some(() => {
        const rng = editor.selection.getRng();
        const startCell = startTableCells[0];
        const endCell = endTableCells[endTableCells.length - 1];
        deleteContentInsideCell(editor, startCell, rng, true);
        deleteContentInsideCell(editor, endCell, rng, false);
        // Only clean empty cells, the first and last cells have the potential to still have content
        const startTableCellsToClean = isEmpty$4(editor.schema, startCell) ? startTableCells : startTableCells.slice(1);
        const endTableCellsToClean = isEmpty$4(editor.schema, endCell) ? endTableCells : endTableCells.slice(0, -1);
        cleanCells(startTableCellsToClean.concat(endTableCellsToClean));
        // Delete all content in between the start table and end table
        betweenRng.deleteContents();
        // This will collapse the selection into the cell of the start table
        collapseAndRestoreCellSelection(editor);
    });
    // Delete the contents of a range inside a cell. Runs on tables that are a single cell or partial selections that need to be cleaned up.
    const deleteCellContents = (editor, rng, cell, moveSelection = true) => Optional.some(() => {
        deleteRangeContents(editor, rng, cell, moveSelection);
    });
    const deleteTableElement = (editor, table) => Optional.some(() => deleteElement$2(editor, false, table));
    const deleteCellRange = (editor, rootElm, rng) => getActionFromRange(rootElm, rng)
        .bind((action) => action.fold(curry(deleteCellContents, editor), curry(deleteTableElement, editor), curry(emptySingleTableCells, editor), curry(emptyMultiTableCells, editor)));
    const deleteCaptionRange = (editor, caption) => emptyElement(editor, caption);
    const deleteTableRange = (editor, rootElm, rng, startElm) => getParentCaption(rootElm, startElm).fold(() => deleteCellRange(editor, rootElm, rng), (caption) => deleteCaptionRange(editor, caption));
    const deleteRange$4 = (editor, startElm, selectedCells) => {
        const rootNode = SugarElement.fromDom(editor.getBody());
        const rng = editor.selection.getRng();
        return selectedCells.length !== 0 ?
            emptySingleTableCells(editor, selectedCells, Optional.none()) :
            deleteTableRange(editor, rootNode, rng, startElm);
    };
    const getParentCell = (rootElm, elm) => find$2(parentsAndSelf(elm, rootElm), isTableCell$2);
    const getParentCaption = (rootElm, elm) => find$2(parentsAndSelf(elm, rootElm), isTag('caption'));
    const deleteBetweenCells = (editor, rootElm, forward, fromCell, from) => 
    // TODO: TINY-8865 - This may not be safe to cast as Node below and alternative solutions need to be looked into
    navigate(forward, editor.getBody(), from)
        .bind((to) => getParentCell(rootElm, SugarElement.fromDom(to.getNode()))
        .bind((toCell) => eq(toCell, fromCell) ? Optional.none() : Optional.some(noop)));
    const emptyElement = (editor, elm) => Optional.some(() => {
        fillWithPaddingBr(elm);
        editor.selection.setCursorLocation(elm.dom, 0);
    });
    const isDeleteOfLastCharPos = (fromCaption, forward, from, to) => firstPositionIn(fromCaption.dom).bind((first) => lastPositionIn(fromCaption.dom).map((last) => forward ?
        from.isEqual(first) && to.isEqual(last) :
        from.isEqual(last) && to.isEqual(first))).getOr(true);
    const emptyCaretCaption = (editor, elm) => emptyElement(editor, elm);
    const validateCaretCaption = (rootElm, fromCaption, to) => 
    // TODO: TINY-8865 - This may not be safe to cast as Node below and alternative solutions need to be looked into
    getParentCaption(rootElm, SugarElement.fromDom(to.getNode()))
        .fold(() => Optional.some(noop), (toCaption) => someIf(!eq(toCaption, fromCaption), noop));
    const deleteCaretInsideCaption = (editor, rootElm, forward, fromCaption, from) => navigate(forward, editor.getBody(), from).fold(() => Optional.some(noop), (to) => isDeleteOfLastCharPos(fromCaption, forward, from, to) ?
        emptyCaretCaption(editor, fromCaption) :
        validateCaretCaption(rootElm, fromCaption, to));
    const deleteCaretCells = (editor, forward, rootElm, startElm) => {
        const from = CaretPosition.fromRangeStart(editor.selection.getRng());
        return getParentCell(rootElm, startElm).bind((fromCell) => isEmpty$4(editor.schema, fromCell, { checkRootAsContent: false }) ?
            emptyElement(editor, fromCell) :
            deleteBetweenCells(editor, rootElm, forward, fromCell, from));
    };
    const deleteCaretCaption = (editor, forward, rootElm, fromCaption) => {
        const from = CaretPosition.fromRangeStart(editor.selection.getRng());
        return isEmpty$4(editor.schema, fromCaption) ?
            emptyElement(editor, fromCaption) :
            deleteCaretInsideCaption(editor, rootElm, forward, fromCaption, from);
    };
    const isNearTable = (forward, pos) => forward ? isBeforeTable(pos) : isAfterTable(pos);
    const isBeforeOrAfterTable = (editor, forward) => {
        const fromPos = CaretPosition.fromRangeStart(editor.selection.getRng());
        return isNearTable(forward, fromPos) || fromPosition(forward, editor.getBody(), fromPos)
            .exists((pos) => isNearTable(forward, pos));
    };
    const deleteCaret$3 = (editor, forward, startElm) => {
        const rootElm = SugarElement.fromDom(editor.getBody());
        return getParentCaption(rootElm, startElm).fold(() => deleteCaretCells(editor, forward, rootElm, startElm)
            .orThunk(() => someIf(isBeforeOrAfterTable(editor, forward), noop)), (fromCaption) => deleteCaretCaption(editor, forward, rootElm, fromCaption));
    };
    const backspaceDelete$d = (editor, forward) => {
        const startElm = SugarElement.fromDom(editor.selection.getStart(true));
        const cells = getCellsFromEditor(editor);
        return editor.selection.isCollapsed() && cells.length === 0 ?
            deleteCaret$3(editor, forward, startElm) :
            deleteRange$4(editor, startElm, cells);
    };

    const getContentEditableRoot$1 = (root, node) => {
        let tempNode = node;
        while (tempNode && tempNode !== root) {
            if (isContentEditableTrue$3(tempNode) || isContentEditableFalse$a(tempNode)) {
                return tempNode;
            }
            tempNode = tempNode.parentNode;
        }
        return null;
    };

    const internalAttributesPrefixes = [
        'data-ephox-',
        'data-mce-',
        'data-alloy-',
        'data-snooker-',
        '_'
    ];
    /**
     * Utility class for various element specific functions.
     *
     * @private
     * @class tinymce.dom.ElementUtils
     */
    const each$9 = Tools.each;
    const ElementUtils = (editor) => {
        const dom = editor.dom;
        const internalAttributes = new Set(editor.serializer.getTempAttrs());
        /**
         * Compares two nodes and checks if it's attributes and styles matches.
         * This doesn't compare classes as items since their order is significant.
         *
         * @method compare
         * @param {Node} node1 First node to compare with.
         * @param {Node} node2 Second node to compare with.
         * @return {Boolean} True/false if the nodes are the same or not.
         */
        const compare = (node1, node2) => {
            // Not the same name or type
            if (node1.nodeName !== node2.nodeName || node1.nodeType !== node2.nodeType) {
                return false;
            }
            /**
             * Returns all the nodes attributes excluding internal ones, styles and classes.
             *
             * @private
             * @param {Node} node Node to get attributes from.
             * @return {Object} Name/value object with attributes and attribute values.
             */
            const getAttribs = (node) => {
                const attribs = {};
                each$9(dom.getAttribs(node), (attr) => {
                    const name = attr.nodeName.toLowerCase();
                    // Don't compare internal attributes or style
                    if (name !== 'style' && !isAttributeInternal(name)) {
                        attribs[name] = dom.getAttrib(node, name);
                    }
                });
                return attribs;
            };
            /**
             * Compares two objects checks if it's key + value exists in the other one.
             *
             * @private
             * @param {Object} obj1 First object to compare.
             * @param {Object} obj2 Second object to compare.
             * @return {Boolean} True/false if the objects matches or not.
             */
            const compareObjects = (obj1, obj2) => {
                for (const name in obj1) {
                    // Obj1 has item obj2 doesn't have
                    if (has$2(obj1, name)) {
                        const value = obj2[name];
                        // Obj2 doesn't have obj1 item
                        if (isUndefined(value)) {
                            return false;
                        }
                        // Obj2 item has a different value
                        if (obj1[name] !== value) {
                            return false;
                        }
                        // Delete similar value
                        delete obj2[name];
                    }
                }
                // Check if obj 2 has something obj 1 doesn't have
                for (const name in obj2) {
                    // Obj2 has item obj1 doesn't have
                    if (has$2(obj2, name)) {
                        return false;
                    }
                }
                return true;
            };
            if (isElement$7(node1) && isElement$7(node2)) {
                // Attribs are not the same
                if (!compareObjects(getAttribs(node1), getAttribs(node2))) {
                    return false;
                }
                // Styles are not the same
                if (!compareObjects(dom.parseStyle(dom.getAttrib(node1, 'style')), dom.parseStyle(dom.getAttrib(node2, 'style')))) {
                    return false;
                }
            }
            return !isBookmarkNode$1(node1) && !isBookmarkNode$1(node2);
        };
        const isAttributeInternal = (attributeName) => exists(internalAttributesPrefixes, (value) => startsWith(attributeName, value)) || internalAttributes.has(attributeName);
        return {
            compare,
            isAttributeInternal
        };
    };

    const getNormalizedPoint$1 = (container, offset) => {
        if (isText$b(container)) {
            return { container, offset };
        }
        const node = RangeUtils.getNode(container, offset);
        if (isText$b(node)) {
            return {
                container: node,
                offset: offset >= container.childNodes.length ? node.data.length : 0
            };
        }
        else if (node.previousSibling && isText$b(node.previousSibling)) {
            return {
                container: node.previousSibling,
                offset: node.previousSibling.data.length
            };
        }
        else if (node.nextSibling && isText$b(node.nextSibling)) {
            return {
                container: node.nextSibling,
                offset: 0
            };
        }
        return { container, offset };
    };
    const normalizeRange$1 = (rng) => {
        const outRng = rng.cloneRange();
        const rangeStart = getNormalizedPoint$1(rng.startContainer, rng.startOffset);
        outRng.setStart(rangeStart.container, rangeStart.offset);
        const rangeEnd = getNormalizedPoint$1(rng.endContainer, rng.endOffset);
        outRng.setEnd(rangeEnd.container, rangeEnd.offset);
        return outRng;
    };

    const DOM$b = DOMUtils.DOM;
    const defaultMarker = () => DOM$b.create('span', { 'data-mce-type': 'bookmark' });
    const setupEndPoint = (container, offset, createMarker) => {
        if (isElement$7(container)) {
            const offsetNode = createMarker();
            if (container.hasChildNodes()) {
                if (offset === container.childNodes.length) {
                    container.appendChild(offsetNode);
                }
                else {
                    container.insertBefore(offsetNode, container.childNodes[offset]);
                }
            }
            else {
                container.appendChild(offsetNode);
            }
            return { container: offsetNode, offset: 0 };
        }
        else {
            return { container, offset };
        }
    };
    const restoreEndPoint = (container, offset) => {
        const nodeIndex = (container) => {
            let node = container.parentNode?.firstChild;
            let idx = 0;
            while (node) {
                if (node === container) {
                    return idx;
                }
                // Skip data-mce-type=bookmark nodes
                if (!isElement$7(node) || node.getAttribute('data-mce-type') !== 'bookmark') {
                    idx++;
                }
                node = node.nextSibling;
            }
            return -1;
        };
        if (isElement$7(container) && isNonNullable(container.parentNode)) {
            const node = container;
            offset = nodeIndex(container);
            container = container.parentNode;
            DOM$b.remove(node);
            if (!container.hasChildNodes() && DOM$b.isBlock(container)) {
                container.appendChild(DOM$b.create('br'));
            }
        }
        return { container, offset };
    };
    const createNormalizedRange = (startContainer, startOffset, endContainer, endOffset) => {
        const rng = DOM$b.createRng();
        rng.setStart(startContainer, startOffset);
        rng.setEnd(endContainer, endOffset);
        return normalizeRange$1(rng);
    };
    /**
     * Returns a range bookmark. This will convert indexed bookmarks into temporary span elements with
     * index 0 so that they can be restored properly after the DOM has been modified. Text bookmarks will not have spans
     * added to them since they can be restored after a dom operation.
     *
     * So this: <p><b>|</b><b>|</b></p>
     * becomes: <p><b><span data-mce-type="bookmark">|</span></b><b data-mce-type="bookmark">|</span></b></p>
     */
    const createBookmark = (rng, createMarker = defaultMarker) => {
        const { container: startContainer, offset: startOffset } = setupEndPoint(rng.startContainer, rng.startOffset, createMarker);
        if (rng.collapsed) {
            return { startContainer, startOffset };
        }
        else {
            const { container: endContainer, offset: endOffset } = setupEndPoint(rng.endContainer, rng.endOffset, createMarker);
            return { startContainer, startOffset, endContainer, endOffset };
        }
    };
    const resolveBookmark = (bookmark) => {
        const { container: startContainer, offset: startOffset } = restoreEndPoint(bookmark.startContainer, bookmark.startOffset);
        if (!isUndefined(bookmark.endContainer) && !isUndefined(bookmark.endOffset)) {
            const { container: endContainer, offset: endOffset } = restoreEndPoint(bookmark.endContainer, bookmark.endOffset);
            return createNormalizedRange(startContainer, startOffset, endContainer, endOffset);
        }
        else {
            return createNormalizedRange(startContainer, startOffset, startContainer, startOffset);
        }
    };

    const applyStyles = (dom, elm, format, vars) => {
        Tools.each(format.styles, (value, name) => {
            dom.setStyle(elm, name, replaceVars(value, vars));
        });
        // Needed for the WebKit span spam bug
        // TODO: Remove this once WebKit/Blink fixes this
        if (format.styles) {
            const styleVal = dom.getAttrib(elm, 'style');
            if (styleVal) {
                dom.setAttrib(elm, 'data-mce-style', styleVal);
            }
        }
    };
    const setElementFormat = (ed, elm, fmt, vars, node) => {
        const dom = ed.dom;
        if (isFunction(fmt.onformat)) {
            fmt.onformat(elm, fmt, vars, node);
        }
        applyStyles(dom, elm, fmt, vars);
        Tools.each(fmt.attributes, (value, name) => {
            dom.setAttrib(elm, name, replaceVars(value, vars));
        });
        Tools.each(fmt.classes, (value) => {
            const newValue = replaceVars(value, vars);
            if (!dom.hasClass(elm, newValue)) {
                dom.addClass(elm, newValue);
            }
        });
    };

    const isApplyFormat = (format) => !isArray$1(format.attributes) && !isArray$1(format.styles);

    const isEq$3 = isEq$5;
    const matchesUnInheritedFormatSelector = (ed, node, name) => {
        const formatList = ed.formatter.get(name);
        if (formatList) {
            for (let i = 0; i < formatList.length; i++) {
                const format = formatList[i];
                if (isSelectorFormat(format) && format.inherit === false && ed.dom.is(node, format.selector)) {
                    return true;
                }
            }
        }
        return false;
    };
    const matchParents = (editor, node, name, vars, similar) => {
        const root = editor.dom.getRoot();
        if (node === root) {
            return false;
        }
        // Find first node with similar format settings
        const matchedNode = editor.dom.getParent(node, (elm) => {
            if (matchesUnInheritedFormatSelector(editor, elm, name)) {
                return true;
            }
            return elm.parentNode === root || !!matchNode$1(editor, elm, name, vars, true);
        });
        // Do an exact check on the similar format element
        return !!matchNode$1(editor, matchedNode, name, vars, similar);
    };
    const matchName = (dom, node, format) => {
        // Check for inline match
        if (isInlineFormat(format) && isEq$3(node, format.inline)) {
            return true;
        }
        // Check for block match
        if (isBlockFormat(format) && isEq$3(node, format.block)) {
            return true;
        }
        // Check for selector match
        if (isSelectorFormat(format)) {
            return isElement$7(node) && dom.is(node, format.selector);
        }
        return false;
    };
    const matchItems = (dom, node, format, itemName, similar, vars) => {
        const items = format[itemName];
        const matchAttributes = itemName === 'attributes';
        // Custom match
        if (isFunction(format.onmatch)) {
            // onmatch is generic in a way that we can't really express without casting
            return format.onmatch(node, format, itemName);
        }
        // Check all items
        if (items) {
            // Non indexed object
            if (!isArrayLike(items)) {
                for (const key in items) {
                    if (has$2(items, key)) {
                        const value = matchAttributes ? dom.getAttrib(node, key) : getStyle(dom, node, key);
                        const expectedValue = replaceVars(items[key], vars);
                        const isEmptyValue = isNullable(value) || isEmpty$5(value);
                        if (isEmptyValue && isNullable(expectedValue)) {
                            continue;
                        }
                        if (similar && isEmptyValue && !format.exact) {
                            return false;
                        }
                        if ((!similar || format.exact) && !isEq$3(value, normalizeStyleValue(expectedValue, key))) {
                            return false;
                        }
                    }
                }
            }
            else {
                // Only one match needed for indexed arrays
                for (let i = 0; i < items.length; i++) {
                    if (matchAttributes ? dom.getAttrib(node, items[i]) : getStyle(dom, node, items[i])) {
                        return true;
                    }
                }
            }
        }
        return true;
    };
    const matchNode$1 = (ed, node, name, vars, similar) => {
        const formatList = ed.formatter.get(name);
        const dom = ed.dom;
        if (formatList && isElement$7(node)) {
            // Check each format in list
            for (let i = 0; i < formatList.length; i++) {
                const format = formatList[i];
                // Name name, attributes, styles and classes
                if (matchName(ed.dom, node, format) && matchItems(dom, node, format, 'attributes', similar, vars) && matchItems(dom, node, format, 'styles', similar, vars)) {
                    // Match classes
                    const classes = format.classes;
                    if (classes) {
                        for (let x = 0; x < classes.length; x++) {
                            if (!ed.dom.hasClass(node, replaceVars(classes[x], vars))) {
                                return;
                            }
                        }
                    }
                    return format;
                }
            }
        }
        return undefined;
    };
    const match$2 = (editor, name, vars, node, similar) => {
        // Check specified node
        if (node) {
            return matchParents(editor, node, name, vars, similar);
        }
        // Check selected node
        node = editor.selection.getNode();
        if (matchParents(editor, node, name, vars, similar)) {
            return true;
        }
        // Check start node if it's different
        const startNode = editor.selection.getStart();
        if (startNode !== node) {
            if (matchParents(editor, startNode, name, vars, similar)) {
                return true;
            }
        }
        return false;
    };
    const matchAll = (editor, names, vars) => {
        const matchedFormatNames = [];
        const checkedMap = {};
        // Check start of selection for formats
        const startElement = editor.selection.getStart();
        editor.dom.getParent(startElement, (node) => {
            for (let i = 0; i < names.length; i++) {
                const name = names[i];
                if (!checkedMap[name] && matchNode$1(editor, node, name, vars)) {
                    checkedMap[name] = true;
                    matchedFormatNames.push(name);
                }
            }
        }, editor.dom.getRoot());
        return matchedFormatNames;
    };
    const closest = (editor, names) => {
        const isRoot = (elm) => eq(elm, SugarElement.fromDom(editor.getBody()));
        const match = (elm, name) => matchNode$1(editor, elm.dom, name) ? Optional.some(name) : Optional.none();
        return Optional.from(editor.selection.getStart(true)).bind((rawElm) => closest$1(SugarElement.fromDom(rawElm), (elm) => findMap(names, (name) => match(elm, name)), isRoot)).getOrNull();
    };
    const canApply = (editor, name) => {
        const formatList = editor.formatter.get(name);
        const dom = editor.dom;
        if (formatList && editor.selection.isEditable()) {
            const startNode = editor.selection.getStart();
            const parents = getParents$2(dom, startNode);
            for (let x = formatList.length - 1; x >= 0; x--) {
                const format = formatList[x];
                // Format is not selector based then always return TRUE
                if (!isSelectorFormat(format)) {
                    return true;
                }
                for (let i = parents.length - 1; i >= 0; i--) {
                    if (dom.is(parents[i], format.selector)) {
                        return true;
                    }
                }
            }
        }
        return false;
    };
    /**
     *  Get all of the format names present on the specified node
     */
    const matchAllOnNode = (editor, node, formatNames) => foldl(formatNames, (acc, name) => {
        const matchSimilar = isVariableFormatName(editor, name);
        if (editor.formatter.matchNode(node, name, {}, matchSimilar)) {
            return acc.concat([name]);
        }
        else {
            return acc;
        }
    }, []);

    const ZWSP = ZWSP$1;
    const importNode = (ownerDocument, node) => {
        return ownerDocument.importNode(node, true);
    };
    const findFirstTextNode = (node) => {
        if (node) {
            const walker = new DomTreeWalker(node, node);
            for (let tempNode = walker.current(); tempNode; tempNode = walker.next()) {
                if (isText$b(tempNode)) {
                    return tempNode;
                }
            }
        }
        return null;
    };
    const createCaretContainer = (fill) => {
        const caretContainer = SugarElement.fromTag('span');
        setAll$1(caretContainer, {
            // style: 'color:red',
            'id': CARET_ID,
            'data-mce-bogus': '1',
            'data-mce-type': 'format-caret'
        });
        if (fill) {
            append$1(caretContainer, SugarElement.fromText(ZWSP));
        }
        return caretContainer;
    };
    const trimZwspFromCaretContainer = (caretContainerNode) => {
        const textNode = findFirstTextNode(caretContainerNode);
        if (textNode && textNode.data.charAt(0) === ZWSP) {
            textNode.deleteData(0, 1);
        }
        return textNode;
    };
    const removeCaretContainerNode = (editor, node, moveCaret) => {
        const dom = editor.dom, selection = editor.selection;
        if (isCaretContainerEmpty(node)) {
            deleteElement$2(editor, false, SugarElement.fromDom(node), moveCaret, true);
        }
        else {
            const rng = selection.getRng();
            const block = dom.getParent(node, dom.isBlock);
            // Store the current selection offsets
            const startContainer = rng.startContainer;
            const startOffset = rng.startOffset;
            const endContainer = rng.endContainer;
            const endOffset = rng.endOffset;
            const textNode = trimZwspFromCaretContainer(node);
            dom.remove(node, true);
            // Restore the selection after unwrapping the node and removing the zwsp
            if (startContainer === textNode && startOffset > 0) {
                rng.setStart(textNode, startOffset - 1);
            }
            if (endContainer === textNode && endOffset > 0) {
                rng.setEnd(textNode, endOffset - 1);
            }
            if (block && dom.isEmpty(block)) {
                fillWithPaddingBr(SugarElement.fromDom(block));
            }
            selection.setRng(rng);
        }
    };
    // Removes the caret container for the specified node or all on the current document
    const removeCaretContainer = (editor, node, moveCaret) => {
        const dom = editor.dom, selection = editor.selection;
        if (!node) {
            node = getParentCaretContainer(editor.getBody(), selection.getStart());
            if (!node) {
                while ((node = dom.get(CARET_ID))) {
                    removeCaretContainerNode(editor, node, moveCaret);
                }
            }
        }
        else {
            removeCaretContainerNode(editor, node, moveCaret);
        }
    };
    const insertCaretContainerNode = (editor, caretContainer, formatNode) => {
        const dom = editor.dom;
        const block = dom.getParent(formatNode, curry(isTextBlock$2, editor.schema));
        if (block && dom.isEmpty(block)) {
            // Replace formatNode with caretContainer when removing format from empty block like <p><b>|</b></p>
            formatNode.parentNode?.replaceChild(caretContainer, formatNode);
        }
        else {
            removeTrailingBr(SugarElement.fromDom(formatNode));
            if (dom.isEmpty(formatNode)) {
                formatNode.parentNode?.replaceChild(caretContainer, formatNode);
            }
            else {
                dom.insertAfter(caretContainer, formatNode);
            }
        }
    };
    const appendNode = (parentNode, node) => {
        parentNode.appendChild(node);
        return node;
    };
    const insertFormatNodesIntoCaretContainer = (formatNodes, caretContainer) => {
        const innerMostFormatNode = foldr(formatNodes, (parentNode, formatNode) => {
            return appendNode(parentNode, formatNode.cloneNode(false));
        }, caretContainer);
        const doc = innerMostFormatNode.ownerDocument ?? document;
        return appendNode(innerMostFormatNode, doc.createTextNode(ZWSP));
    };
    const cleanFormatNode = (editor, caretContainer, formatNode, name, vars, similar) => {
        const formatter = editor.formatter;
        const dom = editor.dom;
        // Find all formats present on the format node
        const validFormats = filter$5(keys(formatter.get()), (formatName) => formatName !== name && !contains$1(formatName, 'removeformat'));
        const matchedFormats = matchAllOnNode(editor, formatNode, validFormats);
        // Filter out any matched formats that are 'visually' equivalent to the 'name' format since they are not unique formats on the node
        const uniqueFormats = filter$5(matchedFormats, (fmtName) => !areSimilarFormats(editor, fmtName, name));
        // If more than one format is present, then there's additional formats that should be retained. So clone the node,
        // remove the format and then return cleaned format node
        if (uniqueFormats.length > 0) {
            const clonedFormatNode = formatNode.cloneNode(false);
            dom.add(caretContainer, clonedFormatNode);
            formatter.remove(name, vars, clonedFormatNode, similar);
            dom.remove(clonedFormatNode);
            return Optional.some(clonedFormatNode);
        }
        else {
            return Optional.none();
        }
    };
    const normalizeNbsps = (node) => set$1(node, get$4(node).replace(new RegExp(`${nbsp}$`), ' '));
    const normalizeNbspsBetween = (editor, caretContainer) => {
        const handler = () => {
            if (caretContainer !== null && !editor.dom.isEmpty(caretContainer)) {
                prevSibling(SugarElement.fromDom(caretContainer)).each((node) => {
                    if (isText$c(node)) {
                        normalizeNbsps(node);
                    }
                    else {
                        descendant$2(node, (e) => isText$c(e)).each((textNode) => {
                            if (isText$c(textNode)) {
                                normalizeNbsps(textNode);
                            }
                        });
                    }
                });
            }
        };
        editor.once('input', (e) => {
            if (e.data && !isWhiteSpace(e.data)) {
                if (!e.isComposing) {
                    handler();
                }
                else {
                    editor.once('compositionend', () => {
                        handler();
                    });
                }
            }
        });
    };
    const applyCaretFormat = (editor, name, vars) => {
        let caretContainer;
        const selection = editor.selection;
        const formatList = editor.formatter.get(name);
        if (!formatList) {
            return;
        }
        const selectionRng = selection.getRng();
        let offset = selectionRng.startOffset;
        const container = selectionRng.startContainer;
        const text = container.nodeValue;
        caretContainer = getParentCaretContainer(editor.getBody(), selection.getStart());
        // Expand to word if caret is in the middle of a text node and the char before/after is a alpha numeric character
        const wordcharRegex = /[^\s\u00a0\u00ad\u200b\ufeff]/;
        if (text && offset > 0 && offset < text.length &&
            wordcharRegex.test(text.charAt(offset)) && wordcharRegex.test(text.charAt(offset - 1))) {
            // Get bookmark of caret position
            const bookmark = selection.getBookmark();
            // Collapse bookmark range (WebKit)
            selectionRng.collapse(true);
            // Expand the range to the closest word and split it at those points
            let rng = expandRng(editor.dom, selectionRng, formatList);
            rng = split(rng);
            // Apply the format to the range
            editor.formatter.apply(name, vars, rng);
            // Move selection back to caret position
            selection.moveToBookmark(bookmark);
        }
        else {
            let textNode = caretContainer ? findFirstTextNode(caretContainer) : null;
            if (!caretContainer || textNode?.data !== ZWSP) {
                // Need to import the node into the document on IE or we get a lovely WrongDocument exception
                caretContainer = importNode(editor.getDoc(), createCaretContainer(true).dom);
                textNode = caretContainer.firstChild;
                selectionRng.insertNode(caretContainer);
                offset = 1;
                normalizeNbspsBetween(editor, caretContainer);
                editor.formatter.apply(name, vars, caretContainer);
            }
            else {
                editor.formatter.apply(name, vars, caretContainer);
            }
            // Move selection to text node
            selection.setCursorLocation(textNode, offset);
        }
    };
    const removeCaretFormat = (editor, name, vars, similar) => {
        const dom = editor.dom;
        const selection = editor.selection;
        let hasContentAfter = false;
        const formatList = editor.formatter.get(name);
        if (!formatList) {
            return;
        }
        const rng = selection.getRng();
        const container = rng.startContainer;
        const offset = rng.startOffset;
        let node = container;
        if (isText$b(container)) {
            if (offset !== container.data.length) {
                hasContentAfter = true;
            }
            node = node.parentNode;
        }
        const parents = [];
        let formatNode;
        while (node) {
            if (matchNode$1(editor, node, name, vars, similar)) {
                formatNode = node;
                break;
            }
            if (node.nextSibling) {
                hasContentAfter = true;
            }
            parents.push(node);
            node = node.parentNode;
        }
        // Node doesn't have the specified format
        if (!formatNode) {
            return;
        }
        // Is there contents after the caret then remove the format on the element
        if (hasContentAfter) {
            const bookmark = selection.getBookmark();
            // Collapse bookmark range (WebKit)
            rng.collapse(true);
            // Expand the range to the closest word and split it at those points
            let expandedRng = expandRng(dom, rng, formatList, { includeTrailingSpace: true });
            expandedRng = split(expandedRng);
            // TODO: Figure out how on earth this works, as it shouldn't since remove format
            //  definitely seems to require an actual Range
            editor.formatter.remove(name, vars, expandedRng, similar);
            selection.moveToBookmark(bookmark);
        }
        else {
            const caretContainer = getParentCaretContainer(editor.getBody(), formatNode);
            const parentsAfter = isNonNullable(caretContainer) ? dom.getParents(formatNode.parentNode, always, caretContainer) : [];
            const newCaretContainer = createCaretContainer(false).dom;
            insertCaretContainerNode(editor, newCaretContainer, caretContainer ?? formatNode);
            const cleanedFormatNode = cleanFormatNode(editor, newCaretContainer, formatNode, name, vars, similar);
            const caretTextNode = insertFormatNodesIntoCaretContainer([
                ...parents,
                ...cleanedFormatNode.toArray(),
                ...parentsAfter
            ], newCaretContainer);
            if (caretContainer) {
                removeCaretContainerNode(editor, caretContainer, isNonNullable(caretContainer));
            }
            selection.setCursorLocation(caretTextNode, 1);
            normalizeNbspsBetween(editor, newCaretContainer);
            if (dom.isEmpty(formatNode)) {
                dom.remove(formatNode);
            }
        }
    };
    const disableCaretContainer = (editor, keyCode, moveCaret) => {
        const selection = editor.selection, body = editor.getBody();
        removeCaretContainer(editor, null, moveCaret);
        // Remove caret container if it's empty
        if ((keyCode === 8 || keyCode === 46) && selection.isCollapsed() && selection.getStart().innerHTML === ZWSP) {
            removeCaretContainer(editor, getParentCaretContainer(body, selection.getStart()), true);
        }
        // Remove caret container on keydown and it's left/right arrow keys
        if (keyCode === 37 || keyCode === 39) {
            removeCaretContainer(editor, getParentCaretContainer(body, selection.getStart()), true);
        }
    };
    const endsWithNbsp = (element) => isText$b(element) && endsWith(element.data, nbsp);
    const setup$B = (editor) => {
        editor.on('mouseup keydown', (e) => {
            disableCaretContainer(editor, e.keyCode, endsWithNbsp(editor.selection.getRng().endContainer));
        });
    };
    const createCaretFormat = (formatNodes) => {
        const caretContainer = createCaretContainer(false);
        const innerMost = insertFormatNodesIntoCaretContainer(formatNodes, caretContainer.dom);
        return { caretContainer, caretPosition: CaretPosition(innerMost, 0) };
    };
    const replaceWithCaretFormat = (targetNode, formatNodes) => {
        const { caretContainer, caretPosition } = createCaretFormat(formatNodes);
        before$4(SugarElement.fromDom(targetNode), caretContainer);
        remove$8(SugarElement.fromDom(targetNode));
        return caretPosition;
    };
    const createCaretFormatAtStart$1 = (rng, formatNodes) => {
        const { caretContainer, caretPosition } = createCaretFormat(formatNodes);
        rng.insertNode(caretContainer.dom);
        return caretPosition;
    };
    const isFormatElement = (editor, element) => {
        if (isCaretNode(element.dom)) {
            return false;
        }
        const inlineElements = editor.schema.getTextInlineElements();
        return has$2(inlineElements, name(element)) && !isCaretNode(element.dom) && !isBogus$1(element.dom);
    };

    const listItemStyles = ['fontWeight', 'fontStyle', 'color', 'fontSize', 'fontFamily'];
    const hasListStyles = (fmt) => isObject(fmt.styles) && exists(keys(fmt.styles), (name) => contains$2(listItemStyles, name));
    const findExpandedListItemFormat = (formats) => find$2(formats, (fmt) => isInlineFormat(fmt) && fmt.inline === 'span' && hasListStyles(fmt));
    const getExpandedListItemFormat = (formatter, format) => {
        const formatList = formatter.get(format);
        return isArray$1(formatList) ? findExpandedListItemFormat(formatList) : Optional.none();
    };
    const isRngStartAtStartOfElement = (rng, elm) => prevPosition(elm, CaretPosition.fromRangeStart(rng)).isNone();
    const isRngEndAtEndOfElement = (rng, elm) => {
        return nextPosition(elm, CaretPosition.fromRangeEnd(rng))
            .exists((pos) => !isBr$7(pos.getNode()) || nextPosition(elm, pos).isSome()) === false;
    };
    const isEditableListItem = (dom) => (elm) => isListItem$3(elm) && dom.isEditable(elm);
    // TINY-13197: If the content is wrapped inside a block element, the first block returned by getSelectedBlocks() is not LI, even when the content is fully selected.
    // However, the second and subsequent do return LI as the selected block so only the first block needs to be adjusted
    const getAndOnlyNormalizeFirstBlockIf = (selection, pred) => map$3(selection.getSelectedBlocks(), (block, i) => {
        if (i === 0 && pred(block)) {
            return selection.dom.getParent(block, isListItem$3) ?? block;
        }
        else {
            return block;
        }
    });
    const getFullySelectedBlocks = (selection) => {
        if (selection.isCollapsed()) {
            return [];
        }
        const rng = selection.getRng();
        const blocks = getAndOnlyNormalizeFirstBlockIf(selection, (el) => isRngStartAtStartOfElement(rng, el) && !isListItem$3(el));
        if (blocks.length === 1) {
            return isRngStartAtStartOfElement(rng, blocks[0]) && isRngEndAtEndOfElement(rng, blocks[0]) ? blocks : [];
        }
        else {
            const first = head(blocks).filter((elm) => isRngStartAtStartOfElement(rng, elm)).toArray();
            const last = last$2(blocks).filter((elm) => isRngEndAtEndOfElement(rng, elm)).toArray();
            const middle = blocks.slice(1, -1);
            return first.concat(middle).concat(last);
        }
    };
    const getFullySelectedListItems = (selection) => filter$5(getFullySelectedBlocks(selection), isEditableListItem(selection.dom));
    const getPartiallySelectedListItems = (selection) => filter$5(getAndOnlyNormalizeFirstBlockIf(selection, (el) => !isListItem$3(el)), isEditableListItem(selection.dom));

    const each$8 = Tools.each;
    const isElementNode = (node) => isElement$7(node) && !isBookmarkNode$1(node) && !isCaretNode(node) && !isBogus$1(node);
    const findElementSibling = (node, siblingName) => {
        for (let sibling = node; sibling; sibling = sibling[siblingName]) {
            if (isText$b(sibling) && isNotEmpty(sibling.data)) {
                return node;
            }
            if (isElement$7(sibling) && !isBookmarkNode$1(sibling)) {
                return sibling;
            }
        }
        return node;
    };
    const mergeSiblingsNodes = (editor, prev, next) => {
        const elementUtils = ElementUtils(editor);
        const isPrevEditable = isHTMLElement(prev) && editor.dom.isEditable(prev);
        const isNextEditable = isHTMLElement(next) && editor.dom.isEditable(next);
        // Check if next/prev exists and that they are elements
        if (isPrevEditable && isNextEditable) {
            // If previous sibling is empty then jump over it
            const prevSibling = findElementSibling(prev, 'previousSibling');
            const nextSibling = findElementSibling(next, 'nextSibling');
            // Compare next and previous nodes
            if (elementUtils.compare(prevSibling, nextSibling)) {
                // Append nodes between
                for (let sibling = prevSibling.nextSibling; sibling && sibling !== nextSibling;) {
                    const tmpSibling = sibling;
                    sibling = sibling.nextSibling;
                    prevSibling.appendChild(tmpSibling);
                }
                editor.dom.remove(nextSibling);
                Tools.each(Tools.grep(nextSibling.childNodes), (node) => {
                    prevSibling.appendChild(node);
                });
                return prevSibling;
            }
        }
        return next;
    };
    const mergeSiblings = (editor, format, vars, node) => {
        // Merge next and previous siblings if they are similar <b>text</b><b>text</b> becomes <b>texttext</b>
        // Note: mergeSiblingNodes attempts to not merge sibilings if they are noneditable
        if (node && format.merge_siblings !== false) {
            // Previous sibling
            const newNode = mergeSiblingsNodes(editor, getNonWhiteSpaceSibling(node), node) ?? node;
            // Next sibling
            mergeSiblingsNodes(editor, newNode, getNonWhiteSpaceSibling(newNode, true));
        }
    };
    const clearChildStyles = (dom, format, node) => {
        if (format.clear_child_styles) {
            const selector = format.links ? '*:not(a)' : '*';
            each$8(dom.select(selector, node), (childNode) => {
                if (isElementNode(childNode) && dom.isEditable(childNode)) {
                    each$8(format.styles, (_value, name) => {
                        dom.setStyle(childNode, name, '');
                    });
                }
            });
        }
    };
    const processChildElements = (node, filter, process) => {
        each$8(node.childNodes, (node) => {
            if (isElementNode(node)) {
                if (filter(node)) {
                    process(node);
                }
                if (node.hasChildNodes()) {
                    processChildElements(node, filter, process);
                }
            }
        });
    };
    const unwrapEmptySpan = (dom, node) => {
        if (node.nodeName === 'SPAN' && dom.getAttribs(node).length === 0) {
            dom.remove(node, true);
        }
    };
    const hasStyle = (dom, name) => (node) => !!(node && getStyle(dom, node, name));
    const applyStyle = (dom, name, value) => (node) => {
        dom.setStyle(node, name, value);
        if (node.getAttribute('style') === '') {
            node.removeAttribute('style');
        }
        unwrapEmptySpan(dom, node);
    };

    const removeResult = Adt.generate([
        { keep: [] },
        { rename: ['name'] },
        { removed: [] }
    ]);
    const MCE_ATTR_RE = /^(src|href|style)$/;
    const each$7 = Tools.each;
    const isEq$2 = isEq$5;
    const isTableCellOrRow = (node) => /^(TR|TH|TD)$/.test(node.nodeName);
    const isChildOfInlineParent = (dom, node, parent) => dom.isChildOf(node, parent) && node !== parent && !dom.isBlock(parent);
    const getContainer = (ed, rng, start) => {
        let container = rng[start ? 'startContainer' : 'endContainer'];
        let offset = rng[start ? 'startOffset' : 'endOffset'];
        if (isElement$7(container)) {
            const lastIdx = container.childNodes.length - 1;
            if (!start && offset) {
                offset--;
            }
            container = container.childNodes[offset > lastIdx ? lastIdx : offset];
        }
        // If start text node is excluded then walk to the next node
        if (isText$b(container) && start && offset >= container.data.length) {
            container = new DomTreeWalker(container, ed.getBody()).next() || container;
        }
        // If end text node is excluded then walk to the previous node
        if (isText$b(container) && !start && offset === 0) {
            container = new DomTreeWalker(container, ed.getBody()).prev() || container;
        }
        return container;
    };
    const normalizeTableSelection = (node, start) => {
        const prop = start ? 'firstChild' : 'lastChild';
        const childNode = node[prop];
        if (isTableCellOrRow(node) && childNode) {
            if (node.nodeName === 'TR') {
                return childNode[prop] || childNode;
            }
            else {
                return childNode;
            }
        }
        return node;
    };
    const wrap$1 = (dom, node, name, attrs) => {
        const wrapper = dom.create(name, attrs);
        node.parentNode?.insertBefore(wrapper, node);
        wrapper.appendChild(node);
        return wrapper;
    };
    const wrapWithSiblings = (dom, node, next, name, attrs) => {
        const start = SugarElement.fromDom(node);
        const wrapper = SugarElement.fromDom(dom.create(name, attrs));
        const siblings = next ? nextSiblings(start) : prevSiblings(start);
        append(wrapper, siblings);
        if (next) {
            before$4(start, wrapper);
            prepend(wrapper, start);
        }
        else {
            after$4(start, wrapper);
            append$1(wrapper, start);
        }
        return wrapper.dom;
    };
    const isColorFormatAndAnchor = (node, format) => format.links && node.nodeName === 'A';
    /**
     * Removes the node and wrap it's children in paragraphs before doing so or
     * appends BR elements to the beginning/end of the block element if forcedRootBlocks is disabled.
     *
     * If the div in the node below gets removed:
     *  text<div>text</div>text
     *
     * Output becomes:
     *  text<div><br />text<br /></div>text
     *
     * So when the div is removed the result is:
     *  text<br />text<br />text
     *
     * @private
     * @param {Node} node Node to remove + apply BR/P elements to.
     * @param {Object} format Format rule.
     * @return {Node} Input node.
     */
    const removeNode = (ed, node, format) => {
        const parentNode = node.parentNode;
        let rootBlockElm;
        const dom = ed.dom;
        const forcedRootBlock = getForcedRootBlock(ed);
        if (isBlockFormat(format)) {
            // Wrap the block in a forcedRootBlock if we are at the root of document
            if (parentNode === dom.getRoot()) {
                if (!format.list_block || !isEq$2(node, format.list_block)) {
                    each$e(from(node.childNodes), (node) => {
                        if (isValid(ed, forcedRootBlock, node.nodeName.toLowerCase())) {
                            if (!rootBlockElm) {
                                rootBlockElm = wrap$1(dom, node, forcedRootBlock);
                                dom.setAttribs(rootBlockElm, getForcedRootBlockAttrs(ed));
                            }
                            else {
                                rootBlockElm.appendChild(node);
                            }
                        }
                        else {
                            rootBlockElm = null;
                        }
                    });
                }
            }
        }
        // Never remove nodes that aren't the specified inline element if a selector is specified too
        if (isMixedFormat(format) && !isEq$2(format.inline, node)) {
            return;
        }
        dom.remove(node, true);
    };
    // Attributes or styles can be either an array of names or an object containing name/value pairs
    const processFormatAttrOrStyle = (name, value, vars) => {
        // Indexed array
        if (isNumber(name)) {
            return {
                name: value,
                value: null
            };
        }
        else {
            return {
                name,
                value: replaceVars(value, vars)
            };
        }
    };
    const removeEmptyStyleAttributeIfNeeded = (dom, elm) => {
        if (dom.getAttrib(elm, 'style') === '') {
            elm.removeAttribute('style');
            elm.removeAttribute('data-mce-style');
        }
    };
    const removeStyles$1 = (dom, elm, format, vars, compareNode) => {
        let stylesModified = false;
        each$7(format.styles, (value, name) => {
            const { name: styleName, value: styleValue } = processFormatAttrOrStyle(name, value, vars);
            const normalizedStyleValue = normalizeStyleValue(styleValue, styleName);
            if (format.remove_similar || isNull(styleValue) || !isElement$7(compareNode) || isEq$2(getStyle(dom, compareNode, styleName), normalizedStyleValue)) {
                dom.setStyle(elm, styleName, '');
            }
            stylesModified = true;
        });
        if (stylesModified) {
            removeEmptyStyleAttributeIfNeeded(dom, elm);
        }
    };
    const removeListStyleFormats = (editor, name, vars) => {
        if (name === 'removeformat') {
            each$e(getPartiallySelectedListItems(editor.selection), (li) => {
                each$e(listItemStyles, (name) => editor.dom.setStyle(li, name, ''));
                removeEmptyStyleAttributeIfNeeded(editor.dom, li);
            });
        }
        else {
            getExpandedListItemFormat(editor.formatter, name).each((liFmt) => {
                each$e(getPartiallySelectedListItems(editor.selection), (li) => removeStyles$1(editor.dom, li, liFmt, vars, null));
            });
        }
    };
    const removeNodeFormatInternal = (ed, format, vars, node, compareNode) => {
        const dom = ed.dom;
        const elementUtils = ElementUtils(ed);
        const schema = ed.schema;
        // Root level block transparents should get converted into regular text blocks
        if (isInlineFormat(format) && isTransparentElementName(schema, format.inline) && isTransparentBlock(schema, node) && node.parentElement === ed.getBody()) {
            removeNode(ed, node, format);
            return removeResult.removed();
        }
        // Check if node is noneditable and can have the format removed from it
        if (!format.ceFalseOverride && node && dom.getContentEditableParent(node) === 'false') {
            return removeResult.keep();
        }
        // Check if node matches format
        if (node && !matchName(dom, node, format) && !isColorFormatAndAnchor(node, format)) {
            return removeResult.keep();
        }
        // "matchName" will made sure we're dealing with an element, so cast as one
        const elm = node;
        // Applies to styling elements like strong, em, i, u, etc. so that if they have styling attributes, the attributes can be kept but the styling element is removed
        const preserveAttributes = format.preserve_attributes;
        if (isInlineFormat(format) && format.remove === 'all' && isArray$1(preserveAttributes)) {
            // Remove all attributes except for the attributes specified in preserve_attributes
            const attrsToPreserve = filter$5(dom.getAttribs(elm), (attr) => contains$2(preserveAttributes, attr.name.toLowerCase()));
            dom.removeAllAttribs(elm);
            each$e(attrsToPreserve, (attr) => dom.setAttrib(elm, attr.name, attr.value));
            // Note: If there are no attributes left, the element will be removed as normal at the end of the function
            if (attrsToPreserve.length > 0) {
                // Convert inline element to span if necessary
                return removeResult.rename('span');
            }
        }
        // Should we compare with format attribs and styles
        if (format.remove !== 'all') {
            removeStyles$1(dom, elm, format, vars, compareNode);
            // Remove attributes
            each$7(format.attributes, (value, name) => {
                const { name: attrName, value: attrValue } = processFormatAttrOrStyle(name, value, vars);
                if (format.remove_similar || isNull(attrValue) || !isElement$7(compareNode) || isEq$2(dom.getAttrib(compareNode, attrName), attrValue)) {
                    // Keep internal classes
                    if (attrName === 'class') {
                        const currentValue = dom.getAttrib(elm, attrName);
                        if (currentValue) {
                            // Build new class value where everything is removed except the internal prefixed classes
                            let valueOut = '';
                            each$e(currentValue.split(/\s+/), (cls) => {
                                if (/mce\-\w+/.test(cls)) {
                                    valueOut += (valueOut ? ' ' : '') + cls;
                                }
                            });
                            // We got some internal classes left
                            if (valueOut) {
                                dom.setAttrib(elm, attrName, valueOut);
                                return;
                            }
                        }
                    }
                    // Remove mce prefixed attributes (must clean before short circuit operations)
                    if (MCE_ATTR_RE.test(attrName)) {
                        elm.removeAttribute('data-mce-' + attrName);
                    }
                    // keep style="list-style-type: none" on <li>s
                    if (attrName === 'style' && matchNodeNames$1(['li'])(elm) && dom.getStyle(elm, 'list-style-type') === 'none') {
                        elm.removeAttribute(attrName);
                        dom.setStyle(elm, 'list-style-type', 'none');
                        return;
                    }
                    // IE6 has a bug where the attribute doesn't get removed correctly
                    if (attrName === 'class') {
                        elm.removeAttribute('className');
                    }
                    elm.removeAttribute(attrName);
                }
            });
            // Remove classes
            each$7(format.classes, (value) => {
                value = replaceVars(value, vars);
                if (!isElement$7(compareNode) || dom.hasClass(compareNode, value)) {
                    dom.removeClass(elm, value);
                }
            });
            // Check for non internal attributes
            const attrs = dom.getAttribs(elm);
            for (let i = 0; i < attrs.length; i++) {
                const attrName = attrs[i].nodeName;
                if (!elementUtils.isAttributeInternal(attrName)) {
                    return removeResult.keep();
                }
            }
        }
        // Remove the inline child if it's empty for example <b> or <span>
        if (format.remove !== 'none') {
            removeNode(ed, elm, format);
            return removeResult.removed();
        }
        return removeResult.keep();
    };
    const findFormatRoot = (editor, container, name, vars, similar) => {
        let formatRoot;
        if (container.parentNode) {
            // Find format root
            each$e(getParents$2(editor.dom, container.parentNode).reverse(), (parent) => {
                // Find format root element
                if (!formatRoot && isElement$7(parent) && parent.id !== '_start' && parent.id !== '_end') {
                    // Is the node matching the format we are looking for
                    const format = matchNode$1(editor, parent, name, vars, similar);
                    if (format && format.split !== false) {
                        formatRoot = parent;
                    }
                }
            });
        }
        return formatRoot;
    };
    const removeNodeFormatFromClone = (editor, format, vars, clone) => removeNodeFormatInternal(editor, format, vars, clone, clone).fold(constant(clone), (newName) => {
        // To rename a node, it needs to be a child of another node
        const fragment = editor.dom.createFragment();
        fragment.appendChild(clone);
        // If renaming we are guaranteed this is a Element, so cast
        return editor.dom.rename(clone, newName);
    }, constant(null));
    const wrapAndSplit = (editor, formatList, formatRoot, container, target, split, format, vars) => {
        let lastClone;
        let firstClone;
        const dom = editor.dom;
        // Format root found then clone formats and split it
        if (formatRoot) {
            const formatRootParent = formatRoot.parentNode;
            for (let parent = container.parentNode; parent && parent !== formatRootParent; parent = parent.parentNode) {
                let clone = dom.clone(parent, false);
                for (let i = 0; i < formatList.length; i++) {
                    clone = removeNodeFormatFromClone(editor, formatList[i], vars, clone);
                    if (clone === null) {
                        break;
                    }
                }
                // Build wrapper node
                if (clone) {
                    if (lastClone) {
                        clone.appendChild(lastClone);
                    }
                    if (!firstClone) {
                        firstClone = clone;
                    }
                    lastClone = clone;
                }
            }
            // Never split block elements if the format is mixed
            if (split && (!format.mixed || !dom.isBlock(formatRoot))) {
                container = dom.split(formatRoot, container) ?? container;
            }
            // Wrap container in cloned formats
            if (lastClone && firstClone) {
                target.parentNode?.insertBefore(lastClone, target);
                firstClone.appendChild(target);
                // After splitting the nodes may match with other siblings so we need to attempt to merge them
                // Note: We can't use MergeFormats, as that'd create a circular dependency
                if (isInlineFormat(format)) {
                    mergeSiblings(editor, format, vars, lastClone);
                }
            }
        }
        return container;
    };
    const removeFormatInternal = (ed, name, vars, node, similar) => {
        const formatList = ed.formatter.get(name);
        const format = formatList[0];
        const dom = ed.dom;
        const selection = ed.selection;
        const splitToFormatRoot = (container) => {
            const formatRoot = findFormatRoot(ed, container, name, vars, similar);
            return wrapAndSplit(ed, formatList, formatRoot, container, container, true, format, vars);
        };
        // Make sure to only check for bookmarks created here (eg _start or _end)
        // as there maybe nested bookmarks
        const isRemoveBookmarkNode = (node) => isBookmarkNode$1(node) && isElement$7(node) && (node.id === '_start' || node.id === '_end');
        const removeFormatOnNode = (node) => exists(formatList, (fmt) => removeNodeFormat(ed, fmt, vars, node, node));
        // Merges the styles for each node
        const process = (node) => {
            // Grab the children first since the nodelist might be changed
            const children = from(node.childNodes);
            // Process current node
            const removed = removeFormatOnNode(node);
            // TINY-6567/TINY-7393: Include the parent if using an expanded selector format and no match was found for the current node
            const currentNodeMatches = removed || exists(formatList, (f) => matchName(dom, node, f));
            const parentNode = node.parentNode;
            if (!currentNodeMatches && isNonNullable(parentNode) && shouldExpandToSelector(format)) {
                removeFormatOnNode(parentNode);
            }
            // Process the children
            if (format.deep) {
                if (children.length) {
                    for (let i = 0; i < children.length; i++) {
                        process(children[i]);
                    }
                }
            }
            // Note: Assists with cleaning up any stray text decorations that may been applied when text decorations
            // and text colors were merged together from an applied format
            // Remove child span if it only contains text-decoration and a parent node also has the same text decoration.
            const textDecorations = ['underline', 'line-through', 'overline'];
            each$e(textDecorations, (decoration) => {
                if (isElement$7(node) && ed.dom.getStyle(node, 'text-decoration') === decoration &&
                    node.parentNode && getTextDecoration(dom, node.parentNode) === decoration) {
                    removeNodeFormat(ed, {
                        deep: false,
                        exact: true,
                        inline: 'span',
                        styles: {
                            textDecoration: decoration
                        }
                    }, undefined, node);
                }
            });
        };
        const unwrap = (start) => {
            const node = dom.get(start ? '_start' : '_end');
            if (node) {
                let out = node[start ? 'firstChild' : 'lastChild'];
                // If the end is placed within the start the result will be removed
                // So this checks if the out node is a bookmark node if it is it
                // checks for another more suitable node
                if (isRemoveBookmarkNode(out)) {
                    out = out[start ? 'firstChild' : 'lastChild'];
                }
                // Since dom.remove removes empty text nodes then we need to try to find a better node
                if (isText$b(out) && out.data.length === 0) {
                    out = start ? node.previousSibling || node.nextSibling : node.nextSibling || node.previousSibling;
                }
                dom.remove(node, true);
                return out;
            }
            else {
                return null;
            }
        };
        const removeRngStyle = (rng) => {
            let startContainer;
            let endContainer;
            let expandedRng = expandRng(dom, rng, formatList, { includeTrailingSpace: rng.collapsed });
            if (format.split) {
                // Split text nodes
                expandedRng = split(expandedRng);
                startContainer = getContainer(ed, expandedRng, true);
                endContainer = getContainer(ed, expandedRng);
                if (startContainer !== endContainer) {
                    // WebKit will render the table incorrectly if we wrap a TH or TD in a SPAN
                    // so let's see if we can use the first/last child instead
                    // This will happen if you triple click a table cell and use remove formatting
                    startContainer = normalizeTableSelection(startContainer, true);
                    endContainer = normalizeTableSelection(endContainer, false);
                    // Wrap and split if nested
                    if (isChildOfInlineParent(dom, startContainer, endContainer)) {
                        const marker = Optional.from(startContainer.firstChild).getOr(startContainer);
                        splitToFormatRoot(wrapWithSiblings(dom, marker, true, 'span', { 'id': '_start', 'data-mce-type': 'bookmark' }));
                        unwrap(true);
                        return;
                    }
                    // Wrap and split if nested
                    if (isChildOfInlineParent(dom, endContainer, startContainer)) {
                        const marker = Optional.from(endContainer.lastChild).getOr(endContainer);
                        splitToFormatRoot(wrapWithSiblings(dom, marker, false, 'span', { 'id': '_end', 'data-mce-type': 'bookmark' }));
                        unwrap(false);
                        return;
                    }
                    // Wrap start/end nodes in span element since these might be cloned/moved
                    startContainer = wrap$1(dom, startContainer, 'span', { 'id': '_start', 'data-mce-type': 'bookmark' });
                    endContainer = wrap$1(dom, endContainer, 'span', { 'id': '_end', 'data-mce-type': 'bookmark' });
                    // Split start/end and anything in between
                    const newRng = dom.createRng();
                    newRng.setStartAfter(startContainer);
                    newRng.setEndBefore(endContainer);
                    walk$3(dom, newRng, (nodes) => {
                        each$e(nodes, (n) => {
                            if (!isBookmarkNode$1(n) && !isBookmarkNode$1(n.parentNode)) {
                                splitToFormatRoot(n);
                            }
                        });
                    });
                    splitToFormatRoot(startContainer);
                    splitToFormatRoot(endContainer);
                    // Unwrap start/end to get real elements again
                    // Note that the return value should always be a node since it's wrapped above
                    startContainer = unwrap(true);
                    endContainer = unwrap();
                }
                else {
                    startContainer = endContainer = splitToFormatRoot(startContainer);
                }
                // Update range positions since they might have changed after the split operations
                expandedRng.startContainer = startContainer.parentNode ? startContainer.parentNode : startContainer;
                expandedRng.startOffset = dom.nodeIndex(startContainer);
                expandedRng.endContainer = endContainer.parentNode ? endContainer.parentNode : endContainer;
                expandedRng.endOffset = dom.nodeIndex(endContainer) + 1;
            }
            // Remove items between start/end
            walk$3(dom, expandedRng, (nodes) => {
                each$e(nodes, process);
            });
        };
        // Handle node
        if (node) {
            if (isNode(node)) {
                const rng = dom.createRng();
                rng.setStartBefore(node);
                rng.setEndAfter(node);
                removeRngStyle(rng);
            }
            else {
                removeRngStyle(node);
            }
            fireFormatRemove(ed, name, node, vars);
            return;
        }
        if (!selection.isCollapsed() || !isInlineFormat(format) || getCellsFromEditor(ed).length) {
            // Remove formatting on the selection
            preserveSelection(ed, () => runOnRanges(ed, removeRngStyle), 
            // Before trying to move the start of the selection, check if start element still has formatting then we are at: "<b>text|</b>text"
            // and need to move the start into the next text node
            (startNode) => isInlineFormat(format) && match$2(ed, name, vars, startNode));
            ed.nodeChanged();
        }
        else {
            removeCaretFormat(ed, name, vars, similar);
        }
        removeListStyleFormats(ed, name, vars);
        fireFormatRemove(ed, name, node, vars);
    };
    const removeFormat$1 = (ed, name, vars, node, similar) => {
        if (node || ed.selection.isEditable()) {
            removeFormatInternal(ed, name, vars, node, similar);
        }
    };
    const removeFormatOnElement = (editor, format, vars, node) => {
        return removeNodeFormatInternal(editor, format, vars, node).fold(() => Optional.some(node), (newName) => Optional.some(editor.dom.rename(node, newName)), Optional.none);
    };
    /**
     * Removes the specified format for the specified node. It will also remove the node if it doesn't have
     * any attributes if the format specifies it to do so.
     *
     * @private
     * @param {Object} format Format object with items to remove from node.
     * @param {Object} vars Name/value object with variables to apply to format.
     * @param {Node} node Node to remove the format styles on.
     * @param {Node} compareNode Optional compare node, if specified the styles will be compared to that node.
     * @return {Boolean} True/false if the node was removed or not.
     */
    const removeNodeFormat = (editor, format, vars, node, compareNode) => {
        return removeNodeFormatInternal(editor, format, vars, node, compareNode).fold(never, (newName) => {
            // If renaming we are guaranteed this is a Element, so cast
            editor.dom.rename(node, newName);
            return true;
        }, always);
    };

    const fontSizeAlteringFormats = ['fontsize', 'subscript', 'superscript'];
    const formatsToActOn = ['strikethrough', ...fontSizeAlteringFormats];
    const hasFormat = (formatter, el, format) => isNonNullable(formatter.matchNode(el.dom, format, {}, format === 'fontsize'));
    const isFontSizeAlteringElement = (formatter, el) => exists(fontSizeAlteringFormats, (format) => hasFormat(formatter, el, format));
    const isNormalizingFormat = (format) => contains$2(formatsToActOn, format);
    const gatherWrapperData = (isRoot, scope, hasFormat, createFormatElement, removeFormatFromElement) => {
        const parents = parents$1(scope, isRoot).filter(isElement$8);
        return findLastIndex(parents, hasFormat).map((index) => {
            const container = parents[index];
            const innerWrapper = createFormatElement(container);
            const outerWrappers = [
                ...removeFormatFromElement(shallow(container)).toArray(),
                ...bind$3(parents.slice(0, index), (wrapper) => {
                    if (hasFormat(wrapper)) {
                        return removeFormatFromElement(wrapper).toArray();
                    }
                    else {
                        return [shallow(wrapper)];
                    }
                })
            ];
            return { container, innerWrapper, outerWrappers };
        });
    };
    const wrapChildrenInInnerWrapper = (target, wrapper, hasFormat, removeFormatFromElement) => {
        each$e(children$1(target), (child) => {
            if (isElement$8(child) && hasFormat(child)) {
                if (removeFormatFromElement(child).isNone()) {
                    unwrap(child);
                }
            }
        });
        each$e(children$1(target), (child) => append$1(wrapper, child));
        prepend(target, wrapper);
    };
    const wrapInOuterWrappers = (target, wrappers) => {
        if (wrappers.length > 0) {
            const outermost = wrappers[wrappers.length - 1];
            before$4(target, outermost);
            const innerMost = foldl(wrappers.slice(0, wrappers.length - 1), (acc, wrapper) => {
                append$1(acc, wrapper);
                return wrapper;
            }, outermost);
            append$1(innerMost, target);
        }
    };
    const normalizeFontSizeElementsInternal = (domUtils, fontSizeElements, hasFormat, createFormatElement, removeFormatFromElement) => {
        const isRoot = (el) => eq(SugarElement.fromDom(domUtils.getRoot()), el) || domUtils.isBlock(el.dom);
        each$e(fontSizeElements, (fontSizeElement) => {
            gatherWrapperData(isRoot, fontSizeElement, hasFormat, createFormatElement, removeFormatFromElement).each(({ container, innerWrapper, outerWrappers }) => {
                domUtils.split(container.dom, fontSizeElement.dom);
                wrapChildrenInInnerWrapper(fontSizeElement, innerWrapper, hasFormat, removeFormatFromElement);
                wrapInOuterWrappers(fontSizeElement, outerWrappers);
            });
        });
    };
    const normalizeFontSizeElementsWithFormat = (editor, formatName, fontSizeElements) => {
        const hasFormat = (el) => isNonNullable(matchNode$1(editor, el.dom, formatName));
        const createFormatElement = (el) => {
            const newEl = SugarElement.fromTag(name(el));
            const format = matchNode$1(editor, el.dom, formatName, {});
            if (isNonNullable(format) && isApplyFormat(format)) {
                setElementFormat(editor, newEl.dom, format);
            }
            return newEl;
        };
        const removeFormatFromElement = (el) => {
            const format = matchNode$1(editor, el.dom, formatName, {});
            if (isNonNullable(format)) {
                return removeFormatOnElement(editor, format, {}, el.dom).map(SugarElement.fromDom);
            }
            else {
                return Optional.some(el);
            }
        };
        const bookmark = createBookmark(editor.selection.getRng());
        normalizeFontSizeElementsInternal(editor.dom, fontSizeElements, hasFormat, createFormatElement, removeFormatFromElement);
        editor.selection.setRng(resolveBookmark(bookmark));
    };
    const collectFontSizeElements = (formatter, wrappers) => bind$3(wrappers, (wrapper) => {
        const fontSizeDescendants = descendants$1(wrapper, (el) => isFontSizeAlteringElement(formatter, el));
        return isFontSizeAlteringElement(formatter, wrapper) ? [wrapper, ...fontSizeDescendants] : fontSizeDescendants;
    });
    const normalizeFontSizeElementsAfterApply = (editor, appliedFormat, wrappers) => {
        if (isNormalizingFormat(appliedFormat)) {
            const fontSizeElements = collectFontSizeElements(editor.formatter, wrappers);
            normalizeFontSizeElementsWithFormat(editor, 'strikethrough', fontSizeElements);
        }
    };
    const normalizeElements = (editor, elements) => {
        const fontSizeElements = filter$5(elements, (el) => isFontSizeAlteringElement(editor.formatter, el));
        normalizeFontSizeElementsWithFormat(editor, 'strikethrough', fontSizeElements);
    };

    const isHeading = (node) => ['h1', 'h2', 'h3', 'h4', 'h5', 'h6'].includes(node.name);
    const isSummary = (node) => node.name === 'summary';

    const traverse = (root, fn) => {
        let node = root;
        while ((node = node.walk())) {
            fn(node);
        }
    };
    // Test a single node against the current filters, and add it to any match lists if necessary
    const matchNode = (nodeFilters, attributeFilters, node, matches) => {
        const name = node.name;
        // Match node filters
        for (let ni = 0, nl = nodeFilters.length; ni < nl; ni++) {
            const filter = nodeFilters[ni];
            if (filter.name === name) {
                const match = matches.nodes[name];
                if (match) {
                    match.nodes.push(node);
                }
                else {
                    matches.nodes[name] = { filter, nodes: [node] };
                }
            }
        }
        // Match attribute filters
        if (node.attributes) {
            for (let ai = 0, al = attributeFilters.length; ai < al; ai++) {
                const filter = attributeFilters[ai];
                const attrName = filter.name;
                if (attrName in node.attributes.map) {
                    const match = matches.attributes[attrName];
                    if (match) {
                        match.nodes.push(node);
                    }
                    else {
                        matches.attributes[attrName] = { filter, nodes: [node] };
                    }
                }
            }
        }
    };
    const findMatchingNodes = (nodeFilters, attributeFilters, node) => {
        const matches = { nodes: {}, attributes: {} };
        if (node.firstChild) {
            traverse(node, (childNode) => {
                matchNode(nodeFilters, attributeFilters, childNode, matches);
            });
        }
        return matches;
    };
    // Run all necessary node filters and attribute filters, based on a match set
    const runFilters = (matches, args) => {
        const run = (matchRecord, filteringAttributes) => {
            each$d(matchRecord, (match) => {
                // in theory we don't need to copy the array, it was created purely for this filtering, but the method is exported so we can't guarantee that
                const nodes = from(match.nodes);
                each$e(match.filter.callbacks, (callback) => {
                    // very very carefully mutate the nodes array based on whether the filter still matches them
                    for (let i = nodes.length - 1; i >= 0; i--) {
                        const node = nodes[i];
                        // Remove already removed children, and nodes that no longer match the filter
                        const valueMatches = filteringAttributes ? node.attr(match.filter.name) !== undefined : node.name === match.filter.name;
                        if (!valueMatches || isNullable(node.parent)) {
                            nodes.splice(i, 1);
                        }
                    }
                    if (nodes.length > 0) {
                        callback(nodes, match.filter.name, args);
                    }
                });
            });
        };
        run(matches.nodes, false);
        run(matches.attributes, true);
    };
    const filter$1 = (nodeFilters, attributeFilters, node, args = {}) => {
        const matches = findMatchingNodes(nodeFilters, attributeFilters, node);
        runFilters(matches, args);
    };

    const paddEmptyNode = (settings, args, isBlock, node) => {
        const brPreferred = settings.pad_empty_with_br || args.insert;
        if (brPreferred && isBlock(node)) {
            const astNode = new AstNode('br', 1);
            if (args.insert) {
                astNode.attr('data-mce-bogus', '1');
            }
            node.empty().append(astNode);
        }
        else {
            node.empty().append(new AstNode('#text', 3)).value = nbsp;
        }
    };
    const isPaddedWithNbsp = (node) => hasOnlyChild(node, '#text') && node?.firstChild?.value === nbsp;
    const hasOnlyChild = (node, name) => {
        const firstChild = node?.firstChild;
        return isNonNullable(firstChild) && firstChild === node.lastChild && firstChild.name === name;
    };
    const isPadded = (schema, node) => {
        const rule = schema.getElementRule(node.name);
        return rule?.paddEmpty === true;
    };
    const isEmpty$2 = (schema, nonEmptyElements, whitespaceElements, node) => node.isEmpty(nonEmptyElements, whitespaceElements, (node) => isPadded(schema, node));
    const isLineBreakNode = (node, isBlock) => isNonNullable(node) && (isBlock(node) || node.name === 'br');
    const findClosestEditingHost = (scope) => {
        let editableNode;
        for (let node = scope; node; node = node.parent) {
            const contentEditable = node.attr('contenteditable');
            if (contentEditable === 'false') {
                break;
            }
            else if (contentEditable === 'true') {
                editableNode = node;
            }
        }
        return Optional.from(editableNode);
    };
    const getAllDescendants = (scope) => {
        const collection = [];
        for (let node = scope.firstChild; isNonNullable(node); node = node.walk()) {
            collection.push(node);
        }
        return collection;
    };

    const removeOrUnwrapInvalidNode = (node, schema, originalNodeParent = node.parent) => {
        if (schema.getSpecialElements()[node.name]) {
            node.empty().remove();
        }
        else {
            // are the children of `node` valid children of the top level parent?
            // if not, remove or unwrap them too
            const children = node.children();
            for (const childNode of children) {
                if (originalNodeParent && !schema.isValidChild(originalNodeParent.name, childNode.name)) {
                    removeOrUnwrapInvalidNode(childNode, schema, originalNodeParent);
                }
            }
            node.unwrap();
        }
    };
    const cleanInvalidNodes = (nodes, schema, rootNode, onCreate = noop) => {
        const textBlockElements = schema.getTextBlockElements();
        const nonEmptyElements = schema.getNonEmptyElements();
        const whitespaceElements = schema.getWhitespaceElements();
        const nonSplittableElements = Tools.makeMap('tr,td,th,tbody,thead,tfoot,table,summary');
        const fixed = new Set();
        const isSplittableElement = (node) => node !== rootNode && !nonSplittableElements[node.name];
        for (let ni = 0; ni < nodes.length; ni++) {
            const node = nodes[ni];
            let parent;
            let newParent;
            let tempNode;
            // Don't bother if it's detached from the tree
            if (!node.parent || fixed.has(node)) {
                continue;
            }
            // If the invalid element is a text block, and the text block is within a parent LI element
            // Then unwrap the first text block and convert other sibling text blocks to LI elements similar to Word/Open Office
            if (textBlockElements[node.name] && node.parent.name === 'li') {
                // Move sibling text blocks after LI element
                let sibling = node.next;
                while (sibling) {
                    if (textBlockElements[sibling.name]) {
                        sibling.name = 'li';
                        fixed.add(sibling);
                        node.parent.insert(sibling, node.parent);
                    }
                    else {
                        break;
                    }
                    sibling = sibling.next;
                }
                // Unwrap current text block
                node.unwrap();
                continue;
            }
            // Get list of all parent nodes until we find a valid parent to stick the child into
            const parents = [node];
            for (parent = node.parent; parent && !schema.isValidChild(parent.name, node.name) && isSplittableElement(parent); parent = parent.parent) {
                parents.push(parent);
            }
            // Found a suitable parent
            if (parent && parents.length > 1) {
                // If the node is a valid child of the parent, then try to move it. Otherwise unwrap it
                if (!isInvalid(schema, node, parent)) {
                    // Reverse the array since it makes looping easier
                    parents.reverse();
                    // Clone the related parent and insert that after the moved node
                    newParent = parents[0].clone();
                    onCreate(newParent);
                    // Start cloning and moving children on the left side of the target node
                    let currentNode = newParent;
                    for (let i = 0; i < parents.length - 1; i++) {
                        if (schema.isValidChild(currentNode.name, parents[i].name) && i > 0) {
                            tempNode = parents[i].clone();
                            onCreate(tempNode);
                            currentNode.append(tempNode);
                        }
                        else {
                            tempNode = currentNode;
                        }
                        for (let childNode = parents[i].firstChild; childNode && childNode !== parents[i + 1];) {
                            const nextNode = childNode.next;
                            tempNode.append(childNode);
                            childNode = nextNode;
                        }
                        currentNode = tempNode;
                    }
                    if (!isEmpty$2(schema, nonEmptyElements, whitespaceElements, newParent)) {
                        parent.insert(newParent, parents[0], true);
                        parent.insert(node, newParent);
                    }
                    else {
                        parent.insert(node, parents[0], true);
                    }
                    // Check if the element is empty by looking through its contents, with special treatment for <p><br /></p>
                    parent = parents[0];
                    if (isEmpty$2(schema, nonEmptyElements, whitespaceElements, parent) || hasOnlyChild(parent, 'br')) {
                        parent.empty().remove();
                    }
                }
                else {
                    removeOrUnwrapInvalidNode(node, schema);
                }
            }
            else if (node.parent) {
                // If it's an LI try to find a UL/OL for it or wrap it
                if (node.name === 'li') {
                    let sibling = node.prev;
                    if (sibling && (sibling.name === 'ul' || sibling.name === 'ol')) {
                        sibling.append(node);
                        continue;
                    }
                    sibling = node.next;
                    if (sibling && (sibling.name === 'ul' || sibling.name === 'ol') && sibling.firstChild) {
                        sibling.insert(node, sibling.firstChild, true);
                        continue;
                    }
                    const wrapper = new AstNode('ul', 1);
                    onCreate(wrapper);
                    node.wrap(wrapper);
                    continue;
                }
                // Try wrapping the element in a DIV
                if (schema.isValidChild(node.parent.name, 'div') && schema.isValidChild('div', node.name)) {
                    const wrapper = new AstNode('div', 1);
                    onCreate(wrapper);
                    node.wrap(wrapper);
                }
                else {
                    // We failed wrapping it, remove or unwrap it
                    removeOrUnwrapInvalidNode(node, schema);
                }
            }
        }
    };
    const hasClosest = (node, parentName) => {
        let tempNode = node;
        while (tempNode) {
            if (tempNode.name === parentName) {
                return true;
            }
            tempNode = tempNode.parent;
        }
        return false;
    };
    // The `parent` parameter of `isInvalid` function represents the closest valid parent
    // under which the `node` is intended to be moved.
    const isInvalid = (schema, node, parent = node.parent) => {
        if (!parent) {
            return false;
        }
        // Check if the node is a valid child of the parent node. If the child is
        // unknown we don't collect it since it's probably a custom element
        if (schema.children[node.name] && !schema.isValidChild(parent.name, node.name)) {
            return true;
        }
        // Anchors are a special case and cannot be nested
        if (node.name === 'a' && hasClosest(parent, 'a')) {
            return true;
        }
        // heading element is valid if it is the only one child of summary
        if (isSummary(parent) && isHeading(node)) {
            return !(parent?.firstChild === node && parent?.lastChild === node);
        }
        return false;
    };

    const createRange = (sc, so, ec, eo) => {
        const rng = document.createRange();
        rng.setStart(sc, so);
        rng.setEnd(ec, eo);
        return rng;
    };
    // If you triple click a paragraph in this case:
    //   <blockquote><p>a</p></blockquote><p>b</p>
    // It would become this range in webkit:
    //   <blockquote><p>[a</p></blockquote><p>]b</p>
    // We would want it to be:
    //   <blockquote><p>[a]</p></blockquote><p>b</p>
    // Since it would otherwise produces spans out of thin air on insertContent for example.
    const normalizeBlockSelectionRange = (rng) => {
        const startPos = CaretPosition.fromRangeStart(rng);
        const endPos = CaretPosition.fromRangeEnd(rng);
        const rootNode = rng.commonAncestorContainer;
        return fromPosition(false, rootNode, endPos)
            .map((newEndPos) => {
            if (!isInSameBlock(startPos, endPos, rootNode) && isInSameBlock(startPos, newEndPos, rootNode)) {
                return createRange(startPos.container(), startPos.offset(), newEndPos.container(), newEndPos.offset());
            }
            else {
                return rng;
            }
        }).getOr(rng);
    };
    const normalize = (rng) => rng.collapsed ? rng : normalizeBlockSelectionRange(rng);

    const explode$1 = Tools.explode;
    const create$8 = () => {
        const filters = {};
        const addFilter = (name, callback) => {
            each$e(explode$1(name), (name) => {
                if (!has$2(filters, name)) {
                    filters[name] = { name, callbacks: [] };
                }
                filters[name].callbacks.push(callback);
            });
        };
        const getFilters = () => values(filters);
        const removeFilter = (name, callback) => {
            each$e(explode$1(name), (name) => {
                if (has$2(filters, name)) {
                    if (isNonNullable(callback)) {
                        const filter = filters[name];
                        const newCallbacks = filter$5(filter.callbacks, (c) => c !== callback);
                        // If all callbacks have been removed then remove the filter reference
                        if (newCallbacks.length > 0) {
                            filter.callbacks = newCallbacks;
                        }
                        else {
                            delete filters[name];
                        }
                    }
                    else {
                        delete filters[name];
                    }
                }
            });
        };
        return {
            addFilter,
            getFilters,
            removeFilter
        };
    };

    const encodeData = (data) => data.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
    const decodeData$1 = (data) => data.replace(/&lt;/g, '<').replace(/&gt;/g, '>').replace(/&amp;/g, '&');

    const removeAttrs = (node, names) => {
        each$e(names, (name) => {
            node.attr(name, null);
        });
    };
    const addFontToSpansFilter = (domParser, styles, fontSizes) => {
        domParser.addNodeFilter('font', (nodes) => {
            each$e(nodes, (node) => {
                const props = styles.parse(node.attr('style'));
                const color = node.attr('color');
                const face = node.attr('face');
                const size = node.attr('size');
                if (color) {
                    props.color = color;
                }
                if (face) {
                    props['font-family'] = face;
                }
                if (size) {
                    toInt(size).each((num) => {
                        props['font-size'] = fontSizes[num - 1];
                    });
                }
                node.name = 'span';
                node.attr('style', styles.serialize(props));
                removeAttrs(node, ['color', 'face', 'size']);
            });
        });
    };
    const addStrikeFilter = (domParser, schema, styles) => {
        domParser.addNodeFilter('strike', (nodes) => {
            const convertToSTag = schema.type !== 'html4';
            each$e(nodes, (node) => {
                if (convertToSTag) {
                    node.name = 's';
                }
                else {
                    const props = styles.parse(node.attr('style'));
                    props['text-decoration'] = 'line-through';
                    node.name = 'span';
                    node.attr('style', styles.serialize(props));
                }
            });
        });
    };
    const addFilters = (domParser, settings, schema) => {
        const styles = Styles();
        if (settings.convert_fonts_to_spans) {
            addFontToSpansFilter(domParser, styles, Tools.explode(settings.font_size_legacy_values ?? ''));
        }
        addStrikeFilter(domParser, schema, styles);
    };
    const register$5 = (domParser, settings, schema) => {
        if (settings.inline_styles) {
            addFilters(domParser, settings, schema);
        }
    };

    const blobUriToBlob = (url) => fetch(url)
        .then((res) => res.ok ? res.blob() : Promise.reject())
        .catch(() => Promise.reject({
        message: `Cannot convert ${url} to Blob. Resource might not exist or is inaccessible.`,
        uriType: 'blob'
    }));
    const extractBase64Data = (data) => {
        const matches = /([a-z0-9+\/=\s]+)/i.exec(data);
        return matches ? matches[1] : '';
    };
    const decodeData = (data) => {
        try {
            return decodeURIComponent(data);
        }
        catch {
            return data;
        }
    };
    const parseDataUri = (uri) => {
        const [type, ...rest] = uri.split(',');
        const data = rest.join(',');
        const matches = /data:([^/]+\/[^;]+)(;.+)?/.exec(type);
        if (matches) {
            const base64Encoded = matches[2] === ';base64';
            const decodedData = decodeData(data);
            const extractedData = base64Encoded ? extractBase64Data(decodedData) : decodedData;
            return Optional.some({
                type: matches[1],
                data: extractedData,
                base64Encoded
            });
        }
        else {
            return Optional.none();
        }
    };
    const buildBlob = (type, data, base64Encoded = true) => {
        let str = data;
        if (base64Encoded) {
            // Might throw error if data isn't proper base64
            try {
                str = atob(data);
            }
            catch {
                return Optional.none();
            }
        }
        const arr = new Uint8Array(str.length);
        for (let i = 0; i < arr.length; i++) {
            arr[i] = str.charCodeAt(i);
        }
        return Optional.some(new Blob([arr], { type }));
    };
    const dataUriToBlob = (uri) => {
        return new Promise((resolve, reject) => {
            parseDataUri(uri)
                .bind(({ type, data, base64Encoded }) => buildBlob(type, data, base64Encoded))
                .fold(() => reject('Invalid data URI'), resolve);
        });
    };
    const uriToBlob = (url) => {
        if (startsWith(url, 'blob:')) {
            return blobUriToBlob(url);
        }
        else if (startsWith(url, 'data:')) {
            return dataUriToBlob(url);
        }
        else {
            return Promise.reject('Unknown URI format');
        }
    };
    const blobToDataUri = (blob) => {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onloadend = () => {
                resolve(reader.result);
            };
            reader.onerror = () => {
                reject(reader.error?.message);
            };
            reader.readAsDataURL(blob);
        });
    };

    let count$1 = 0;
    const uniqueId$1 = (prefix) => {
        return (prefix || 'blobid') + (count$1++);
    };
    const processDataUri = (dataUri, base64Only, generateBlobInfo) => {
        return parseDataUri(dataUri).bind(({ data, type, base64Encoded }) => {
            if (base64Only && !base64Encoded) {
                return Optional.none();
            }
            else {
                const base64 = base64Encoded ? data : btoa(data);
                return generateBlobInfo(base64, type);
            }
        });
    };
    const createBlobInfo$1 = (blobCache, blob, base64) => {
        const blobInfo = blobCache.create(uniqueId$1(), blob, base64);
        blobCache.add(blobInfo);
        return blobInfo;
    };
    const dataUriToBlobInfo = (blobCache, dataUri, base64Only = false) => {
        return processDataUri(dataUri, base64Only, (base64, type) => Optional.from(blobCache.getByData(base64, type)).orThunk(() => buildBlob(type, base64).map((blob) => createBlobInfo$1(blobCache, blob, base64))));
    };
    const imageToBlobInfo = (blobCache, imageSrc) => {
        const invalidDataUri = () => Promise.reject('Invalid data URI');
        if (startsWith(imageSrc, 'blob:')) {
            const blobInfo = blobCache.getByUri(imageSrc);
            if (isNonNullable(blobInfo)) {
                return Promise.resolve(blobInfo);
            }
            else {
                return uriToBlob(imageSrc).then((blob) => {
                    return blobToDataUri(blob).then((dataUri) => {
                        return processDataUri(dataUri, false, (base64) => {
                            return Optional.some(createBlobInfo$1(blobCache, blob, base64));
                        }).getOrThunk(invalidDataUri);
                    });
                });
            }
        }
        else if (startsWith(imageSrc, 'data:')) {
            return dataUriToBlobInfo(blobCache, imageSrc).fold(invalidDataUri, (blobInfo) => Promise.resolve(blobInfo));
        }
        else {
            // Not a blob or data URI so the image isn't a local image and isn't something that can be processed
            return Promise.reject('Unknown image data format');
        }
    };

    // TINY-10350: A modification of the Regexes.link regex to specifically capture host.
    // eslint-disable-next-line max-len
    const hostCaptureRegex = /^(?:(?:(?:[A-Za-z][A-Za-z\d.+-]{0,14}:\/\/(?:[-.~*+=!&;:'%@?^${}(),\w]+@)?|www\.|[-;:&=+$,.\w]+@)([A-Za-z\d-]+(?:\.[A-Za-z\d-]+)*))(?::\d+)?(?:\/(?:[-.~*+=!;:'%@$(),\/\w]*[-~*+=%@$()\/\w])?)?(?:\?(?:[-.~*+=!&;:'%@?^${}(),\/\w]+)?)?(?:#(?:[-.~*+=!&;:'%@?^${}(),\/\w]+)?)?)$/;
    const extractHost = (url) => Optional.from(url.match(hostCaptureRegex)).bind((ms) => get$b(ms, 1)).map((h) => startsWith(h, 'www.') ? h.substring(4) : h);

    const sandboxIframe = (iframeNode, exclusions) => {
        if (Optional.from(iframeNode.attr('src')).bind(extractHost).forall((host) => !contains$2(exclusions, host))) {
            iframeNode.attr('sandbox', '');
        }
    };
    const isMimeType = (mime, type) => startsWith(mime, `${type}/`);
    const getEmbedType = (type) => {
        if (isUndefined(type)) {
            return 'iframe';
        }
        else if (isMimeType(type, 'image')) {
            return 'img';
        }
        else if (isMimeType(type, 'video')) {
            return 'video';
        }
        else if (isMimeType(type, 'audio')) {
            return 'audio';
        }
        else {
            return 'iframe';
        }
    };
    const createSafeEmbed = ({ type, src, width, height } = {}, sandboxIframes, sandboxIframesExclusions) => {
        const name = getEmbedType(type);
        const embed = new AstNode(name, 1);
        embed.attr(name === 'audio' ? { src } : { src, width, height });
        // TINY-10349: Show controls for audio and video so the replaced embed is visible in editor.
        if (name === 'audio' || name === 'video') {
            embed.attr('controls', '');
        }
        if (name === 'iframe' && sandboxIframes) {
            sandboxIframe(embed, sandboxIframesExclusions);
        }
        return embed;
    };

    const isBogusImage = (img) => isNonNullable(img.attr('data-mce-bogus'));
    const isInternalImageSource = (img) => img.attr('src') === Env.transparentSrc || isNonNullable(img.attr('data-mce-placeholder'));
    const registerBase64ImageFilter = (parser, settings) => {
        const { blob_cache: blobCache } = settings;
        if (blobCache) {
            const processImage = (img) => {
                const inputSrc = img.attr('src');
                if (isInternalImageSource(img) || isBogusImage(img) || isNullable(inputSrc)) {
                    return;
                }
                dataUriToBlobInfo(blobCache, inputSrc, true).each((blobInfo) => {
                    img.attr('src', blobInfo.blobUri());
                });
            };
            parser.addAttributeFilter('src', (nodes) => each$e(nodes, processImage));
        }
    };
    const register$4 = (parser, settings) => {
        const schema = parser.schema;
        parser.addAttributeFilter('href', (nodes) => {
            let i = nodes.length;
            const appendRel = (rel) => {
                const parts = rel.split(' ').filter((p) => p.length > 0);
                return parts.concat(['noopener']).sort().join(' ');
            };
            const addNoOpener = (rel) => {
                const newRel = rel ? Tools.trim(rel) : '';
                if (!/\b(noopener)\b/g.test(newRel)) {
                    return appendRel(newRel);
                }
                else {
                    return newRel;
                }
            };
            if (!settings.allow_unsafe_link_target) {
                while (i--) {
                    const node = nodes[i];
                    if (node.name === 'a' && node.attr('target') === '_blank') {
                        node.attr('rel', addNoOpener(node.attr('rel')));
                    }
                }
            }
        });
        // Force anchor names closed, unless the setting "allow_html_in_named_anchor" is explicitly included.
        if (!settings.allow_html_in_named_anchor) {
            parser.addAttributeFilter('id,name', (nodes) => {
                let i = nodes.length, sibling, prevSibling, parent, node;
                while (i--) {
                    node = nodes[i];
                    if (node.name === 'a' && node.firstChild && !node.attr('href')) {
                        parent = node.parent;
                        // Move children after current node
                        sibling = node.lastChild;
                        while (sibling && parent) {
                            prevSibling = sibling.prev;
                            parent.insert(sibling, node);
                            sibling = prevSibling;
                        }
                    }
                }
            });
        }
        if (settings.fix_list_elements) {
            parser.addNodeFilter('ul,ol', (nodes) => {
                let i = nodes.length, node, parentNode;
                while (i--) {
                    node = nodes[i];
                    parentNode = node.parent;
                    if (parentNode && (parentNode.name === 'ul' || parentNode.name === 'ol')) {
                        if (node.prev && node.prev.name === 'li') {
                            node.prev.append(node);
                        }
                        else {
                            const li = new AstNode('li', 1);
                            li.attr('style', 'list-style-type: none');
                            node.wrap(li);
                        }
                    }
                }
            });
        }
        const validClasses = schema.getValidClasses();
        if (settings.validate && validClasses) {
            parser.addAttributeFilter('class', (nodes) => {
                let i = nodes.length;
                while (i--) {
                    const node = nodes[i];
                    const clazz = node.attr('class') ?? '';
                    const classList = Tools.explode(clazz, ' ');
                    let classValue = '';
                    for (let ci = 0; ci < classList.length; ci++) {
                        const className = classList[ci];
                        let valid = false;
                        let validClassesMap = validClasses['*'];
                        if (validClassesMap && validClassesMap[className]) {
                            valid = true;
                        }
                        validClassesMap = validClasses[node.name];
                        if (!valid && validClassesMap && validClassesMap[className]) {
                            valid = true;
                        }
                        if (valid) {
                            if (classValue) {
                                classValue += ' ';
                            }
                            classValue += className;
                        }
                    }
                    if (!classValue.length) {
                        classValue = null;
                    }
                    node.attr('class', classValue);
                }
            });
        }
        registerBase64ImageFilter(parser, settings);
        const shouldSandboxIframes = settings.sandbox_iframes ?? false;
        const sandboxIframesExclusions = unique$1(settings.sandbox_iframes_exclusions ?? []);
        if (settings.convert_unsafe_embeds) {
            parser.addNodeFilter('object,embed', (nodes) => each$e(nodes, (node) => {
                node.replace(createSafeEmbed({
                    type: node.attr('type'),
                    src: node.name === 'object' ? node.attr('data') : node.attr('src'),
                    width: node.attr('width'),
                    height: node.attr('height'),
                }, shouldSandboxIframes, sandboxIframesExclusions));
            }));
        }
        if (shouldSandboxIframes) {
            parser.addNodeFilter('iframe', (nodes) => each$e(nodes, (node) => sandboxIframe(node, sandboxIframesExclusions)));
        }
    };

    /*! @license DOMPurify 3.2.6 | (c) Cure53 and other contributors | Released under the Apache license 2.0 and Mozilla Public License 2.0 | github.com/cure53/DOMPurify/blob/3.2.6/LICENSE */

    const {
      entries,
      setPrototypeOf,
      isFrozen,
      getPrototypeOf,
      getOwnPropertyDescriptor
    } = Object;
    let {
      freeze,
      seal,
      create: create$7
    } = Object; // eslint-disable-line import/no-mutable-exports
    let {
      apply,
      construct
    } = typeof Reflect !== 'undefined' && Reflect;
    if (!freeze) {
      freeze = function freeze(x) {
        return x;
      };
    }
    if (!seal) {
      seal = function seal(x) {
        return x;
      };
    }
    if (!apply) {
      apply = function apply(fun, thisValue, args) {
        return fun.apply(thisValue, args);
      };
    }
    if (!construct) {
      construct = function construct(Func, args) {
        return new Func(...args);
      };
    }
    const arrayForEach = unapply(Array.prototype.forEach);
    const arrayLastIndexOf = unapply(Array.prototype.lastIndexOf);
    const arrayPop = unapply(Array.prototype.pop);
    const arrayPush = unapply(Array.prototype.push);
    const arraySplice = unapply(Array.prototype.splice);
    const stringToLowerCase = unapply(String.prototype.toLowerCase);
    const stringToString = unapply(String.prototype.toString);
    const stringMatch = unapply(String.prototype.match);
    const stringReplace = unapply(String.prototype.replace);
    const stringIndexOf = unapply(String.prototype.indexOf);
    const stringTrim = unapply(String.prototype.trim);
    const objectHasOwnProperty = unapply(Object.prototype.hasOwnProperty);
    const regExpTest = unapply(RegExp.prototype.test);
    const typeErrorCreate = unconstruct(TypeError);
    /**
     * Creates a new function that calls the given function with a specified thisArg and arguments.
     *
     * @param func - The function to be wrapped and called.
     * @returns A new function that calls the given function with a specified thisArg and arguments.
     */
    function unapply(func) {
      return function (thisArg) {
        if (thisArg instanceof RegExp) {
          thisArg.lastIndex = 0;
        }
        for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
          args[_key - 1] = arguments[_key];
        }
        return apply(func, thisArg, args);
      };
    }
    /**
     * Creates a new function that constructs an instance of the given constructor function with the provided arguments.
     *
     * @param func - The constructor function to be wrapped and called.
     * @returns A new function that constructs an instance of the given constructor function with the provided arguments.
     */
    function unconstruct(func) {
      return function () {
        for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
          args[_key2] = arguments[_key2];
        }
        return construct(func, args);
      };
    }
    /**
     * Add properties to a lookup table
     *
     * @param set - The set to which elements will be added.
     * @param array - The array containing elements to be added to the set.
     * @param transformCaseFunc - An optional function to transform the case of each element before adding to the set.
     * @returns The modified set with added elements.
     */
    function addToSet(set, array) {
      let transformCaseFunc = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : stringToLowerCase;
      if (setPrototypeOf) {
        // Make 'in' and truthy checks like Boolean(set.constructor)
        // independent of any properties defined on Object.prototype.
        // Prevent prototype setters from intercepting set as a this value.
        setPrototypeOf(set, null);
      }
      let l = array.length;
      while (l--) {
        let element = array[l];
        if (typeof element === 'string') {
          const lcElement = transformCaseFunc(element);
          if (lcElement !== element) {
            // Config presets (e.g. tags.js, attrs.js) are immutable.
            if (!isFrozen(array)) {
              array[l] = lcElement;
            }
            element = lcElement;
          }
        }
        set[element] = true;
      }
      return set;
    }
    /**
     * Clean up an array to harden against CSPP
     *
     * @param array - The array to be cleaned.
     * @returns The cleaned version of the array
     */
    function cleanArray(array) {
      for (let index = 0; index < array.length; index++) {
        const isPropertyExist = objectHasOwnProperty(array, index);
        if (!isPropertyExist) {
          array[index] = null;
        }
      }
      return array;
    }
    /**
     * Shallow clone an object
     *
     * @param object - The object to be cloned.
     * @returns A new object that copies the original.
     */
    function clone(object) {
      const newObject = create$7(null);
      for (const [property, value] of entries(object)) {
        const isPropertyExist = objectHasOwnProperty(object, property);
        if (isPropertyExist) {
          if (Array.isArray(value)) {
            newObject[property] = cleanArray(value);
          } else if (value && typeof value === 'object' && value.constructor === Object) {
            newObject[property] = clone(value);
          } else {
            newObject[property] = value;
          }
        }
      }
      return newObject;
    }
    /**
     * This method automatically checks if the prop is function or getter and behaves accordingly.
     *
     * @param object - The object to look up the getter function in its prototype chain.
     * @param prop - The property name for which to find the getter function.
     * @returns The getter function found in the prototype chain or a fallback function.
     */
    function lookupGetter(object, prop) {
      while (object !== null) {
        const desc = getOwnPropertyDescriptor(object, prop);
        if (desc) {
          if (desc.get) {
            return unapply(desc.get);
          }
          if (typeof desc.value === 'function') {
            return unapply(desc.value);
          }
        }
        object = getPrototypeOf(object);
      }
      function fallbackValue() {
        return null;
      }
      return fallbackValue;
    }

    const html$1 = freeze(['a', 'abbr', 'acronym', 'address', 'area', 'article', 'aside', 'audio', 'b', 'bdi', 'bdo', 'big', 'blink', 'blockquote', 'body', 'br', 'button', 'canvas', 'caption', 'center', 'cite', 'code', 'col', 'colgroup', 'content', 'data', 'datalist', 'dd', 'decorator', 'del', 'details', 'dfn', 'dialog', 'dir', 'div', 'dl', 'dt', 'element', 'em', 'fieldset', 'figcaption', 'figure', 'font', 'footer', 'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'head', 'header', 'hgroup', 'hr', 'html', 'i', 'img', 'input', 'ins', 'kbd', 'label', 'legend', 'li', 'main', 'map', 'mark', 'marquee', 'menu', 'menuitem', 'meter', 'nav', 'nobr', 'ol', 'optgroup', 'option', 'output', 'p', 'picture', 'pre', 'progress', 'q', 'rp', 'rt', 'ruby', 's', 'samp', 'section', 'select', 'shadow', 'small', 'source', 'spacer', 'span', 'strike', 'strong', 'style', 'sub', 'summary', 'sup', 'table', 'tbody', 'td', 'template', 'textarea', 'tfoot', 'th', 'thead', 'time', 'tr', 'track', 'tt', 'u', 'ul', 'var', 'video', 'wbr']);
    const svg$1 = freeze(['svg', 'a', 'altglyph', 'altglyphdef', 'altglyphitem', 'animatecolor', 'animatemotion', 'animatetransform', 'circle', 'clippath', 'defs', 'desc', 'ellipse', 'filter', 'font', 'g', 'glyph', 'glyphref', 'hkern', 'image', 'line', 'lineargradient', 'marker', 'mask', 'metadata', 'mpath', 'path', 'pattern', 'polygon', 'polyline', 'radialgradient', 'rect', 'stop', 'style', 'switch', 'symbol', 'text', 'textpath', 'title', 'tref', 'tspan', 'view', 'vkern']);
    const svgFilters = freeze(['feBlend', 'feColorMatrix', 'feComponentTransfer', 'feComposite', 'feConvolveMatrix', 'feDiffuseLighting', 'feDisplacementMap', 'feDistantLight', 'feDropShadow', 'feFlood', 'feFuncA', 'feFuncB', 'feFuncG', 'feFuncR', 'feGaussianBlur', 'feImage', 'feMerge', 'feMergeNode', 'feMorphology', 'feOffset', 'fePointLight', 'feSpecularLighting', 'feSpotLight', 'feTile', 'feTurbulence']);
    // List of SVG elements that are disallowed by default.
    // We still need to know them so that we can do namespace
    // checks properly in case one wants to add them to
    // allow-list.
    const svgDisallowed = freeze(['animate', 'color-profile', 'cursor', 'discard', 'font-face', 'font-face-format', 'font-face-name', 'font-face-src', 'font-face-uri', 'foreignobject', 'hatch', 'hatchpath', 'mesh', 'meshgradient', 'meshpatch', 'meshrow', 'missing-glyph', 'script', 'set', 'solidcolor', 'unknown', 'use']);
    const mathMl$1 = freeze(['math', 'menclose', 'merror', 'mfenced', 'mfrac', 'mglyph', 'mi', 'mlabeledtr', 'mmultiscripts', 'mn', 'mo', 'mover', 'mpadded', 'mphantom', 'mroot', 'mrow', 'ms', 'mspace', 'msqrt', 'mstyle', 'msub', 'msup', 'msubsup', 'mtable', 'mtd', 'mtext', 'mtr', 'munder', 'munderover', 'mprescripts']);
    // Similarly to SVG, we want to know all MathML elements,
    // even those that we disallow by default.
    const mathMlDisallowed = freeze(['maction', 'maligngroup', 'malignmark', 'mlongdiv', 'mscarries', 'mscarry', 'msgroup', 'mstack', 'msline', 'msrow', 'semantics', 'annotation', 'annotation-xml', 'mprescripts', 'none']);
    const text = freeze(['#text']);

    const html = freeze(['accept', 'action', 'align', 'alt', 'autocapitalize', 'autocomplete', 'autopictureinpicture', 'autoplay', 'background', 'bgcolor', 'border', 'capture', 'cellpadding', 'cellspacing', 'checked', 'cite', 'class', 'clear', 'color', 'cols', 'colspan', 'controls', 'controlslist', 'coords', 'crossorigin', 'datetime', 'decoding', 'default', 'dir', 'disabled', 'disablepictureinpicture', 'disableremoteplayback', 'download', 'draggable', 'enctype', 'enterkeyhint', 'face', 'for', 'headers', 'height', 'hidden', 'high', 'href', 'hreflang', 'id', 'inputmode', 'integrity', 'ismap', 'kind', 'label', 'lang', 'list', 'loading', 'loop', 'low', 'max', 'maxlength', 'media', 'method', 'min', 'minlength', 'multiple', 'muted', 'name', 'nonce', 'noshade', 'novalidate', 'nowrap', 'open', 'optimum', 'pattern', 'placeholder', 'playsinline', 'popover', 'popovertarget', 'popovertargetaction', 'poster', 'preload', 'pubdate', 'radiogroup', 'readonly', 'rel', 'required', 'rev', 'reversed', 'role', 'rows', 'rowspan', 'spellcheck', 'scope', 'selected', 'shape', 'size', 'sizes', 'span', 'srclang', 'start', 'src', 'srcset', 'step', 'style', 'summary', 'tabindex', 'title', 'translate', 'type', 'usemap', 'valign', 'value', 'width', 'wrap', 'xmlns', 'slot']);
    const svg = freeze(['accent-height', 'accumulate', 'additive', 'alignment-baseline', 'amplitude', 'ascent', 'attributename', 'attributetype', 'azimuth', 'basefrequency', 'baseline-shift', 'begin', 'bias', 'by', 'class', 'clip', 'clippathunits', 'clip-path', 'clip-rule', 'color', 'color-interpolation', 'color-interpolation-filters', 'color-profile', 'color-rendering', 'cx', 'cy', 'd', 'dx', 'dy', 'diffuseconstant', 'direction', 'display', 'divisor', 'dur', 'edgemode', 'elevation', 'end', 'exponent', 'fill', 'fill-opacity', 'fill-rule', 'filter', 'filterunits', 'flood-color', 'flood-opacity', 'font-family', 'font-size', 'font-size-adjust', 'font-stretch', 'font-style', 'font-variant', 'font-weight', 'fx', 'fy', 'g1', 'g2', 'glyph-name', 'glyphref', 'gradientunits', 'gradienttransform', 'height', 'href', 'id', 'image-rendering', 'in', 'in2', 'intercept', 'k', 'k1', 'k2', 'k3', 'k4', 'kerning', 'keypoints', 'keysplines', 'keytimes', 'lang', 'lengthadjust', 'letter-spacing', 'kernelmatrix', 'kernelunitlength', 'lighting-color', 'local', 'marker-end', 'marker-mid', 'marker-start', 'markerheight', 'markerunits', 'markerwidth', 'maskcontentunits', 'maskunits', 'max', 'mask', 'media', 'method', 'mode', 'min', 'name', 'numoctaves', 'offset', 'operator', 'opacity', 'order', 'orient', 'orientation', 'origin', 'overflow', 'paint-order', 'path', 'pathlength', 'patterncontentunits', 'patterntransform', 'patternunits', 'points', 'preservealpha', 'preserveaspectratio', 'primitiveunits', 'r', 'rx', 'ry', 'radius', 'refx', 'refy', 'repeatcount', 'repeatdur', 'restart', 'result', 'rotate', 'scale', 'seed', 'shape-rendering', 'slope', 'specularconstant', 'specularexponent', 'spreadmethod', 'startoffset', 'stddeviation', 'stitchtiles', 'stop-color', 'stop-opacity', 'stroke-dasharray', 'stroke-dashoffset', 'stroke-linecap', 'stroke-linejoin', 'stroke-miterlimit', 'stroke-opacity', 'stroke', 'stroke-width', 'style', 'surfacescale', 'systemlanguage', 'tabindex', 'tablevalues', 'targetx', 'targety', 'transform', 'transform-origin', 'text-anchor', 'text-decoration', 'text-rendering', 'textlength', 'type', 'u1', 'u2', 'unicode', 'values', 'viewbox', 'visibility', 'version', 'vert-adv-y', 'vert-origin-x', 'vert-origin-y', 'width', 'word-spacing', 'wrap', 'writing-mode', 'xchannelselector', 'ychannelselector', 'x', 'x1', 'x2', 'xmlns', 'y', 'y1', 'y2', 'z', 'zoomandpan']);
    const mathMl = freeze(['accent', 'accentunder', 'align', 'bevelled', 'close', 'columnsalign', 'columnlines', 'columnspan', 'denomalign', 'depth', 'dir', 'display', 'displaystyle', 'encoding', 'fence', 'frame', 'height', 'href', 'id', 'largeop', 'length', 'linethickness', 'lspace', 'lquote', 'mathbackground', 'mathcolor', 'mathsize', 'mathvariant', 'maxsize', 'minsize', 'movablelimits', 'notation', 'numalign', 'open', 'rowalign', 'rowlines', 'rowspacing', 'rowspan', 'rspace', 'rquote', 'scriptlevel', 'scriptminsize', 'scriptsizemultiplier', 'selection', 'separator', 'separators', 'stretchy', 'subscriptshift', 'supscriptshift', 'symmetric', 'voffset', 'width', 'xmlns']);
    const xml = freeze(['xlink:href', 'xml:id', 'xlink:title', 'xml:space', 'xmlns:xlink']);

    // eslint-disable-next-line unicorn/better-regex
    const MUSTACHE_EXPR = seal(/\{\{[\w\W]*|[\w\W]*\}\}/gm); // Specify template detection regex for SAFE_FOR_TEMPLATES mode
    const ERB_EXPR = seal(/<%[\w\W]*|[\w\W]*%>/gm);
    const TMPLIT_EXPR = seal(/\$\{[\w\W]*/gm); // eslint-disable-line unicorn/better-regex
    const DATA_ATTR = seal(/^data-[\-\w.\u00B7-\uFFFF]+$/); // eslint-disable-line no-useless-escape
    const ARIA_ATTR = seal(/^aria-[\-\w]+$/); // eslint-disable-line no-useless-escape
    const IS_ALLOWED_URI = seal(/^(?:(?:(?:f|ht)tps?|mailto|tel|callto|sms|cid|xmpp|matrix):|[^a-z]|[a-z+.\-]+(?:[^a-z+.\-:]|$))/i // eslint-disable-line no-useless-escape
    );
    const IS_SCRIPT_OR_DATA = seal(/^(?:\w+script|data):/i);
    const ATTR_WHITESPACE = seal(/[\u0000-\u0020\u00A0\u1680\u180E\u2000-\u2029\u205F\u3000]/g // eslint-disable-line no-control-regex
    );
    const DOCTYPE_NAME = seal(/^html$/i);
    const CUSTOM_ELEMENT = seal(/^[a-z][.\w]*(-[.\w]+)+$/i);

    var EXPRESSIONS = /*#__PURE__*/Object.freeze({
      __proto__: null,
      ARIA_ATTR: ARIA_ATTR,
      ATTR_WHITESPACE: ATTR_WHITESPACE,
      CUSTOM_ELEMENT: CUSTOM_ELEMENT,
      DATA_ATTR: DATA_ATTR,
      DOCTYPE_NAME: DOCTYPE_NAME,
      ERB_EXPR: ERB_EXPR,
      IS_ALLOWED_URI: IS_ALLOWED_URI,
      IS_SCRIPT_OR_DATA: IS_SCRIPT_OR_DATA,
      MUSTACHE_EXPR: MUSTACHE_EXPR,
      TMPLIT_EXPR: TMPLIT_EXPR
    });

    /* eslint-disable @typescript-eslint/indent */
    // https://developer.mozilla.org/en-US/docs/Web/API/Node/nodeType
    const NODE_TYPE = {
      element: 1,
      attribute: 2,
      text: 3,
      cdataSection: 4,
      entityReference: 5,
      // Deprecated
      entityNode: 6,
      // Deprecated
      progressingInstruction: 7,
      comment: 8,
      document: 9,
      documentType: 10,
      documentFragment: 11,
      notation: 12 // Deprecated
    };
    const getGlobal = function getGlobal() {
      return typeof window === 'undefined' ? null : window;
    };
    /**
     * Creates a no-op policy for internal use only.
     * Don't export this function outside this module!
     * @param trustedTypes The policy factory.
     * @param purifyHostElement The Script element used to load DOMPurify (to determine policy name suffix).
     * @return The policy created (or null, if Trusted Types
     * are not supported or creating the policy failed).
     */
    const _createTrustedTypesPolicy = function _createTrustedTypesPolicy(trustedTypes, purifyHostElement) {
      if (typeof trustedTypes !== 'object' || typeof trustedTypes.createPolicy !== 'function') {
        return null;
      }
      // Allow the callers to control the unique policy name
      // by adding a data-tt-policy-suffix to the script element with the DOMPurify.
      // Policy creation with duplicate names throws in Trusted Types.
      let suffix = null;
      const ATTR_NAME = 'data-tt-policy-suffix';
      if (purifyHostElement && purifyHostElement.hasAttribute(ATTR_NAME)) {
        suffix = purifyHostElement.getAttribute(ATTR_NAME);
      }
      const policyName = 'dompurify' + (suffix ? '#' + suffix : '');
      try {
        return trustedTypes.createPolicy(policyName, {
          createHTML(html) {
            return html;
          },
          createScriptURL(scriptUrl) {
            return scriptUrl;
          }
        });
      } catch (_) {
        // Policy creation failed (most likely another DOMPurify script has
        // already run). Skip creating the policy, as this will only cause errors
        // if TT are enforced.
        console.warn('TrustedTypes policy ' + policyName + ' could not be created.');
        return null;
      }
    };
    const _createHooksMap = function _createHooksMap() {
      return {
        afterSanitizeAttributes: [],
        afterSanitizeElements: [],
        afterSanitizeShadowDOM: [],
        beforeSanitizeAttributes: [],
        beforeSanitizeElements: [],
        beforeSanitizeShadowDOM: [],
        uponSanitizeAttribute: [],
        uponSanitizeElement: [],
        uponSanitizeShadowNode: []
      };
    };
    function createDOMPurify() {
      let window = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : getGlobal();
      const DOMPurify = root => createDOMPurify(root);
      DOMPurify.version = '3.2.6';
      DOMPurify.removed = [];
      if (!window || !window.document || window.document.nodeType !== NODE_TYPE.document || !window.Element) {
        // Not running in a browser, provide a factory function
        // so that you can pass your own Window
        DOMPurify.isSupported = false;
        return DOMPurify;
      }
      let {
        document
      } = window;
      const originalDocument = document;
      const currentScript = originalDocument.currentScript;
      const {
        DocumentFragment,
        HTMLTemplateElement,
        Node,
        Element,
        NodeFilter,
        NamedNodeMap = window.NamedNodeMap || window.MozNamedAttrMap,
        HTMLFormElement,
        DOMParser,
        trustedTypes
      } = window;
      const ElementPrototype = Element.prototype;
      const cloneNode = lookupGetter(ElementPrototype, 'cloneNode');
      const remove = lookupGetter(ElementPrototype, 'remove');
      const getNextSibling = lookupGetter(ElementPrototype, 'nextSibling');
      const getChildNodes = lookupGetter(ElementPrototype, 'childNodes');
      const getParentNode = lookupGetter(ElementPrototype, 'parentNode');
      // As per issue #47, the web-components registry is inherited by a
      // new document created via createHTMLDocument. As per the spec
      // (http://w3c.github.io/webcomponents/spec/custom/#creating-and-passing-registries)
      // a new empty registry is used when creating a template contents owner
      // document, so we use that as our parent document to ensure nothing
      // is inherited.
      if (typeof HTMLTemplateElement === 'function') {
        const template = document.createElement('template');
        if (template.content && template.content.ownerDocument) {
          document = template.content.ownerDocument;
        }
      }
      let trustedTypesPolicy;
      let emptyHTML = '';
      const {
        implementation,
        createNodeIterator,
        createDocumentFragment,
        getElementsByTagName
      } = document;
      const {
        importNode
      } = originalDocument;
      let hooks = _createHooksMap();
      /**
       * Expose whether this browser supports running the full DOMPurify.
       */
      DOMPurify.isSupported = typeof entries === 'function' && typeof getParentNode === 'function' && implementation && implementation.createHTMLDocument !== undefined;
      const {
        MUSTACHE_EXPR,
        ERB_EXPR,
        TMPLIT_EXPR,
        DATA_ATTR,
        ARIA_ATTR,
        IS_SCRIPT_OR_DATA,
        ATTR_WHITESPACE,
        CUSTOM_ELEMENT
      } = EXPRESSIONS;
      let {
        IS_ALLOWED_URI: IS_ALLOWED_URI$1
      } = EXPRESSIONS;
      /**
       * We consider the elements and attributes below to be safe. Ideally
       * don't add any new ones but feel free to remove unwanted ones.
       */
      /* allowed element names */
      let ALLOWED_TAGS = null;
      const DEFAULT_ALLOWED_TAGS = addToSet({}, [...html$1, ...svg$1, ...svgFilters, ...mathMl$1, ...text]);
      /* Allowed attribute names */
      let ALLOWED_ATTR = null;
      const DEFAULT_ALLOWED_ATTR = addToSet({}, [...html, ...svg, ...mathMl, ...xml]);
      /*
       * Configure how DOMPurify should handle custom elements and their attributes as well as customized built-in elements.
       * @property {RegExp|Function|null} tagNameCheck one of [null, regexPattern, predicate]. Default: `null` (disallow any custom elements)
       * @property {RegExp|Function|null} attributeNameCheck one of [null, regexPattern, predicate]. Default: `null` (disallow any attributes not on the allow list)
       * @property {boolean} allowCustomizedBuiltInElements allow custom elements derived from built-ins if they pass CUSTOM_ELEMENT_HANDLING.tagNameCheck. Default: `false`.
       */
      let CUSTOM_ELEMENT_HANDLING = Object.seal(create$7(null, {
        tagNameCheck: {
          writable: true,
          configurable: false,
          enumerable: true,
          value: null
        },
        attributeNameCheck: {
          writable: true,
          configurable: false,
          enumerable: true,
          value: null
        },
        allowCustomizedBuiltInElements: {
          writable: true,
          configurable: false,
          enumerable: true,
          value: false
        }
      }));
      /* Explicitly forbidden tags (overrides ALLOWED_TAGS/ADD_TAGS) */
      let FORBID_TAGS = null;
      /* Explicitly forbidden attributes (overrides ALLOWED_ATTR/ADD_ATTR) */
      let FORBID_ATTR = null;
      /* Decide if ARIA attributes are okay */
      let ALLOW_ARIA_ATTR = true;
      /* Decide if custom data attributes are okay */
      let ALLOW_DATA_ATTR = true;
      /* Decide if unknown protocols are okay */
      let ALLOW_UNKNOWN_PROTOCOLS = false;
      /* Decide if self-closing tags in attributes are allowed.
       * Usually removed due to a mXSS issue in jQuery 3.0 */
      let ALLOW_SELF_CLOSE_IN_ATTR = true;
      /* Output should be safe for common template engines.
       * This means, DOMPurify removes data attributes, mustaches and ERB
       */
      let SAFE_FOR_TEMPLATES = false;
      /* Output should be safe even for XML used within HTML and alike.
       * This means, DOMPurify removes comments when containing risky content.
       */
      let SAFE_FOR_XML = true;
      /* Decide if document with <html>... should be returned */
      let WHOLE_DOCUMENT = false;
      /* Track whether config is already set on this instance of DOMPurify. */
      let SET_CONFIG = false;
      /* Decide if all elements (e.g. style, script) must be children of
       * document.body. By default, browsers might move them to document.head */
      let FORCE_BODY = false;
      /* Decide if a DOM `HTMLBodyElement` should be returned, instead of a html
       * string (or a TrustedHTML object if Trusted Types are supported).
       * If `WHOLE_DOCUMENT` is enabled a `HTMLHtmlElement` will be returned instead
       */
      let RETURN_DOM = false;
      /* Decide if a DOM `DocumentFragment` should be returned, instead of a html
       * string  (or a TrustedHTML object if Trusted Types are supported) */
      let RETURN_DOM_FRAGMENT = false;
      /* Try to return a Trusted Type object instead of a string, return a string in
       * case Trusted Types are not supported  */
      let RETURN_TRUSTED_TYPE = false;
      /* Output should be free from DOM clobbering attacks?
       * This sanitizes markups named with colliding, clobberable built-in DOM APIs.
       */
      let SANITIZE_DOM = true;
      /* Achieve full DOM Clobbering protection by isolating the namespace of named
       * properties and JS variables, mitigating attacks that abuse the HTML/DOM spec rules.
       *
       * HTML/DOM spec rules that enable DOM Clobbering:
       *   - Named Access on Window (§7.3.3)
       *   - DOM Tree Accessors (§3.1.5)
       *   - Form Element Parent-Child Relations (§4.10.3)
       *   - Iframe srcdoc / Nested WindowProxies (§4.8.5)
       *   - HTMLCollection (§4.2.10.2)
       *
       * Namespace isolation is implemented by prefixing `id` and `name` attributes
       * with a constant string, i.e., `user-content-`
       */
      let SANITIZE_NAMED_PROPS = false;
      const SANITIZE_NAMED_PROPS_PREFIX = 'user-content-';
      /* Keep element content when removing element? */
      let KEEP_CONTENT = true;
      /* If a `Node` is passed to sanitize(), then performs sanitization in-place instead
       * of importing it into a new Document and returning a sanitized copy */
      let IN_PLACE = false;
      /* Allow usage of profiles like html, svg and mathMl */
      let USE_PROFILES = {};
      /* Tags to ignore content of when KEEP_CONTENT is true */
      let FORBID_CONTENTS = null;
      const DEFAULT_FORBID_CONTENTS = addToSet({}, ['annotation-xml', 'audio', 'colgroup', 'desc', 'foreignobject', 'head', 'iframe', 'math', 'mi', 'mn', 'mo', 'ms', 'mtext', 'noembed', 'noframes', 'noscript', 'plaintext', 'script', 'style', 'svg', 'template', 'thead', 'title', 'video', 'xmp']);
      /* Tags that are safe for data: URIs */
      let DATA_URI_TAGS = null;
      const DEFAULT_DATA_URI_TAGS = addToSet({}, ['audio', 'video', 'img', 'source', 'image', 'track']);
      /* Attributes safe for values like "javascript:" */
      let URI_SAFE_ATTRIBUTES = null;
      const DEFAULT_URI_SAFE_ATTRIBUTES = addToSet({}, ['alt', 'class', 'for', 'id', 'label', 'name', 'pattern', 'placeholder', 'role', 'summary', 'title', 'value', 'style', 'xmlns']);
      const MATHML_NAMESPACE = 'http://www.w3.org/1998/Math/MathML';
      const SVG_NAMESPACE = 'http://www.w3.org/2000/svg';
      const HTML_NAMESPACE = 'http://www.w3.org/1999/xhtml';
      /* Document namespace */
      let NAMESPACE = HTML_NAMESPACE;
      let IS_EMPTY_INPUT = false;
      /* Allowed XHTML+XML namespaces */
      let ALLOWED_NAMESPACES = null;
      const DEFAULT_ALLOWED_NAMESPACES = addToSet({}, [MATHML_NAMESPACE, SVG_NAMESPACE, HTML_NAMESPACE], stringToString);
      let MATHML_TEXT_INTEGRATION_POINTS = addToSet({}, ['mi', 'mo', 'mn', 'ms', 'mtext']);
      let HTML_INTEGRATION_POINTS = addToSet({}, ['annotation-xml']);
      // Certain elements are allowed in both SVG and HTML
      // namespace. We need to specify them explicitly
      // so that they don't get erroneously deleted from
      // HTML namespace.
      const COMMON_SVG_AND_HTML_ELEMENTS = addToSet({}, ['title', 'style', 'font', 'a', 'script']);
      /* Parsing of strict XHTML documents */
      let PARSER_MEDIA_TYPE = null;
      const SUPPORTED_PARSER_MEDIA_TYPES = ['application/xhtml+xml', 'text/html'];
      const DEFAULT_PARSER_MEDIA_TYPE = 'text/html';
      let transformCaseFunc = null;
      /* Keep a reference to config to pass to hooks */
      let CONFIG = null;
      /* Ideally, do not touch anything below this line */
      /* ______________________________________________ */
      const formElement = document.createElement('form');
      const isRegexOrFunction = function isRegexOrFunction(testValue) {
        return testValue instanceof RegExp || testValue instanceof Function;
      };
      /**
       * _parseConfig
       *
       * @param cfg optional config literal
       */
      // eslint-disable-next-line complexity
      const _parseConfig = function _parseConfig() {
        let cfg = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        if (CONFIG && CONFIG === cfg) {
          return;
        }
        /* Shield configuration object from tampering */
        if (!cfg || typeof cfg !== 'object') {
          cfg = {};
        }
        /* Shield configuration object from prototype pollution */
        cfg = clone(cfg);
        PARSER_MEDIA_TYPE =
        // eslint-disable-next-line unicorn/prefer-includes
        SUPPORTED_PARSER_MEDIA_TYPES.indexOf(cfg.PARSER_MEDIA_TYPE) === -1 ? DEFAULT_PARSER_MEDIA_TYPE : cfg.PARSER_MEDIA_TYPE;
        // HTML tags and attributes are not case-sensitive, converting to lowercase. Keeping XHTML as is.
        transformCaseFunc = PARSER_MEDIA_TYPE === 'application/xhtml+xml' ? stringToString : stringToLowerCase;
        /* Set configuration parameters */
        ALLOWED_TAGS = objectHasOwnProperty(cfg, 'ALLOWED_TAGS') ? addToSet({}, cfg.ALLOWED_TAGS, transformCaseFunc) : DEFAULT_ALLOWED_TAGS;
        ALLOWED_ATTR = objectHasOwnProperty(cfg, 'ALLOWED_ATTR') ? addToSet({}, cfg.ALLOWED_ATTR, transformCaseFunc) : DEFAULT_ALLOWED_ATTR;
        ALLOWED_NAMESPACES = objectHasOwnProperty(cfg, 'ALLOWED_NAMESPACES') ? addToSet({}, cfg.ALLOWED_NAMESPACES, stringToString) : DEFAULT_ALLOWED_NAMESPACES;
        URI_SAFE_ATTRIBUTES = objectHasOwnProperty(cfg, 'ADD_URI_SAFE_ATTR') ? addToSet(clone(DEFAULT_URI_SAFE_ATTRIBUTES), cfg.ADD_URI_SAFE_ATTR, transformCaseFunc) : DEFAULT_URI_SAFE_ATTRIBUTES;
        DATA_URI_TAGS = objectHasOwnProperty(cfg, 'ADD_DATA_URI_TAGS') ? addToSet(clone(DEFAULT_DATA_URI_TAGS), cfg.ADD_DATA_URI_TAGS, transformCaseFunc) : DEFAULT_DATA_URI_TAGS;
        FORBID_CONTENTS = objectHasOwnProperty(cfg, 'FORBID_CONTENTS') ? addToSet({}, cfg.FORBID_CONTENTS, transformCaseFunc) : DEFAULT_FORBID_CONTENTS;
        FORBID_TAGS = objectHasOwnProperty(cfg, 'FORBID_TAGS') ? addToSet({}, cfg.FORBID_TAGS, transformCaseFunc) : clone({});
        FORBID_ATTR = objectHasOwnProperty(cfg, 'FORBID_ATTR') ? addToSet({}, cfg.FORBID_ATTR, transformCaseFunc) : clone({});
        USE_PROFILES = objectHasOwnProperty(cfg, 'USE_PROFILES') ? cfg.USE_PROFILES : false;
        ALLOW_ARIA_ATTR = cfg.ALLOW_ARIA_ATTR !== false; // Default true
        ALLOW_DATA_ATTR = cfg.ALLOW_DATA_ATTR !== false; // Default true
        ALLOW_UNKNOWN_PROTOCOLS = cfg.ALLOW_UNKNOWN_PROTOCOLS || false; // Default false
        ALLOW_SELF_CLOSE_IN_ATTR = cfg.ALLOW_SELF_CLOSE_IN_ATTR !== false; // Default true
        SAFE_FOR_TEMPLATES = cfg.SAFE_FOR_TEMPLATES || false; // Default false
        SAFE_FOR_XML = cfg.SAFE_FOR_XML !== false; // Default true
        WHOLE_DOCUMENT = cfg.WHOLE_DOCUMENT || false; // Default false
        RETURN_DOM = cfg.RETURN_DOM || false; // Default false
        RETURN_DOM_FRAGMENT = cfg.RETURN_DOM_FRAGMENT || false; // Default false
        RETURN_TRUSTED_TYPE = cfg.RETURN_TRUSTED_TYPE || false; // Default false
        FORCE_BODY = cfg.FORCE_BODY || false; // Default false
        SANITIZE_DOM = cfg.SANITIZE_DOM !== false; // Default true
        SANITIZE_NAMED_PROPS = cfg.SANITIZE_NAMED_PROPS || false; // Default false
        KEEP_CONTENT = cfg.KEEP_CONTENT !== false; // Default true
        IN_PLACE = cfg.IN_PLACE || false; // Default false
        IS_ALLOWED_URI$1 = cfg.ALLOWED_URI_REGEXP || IS_ALLOWED_URI;
        NAMESPACE = cfg.NAMESPACE || HTML_NAMESPACE;
        MATHML_TEXT_INTEGRATION_POINTS = cfg.MATHML_TEXT_INTEGRATION_POINTS || MATHML_TEXT_INTEGRATION_POINTS;
        HTML_INTEGRATION_POINTS = cfg.HTML_INTEGRATION_POINTS || HTML_INTEGRATION_POINTS;
        CUSTOM_ELEMENT_HANDLING = cfg.CUSTOM_ELEMENT_HANDLING || {};
        if (cfg.CUSTOM_ELEMENT_HANDLING && isRegexOrFunction(cfg.CUSTOM_ELEMENT_HANDLING.tagNameCheck)) {
          CUSTOM_ELEMENT_HANDLING.tagNameCheck = cfg.CUSTOM_ELEMENT_HANDLING.tagNameCheck;
        }
        if (cfg.CUSTOM_ELEMENT_HANDLING && isRegexOrFunction(cfg.CUSTOM_ELEMENT_HANDLING.attributeNameCheck)) {
          CUSTOM_ELEMENT_HANDLING.attributeNameCheck = cfg.CUSTOM_ELEMENT_HANDLING.attributeNameCheck;
        }
        if (cfg.CUSTOM_ELEMENT_HANDLING && typeof cfg.CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements === 'boolean') {
          CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements = cfg.CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements;
        }
        if (SAFE_FOR_TEMPLATES) {
          ALLOW_DATA_ATTR = false;
        }
        if (RETURN_DOM_FRAGMENT) {
          RETURN_DOM = true;
        }
        /* Parse profile info */
        if (USE_PROFILES) {
          ALLOWED_TAGS = addToSet({}, text);
          ALLOWED_ATTR = [];
          if (USE_PROFILES.html === true) {
            addToSet(ALLOWED_TAGS, html$1);
            addToSet(ALLOWED_ATTR, html);
          }
          if (USE_PROFILES.svg === true) {
            addToSet(ALLOWED_TAGS, svg$1);
            addToSet(ALLOWED_ATTR, svg);
            addToSet(ALLOWED_ATTR, xml);
          }
          if (USE_PROFILES.svgFilters === true) {
            addToSet(ALLOWED_TAGS, svgFilters);
            addToSet(ALLOWED_ATTR, svg);
            addToSet(ALLOWED_ATTR, xml);
          }
          if (USE_PROFILES.mathMl === true) {
            addToSet(ALLOWED_TAGS, mathMl$1);
            addToSet(ALLOWED_ATTR, mathMl);
            addToSet(ALLOWED_ATTR, xml);
          }
        }
        /* Merge configuration parameters */
        if (cfg.ADD_TAGS) {
          if (ALLOWED_TAGS === DEFAULT_ALLOWED_TAGS) {
            ALLOWED_TAGS = clone(ALLOWED_TAGS);
          }
          addToSet(ALLOWED_TAGS, cfg.ADD_TAGS, transformCaseFunc);
        }
        if (cfg.ADD_ATTR) {
          if (ALLOWED_ATTR === DEFAULT_ALLOWED_ATTR) {
            ALLOWED_ATTR = clone(ALLOWED_ATTR);
          }
          addToSet(ALLOWED_ATTR, cfg.ADD_ATTR, transformCaseFunc);
        }
        if (cfg.ADD_URI_SAFE_ATTR) {
          addToSet(URI_SAFE_ATTRIBUTES, cfg.ADD_URI_SAFE_ATTR, transformCaseFunc);
        }
        if (cfg.FORBID_CONTENTS) {
          if (FORBID_CONTENTS === DEFAULT_FORBID_CONTENTS) {
            FORBID_CONTENTS = clone(FORBID_CONTENTS);
          }
          addToSet(FORBID_CONTENTS, cfg.FORBID_CONTENTS, transformCaseFunc);
        }
        /* Add #text in case KEEP_CONTENT is set to true */
        if (KEEP_CONTENT) {
          ALLOWED_TAGS['#text'] = true;
        }
        /* Add html, head and body to ALLOWED_TAGS in case WHOLE_DOCUMENT is true */
        if (WHOLE_DOCUMENT) {
          addToSet(ALLOWED_TAGS, ['html', 'head', 'body']);
        }
        /* Add tbody to ALLOWED_TAGS in case tables are permitted, see #286, #365 */
        if (ALLOWED_TAGS.table) {
          addToSet(ALLOWED_TAGS, ['tbody']);
          delete FORBID_TAGS.tbody;
        }
        if (cfg.TRUSTED_TYPES_POLICY) {
          if (typeof cfg.TRUSTED_TYPES_POLICY.createHTML !== 'function') {
            throw typeErrorCreate('TRUSTED_TYPES_POLICY configuration option must provide a "createHTML" hook.');
          }
          if (typeof cfg.TRUSTED_TYPES_POLICY.createScriptURL !== 'function') {
            throw typeErrorCreate('TRUSTED_TYPES_POLICY configuration option must provide a "createScriptURL" hook.');
          }
          // Overwrite existing TrustedTypes policy.
          trustedTypesPolicy = cfg.TRUSTED_TYPES_POLICY;
          // Sign local variables required by `sanitize`.
          emptyHTML = trustedTypesPolicy.createHTML('');
        } else {
          // Uninitialized policy, attempt to initialize the internal dompurify policy.
          if (trustedTypesPolicy === undefined) {
            trustedTypesPolicy = _createTrustedTypesPolicy(trustedTypes, currentScript);
          }
          // If creating the internal policy succeeded sign internal variables.
          if (trustedTypesPolicy !== null && typeof emptyHTML === 'string') {
            emptyHTML = trustedTypesPolicy.createHTML('');
          }
        }
        // Prevent further manipulation of configuration.
        // Not available in IE8, Safari 5, etc.
        if (freeze) {
          freeze(cfg);
        }
        CONFIG = cfg;
      };
      /* Keep track of all possible SVG and MathML tags
       * so that we can perform the namespace checks
       * correctly. */
      const ALL_SVG_TAGS = addToSet({}, [...svg$1, ...svgFilters, ...svgDisallowed]);
      const ALL_MATHML_TAGS = addToSet({}, [...mathMl$1, ...mathMlDisallowed]);
      /**
       * @param element a DOM element whose namespace is being checked
       * @returns Return false if the element has a
       *  namespace that a spec-compliant parser would never
       *  return. Return true otherwise.
       */
      const _checkValidNamespace = function _checkValidNamespace(element) {
        let parent = getParentNode(element);
        // In JSDOM, if we're inside shadow DOM, then parentNode
        // can be null. We just simulate parent in this case.
        if (!parent || !parent.tagName) {
          parent = {
            namespaceURI: NAMESPACE,
            tagName: 'template'
          };
        }
        const tagName = stringToLowerCase(element.tagName);
        const parentTagName = stringToLowerCase(parent.tagName);
        if (!ALLOWED_NAMESPACES[element.namespaceURI]) {
          return false;
        }
        if (element.namespaceURI === SVG_NAMESPACE) {
          // The only way to switch from HTML namespace to SVG
          // is via <svg>. If it happens via any other tag, then
          // it should be killed.
          if (parent.namespaceURI === HTML_NAMESPACE) {
            return tagName === 'svg';
          }
          // The only way to switch from MathML to SVG is via`
          // svg if parent is either <annotation-xml> or MathML
          // text integration points.
          if (parent.namespaceURI === MATHML_NAMESPACE) {
            return tagName === 'svg' && (parentTagName === 'annotation-xml' || MATHML_TEXT_INTEGRATION_POINTS[parentTagName]);
          }
          // We only allow elements that are defined in SVG
          // spec. All others are disallowed in SVG namespace.
          return Boolean(ALL_SVG_TAGS[tagName]);
        }
        if (element.namespaceURI === MATHML_NAMESPACE) {
          // The only way to switch from HTML namespace to MathML
          // is via <math>. If it happens via any other tag, then
          // it should be killed.
          if (parent.namespaceURI === HTML_NAMESPACE) {
            return tagName === 'math';
          }
          // The only way to switch from SVG to MathML is via
          // <math> and HTML integration points
          if (parent.namespaceURI === SVG_NAMESPACE) {
            return tagName === 'math' && HTML_INTEGRATION_POINTS[parentTagName];
          }
          // We only allow elements that are defined in MathML
          // spec. All others are disallowed in MathML namespace.
          return Boolean(ALL_MATHML_TAGS[tagName]);
        }
        if (element.namespaceURI === HTML_NAMESPACE) {
          // The only way to switch from SVG to HTML is via
          // HTML integration points, and from MathML to HTML
          // is via MathML text integration points
          if (parent.namespaceURI === SVG_NAMESPACE && !HTML_INTEGRATION_POINTS[parentTagName]) {
            return false;
          }
          if (parent.namespaceURI === MATHML_NAMESPACE && !MATHML_TEXT_INTEGRATION_POINTS[parentTagName]) {
            return false;
          }
          // We disallow tags that are specific for MathML
          // or SVG and should never appear in HTML namespace
          return !ALL_MATHML_TAGS[tagName] && (COMMON_SVG_AND_HTML_ELEMENTS[tagName] || !ALL_SVG_TAGS[tagName]);
        }
        // For XHTML and XML documents that support custom namespaces
        if (PARSER_MEDIA_TYPE === 'application/xhtml+xml' && ALLOWED_NAMESPACES[element.namespaceURI]) {
          return true;
        }
        // The code should never reach this place (this means
        // that the element somehow got namespace that is not
        // HTML, SVG, MathML or allowed via ALLOWED_NAMESPACES).
        // Return false just in case.
        return false;
      };
      /**
       * _forceRemove
       *
       * @param node a DOM node
       */
      const _forceRemove = function _forceRemove(node) {
        arrayPush(DOMPurify.removed, {
          element: node
        });
        try {
          // eslint-disable-next-line unicorn/prefer-dom-node-remove
          getParentNode(node).removeChild(node);
        } catch (_) {
          remove(node);
        }
      };
      /**
       * _removeAttribute
       *
       * @param name an Attribute name
       * @param element a DOM node
       */
      const _removeAttribute = function _removeAttribute(name, element) {
        try {
          arrayPush(DOMPurify.removed, {
            attribute: element.getAttributeNode(name),
            from: element
          });
        } catch (_) {
          arrayPush(DOMPurify.removed, {
            attribute: null,
            from: element
          });
        }
        element.removeAttribute(name);
        // We void attribute values for unremovable "is" attributes
        if (name === 'is') {
          if (RETURN_DOM || RETURN_DOM_FRAGMENT) {
            try {
              _forceRemove(element);
            } catch (_) {}
          } else {
            try {
              element.setAttribute(name, '');
            } catch (_) {}
          }
        }
      };
      /**
       * _initDocument
       *
       * @param dirty - a string of dirty markup
       * @return a DOM, filled with the dirty markup
       */
      const _initDocument = function _initDocument(dirty) {
        /* Create a HTML document */
        let doc = null;
        let leadingWhitespace = null;
        if (FORCE_BODY) {
          dirty = '<remove></remove>' + dirty;
        } else {
          /* If FORCE_BODY isn't used, leading whitespace needs to be preserved manually */
          const matches = stringMatch(dirty, /^[\r\n\t ]+/);
          leadingWhitespace = matches && matches[0];
        }
        if (PARSER_MEDIA_TYPE === 'application/xhtml+xml' && NAMESPACE === HTML_NAMESPACE) {
          // Root of XHTML doc must contain xmlns declaration (see https://www.w3.org/TR/xhtml1/normative.html#strict)
          dirty = '<html xmlns="http://www.w3.org/1999/xhtml"><head></head><body>' + dirty + '</body></html>';
        }
        const dirtyPayload = trustedTypesPolicy ? trustedTypesPolicy.createHTML(dirty) : dirty;
        /*
         * Use the DOMParser API by default, fallback later if needs be
         * DOMParser not work for svg when has multiple root element.
         */
        if (NAMESPACE === HTML_NAMESPACE) {
          try {
            doc = new DOMParser().parseFromString(dirtyPayload, PARSER_MEDIA_TYPE);
          } catch (_) {}
        }
        /* Use createHTMLDocument in case DOMParser is not available */
        if (!doc || !doc.documentElement) {
          doc = implementation.createDocument(NAMESPACE, 'template', null);
          try {
            doc.documentElement.innerHTML = IS_EMPTY_INPUT ? emptyHTML : dirtyPayload;
          } catch (_) {
            // Syntax error if dirtyPayload is invalid xml
          }
        }
        const body = doc.body || doc.documentElement;
        if (dirty && leadingWhitespace) {
          body.insertBefore(document.createTextNode(leadingWhitespace), body.childNodes[0] || null);
        }
        /* Work on whole document or just its body */
        if (NAMESPACE === HTML_NAMESPACE) {
          return getElementsByTagName.call(doc, WHOLE_DOCUMENT ? 'html' : 'body')[0];
        }
        return WHOLE_DOCUMENT ? doc.documentElement : body;
      };
      /**
       * Creates a NodeIterator object that you can use to traverse filtered lists of nodes or elements in a document.
       *
       * @param root The root element or node to start traversing on.
       * @return The created NodeIterator
       */
      const _createNodeIterator = function _createNodeIterator(root) {
        return createNodeIterator.call(root.ownerDocument || root, root,
        // eslint-disable-next-line no-bitwise
        NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_COMMENT | NodeFilter.SHOW_TEXT | NodeFilter.SHOW_PROCESSING_INSTRUCTION | NodeFilter.SHOW_CDATA_SECTION, null);
      };
      /**
       * _isClobbered
       *
       * @param element element to check for clobbering attacks
       * @return true if clobbered, false if safe
       */
      const _isClobbered = function _isClobbered(element) {
        return element instanceof HTMLFormElement && (typeof element.nodeName !== 'string' || typeof element.textContent !== 'string' || typeof element.removeChild !== 'function' || !(element.attributes instanceof NamedNodeMap) || typeof element.removeAttribute !== 'function' || typeof element.setAttribute !== 'function' || typeof element.namespaceURI !== 'string' || typeof element.insertBefore !== 'function' || typeof element.hasChildNodes !== 'function');
      };
      /**
       * Checks whether the given object is a DOM node.
       *
       * @param value object to check whether it's a DOM node
       * @return true is object is a DOM node
       */
      const _isNode = function _isNode(value) {
        return typeof Node === 'function' && value instanceof Node;
      };
      function _executeHooks(hooks, currentNode, data) {
        arrayForEach(hooks, hook => {
          hook.call(DOMPurify, currentNode, data, CONFIG);
        });
      }
      /**
       * _sanitizeElements
       *
       * @protect nodeName
       * @protect textContent
       * @protect removeChild
       * @param currentNode to check for permission to exist
       * @return true if node was killed, false if left alive
       */
      const _sanitizeElements = function _sanitizeElements(currentNode) {
        let content = null;
        /* Execute a hook if present */
        _executeHooks(hooks.beforeSanitizeElements, currentNode, null);
        /* Check if element is clobbered or can clobber */
        if (_isClobbered(currentNode)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Now let's check the element's type and name */
        const tagName = transformCaseFunc(currentNode.nodeName);
        /* Execute a hook if present */
        _executeHooks(hooks.uponSanitizeElement, currentNode, {
          tagName,
          allowedTags: ALLOWED_TAGS
        });
        /* Detect mXSS attempts abusing namespace confusion */
        if (SAFE_FOR_XML && currentNode.hasChildNodes() && !_isNode(currentNode.firstElementChild) && regExpTest(/<[/\w!]/g, currentNode.innerHTML) && regExpTest(/<[/\w!]/g, currentNode.textContent)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Remove any occurrence of processing instructions */
        if (currentNode.nodeType === NODE_TYPE.progressingInstruction) {
          _forceRemove(currentNode);
          return true;
        }
        /* Remove any kind of possibly harmful comments */
        if (SAFE_FOR_XML && currentNode.nodeType === NODE_TYPE.comment && regExpTest(/<[/\w]/g, currentNode.data)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Remove element if anything forbids its presence */
        if (!ALLOWED_TAGS[tagName] || FORBID_TAGS[tagName]) {
          /* Check if we have a custom element to handle */
          if (!FORBID_TAGS[tagName] && _isBasicCustomElement(tagName)) {
            if (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.tagNameCheck, tagName)) {
              return false;
            }
            if (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.tagNameCheck(tagName)) {
              return false;
            }
          }
          /* Keep content except for bad-listed elements */
          if (KEEP_CONTENT && !FORBID_CONTENTS[tagName]) {
            const parentNode = getParentNode(currentNode) || currentNode.parentNode;
            const childNodes = getChildNodes(currentNode) || currentNode.childNodes;
            if (childNodes && parentNode) {
              const childCount = childNodes.length;
              for (let i = childCount - 1; i >= 0; --i) {
                const childClone = cloneNode(childNodes[i], true);
                childClone.__removalCount = (currentNode.__removalCount || 0) + 1;
                parentNode.insertBefore(childClone, getNextSibling(currentNode));
              }
            }
          }
          _forceRemove(currentNode);
          return true;
        }
        /* Check whether element has a valid namespace */
        if (currentNode instanceof Element && !_checkValidNamespace(currentNode)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Make sure that older browsers don't get fallback-tag mXSS */
        if ((tagName === 'noscript' || tagName === 'noembed' || tagName === 'noframes') && regExpTest(/<\/no(script|embed|frames)/i, currentNode.innerHTML)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Sanitize element content to be template-safe */
        if (SAFE_FOR_TEMPLATES && currentNode.nodeType === NODE_TYPE.text) {
          /* Get the element's text content */
          content = currentNode.textContent;
          arrayForEach([MUSTACHE_EXPR, ERB_EXPR, TMPLIT_EXPR], expr => {
            content = stringReplace(content, expr, ' ');
          });
          if (currentNode.textContent !== content) {
            arrayPush(DOMPurify.removed, {
              element: currentNode.cloneNode()
            });
            currentNode.textContent = content;
          }
        }
        /* Execute a hook if present */
        _executeHooks(hooks.afterSanitizeElements, currentNode, null);
        return false;
      };
      /**
       * _isValidAttribute
       *
       * @param lcTag Lowercase tag name of containing element.
       * @param lcName Lowercase attribute name.
       * @param value Attribute value.
       * @return Returns true if `value` is valid, otherwise false.
       */
      // eslint-disable-next-line complexity
      const _isValidAttribute = function _isValidAttribute(lcTag, lcName, value) {
        /* Make sure attribute cannot clobber */
        if (SANITIZE_DOM && (lcName === 'id' || lcName === 'name') && (value in document || value in formElement)) {
          return false;
        }
        /* Allow valid data-* attributes: At least one character after "-"
            (https://html.spec.whatwg.org/multipage/dom.html#embedding-custom-non-visible-data-with-the-data-*-attributes)
            XML-compatible (https://html.spec.whatwg.org/multipage/infrastructure.html#xml-compatible and http://www.w3.org/TR/xml/#d0e804)
            We don't need to check the value; it's always URI safe. */
        if (ALLOW_DATA_ATTR && !FORBID_ATTR[lcName] && regExpTest(DATA_ATTR, lcName)) ; else if (ALLOW_ARIA_ATTR && regExpTest(ARIA_ATTR, lcName)) ; else if (!ALLOWED_ATTR[lcName] || FORBID_ATTR[lcName]) {
          if (
          // First condition does a very basic check if a) it's basically a valid custom element tagname AND
          // b) if the tagName passes whatever the user has configured for CUSTOM_ELEMENT_HANDLING.tagNameCheck
          // and c) if the attribute name passes whatever the user has configured for CUSTOM_ELEMENT_HANDLING.attributeNameCheck
          _isBasicCustomElement(lcTag) && (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.tagNameCheck, lcTag) || CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.tagNameCheck(lcTag)) && (CUSTOM_ELEMENT_HANDLING.attributeNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.attributeNameCheck, lcName) || CUSTOM_ELEMENT_HANDLING.attributeNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.attributeNameCheck(lcName)) ||
          // Alternative, second condition checks if it's an `is`-attribute, AND
          // the value passes whatever the user has configured for CUSTOM_ELEMENT_HANDLING.tagNameCheck
          lcName === 'is' && CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements && (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.tagNameCheck, value) || CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.tagNameCheck(value))) ; else {
            return false;
          }
          /* Check value is safe. First, is attr inert? If so, is safe */
        } else if (URI_SAFE_ATTRIBUTES[lcName]) ; else if (regExpTest(IS_ALLOWED_URI$1, stringReplace(value, ATTR_WHITESPACE, ''))) ; else if ((lcName === 'src' || lcName === 'xlink:href' || lcName === 'href') && lcTag !== 'script' && stringIndexOf(value, 'data:') === 0 && DATA_URI_TAGS[lcTag]) ; else if (ALLOW_UNKNOWN_PROTOCOLS && !regExpTest(IS_SCRIPT_OR_DATA, stringReplace(value, ATTR_WHITESPACE, ''))) ; else if (value) {
          return false;
        } else ;
        return true;
      };
      /**
       * _isBasicCustomElement
       * checks if at least one dash is included in tagName, and it's not the first char
       * for more sophisticated checking see https://github.com/sindresorhus/validate-element-name
       *
       * @param tagName name of the tag of the node to sanitize
       * @returns Returns true if the tag name meets the basic criteria for a custom element, otherwise false.
       */
      const _isBasicCustomElement = function _isBasicCustomElement(tagName) {
        return tagName !== 'annotation-xml' && stringMatch(tagName, CUSTOM_ELEMENT);
      };
      /**
       * _sanitizeAttributes
       *
       * @protect attributes
       * @protect nodeName
       * @protect removeAttribute
       * @protect setAttribute
       *
       * @param currentNode to sanitize
       */
      const _sanitizeAttributes = function _sanitizeAttributes(currentNode) {
        /* Execute a hook if present */
        _executeHooks(hooks.beforeSanitizeAttributes, currentNode, null);
        const {
          attributes
        } = currentNode;
        /* Check if we have attributes; if not we might have a text node */
        if (!attributes || _isClobbered(currentNode)) {
          return;
        }
        const hookEvent = {
          attrName: '',
          attrValue: '',
          keepAttr: true,
          allowedAttributes: ALLOWED_ATTR,
          forceKeepAttr: undefined
        };
        let l = attributes.length;
        /* Go backwards over all attributes; safely remove bad ones */
        while (l--) {
          const attr = attributes[l];
          const {
            name,
            namespaceURI,
            value: attrValue
          } = attr;
          const lcName = transformCaseFunc(name);
          const initValue = attrValue;
          let value = name === 'value' ? initValue : stringTrim(initValue);
          /* Execute a hook if present */
          hookEvent.attrName = lcName;
          hookEvent.attrValue = value;
          hookEvent.keepAttr = true;
          hookEvent.forceKeepAttr = undefined; // Allows developers to see this is a property they can set
          _executeHooks(hooks.uponSanitizeAttribute, currentNode, hookEvent);
          value = hookEvent.attrValue;
          /* Full DOM Clobbering protection via namespace isolation,
           * Prefix id and name attributes with `user-content-`
           */
          if (SANITIZE_NAMED_PROPS && (lcName === 'id' || lcName === 'name')) {
            // Remove the attribute with this value
            _removeAttribute(name, currentNode);
            // Prefix the value and later re-create the attribute with the sanitized value
            value = SANITIZE_NAMED_PROPS_PREFIX + value;
          }
          /* Work around a security issue with comments inside attributes */
          if (SAFE_FOR_XML && regExpTest(/((--!?|])>)|<\/(style|title)/i, value)) {
            _removeAttribute(name, currentNode);
            continue;
          }
          /* Did the hooks approve of the attribute? */
          if (hookEvent.forceKeepAttr) {
            continue;
          }
          /* Did the hooks approve of the attribute? */
          if (!hookEvent.keepAttr) {
            _removeAttribute(name, currentNode);
            continue;
          }
          /* Work around a security issue in jQuery 3.0 */
          if (!ALLOW_SELF_CLOSE_IN_ATTR && regExpTest(/\/>/i, value)) {
            _removeAttribute(name, currentNode);
            continue;
          }
          /* Sanitize attribute content to be template-safe */
          if (SAFE_FOR_TEMPLATES) {
            arrayForEach([MUSTACHE_EXPR, ERB_EXPR, TMPLIT_EXPR], expr => {
              value = stringReplace(value, expr, ' ');
            });
          }
          /* Is `value` valid for this attribute? */
          const lcTag = transformCaseFunc(currentNode.nodeName);
          if (!_isValidAttribute(lcTag, lcName, value)) {
            _removeAttribute(name, currentNode);
            continue;
          }
          /* Handle attributes that require Trusted Types */
          if (trustedTypesPolicy && typeof trustedTypes === 'object' && typeof trustedTypes.getAttributeType === 'function') {
            if (namespaceURI) ; else {
              switch (trustedTypes.getAttributeType(lcTag, lcName)) {
                case 'TrustedHTML':
                  {
                    value = trustedTypesPolicy.createHTML(value);
                    break;
                  }
                case 'TrustedScriptURL':
                  {
                    value = trustedTypesPolicy.createScriptURL(value);
                    break;
                  }
              }
            }
          }
          /* Handle invalid data-* attribute set by try-catching it */
          if (value !== initValue) {
            try {
              if (namespaceURI) {
                currentNode.setAttributeNS(namespaceURI, name, value);
              } else {
                /* Fallback to setAttribute() for browser-unrecognized namespaces e.g. "x-schema". */
                currentNode.setAttribute(name, value);
              }
              if (_isClobbered(currentNode)) {
                _forceRemove(currentNode);
              } else {
                arrayPop(DOMPurify.removed);
              }
            } catch (_) {
              _removeAttribute(name, currentNode);
            }
          }
        }
        /* Execute a hook if present */
        _executeHooks(hooks.afterSanitizeAttributes, currentNode, null);
      };
      /**
       * _sanitizeShadowDOM
       *
       * @param fragment to iterate over recursively
       */
      const _sanitizeShadowDOM = function _sanitizeShadowDOM(fragment) {
        let shadowNode = null;
        const shadowIterator = _createNodeIterator(fragment);
        /* Execute a hook if present */
        _executeHooks(hooks.beforeSanitizeShadowDOM, fragment, null);
        while (shadowNode = shadowIterator.nextNode()) {
          /* Execute a hook if present */
          _executeHooks(hooks.uponSanitizeShadowNode, shadowNode, null);
          /* Sanitize tags and elements */
          _sanitizeElements(shadowNode);
          /* Check attributes next */
          _sanitizeAttributes(shadowNode);
          /* Deep shadow DOM detected */
          if (shadowNode.content instanceof DocumentFragment) {
            _sanitizeShadowDOM(shadowNode.content);
          }
        }
        /* Execute a hook if present */
        _executeHooks(hooks.afterSanitizeShadowDOM, fragment, null);
      };
      // eslint-disable-next-line complexity
      DOMPurify.sanitize = function (dirty) {
        let cfg = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
        let body = null;
        let importedNode = null;
        let currentNode = null;
        let returnNode = null;
        /* Make sure we have a string to sanitize.
          DO NOT return early, as this will return the wrong type if
          the user has requested a DOM object rather than a string */
        IS_EMPTY_INPUT = !dirty;
        if (IS_EMPTY_INPUT) {
          dirty = '<!-->';
        }
        /* Stringify, in case dirty is an object */
        if (typeof dirty !== 'string' && !_isNode(dirty)) {
          if (typeof dirty.toString === 'function') {
            dirty = dirty.toString();
            if (typeof dirty !== 'string') {
              throw typeErrorCreate('dirty is not a string, aborting');
            }
          } else {
            throw typeErrorCreate('toString is not a function');
          }
        }
        /* Return dirty HTML if DOMPurify cannot run */
        if (!DOMPurify.isSupported) {
          return dirty;
        }
        /* Assign config vars */
        if (!SET_CONFIG) {
          _parseConfig(cfg);
        }
        /* Clean up removed elements */
        DOMPurify.removed = [];
        /* Check if dirty is correctly typed for IN_PLACE */
        if (typeof dirty === 'string') {
          IN_PLACE = false;
        }
        if (IN_PLACE) {
          /* Do some early pre-sanitization to avoid unsafe root nodes */
          if (dirty.nodeName) {
            const tagName = transformCaseFunc(dirty.nodeName);
            if (!ALLOWED_TAGS[tagName] || FORBID_TAGS[tagName]) {
              throw typeErrorCreate('root node is forbidden and cannot be sanitized in-place');
            }
          }
        } else if (dirty instanceof Node) {
          /* If dirty is a DOM element, append to an empty document to avoid
             elements being stripped by the parser */
          body = _initDocument('<!---->');
          importedNode = body.ownerDocument.importNode(dirty, true);
          if (importedNode.nodeType === NODE_TYPE.element && importedNode.nodeName === 'BODY') {
            /* Node is already a body, use as is */
            body = importedNode;
          } else if (importedNode.nodeName === 'HTML') {
            body = importedNode;
          } else {
            // eslint-disable-next-line unicorn/prefer-dom-node-append
            body.appendChild(importedNode);
          }
        } else {
          /* Exit directly if we have nothing to do */
          if (!RETURN_DOM && !SAFE_FOR_TEMPLATES && !WHOLE_DOCUMENT &&
          // eslint-disable-next-line unicorn/prefer-includes
          dirty.indexOf('<') === -1) {
            return trustedTypesPolicy && RETURN_TRUSTED_TYPE ? trustedTypesPolicy.createHTML(dirty) : dirty;
          }
          /* Initialize the document to work on */
          body = _initDocument(dirty);
          /* Check we have a DOM node from the data */
          if (!body) {
            return RETURN_DOM ? null : RETURN_TRUSTED_TYPE ? emptyHTML : '';
          }
        }
        /* Remove first element node (ours) if FORCE_BODY is set */
        if (body && FORCE_BODY) {
          _forceRemove(body.firstChild);
        }
        /* Get node iterator */
        const nodeIterator = _createNodeIterator(IN_PLACE ? dirty : body);
        /* Now start iterating over the created document */
        while (currentNode = nodeIterator.nextNode()) {
          /* Sanitize tags and elements */
          _sanitizeElements(currentNode);
          /* Check attributes next */
          _sanitizeAttributes(currentNode);
          /* Shadow DOM detected, sanitize it */
          if (currentNode.content instanceof DocumentFragment) {
            _sanitizeShadowDOM(currentNode.content);
          }
        }
        /* If we sanitized `dirty` in-place, return it. */
        if (IN_PLACE) {
          return dirty;
        }
        /* Return sanitized string or DOM */
        if (RETURN_DOM) {
          if (RETURN_DOM_FRAGMENT) {
            returnNode = createDocumentFragment.call(body.ownerDocument);
            while (body.firstChild) {
              // eslint-disable-next-line unicorn/prefer-dom-node-append
              returnNode.appendChild(body.firstChild);
            }
          } else {
            returnNode = body;
          }
          if (ALLOWED_ATTR.shadowroot || ALLOWED_ATTR.shadowrootmode) {
            /*
              AdoptNode() is not used because internal state is not reset
              (e.g. the past names map of a HTMLFormElement), this is safe
              in theory but we would rather not risk another attack vector.
              The state that is cloned by importNode() is explicitly defined
              by the specs.
            */
            returnNode = importNode.call(originalDocument, returnNode, true);
          }
          return returnNode;
        }
        let serializedHTML = WHOLE_DOCUMENT ? body.outerHTML : body.innerHTML;
        /* Serialize doctype if allowed */
        if (WHOLE_DOCUMENT && ALLOWED_TAGS['!doctype'] && body.ownerDocument && body.ownerDocument.doctype && body.ownerDocument.doctype.name && regExpTest(DOCTYPE_NAME, body.ownerDocument.doctype.name)) {
          serializedHTML = '<!DOCTYPE ' + body.ownerDocument.doctype.name + '>\n' + serializedHTML;
        }
        /* Sanitize final string template-safe */
        if (SAFE_FOR_TEMPLATES) {
          arrayForEach([MUSTACHE_EXPR, ERB_EXPR, TMPLIT_EXPR], expr => {
            serializedHTML = stringReplace(serializedHTML, expr, ' ');
          });
        }
        return trustedTypesPolicy && RETURN_TRUSTED_TYPE ? trustedTypesPolicy.createHTML(serializedHTML) : serializedHTML;
      };
      DOMPurify.setConfig = function () {
        let cfg = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        _parseConfig(cfg);
        SET_CONFIG = true;
      };
      DOMPurify.clearConfig = function () {
        CONFIG = null;
        SET_CONFIG = false;
      };
      DOMPurify.isValidAttribute = function (tag, attr, value) {
        /* Initialize shared config vars if necessary. */
        if (!CONFIG) {
          _parseConfig({});
        }
        const lcTag = transformCaseFunc(tag);
        const lcName = transformCaseFunc(attr);
        return _isValidAttribute(lcTag, lcName, value);
      };
      DOMPurify.addHook = function (entryPoint, hookFunction) {
        if (typeof hookFunction !== 'function') {
          return;
        }
        arrayPush(hooks[entryPoint], hookFunction);
      };
      DOMPurify.removeHook = function (entryPoint, hookFunction) {
        if (hookFunction !== undefined) {
          const index = arrayLastIndexOf(hooks[entryPoint], hookFunction);
          return index === -1 ? undefined : arraySplice(hooks[entryPoint], index, 1)[0];
        }
        return arrayPop(hooks[entryPoint]);
      };
      DOMPurify.removeHooks = function (entryPoint) {
        hooks[entryPoint] = [];
      };
      DOMPurify.removeAllHooks = function () {
        hooks = _createHooksMap();
      };
      return DOMPurify;
    }
    var purify = createDOMPurify();

    /**
     * This class handles parsing, modification and serialization of URI/URL strings.
     * @class tinymce.util.URI
     */
    const each$6 = Tools.each, trim = Tools.trim;
    const queryParts = [
        'source', 'protocol', 'authority', 'userInfo', 'user', 'password', 'host',
        'port', 'relative', 'path', 'directory', 'file', 'query', 'anchor'
    ];
    const DEFAULT_PORTS = {
        ftp: 21,
        http: 80,
        https: 443,
        mailto: 25
    };
    const safeSvgDataUrlElements = ['img', 'video'];
    const blockSvgDataUris = (allowSvgDataUrls, tagName) => {
        if (isNonNullable(allowSvgDataUrls)) {
            return !allowSvgDataUrls;
        }
        else {
            // Only allow SVGs by default on images/videos since the browser won't execute scripts on those elements
            return isNonNullable(tagName) ? !contains$2(safeSvgDataUrlElements, tagName) : true;
        }
    };
    const decodeUri = (encodedUri) => {
        try {
            // Might throw malformed URI sequence
            return decodeURIComponent(encodedUri);
        }
        catch {
            // Fallback to non UTF-8 decoder
            return unescape(encodedUri);
        }
    };
    const isInvalidUri = (settings, uri, tagName) => {
        // remove all whitespaces from decoded uri to prevent impact on regex matching
        const decodedUri = decodeUri(uri).replace(/\s/g, '');
        if (settings.allow_script_urls) {
            return false;
            // Ensure we don't have a javascript URI, as that is not safe since it allows arbitrary JavaScript execution
        }
        else if (/((java|vb)script|mhtml):/i.test(decodedUri)) {
            return true;
        }
        else if (settings.allow_html_data_urls) {
            return false;
        }
        else if (/^data:image\//i.test(decodedUri)) {
            return blockSvgDataUris(settings.allow_svg_data_urls, tagName) && /^data:image\/svg\+xml/i.test(decodedUri);
        }
        else {
            return /^data:/i.test(decodedUri);
        }
    };
    class URI {
        static parseDataUri(uri) {
            let type;
            const uriComponents = decodeURIComponent(uri).split(',');
            const matches = /data:([^;]+)/.exec(uriComponents[0]);
            if (matches) {
                type = matches[1];
            }
            return {
                type,
                data: uriComponents[1]
            };
        }
        /**
         * Check to see if a URI is safe to use in the Document Object Model (DOM). This will return
         * true if the URI can be used in the DOM without potentially triggering a security issue.
         *
         * @method isDomSafe
         * @static
         * @param {String} uri The URI to be validated.
         * @param {Object} context An optional HTML tag name where the element is being used.
         * @param {Object} options An optional set of options to use when determining if the URI is safe.
         * @return {Boolean} True if the URI is safe, otherwise false.
         */
        static isDomSafe(uri, context, options = {}) {
            if (options.allow_script_urls) {
                return true;
            }
            else {
                const decodedUri = Entities.decode(uri).replace(/[\s\u0000-\u001F]+/g, '');
                return !isInvalidUri(options, decodedUri, context);
            }
        }
        static getDocumentBaseUrl(loc) {
            let baseUrl;
            // Pass applewebdata:// and other non web protocols though
            if (loc.protocol.indexOf('http') !== 0 && loc.protocol !== 'file:') {
                baseUrl = loc.href ?? '';
            }
            else {
                baseUrl = loc.protocol + '//' + loc.host + loc.pathname;
            }
            if (/^[^:]+:\/\/\/?[^\/]+\//.test(baseUrl)) {
                baseUrl = baseUrl.replace(/[\?#].*$/, '').replace(/[\/\\][^\/]+$/, '');
                if (!/[\/\\]$/.test(baseUrl)) {
                    baseUrl += '/';
                }
            }
            return baseUrl;
        }
        source;
        protocol;
        authority;
        userInfo;
        user;
        password;
        host;
        port;
        relative;
        path = '';
        directory = '';
        file;
        query;
        anchor;
        settings;
        /**
         * Constructs a new URI instance.
         *
         * @constructor
         * @method URI
         * @param {String} url URI string to parse.
         * @param {Object} settings Optional settings object.
         */
        constructor(url, settings = {}) {
            url = trim(url);
            this.settings = settings;
            const baseUri = settings.base_uri;
            const self = this;
            // Strange app protocol that isn't http/https or local anchor
            // For example: mailto,skype,tel etc.
            if (/^([\w\-]+):([^\/]{2})/i.test(url) || /^\s*#/.test(url)) {
                self.source = url;
                return;
            }
            const isProtocolRelative = url.indexOf('//') === 0;
            // Absolute path with no host, fake host and protocol
            if (url.indexOf('/') === 0 && !isProtocolRelative) {
                url = (baseUri ? baseUri.protocol || 'http' : 'http') + '://mce_host' + url;
            }
            // Relative path http:// or protocol relative //path
            if (!/^[\w\-]*:?\/\//.test(url)) {
                const baseUrl = baseUri ? baseUri.path : new URI(document.location.href).directory;
                if (baseUri?.protocol === '') {
                    url = '//mce_host' + self.toAbsPath(baseUrl, url);
                }
                else {
                    const match = /([^#?]*)([#?]?.*)/.exec(url);
                    if (match) {
                        url = ((baseUri && baseUri.protocol) || 'http') + '://mce_host' + self.toAbsPath(baseUrl, match[1]) + match[2];
                    }
                }
            }
            // Parse URL (Credits goes to Steave, http://blog.stevenlevithan.com/archives/parseuri)
            url = url.replace(/@@/g, '(mce_at)'); // Zope 3 workaround, they use @@something
            const urlMatch = /^(?:(?![^:@]+:[^:@\/]*@)([^:\/?#.]+):)?(?:\/\/)?((?:(([^:@\/]*):?([^:@\/]*))?@)?(\[[a-zA-Z0-9:.%]+\]|[^:\/?#]*)(?::(\d*))?)(((\/(?:[^?#](?![^?#\/]*\.[^?#\/.]+(?:[?#]|$)))*\/?)?([^?#\/]*))(?:\?([^#]*))?(?:#(.*))?)/.exec(url);
            if (urlMatch) {
                each$6(queryParts, (v, i) => {
                    let part = urlMatch[i];
                    // Zope 3 workaround, they use @@something
                    if (part) {
                        part = part.replace(/\(mce_at\)/g, '@@');
                    }
                    self[v] = part;
                });
            }
            if (baseUri) {
                if (!self.protocol) {
                    self.protocol = baseUri.protocol;
                }
                if (!self.userInfo) {
                    self.userInfo = baseUri.userInfo;
                }
                if (!self.port && self.host === 'mce_host') {
                    self.port = baseUri.port;
                }
                if (!self.host || self.host === 'mce_host') {
                    self.host = baseUri.host;
                }
                self.source = '';
            }
            if (isProtocolRelative) {
                self.protocol = '';
            }
        }
        /**
         * Sets the internal path part of the URI.
         *
         * @method setPath
         * @param {String} path Path string to set.
         */
        setPath(path) {
            const pathMatch = /^(.*?)\/?(\w+)?$/.exec(path);
            // Update path parts
            if (pathMatch) {
                this.path = pathMatch[0];
                this.directory = pathMatch[1];
                this.file = pathMatch[2];
            }
            // Rebuild source
            this.source = '';
            this.getURI();
        }
        /**
         * Converts the specified URI into a relative URI based on the current URI instance location.
         *
         * @method toRelative
         * @param {String} uri URI to convert into a relative path/URI.
         * @return {String} Relative URI from the point specified in the current URI instance.
         * @example
         * // Converts an absolute URL to an relative URL url will be somedir/somefile.htm
         * const url = new tinymce.util.URI('http://www.site.com/dir/').toRelative('http://www.site.com/dir/somedir/somefile.htm');
         */
        toRelative(uri) {
            if (uri === './') {
                return uri;
            }
            const relativeUri = new URI(uri, { base_uri: this });
            // Not on same domain/port or protocol
            if ((relativeUri.host !== 'mce_host' && this.host !== relativeUri.host && relativeUri.host) || this.port !== relativeUri.port ||
                (this.protocol !== relativeUri.protocol && relativeUri.protocol !== '')) {
                return relativeUri.getURI();
            }
            const tu = this.getURI(), uu = relativeUri.getURI();
            // Allow usage of the base_uri when relative_urls = true
            if (tu === uu || (tu.charAt(tu.length - 1) === '/' && tu.substr(0, tu.length - 1) === uu)) {
                return tu;
            }
            let output = this.toRelPath(this.path, relativeUri.path);
            // Add query
            if (relativeUri.query) {
                output += '?' + relativeUri.query;
            }
            // Add anchor
            if (relativeUri.anchor) {
                output += '#' + relativeUri.anchor;
            }
            return output;
        }
        /**
         * Converts the specified URI into a absolute URI based on the current URI instance location.
         *
         * @method toAbsolute
         * @param {String} uri URI to convert into a relative path/URI.
         * @param {Boolean} noHost No host and protocol prefix.
         * @return {String} Absolute URI from the point specified in the current URI instance.
         * @example
         * // Converts an relative URL to an absolute URL url will be http://www.site.com/dir/somedir/somefile.htm
         * const url = new tinymce.util.URI('http://www.site.com/dir/').toAbsolute('somedir/somefile.htm');
         */
        toAbsolute(uri, noHost) {
            const absoluteUri = new URI(uri, { base_uri: this });
            return absoluteUri.getURI(noHost && this.isSameOrigin(absoluteUri));
        }
        /**
         * Determine whether the given URI has the same origin as this URI.  Based on RFC-6454.
         * Supports default ports for protocols listed in DEFAULT_PORTS.  Unsupported protocols will fail safe: they
         * won't match, if the port specifications differ.
         *
         * @method isSameOrigin
         * @param {tinymce.util.URI} uri Uri instance to compare.
         * @returns {Boolean} True if the origins are the same.
         */
        isSameOrigin(uri) {
            // eslint-disable-next-line eqeqeq
            if (this.host == uri.host && this.protocol == uri.protocol) {
                // eslint-disable-next-line eqeqeq
                if (this.port == uri.port) {
                    return true;
                }
                const defaultPort = this.protocol ? DEFAULT_PORTS[this.protocol] : null;
                // eslint-disable-next-line eqeqeq
                if (defaultPort && ((this.port || defaultPort) == (uri.port || defaultPort))) {
                    return true;
                }
            }
            return false;
        }
        /**
         * Converts a absolute path into a relative path.
         *
         * @method toRelPath
         * @param {String} base Base point to convert the path from.
         * @param {String} path Absolute path to convert into a relative path.
         */
        toRelPath(base, path) {
            let breakPoint = 0, out = '', i, l;
            // Split the paths
            const normalizedBase = base.substring(0, base.lastIndexOf('/')).split('/');
            const items = path.split('/');
            if (normalizedBase.length >= items.length) {
                for (i = 0, l = normalizedBase.length; i < l; i++) {
                    if (i >= items.length || normalizedBase[i] !== items[i]) {
                        breakPoint = i + 1;
                        break;
                    }
                }
            }
            if (normalizedBase.length < items.length) {
                for (i = 0, l = items.length; i < l; i++) {
                    if (i >= normalizedBase.length || normalizedBase[i] !== items[i]) {
                        breakPoint = i + 1;
                        break;
                    }
                }
            }
            if (breakPoint === 1) {
                return path;
            }
            for (i = 0, l = normalizedBase.length - (breakPoint - 1); i < l; i++) {
                out += '../';
            }
            for (i = breakPoint - 1, l = items.length; i < l; i++) {
                if (i !== breakPoint - 1) {
                    out += '/' + items[i];
                }
                else {
                    out += items[i];
                }
            }
            return out;
        }
        /**
         * Converts a relative path into a absolute path.
         *
         * @method toAbsPath
         * @param {String} base Base point to convert the path from.
         * @param {String} path Relative path to convert into an absolute path.
         */
        toAbsPath(base, path) {
            let nb = 0;
            // Split paths
            const tr = /\/$/.test(path) ? '/' : '';
            const normalizedBase = base.split('/');
            const normalizedPath = path.split('/');
            // Remove empty chunks
            const baseParts = [];
            each$6(normalizedBase, (k) => {
                if (k) {
                    baseParts.push(k);
                }
            });
            // Merge relURLParts chunks
            const pathParts = [];
            for (let i = normalizedPath.length - 1; i >= 0; i--) {
                // Ignore empty or .
                if (normalizedPath[i].length === 0 || normalizedPath[i] === '.') {
                    continue;
                }
                // Is parent
                if (normalizedPath[i] === '..') {
                    nb++;
                    continue;
                }
                // Move up
                if (nb > 0) {
                    nb--;
                    continue;
                }
                pathParts.push(normalizedPath[i]);
            }
            const i = baseParts.length - nb;
            // If /a/b/c or /
            let outPath;
            if (i <= 0) {
                outPath = reverse(pathParts).join('/');
            }
            else {
                outPath = baseParts.slice(0, i).join('/') + '/' + reverse(pathParts).join('/');
            }
            // Add front / if it's needed
            if (outPath.indexOf('/') !== 0) {
                outPath = '/' + outPath;
            }
            // Add trailing / if it's needed
            if (tr && outPath.lastIndexOf('/') !== outPath.length - 1) {
                outPath += tr;
            }
            return outPath;
        }
        /**
         * Returns the full URI of the internal structure.
         *
         * @method getURI
         * @param {Boolean} noProtoHost Optional no host and protocol part. Defaults to false.
         */
        getURI(noProtoHost = false) {
            let s;
            // Rebuild source
            if (!this.source || noProtoHost) {
                s = '';
                if (!noProtoHost) {
                    if (this.protocol) {
                        s += this.protocol + '://';
                    }
                    else {
                        s += '//';
                    }
                    if (this.userInfo) {
                        s += this.userInfo + '@';
                    }
                    if (this.host) {
                        s += this.host;
                    }
                    if (this.port) {
                        s += ':' + this.port;
                    }
                }
                if (this.path) {
                    s += this.path;
                }
                if (this.query) {
                    s += '?' + this.query;
                }
                if (this.anchor) {
                    s += '#' + this.anchor;
                }
                this.source = s;
            }
            return this.source;
        }
    }

    // A list of attributes that should be filtered further based on the parser settings
    const filteredUrlAttrs = Tools.makeMap('src,href,data,background,action,formaction,poster,xlink:href');
    const internalElementAttr = 'data-mce-type';
    let uid = 0;
    const processNode = (node, settings, schema, scope, evt) => {
        const validate = settings.validate;
        const specialElements = schema.getSpecialElements();
        if (node.nodeType === COMMENT) {
            // Pad conditional comments if they aren't allowed
            if (!settings.allow_conditional_comments && /^\[if/i.test(node.nodeValue ?? '')) {
                node.nodeValue = ' ' + node.nodeValue;
            }
            if (settings.sanitize && settings.allow_html_in_comments && isString(node.nodeValue)) {
                node.nodeValue = encodeData(node.nodeValue);
            }
        }
        const lcTagName = evt?.tagName ?? node.nodeName.toLowerCase();
        if (scope !== 'html' && schema.isValid(scope)) {
            if (isNonNullable(evt)) {
                evt.allowedTags[lcTagName] = true;
            }
            return;
        }
        // Just leave non-elements such as text and comments up to dompurify
        if (node.nodeType !== ELEMENT || lcTagName === 'body') {
            return;
        }
        // Construct the sugar element wrapper
        const element = SugarElement.fromDom(node);
        // Determine if we're dealing with an internal attribute
        const isInternalElement = has$1(element, internalElementAttr);
        // Cleanup bogus elements
        const bogus = get$9(element, 'data-mce-bogus');
        if (!isInternalElement && isString(bogus)) {
            if (bogus === 'all') {
                remove$8(element);
            }
            else {
                unwrap(element);
            }
            return;
        }
        // Determine if the schema allows the element and either add it or remove it
        const rule = schema.getElementRule(lcTagName);
        if (validate && !rule) {
            // If a special element is invalid, then remove the entire element instead of unwrapping
            if (has$2(specialElements, lcTagName)) {
                remove$8(element);
            }
            else {
                unwrap(element);
            }
            return;
        }
        else {
            if (isNonNullable(evt)) {
                evt.allowedTags[lcTagName] = true;
            }
        }
        // Validate the element using the attribute rules
        if (validate && rule && !isInternalElement) {
            // Fix the attributes for the element, unwrapping it if we have to
            each$e(rule.attributesForced ?? [], (attr) => {
                set$4(element, attr.name, attr.value === '{$uid}' ? `mce_${uid++}` : attr.value);
            });
            each$e(rule.attributesDefault ?? [], (attr) => {
                if (!has$1(element, attr.name)) {
                    set$4(element, attr.name, attr.value === '{$uid}' ? `mce_${uid++}` : attr.value);
                }
            });
            // If none of the required attributes were found then remove
            if (rule.attributesRequired && !exists(rule.attributesRequired, (attr) => has$1(element, attr))) {
                unwrap(element);
                return;
            }
            // If there are no attributes then remove
            if (rule.removeEmptyAttrs && hasNone(element)) {
                unwrap(element);
                return;
            }
            // Change the node name if the schema says to
            if (rule.outputName && rule.outputName !== lcTagName) {
                mutate(element, rule.outputName);
            }
        }
    };
    const processAttr = (ele, settings, schema, scope, evt) => {
        const tagName = ele.tagName.toLowerCase();
        const { attrName, attrValue } = evt;
        evt.keepAttr = shouldKeepAttribute(settings, schema, scope, tagName, attrName, attrValue);
        if (evt.keepAttr) {
            evt.allowedAttributes[attrName] = true;
            if (isBooleanAttributeOfNonCustomElement(attrName, schema, ele.nodeName)) {
                evt.attrValue = attrName;
            }
            // We need to tell DOMPurify to forcibly keep the attribute if it's an SVG data URI and svg data URIs are allowed
            if (settings.allow_svg_data_urls && startsWith(attrValue, 'data:image/svg+xml')) {
                evt.forceKeepAttr = true;
            }
            // For internal elements always keep the attribute if the attribute name is id, class or style
        }
        else if (isRequiredAttributeOfInternalElement(ele, attrName)) {
            evt.forceKeepAttr = true;
        }
    };
    const shouldKeepAttribute = (settings, schema, scope, tagName, attrName, attrValue) => {
        // All attributes within non HTML namespaces elements are considered valid
        if (scope !== 'html' && !isNonHtmlElementRootName(tagName)) {
            return true;
        }
        return !(attrName in filteredUrlAttrs && isInvalidUri(settings, attrValue, tagName)) &&
            (!settings.validate || schema.isValid(tagName, attrName) || startsWith(attrName, 'data-') || startsWith(attrName, 'aria-'));
    };
    const isRequiredAttributeOfInternalElement = (ele, attrName) => ele.hasAttribute(internalElementAttr) && (attrName === 'id' || attrName === 'class' || attrName === 'style');
    const isBooleanAttributeOfNonCustomElement = (attrName, schema, nodeName) => attrName in schema.getBoolAttrs() && !has$2(schema.getCustomElements(), nodeName.toLowerCase());
    const filterAttributes = (ele, settings, schema, scope) => {
        const { attributes } = ele;
        for (let i = attributes.length - 1; i >= 0; i--) {
            const attr = attributes[i];
            const attrName = attr.name;
            const attrValue = attr.value;
            if (!shouldKeepAttribute(settings, schema, scope, ele.tagName.toLowerCase(), attrName, attrValue) && !isRequiredAttributeOfInternalElement(ele, attrName)) {
                ele.removeAttribute(attrName);
            }
            else if (isBooleanAttributeOfNonCustomElement(attrName, schema, ele.nodeName)) {
                ele.setAttribute(attrName, attrName);
            }
        }
    };
    const setupPurify = (settings, schema, namespaceTracker) => {
        const purify$1 = purify();
        // We use this to add new tags to the allow-list as we parse, if we notice that a tag has been banned but it's still in the schema
        purify$1.addHook('uponSanitizeElement', (ele, evt) => {
            processNode(ele, settings, schema, namespaceTracker.track(ele), evt);
        });
        // Let's do the same thing for attributes
        purify$1.addHook('uponSanitizeAttribute', (ele, evt) => {
            processAttr(ele, settings, schema, namespaceTracker.current(), evt);
        });
        return purify$1;
    };
    const getPurifyConfig = (settings, mimeType) => {
        const basePurifyConfig = {
            IN_PLACE: true,
            ALLOW_UNKNOWN_PROTOCOLS: true,
            // Deliberately ban all tags and attributes by default, and then un-ban them on demand in hooks
            // #comment and #cdata-section are always allowed as they aren't controlled via the schema
            // body is also allowed due to the DOMPurify checking the root node before sanitizing
            ALLOWED_TAGS: ['#comment', '#cdata-section', 'body', 'html'],
            ALLOWED_ATTR: []
        };
        const config = { ...basePurifyConfig };
        // Set the relevant parser mimetype
        config.PARSER_MEDIA_TYPE = mimeType;
        // Allow any URI when allowing script urls
        if (settings.allow_script_urls) {
            config.ALLOWED_URI_REGEXP = /.*/;
            // Allow anything except javascript (or similar) URIs if all html data urls are allowed
        }
        else if (settings.allow_html_data_urls) {
            config.ALLOWED_URI_REGEXP = /^(?!(\w+script|mhtml):)/i;
        }
        return config;
    };
    const sanitizeSvgElement = (ele) => {
        // xlink:href used to be the way to do links in SVG 1.x https://developer.mozilla.org/en-US/docs/Web/SVG/Attribute/xlink:href
        const xlinkAttrs = ['type', 'href', 'role', 'arcrole', 'title', 'show', 'actuate', 'label', 'from', 'to'].map((name) => `xlink:${name}`);
        const config = {
            IN_PLACE: true,
            USE_PROFILES: {
                html: true,
                svg: true,
                svgFilters: true
            },
            ALLOWED_ATTR: xlinkAttrs
        };
        purify().sanitize(ele, config);
    };
    const sanitizeMathmlElement = (node, settings) => {
        const config = {
            IN_PLACE: true,
            USE_PROFILES: {
                mathMl: true
            },
        };
        const purify$1 = purify();
        const allowedEncodings = settings.allow_mathml_annotation_encodings;
        const hasAllowedEncodings = isArray$1(allowedEncodings) && allowedEncodings.length > 0;
        const hasValidEncoding = (el) => {
            const encoding = el.getAttribute('encoding');
            return hasAllowedEncodings && isString(encoding) && contains$2(allowedEncodings, encoding);
        };
        const isValidElementOpt = (node, lcTagName) => {
            if (hasAllowedEncodings && lcTagName === 'semantics') {
                return Optional.some(true);
            }
            else if (lcTagName === 'annotation') {
                return Optional.some(isElement$7(node) && hasValidEncoding(node));
            }
            else if (isArray$1(settings.extended_mathml_elements)) {
                if (settings.extended_mathml_elements.includes(lcTagName)) {
                    return Optional.from(true);
                }
                else {
                    return Optional.none();
                }
            }
            else {
                return Optional.none();
            }
        };
        purify$1.addHook('uponSanitizeElement', (node, evt) => {
            // We know the node is an element as we have
            // passed an element to the purify.sanitize function below
            const lcTagName = evt.tagName ?? node.nodeName.toLowerCase();
            const keepElementOpt = isValidElementOpt(node, lcTagName);
            keepElementOpt.each((keepElement) => {
                evt.allowedTags[lcTagName] = keepElement;
                if (!keepElement && settings.sanitize) {
                    if (isElement$7(node)) {
                        node.remove();
                    }
                }
            });
        });
        purify$1.addHook('uponSanitizeAttribute', (_node, event) => {
            if (isArray$1(settings.extended_mathml_attributes)) {
                const keepAttribute = settings.extended_mathml_attributes.includes(event.attrName);
                if (keepAttribute) {
                    event.forceKeepAttr = true;
                }
            }
        });
        purify$1.sanitize(node, config);
    };
    const mkSanitizeNamespaceElement = (settings) => (ele) => {
        const namespaceType = toScopeType(ele);
        if (namespaceType === 'svg') {
            sanitizeSvgElement(ele);
        }
        else if (namespaceType === 'math') {
            sanitizeMathmlElement(ele, settings);
        }
        else {
            throw new Error('Not a namespace element');
        }
    };
    const getSanitizer = (settings, schema) => {
        const namespaceTracker = createNamespaceTracker();
        if (settings.sanitize) {
            const purify = setupPurify(settings, schema, namespaceTracker);
            const sanitizeHtmlElement = (body, mimeType) => {
                purify.sanitize(body, getPurifyConfig(settings, mimeType));
                purify.removed = [];
                namespaceTracker.reset();
            };
            return {
                sanitizeHtmlElement,
                sanitizeNamespaceElement: mkSanitizeNamespaceElement(settings)
            };
        }
        else {
            const sanitizeHtmlElement = (body, _mimeType) => {
                // eslint-disable-next-line no-bitwise
                const nodeIterator = document.createNodeIterator(body, NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_COMMENT | NodeFilter.SHOW_TEXT);
                let node;
                while ((node = nodeIterator.nextNode())) {
                    const currentScope = namespaceTracker.track(node);
                    processNode(node, settings, schema, currentScope);
                    if (isElement$7(node)) {
                        filterAttributes(node, settings, schema, currentScope);
                    }
                }
                namespaceTracker.reset();
            };
            const sanitizeNamespaceElement = noop;
            return {
                sanitizeHtmlElement,
                sanitizeNamespaceElement
            };
        }
    };

    /**
     * @summary
     * This class parses HTML code into a DOM like structure of nodes it will remove redundant whitespace and make
     * sure that the node tree is valid according to the specified schema.
     * So for example: `<p>a<p>b</p>c</p>` will become `<p>a</p><p>b</p><p>c</p>`.
     *
     * @example
     * const parser = tinymce.html.DomParser({ validate: true }, schema);
     * const rootNode = parser.parse('<h1>content</h1>');
     *
     * @class tinymce.html.DomParser
     * @version 3.4
     */
    const extraBlockLikeElements = ['script', 'style', 'template', 'param', 'meta', 'title', 'link'];
    const makeMap = Tools.makeMap, extend$1 = Tools.extend;
    const transferChildren = (parent, nativeParent, specialElements, nsSanitizer, decodeComments) => {
        const parentName = parent.name;
        // Exclude the special elements where the content is RCDATA as their content needs to be parsed instead of being left as plain text
        // See: https://html.spec.whatwg.org/multipage/parsing.html#parsing-html-fragments
        const isSpecial = parentName in specialElements && parentName !== 'title' && parentName !== 'textarea' && parentName !== 'noscript';
        const childNodes = nativeParent.childNodes;
        for (let ni = 0, nl = childNodes.length; ni < nl; ni++) {
            const nativeChild = childNodes[ni];
            const child = new AstNode(nativeChild.nodeName.toLowerCase(), nativeChild.nodeType);
            if (isElement$7(nativeChild)) {
                const attributes = nativeChild.attributes;
                for (let ai = 0, al = attributes.length; ai < al; ai++) {
                    const attr = attributes[ai];
                    child.attr(attr.name, attr.value);
                }
                if (isNonHtmlElementRootName(child.name)) {
                    nsSanitizer(nativeChild);
                    child.value = nativeChild.innerHTML;
                }
            }
            else if (isText$b(nativeChild)) {
                child.value = nativeChild.data;
                if (isSpecial) {
                    child.raw = true;
                }
            }
            else if (isComment(nativeChild)) {
                child.value = decodeComments ? decodeData$1(nativeChild.data) : nativeChild.data;
            }
            else if (isCData(nativeChild) || isPi(nativeChild)) {
                child.value = nativeChild.data;
            }
            if (isTemplate(nativeChild)) {
                const content = AstNode.create('#text');
                content.value = nativeChild.innerHTML;
                content.raw = true;
                child.append(content);
            }
            else if (!isNonHtmlElementRootName(child.name)) {
                transferChildren(child, nativeChild, specialElements, nsSanitizer, decodeComments);
            }
            parent.append(child);
        }
    };
    const walkTree = (root, preprocessors, postprocessors) => {
        const traverseOrder = [];
        for (let node = root, lastNode = node; node; lastNode = node, node = node.walk()) {
            const tempNode = node;
            each$e(preprocessors, (preprocess) => preprocess(tempNode));
            if (isNullable(tempNode.parent) && tempNode !== root) {
                // The node has been detached, so rewind a little and don't add it to our traversal
                node = lastNode;
            }
            else {
                traverseOrder.push(tempNode);
            }
        }
        for (let i = traverseOrder.length - 1; i >= 0; i--) {
            const node = traverseOrder[i];
            each$e(postprocessors, (postprocess) => postprocess(node));
        }
    };
    // All the dom operations we want to perform, regardless of whether we're trying to properly validate things
    // e.g. removing excess whitespace
    // e.g. removing empty nodes (or padding them with <br>)
    //
    // Returns [ preprocess, postprocess ]
    const whitespaceCleaner = (root, schema, settings, args) => {
        const validate = settings.validate;
        const nonEmptyElements = schema.getNonEmptyElements();
        const whitespaceElements = schema.getWhitespaceElements();
        const blockElements = extend$1(makeMap(extraBlockLikeElements), schema.getBlockElements());
        const textRootBlockElements = getTextRootBlockElements(schema);
        const allWhiteSpaceRegExp = /[ \t\r\n]+/g;
        const startWhiteSpaceRegExp = /^[ \t\r\n]+/;
        const endWhiteSpaceRegExp = /[ \t\r\n]+$/;
        const hasWhitespaceParent = (node) => {
            let tempNode = node.parent;
            while (isNonNullable(tempNode)) {
                if (tempNode.name in whitespaceElements) {
                    return true;
                }
                else {
                    tempNode = tempNode.parent;
                }
            }
            return false;
        };
        const isTextRootBlockEmpty = (node) => {
            let tempNode = node;
            while (isNonNullable(tempNode)) {
                if (tempNode.name in textRootBlockElements) {
                    return isEmpty$2(schema, nonEmptyElements, whitespaceElements, tempNode);
                }
                else {
                    tempNode = tempNode.parent;
                }
            }
            return false;
        };
        const isBlock = (node) => node.name in blockElements || isTransparentAstBlock(schema, node) || (isNonHtmlElementRootName(node.name) && node.parent === root);
        const isAtEdgeOfBlock = (node, start) => {
            const neighbour = start ? node.prev : node.next;
            if (isNonNullable(neighbour) || isNullable(node.parent)) {
                return false;
            }
            // Make sure our parent is actually a block, and also make sure it isn't a temporary "context" element
            // that we're probably going to unwrap as soon as we insert this content into the editor
            return isBlock(node.parent) && (node.parent !== root || args.isRootContent === true);
        };
        const preprocess = (node) => {
            if (node.type === 3) {
                // Remove leading whitespace here, so that all whitespace in nodes to the left of us has already been fixed
                if (!hasWhitespaceParent(node)) {
                    let text = node.value ?? '';
                    text = text.replace(allWhiteSpaceRegExp, ' ');
                    if (isLineBreakNode(node.prev, isBlock) || isAtEdgeOfBlock(node, true)) {
                        text = text.replace(startWhiteSpaceRegExp, '');
                    }
                    if (text.length === 0) {
                        node.remove();
                    }
                    else if (text === ' ' && node.prev && node.prev.type === COMMENT && node.next && node.next.type === COMMENT) {
                        node.remove();
                    }
                    else {
                        node.value = text;
                    }
                }
            }
        };
        const postprocess = (node) => {
            if (node.type === 1) {
                // Check for empty nodes here, because children will have been processed and (if necessary) emptied / removed already
                const elementRule = schema.getElementRule(node.name);
                if (validate && elementRule) {
                    const isNodeEmpty = isEmpty$2(schema, nonEmptyElements, whitespaceElements, node);
                    if (elementRule.paddInEmptyBlock && isNodeEmpty && isTextRootBlockEmpty(node)) {
                        paddEmptyNode(settings, args, isBlock, node);
                    }
                    else if (elementRule.removeEmpty && isNodeEmpty) {
                        if (isBlock(node)) {
                            node.remove();
                        }
                        else {
                            node.unwrap();
                        }
                    }
                    else if (elementRule.paddEmpty && (isNodeEmpty || isPaddedWithNbsp(node))) {
                        paddEmptyNode(settings, args, isBlock, node);
                    }
                }
            }
            else if (node.type === 3) {
                // Removing trailing whitespace here, so that all whitespace in nodes to the right of us has already been fixed
                if (!hasWhitespaceParent(node)) {
                    let text = node.value ?? '';
                    if (node.next && isBlock(node.next) || isAtEdgeOfBlock(node, false)) {
                        text = text.replace(endWhiteSpaceRegExp, '');
                    }
                    if (text.length === 0) {
                        node.remove();
                    }
                    else {
                        node.value = text;
                    }
                }
            }
        };
        return [preprocess, postprocess];
    };
    const getRootBlockName = (settings, args) => {
        const name = args.forced_root_block ?? settings.forced_root_block;
        if (name === false) {
            return '';
        }
        else if (name === true) {
            return 'p';
        }
        else {
            return name;
        }
    };
    const xhtmlAttribte = ' xmlns="http://www.w3.org/1999/xhtml"';
    const DomParser = (settings = {}, schema = Schema()) => {
        const nodeFilterRegistry = create$8();
        const attributeFilterRegistry = create$8();
        // Apply setting defaults
        const defaultedSettings = {
            validate: true,
            root_name: 'body',
            sanitize: true,
            allow_html_in_comments: false,
            ...settings
        };
        const parser = new DOMParser();
        const sanitizer = getSanitizer(defaultedSettings, schema);
        const parseAndSanitizeWithContext = (html, rootName, format = 'html', useDocumentNotBody = false) => {
            const isxhtml = format === 'xhtml';
            const mimeType = isxhtml ? 'application/xhtml+xml' : 'text/html';
            // Determine the root element to wrap the HTML in when parsing. If we're dealing with a
            // special element then we need to wrap it so the internal content is handled appropriately.
            const isSpecialRoot = has$2(schema.getSpecialElements(), rootName.toLowerCase());
            const content = isSpecialRoot ? `<${rootName}>${html}</${rootName}>` : html;
            const makeWrap = () => {
                if (/^[\s]*<head/i.test(html) || /^[\s]*<html/i.test(html) || /^[\s]*<!DOCTYPE/i.test(html)) {
                    return `<html${isxhtml ? xhtmlAttribte : ''}>${content}</html>`;
                }
                else {
                    if (isxhtml) {
                        return `<html${xhtmlAttribte}><head></head><body>${content}</body></html>`;
                    }
                    else {
                        return `<body>${content}</body>`;
                    }
                }
            };
            const document = parser.parseFromString(makeWrap(), mimeType);
            const body = useDocumentNotBody ? document.documentElement : document.body;
            sanitizer.sanitizeHtmlElement(body, mimeType);
            return isSpecialRoot ? body.firstChild : body;
        };
        /**
         * Adds a node filter function to the parser, the parser will collect the specified nodes by name
         * and then execute the callback once it has finished parsing the document.
         *
         * @method addNodeFilter
         * @param {String} name Comma separated list of nodes to collect.
         * @param {Function} callback Callback function to execute once it has collected nodes.
         * @example
         * parser.addNodeFilter('p,h1', (nodes, name) => {
         *   for (var i = 0; i < nodes.length; i++) {
         *     console.log(nodes[i].name);
         *   }
         * });
         */
        const addNodeFilter = nodeFilterRegistry.addFilter;
        const getNodeFilters = nodeFilterRegistry.getFilters;
        /**
         * Removes a node filter function or removes all filter functions from the parser for the node names provided.
         *
         * @method removeNodeFilter
         * @param {String} name Comma separated list of node names to remove filters for.
         * @param {Function} callback Optional callback function to only remove a specific callback.
         * @example
         * // Remove a single filter
         * parser.removeNodeFilter('p,h1', someCallback);
         *
         * // Remove all filters
         * parser.removeNodeFilter('p,h1');
         */
        const removeNodeFilter = nodeFilterRegistry.removeFilter;
        /**
         * Adds an attribute filter function to the parser, the parser will collect nodes that has the specified attributes
         * and then execute the callback once it has finished parsing the document.
         *
         * @method addAttributeFilter
         * @param {String} name Comma separated list of attributes to collect.
         * @param {Function} callback Callback function to execute once it has collected nodes.
         * @example
         * parser.addAttributeFilter('src,href', (nodes, name) => {
         *   for (let i = 0; i < nodes.length; i++) {
         *     console.log(nodes[i].name);
         *   }
         * });
         */
        const addAttributeFilter = attributeFilterRegistry.addFilter;
        const getAttributeFilters = attributeFilterRegistry.getFilters;
        /**
         * Removes an attribute filter function or removes all filter functions from the parser for the attribute names provided.
         *
         * @method removeAttributeFilter
         * @param {String} name Comma separated list of attribute names to remove filters for.
         * @param {Function} callback Optional callback function to only remove a specific callback.
         * @example
         * // Remove a single filter
         * parser.removeAttributeFilter('src,href', someCallback);
         *
         * // Remove all filters
         * parser.removeAttributeFilter('src,href');
         */
        const removeAttributeFilter = attributeFilterRegistry.removeFilter;
        const findInvalidChildren = (node, invalidChildren) => {
            if (isInvalid(schema, node)) {
                invalidChildren.push(node);
            }
        };
        const isWrappableNode = (blockElements, node) => {
            const isInternalElement = isString(node.attr(internalElementAttr));
            const isInlineElement = node.type === 1 && (!has$2(blockElements, node.name) && !isTransparentAstBlock(schema, node)) && !isNonHtmlElementRootName(node.name);
            return node.type === 3 || (isInlineElement && !isInternalElement);
        };
        const addRootBlocks = (rootNode, rootBlockName) => {
            const blockElements = extend$1(makeMap(extraBlockLikeElements), schema.getBlockElements());
            const startWhiteSpaceRegExp = /^[ \t\r\n]+/;
            const endWhiteSpaceRegExp = /[ \t\r\n]+$/;
            let node = rootNode.firstChild, rootBlockNode = null;
            // Removes whitespace at beginning and end of block so:
            // <p> x </p> -> <p>x</p>
            const trim = (rootBlock) => {
                if (rootBlock) {
                    node = rootBlock.firstChild;
                    if (node && node.type === 3) {
                        node.value = node.value?.replace(startWhiteSpaceRegExp, '');
                    }
                    node = rootBlock.lastChild;
                    if (node && node.type === 3) {
                        node.value = node.value?.replace(endWhiteSpaceRegExp, '');
                    }
                }
            };
            // Check if rootBlock is valid within rootNode for example if P is valid in H1 if H1 is the contentEditable root
            if (!schema.isValidChild(rootNode.name, rootBlockName.toLowerCase())) {
                return;
            }
            while (node) {
                const next = node.next;
                if (isWrappableNode(blockElements, node)) {
                    if (!rootBlockNode) {
                        // Create a new root block element
                        rootBlockNode = new AstNode(rootBlockName, 1);
                        rootBlockNode.attr(defaultedSettings.forced_root_block_attrs);
                        rootNode.insert(rootBlockNode, node);
                        rootBlockNode.append(node);
                    }
                    else {
                        rootBlockNode.append(node);
                    }
                }
                else {
                    trim(rootBlockNode);
                    rootBlockNode = null;
                }
                node = next;
            }
            trim(rootBlockNode);
        };
        /**
         * Parses the specified HTML string into a DOM like node tree and returns the result.
         *
         * @method parse
         * @param {String} html Html string to sax parse.
         * @param {Object} args Optional args object that gets passed to all filter functions.
         * @return {tinymce.html.Node} Root node containing the tree.
         * @example
         * const rootNode = tinymce.html.DomParser({...}).parse('<b>text</b>');
         */
        const parse = (html, args = {}) => {
            const validate = defaultedSettings.validate;
            const preferFullDocument = (args.context ?? defaultedSettings.root_name) === '#document';
            const rootName = args.context ?? (preferFullDocument ? 'html' : defaultedSettings.root_name);
            // Parse and sanitize the content
            const element = parseAndSanitizeWithContext(html, rootName, args.format, preferFullDocument);
            updateChildren(schema, element);
            // Create the AST representation
            const rootNode = new AstNode(rootName, 11);
            transferChildren(rootNode, element, schema.getSpecialElements(), sanitizer.sanitizeNamespaceElement, defaultedSettings.sanitize && defaultedSettings.allow_html_in_comments);
            // This next line is needed to fix a memory leak in chrome and firefox.
            // For more information see TINY-9186
            element.innerHTML = '';
            // Set up whitespace fixes
            const [whitespacePre, whitespacePost] = whitespaceCleaner(rootNode, schema, defaultedSettings, args);
            // Find the invalid children in the tree
            const invalidChildren = [];
            const invalidFinder = validate ? (node) => findInvalidChildren(node, invalidChildren) : noop;
            // Set up attribute and node matching
            const matches = { nodes: {}, attributes: {} };
            const matchFinder = (node) => matchNode(getNodeFilters(), getAttributeFilters(), node, matches);
            // Walk the dom, apply all of the above things
            walkTree(rootNode, [whitespacePre, matchFinder], [whitespacePost, invalidFinder]);
            // Because we collected invalid children while walking backwards, we need to reverse the list before operating on them
            invalidChildren.reverse();
            // Fix invalid children or report invalid children in a contextual parsing
            if (validate && invalidChildren.length > 0) {
                if (args.context) {
                    args.invalid = true;
                }
                else {
                    cleanInvalidNodes(invalidChildren, schema, rootNode, matchFinder);
                }
            }
            // Wrap nodes in the root into block elements if the root is body
            const rootBlockName = getRootBlockName(defaultedSettings, args);
            if (rootBlockName && (rootNode.name === 'body' || args.isRootContent)) {
                addRootBlocks(rootNode, rootBlockName);
            }
            // Run filters only when the contents is valid
            if (!args.invalid) {
                runFilters(matches, args);
            }
            return rootNode;
        };
        const exports = {
            schema,
            addAttributeFilter,
            getAttributeFilters,
            removeAttributeFilter,
            addNodeFilter,
            getNodeFilters,
            removeNodeFilter,
            parse
        };
        register$4(exports, defaultedSettings);
        register$5(exports, defaultedSettings, schema);
        return exports;
    };

    const isTreeNode = (content) => content instanceof AstNode;

    const serializeContent = (content) => isTreeNode(content) ? HtmlSerializer({ validate: false }).serialize(content) : content;
    const withSerializedContent = (content, fireEvent, parserSettings) => {
        const serializedContent = serializeContent(content);
        const eventArgs = fireEvent(serializedContent);
        if (eventArgs.isDefaultPrevented()) {
            return eventArgs;
        }
        else if (isTreeNode(content)) {
            // Restore the content type back to being an AstNode. If the content has changed we need to
            // re-parse the new content, otherwise we can return the input.
            if (eventArgs.content !== serializedContent) {
                const rootNode = DomParser({ validate: false, forced_root_block: false, ...parserSettings }).parse(eventArgs.content, { context: content.name });
                return { ...eventArgs, content: rootNode };
            }
            else {
                return { ...eventArgs, content };
            }
        }
        else {
            return eventArgs;
        }
    };
    const makeParserSettings = (editor) => ({
        sanitize: shouldSanitizeXss(editor),
        sandbox_iframes: shouldSandboxIframes(editor),
        sandbox_iframes_exclusions: getSandboxIframesExclusions(editor)
    });
    const preProcessGetContent = (editor, args) => {
        if (args.no_events) {
            return Result.value(args);
        }
        else {
            const eventArgs = fireBeforeGetContent(editor, args);
            if (eventArgs.isDefaultPrevented()) {
                return Result.error(fireGetContent(editor, { content: '', ...eventArgs }).content);
            }
            else {
                return Result.value(eventArgs);
            }
        }
    };
    const postProcessGetContent = (editor, content, args) => {
        if (args.no_events) {
            return content;
        }
        else {
            const processedEventArgs = withSerializedContent(content, (content) => fireGetContent(editor, { ...args, content }), makeParserSettings(editor));
            return processedEventArgs.content;
        }
    };
    const preProcessSetContent = (editor, args) => {
        if (args.no_events) {
            return Result.value(args);
        }
        else {
            const processedEventArgs = withSerializedContent(args.content, (content) => fireBeforeSetContent(editor, { ...args, content }), makeParserSettings(editor));
            if (processedEventArgs.isDefaultPrevented()) {
                fireSetContent(editor, processedEventArgs);
                return Result.error(undefined);
            }
            else {
                return Result.value(processedEventArgs);
            }
        }
    };
    const postProcessSetContent = (editor, content, args) => {
        if (!args.no_events) {
            fireSetContent(editor, { ...args, content });
        }
    };

    const removedOptions = ('autoresize_on_init,content_editable_state,padd_empty_with_br,block_elements,' +
        'boolean_attributes,editor_deselector,editor_selector,elements,file_browser_callback_types,filepicker_validator_handler,' +
        'force_hex_style_colors,force_p_newlines,gecko_spellcheck,images_dataimg_filter,media_scripts,mode,move_caret_before_on_enter_elements,' +
        'non_empty_elements,self_closing_elements,short_ended_elements,special,spellchecker_select_languages,spellchecker_whitelist,' +
        'tab_focus,tabfocus_elements,table_responsive_width,text_block_elements,text_inline_elements,toolbar_drawer,types,validate,whitespace_elements,' +
        'paste_enable_default_filters,paste_filter_drop,paste_word_valid_elements,paste_retain_style_properties,paste_convert_word_fake_lists,' +
        'template_cdate_classes,template_mdate_classes,template_selected_content_classes,template_preview_replace_values,template_replace_values,templates,template_cdate_format,template_mdate_format').split(',');
    const deprecatedOptions = ['content_css_cors'];
    const removedPlugins = 'bbcode,colorpicker,contextmenu,fullpage,legacyoutput,spellchecker,template,textcolor,rtc'.split(',');
    const deprecatedPlugins = [
        {
            name: 'export',
            replacedWith: 'Export to PDF'
        },
    ];
    const getMatchingOptions = (options, searchingFor) => {
        const settingNames = filter$5(searchingFor, (setting) => has$2(options, setting));
        return sort(settingNames);
    };
    const getRemovedOptions = (options) => {
        const settingNames = getMatchingOptions(options, removedOptions);
        // Forced root block is a special case whereby only the empty/false value is deprecated
        const forcedRootBlock = options.forced_root_block;
        // Note: This cast is required for old configurations as forced root block used to allow a boolean
        if (forcedRootBlock === false || forcedRootBlock === '') {
            settingNames.push('forced_root_block (false only)');
        }
        return sort(settingNames);
    };
    const getDeprecatedOptions = (options) => getMatchingOptions(options, deprecatedOptions);
    const getMatchingPlugins = (options, searchingFor) => {
        const plugins = Tools.makeMap(options.plugins, ' ');
        const hasPlugin = (plugin) => has$2(plugins, plugin);
        const pluginNames = filter$5(searchingFor, hasPlugin);
        return sort(pluginNames);
    };
    const getRemovedPlugins = (options) => getMatchingPlugins(options, removedPlugins);
    const getDeprecatedPlugins = (options) => getMatchingPlugins(options, deprecatedPlugins.map((entry) => entry.name));
    const logRemovedWarnings = (rawOptions, normalizedOptions) => {
        // Note: Ensure we use the original user settings, not the final when logging
        const removedOptions = getRemovedOptions(rawOptions);
        const removedPlugins = getRemovedPlugins(normalizedOptions);
        const hasRemovedPlugins = removedPlugins.length > 0;
        const hasRemovedOptions = removedOptions.length > 0;
        const isLegacyMobileTheme = normalizedOptions.theme === 'mobile';
        if (hasRemovedPlugins || hasRemovedOptions || isLegacyMobileTheme) {
            const listJoiner = '\n- ';
            const themesMessage = isLegacyMobileTheme ? `\n\nThemes:${listJoiner}mobile` : '';
            const pluginsMessage = hasRemovedPlugins ? `\n\nPlugins:${listJoiner}${removedPlugins.join(listJoiner)}` : '';
            const optionsMessage = hasRemovedOptions ? `\n\nOptions:${listJoiner}${removedOptions.join(listJoiner)}` : '';
            // eslint-disable-next-line no-console
            console.warn('The following deprecated features are currently enabled and have been removed in TinyMCE 8.0. These features will no longer work and should be removed from the TinyMCE configuration. ' +
                'See https://www.tiny.cloud/docs/tinymce/8/migration-from-7x/ for more information.' +
                themesMessage +
                pluginsMessage +
                optionsMessage);
        }
    };
    const getPluginDescription = (name) => find$2(deprecatedPlugins, (entry) => entry.name === name).fold(() => name, (entry) => {
        if (entry.replacedWith) {
            return `${name}, replaced by ${entry.replacedWith}`;
        }
        else {
            return name;
        }
    });
    const logDeprecatedWarnings = (rawOptions, normalizedOptions) => {
        // Note: Ensure we use the original user settings, not the final when logging
        const deprecatedOptions = getDeprecatedOptions(rawOptions);
        const deprecatedPlugins = getDeprecatedPlugins(normalizedOptions);
        const hasDeprecatedPlugins = deprecatedPlugins.length > 0;
        const hasDeprecatedOptions = deprecatedOptions.length > 0;
        if (hasDeprecatedPlugins || hasDeprecatedOptions) {
            const listJoiner = '\n- ';
            const pluginsMessage = hasDeprecatedPlugins ? `\n\nPlugins:${listJoiner}${deprecatedPlugins.map(getPluginDescription).join(listJoiner)}` : '';
            const optionsMessage = hasDeprecatedOptions ? `\n\nOptions:${listJoiner}${deprecatedOptions.join(listJoiner)}` : '';
            // eslint-disable-next-line no-console
            console.warn('The following deprecated features are currently enabled but will be removed soon.' +
                pluginsMessage +
                optionsMessage);
        }
    };
    const logWarnings = (rawOptions, normalizedOptions) => {
        logRemovedWarnings(rawOptions, normalizedOptions);
        logDeprecatedWarnings(rawOptions, normalizedOptions);
    };
    const deprecatedFeatures = {
        fire: 'The "fire" event api has been deprecated and will be removed in TinyMCE 9. Use "dispatch" instead.',
        selectionSetContent: 'The "editor.selection.setContent" method has been deprecated and will be removed in TinyMCE 9. Use "editor.insertContent" instead.'
    };
    const logFeatureDeprecationWarning = (feature) => {
        // eslint-disable-next-line no-console
        console.warn(deprecatedFeatures[feature], new Error().stack);
    };

    const removeEmpty = (text) => {
        if (text.dom.length === 0) {
            remove$8(text);
            return Optional.none();
        }
        else {
            return Optional.some(text);
        }
    };
    const walkPastBookmark = (node, start) => node.filter((elm) => BookmarkManager.isBookmarkNode(elm.dom))
        .bind(start ? nextSibling : prevSibling);
    const merge = (outer, inner, rng, start, schema) => {
        const outerElm = outer.dom;
        const innerElm = inner.dom;
        const oldLength = start ? outerElm.length : innerElm.length;
        if (start) {
            mergeTextNodes(outerElm, innerElm, schema, false, !start);
            rng.setStart(innerElm, oldLength);
        }
        else {
            mergeTextNodes(innerElm, outerElm, schema, false, !start);
            rng.setEnd(innerElm, oldLength);
        }
    };
    const normalizeTextIfRequired = (inner, start, schema) => {
        parent(inner).each((root) => {
            const text = inner.dom;
            if (start && needsToBeNbspLeft(root, CaretPosition(text, 0), schema)) {
                normalizeWhitespaceAfter(text, 0, schema);
            }
            else if (!start && needsToBeNbspRight(root, CaretPosition(text, text.length), schema)) {
                normalizeWhitespaceBefore(text, text.length, schema);
            }
        });
    };
    const mergeAndNormalizeText = (outerNode, innerNode, rng, start, schema) => {
        outerNode.bind((outer) => {
            // Normalize the text outside the inserted content
            const normalizer = start ? normalizeWhitespaceBefore : normalizeWhitespaceAfter;
            normalizer(outer.dom, start ? outer.dom.length : 0, schema);
            // Merge the inserted content with other text nodes
            return innerNode.filter(isText$c).map((inner) => merge(outer, inner, rng, start, schema));
        }).orThunk(() => {
            // Note: Attempt to leave the inserted/inner content as is and only adjust if absolutely required
            const innerTextNode = walkPastBookmark(innerNode, start).or(innerNode).filter(isText$c);
            return innerTextNode.map((inner) => normalizeTextIfRequired(inner, start, schema));
        });
    };
    const rngSetContent = (rng, fragment, schema) => {
        const firstChild = Optional.from(fragment.firstChild).map(SugarElement.fromDom);
        const lastChild = Optional.from(fragment.lastChild).map(SugarElement.fromDom);
        rng.deleteContents();
        rng.insertNode(fragment);
        const prevText = firstChild.bind(prevSibling).filter(isText$c).bind(removeEmpty);
        const nextText = lastChild.bind(nextSibling).filter(isText$c).bind(removeEmpty);
        // Join and normalize text
        mergeAndNormalizeText(prevText, firstChild, rng, true, schema);
        mergeAndNormalizeText(nextText, lastChild, rng, false, schema);
        rng.collapse(false);
    };
    const setupArgs$3 = (args, content) => ({
        format: 'html',
        ...args,
        set: true,
        selection: true,
        content
    });
    const cleanContent = (editor, args) => {
        if (args.format !== 'raw') {
            // Find which context to parse the content in
            const rng = editor.selection.getRng();
            const contextBlock = editor.dom.getParent(rng.commonAncestorContainer, editor.dom.isBlock);
            const contextArgs = contextBlock ? { context: contextBlock.nodeName.toLowerCase() } : {};
            const node = editor.parser.parse(args.content, { forced_root_block: false, ...contextArgs, ...args });
            return HtmlSerializer({ validate: false }, editor.schema).serialize(node);
        }
        else {
            return args.content;
        }
    };
    const setContentInternal$1 = (editor, content, args = {}) => {
        const defaultedArgs = setupArgs$3(args, content);
        preProcessSetContent(editor, defaultedArgs).each((updatedArgs) => {
            // Sanitize the content
            const cleanedContent = cleanContent(editor, updatedArgs);
            const rng = editor.selection.getRng();
            rngSetContent(rng, rng.createContextualFragment(cleanedContent), editor.schema);
            editor.selection.setRng(rng);
            scrollRangeIntoView(editor, rng);
            postProcessSetContent(editor, cleanedContent, updatedArgs);
        });
    };
    const setContentExternal = (editor, content, args = {}) => {
        logFeatureDeprecationWarning('selectionSetContent');
        setContentInternal$1(editor, content, args);
    };

    /**
     * Handles inserts of lists into the editor instance.
     *
     * @class tinymce.InsertList
     * @private
     */
    const hasOnlyOneChild$1 = (node) => {
        return isNonNullable(node.firstChild) && node.firstChild === node.lastChild;
    };
    const isPaddingNode = (node) => {
        return node.name === 'br' || node.value === nbsp;
    };
    const isPaddedEmptyBlock = (schema, node) => {
        const blockElements = schema.getBlockElements();
        return blockElements[node.name] && hasOnlyOneChild$1(node) && isPaddingNode(node.firstChild);
    };
    const isEmptyFragmentElement = (schema, node) => {
        const nonEmptyElements = schema.getNonEmptyElements();
        return isNonNullable(node) && (node.isEmpty(nonEmptyElements) || isPaddedEmptyBlock(schema, node));
    };
    const isListFragment = (schema, fragment) => {
        let firstChild = fragment.firstChild;
        let lastChild = fragment.lastChild;
        // Skip meta since it's likely <meta><ul>..</ul>
        if (firstChild && firstChild.name === 'meta') {
            firstChild = firstChild.next;
        }
        // Skip mce_marker since it's likely <ul>..</ul><span id="mce_marker"></span>
        if (lastChild && lastChild.attr('id') === 'mce_marker') {
            lastChild = lastChild.prev;
        }
        // Skip last child if it's an empty block
        if (isEmptyFragmentElement(schema, lastChild)) {
            lastChild = lastChild?.prev;
        }
        if (!firstChild || firstChild !== lastChild) {
            return false;
        }
        return firstChild.name === 'ul' || firstChild.name === 'ol';
    };
    const cleanupDomFragment = (domFragment) => {
        const firstChild = domFragment.firstChild;
        const lastChild = domFragment.lastChild;
        // TODO: remove the meta tag from paste logic
        if (firstChild && firstChild.nodeName === 'META') {
            firstChild.parentNode?.removeChild(firstChild);
        }
        if (lastChild && lastChild.id === 'mce_marker') {
            lastChild.parentNode?.removeChild(lastChild);
        }
        return domFragment;
    };
    const toDomFragment = (dom, serializer, fragment) => {
        const html = serializer.serialize(fragment);
        const domFragment = dom.createFragment(html);
        return cleanupDomFragment(domFragment);
    };
    const listItems = (elm) => {
        return filter$5(elm?.childNodes ?? [], (child) => {
            return child.nodeName === 'LI';
        });
    };
    const isPadding = (node) => {
        return node.data === nbsp || isBr$7(node);
    };
    const isListItemPadded = (node) => {
        return isNonNullable(node?.firstChild) && node.firstChild === node.lastChild && isPadding(node.firstChild);
    };
    const isEmptyOrPadded = (elm) => {
        return !elm.firstChild || isListItemPadded(elm);
    };
    const trimListItems = (elms) => {
        return elms.length > 0 && isEmptyOrPadded(elms[elms.length - 1]) ? elms.slice(0, -1) : elms;
    };
    const getParentLi = (dom, node) => {
        const parentBlock = dom.getParent(node, dom.isBlock);
        return parentBlock && parentBlock.nodeName === 'LI' ? parentBlock : null;
    };
    const isParentBlockLi = (dom, node) => {
        return !!getParentLi(dom, node);
    };
    const getSplit = (parentNode, rng) => {
        const beforeRng = rng.cloneRange();
        const afterRng = rng.cloneRange();
        beforeRng.setStartBefore(parentNode);
        afterRng.setEndAfter(parentNode);
        return [
            beforeRng.cloneContents(),
            afterRng.cloneContents()
        ];
    };
    const findFirstIn = (node, rootNode) => {
        const caretPos = CaretPosition.before(node);
        const caretWalker = CaretWalker(rootNode);
        const newCaretPos = caretWalker.next(caretPos);
        return newCaretPos ? newCaretPos.toRange() : null;
    };
    const findLastOf = (node, rootNode) => {
        const caretPos = CaretPosition.after(node);
        const caretWalker = CaretWalker(rootNode);
        const newCaretPos = caretWalker.prev(caretPos);
        return newCaretPos ? newCaretPos.toRange() : null;
    };
    const insertMiddle = (target, elms, rootNode, rng) => {
        const parts = getSplit(target, rng);
        const parentElm = target.parentNode;
        if (parentElm) {
            parentElm.insertBefore(parts[0], target);
            Tools.each(elms, (li) => {
                parentElm.insertBefore(li, target);
            });
            parentElm.insertBefore(parts[1], target);
            parentElm.removeChild(target);
        }
        return findLastOf(elms[elms.length - 1], rootNode);
    };
    const insertBefore$2 = (target, elms, rootNode) => {
        const parentElm = target.parentNode;
        if (parentElm) {
            Tools.each(elms, (elm) => {
                parentElm.insertBefore(elm, target);
            });
        }
        return findFirstIn(target, rootNode);
    };
    const insertAfter$2 = (target, elms, rootNode, dom) => {
        dom.insertAfter(elms.reverse(), target);
        return findLastOf(elms[0], rootNode);
    };
    const insertAtCaret$1 = (serializer, dom, rng, fragment) => {
        const domFragment = toDomFragment(dom, serializer, fragment);
        const liTarget = getParentLi(dom, rng.startContainer);
        const liElms = trimListItems(listItems(domFragment.firstChild));
        const BEGINNING = 1, END = 2;
        const rootNode = dom.getRoot();
        const isAt = (location) => {
            const caretPos = CaretPosition.fromRangeStart(rng);
            const caretWalker = CaretWalker(dom.getRoot());
            const newPos = location === BEGINNING ? caretWalker.prev(caretPos) : caretWalker.next(caretPos);
            const newPosNode = newPos?.getNode();
            return newPosNode ? getParentLi(dom, newPosNode) !== liTarget : true;
        };
        if (!liTarget) {
            return null;
        }
        else if (isAt(BEGINNING)) {
            return insertBefore$2(liTarget, liElms, rootNode);
        }
        else if (isAt(END)) {
            return insertAfter$2(liTarget, liElms, rootNode, dom);
        }
        else {
            return insertMiddle(liTarget, liElms, rootNode, rng);
        }
    };

    const mergeableWrappedElements = ['pre'];
    const shouldPasteContentOnly = (dom, fragment, parentNode, root) => {
        const firstNode = fragment.firstChild;
        const lastNode = fragment.lastChild;
        const last = lastNode.attr('data-mce-type') === 'bookmark' ? lastNode.prev : lastNode;
        const isPastingSingleElement = firstNode === last;
        const isWrappedElement = contains$2(mergeableWrappedElements, firstNode.name);
        if (isPastingSingleElement && isWrappedElement) {
            const isContentEditable = firstNode.attr('contenteditable') !== 'false';
            const isPastingInTheSameBlockTag = dom.getParent(parentNode, dom.isBlock)?.nodeName.toLowerCase() === firstNode.name;
            const isPastingInContentEditable = Optional.from(getContentEditableRoot$1(root, parentNode)).forall(isContentEditableTrue$3);
            return isContentEditable && isPastingInTheSameBlockTag && isPastingInContentEditable;
        }
        else {
            return false;
        }
    };
    const isTableCell = isTableCell$3;
    const isTableCellContentSelected = (dom, rng, cell) => {
        if (isNonNullable(cell)) {
            const endCell = dom.getParent(rng.endContainer, isTableCell);
            return cell === endCell && hasAllContentsSelected(SugarElement.fromDom(cell), rng);
        }
        else {
            return false;
        }
    };
    const isEditableEmptyBlock = (dom, node) => {
        if (dom.isBlock(node) && dom.isEditable(node)) {
            const childNodes = node.childNodes;
            return (childNodes.length === 1 && isBr$7(childNodes[0])) || childNodes.length === 0;
        }
        else {
            return false;
        }
    };
    const validInsertion = (editor, value, parentNode) => {
        // Should never insert content into bogus elements, since these can
        // be resize handles or similar
        if (parentNode.getAttribute('data-mce-bogus') === 'all') {
            parentNode.parentNode?.insertBefore(editor.dom.createFragment(value), parentNode);
        }
        else {
            if (isEditableEmptyBlock(editor.dom, parentNode)) {
                editor.dom.setHTML(parentNode, value);
            }
            else {
                setContentInternal$1(editor, value, { no_events: true });
            }
        }
    };
    const trimBrsFromTableCell = (dom, elm, schema) => {
        Optional.from(dom.getParent(elm, 'td,th')).map(SugarElement.fromDom).each((el) => trimBlockTrailingBr(el, schema));
    };
    // Remove children nodes that are exactly the same as a parent node - name, attributes, styles
    const reduceInlineTextElements = (editor, merge) => {
        const textInlineElements = editor.schema.getTextInlineElements();
        const dom = editor.dom;
        if (merge) {
            const root = editor.getBody();
            const elementUtils = ElementUtils(editor);
            const fragmentSelector = '*[data-mce-fragment]';
            const fragments = dom.select(fragmentSelector);
            Tools.each(fragments, (node) => {
                const isInline = (currentNode) => isNonNullable(textInlineElements[currentNode.nodeName.toLowerCase()]);
                const hasOneChild = (currentNode) => currentNode.childNodes.length === 1;
                const hasNoNonInheritableStyles = (currentNode) => !(hasNonInheritableStyles(dom, currentNode) || hasConditionalNonInheritableStyles(dom, currentNode));
                if (hasNoNonInheritableStyles(node) && isInline(node) && hasOneChild(node)) {
                    const styles = getStyleProps(dom, node);
                    const isOverridden = (oldStyles, newStyles) => forall(oldStyles, (style) => contains$2(newStyles, style));
                    const overriddenByAllChildren = (childNode) => hasOneChild(node) && dom.is(childNode, fragmentSelector) && isInline(childNode) &&
                        (childNode.nodeName === node.nodeName && isOverridden(styles, getStyleProps(dom, childNode)) || overriddenByAllChildren(childNode.children[0]));
                    const identicalToParent = (parentNode) => isNonNullable(parentNode) && parentNode !== root
                        && (elementUtils.compare(node, parentNode) || identicalToParent(parentNode.parentElement));
                    const conflictWithInsertedParent = (parentNode) => isNonNullable(parentNode) && parentNode !== root
                        && dom.is(parentNode, fragmentSelector) && (hasStyleConflict(dom, node, parentNode) || conflictWithInsertedParent(parentNode.parentElement));
                    if (overriddenByAllChildren(node.children[0]) || (identicalToParent(node.parentElement) && !conflictWithInsertedParent(node.parentElement))) {
                        dom.remove(node, true);
                    }
                }
            });
            normalizeElements(editor, fromDom$1(fragments));
        }
    };
    const markFragmentElements = (fragment) => {
        let node = fragment;
        while ((node = node.walk())) {
            if (node.type === 1) {
                node.attr('data-mce-fragment', '1');
            }
        }
    };
    const unmarkFragmentElements = (elm) => {
        Tools.each(elm.getElementsByTagName('*'), (elm) => {
            elm.removeAttribute('data-mce-fragment');
        });
    };
    const isPartOfFragment = (node) => {
        return !!node.getAttribute('data-mce-fragment');
    };
    const canHaveChildren = (editor, node) => {
        return isNonNullable(node) && !editor.schema.getVoidElements()[node.nodeName];
    };
    const moveSelectionToMarker = (editor, marker) => {
        let nextRng;
        const dom = editor.dom;
        const selection = editor.selection;
        if (!marker) {
            return;
        }
        selection.scrollIntoView(marker);
        // If marker is in cE=false then move selection to that element instead
        const parentEditableElm = getContentEditableRoot$1(editor.getBody(), marker);
        if (parentEditableElm && dom.getContentEditable(parentEditableElm) === 'false') {
            dom.remove(marker);
            selection.select(parentEditableElm);
            return;
        }
        // Move selection before marker and remove it
        let rng = dom.createRng();
        // If previous sibling is a text node set the selection to the end of that node
        const node = marker.previousSibling;
        if (isText$b(node)) {
            rng.setStart(node, node.nodeValue?.length ?? 0);
            const node2 = marker.nextSibling;
            if (isText$b(node2)) {
                node.appendData(node2.data);
                node2.parentNode?.removeChild(node2);
            }
        }
        else {
            // If the previous sibling isn't a text node or doesn't exist set the selection before the marker node
            rng.setStartBefore(marker);
            rng.setEndBefore(marker);
        }
        const findNextCaretRng = (rng) => {
            let caretPos = CaretPosition.fromRangeStart(rng);
            const caretWalker = CaretWalker(editor.getBody());
            caretPos = caretWalker.next(caretPos);
            return caretPos?.toRange();
        };
        // Remove the marker node and set the new range
        const parentBlock = dom.getParent(marker, dom.isBlock);
        dom.remove(marker);
        if (parentBlock && dom.isEmpty(parentBlock)) {
            const isCell = isTableCell(parentBlock);
            empty(SugarElement.fromDom(parentBlock));
            rng.setStart(parentBlock, 0);
            rng.setEnd(parentBlock, 0);
            if (!isCell && !isPartOfFragment(parentBlock) && (nextRng = findNextCaretRng(rng))) {
                rng = nextRng;
                dom.remove(parentBlock);
            }
            else {
                // TINY-9860: If parentBlock is a table cell, add a br without 'data-mce-bogus' attribute.
                dom.add(parentBlock, dom.create('br', isCell ? {} : { 'data-mce-bogus': '1' }));
            }
        }
        selection.setRng(rng);
    };
    const deleteSelectedContent = (editor) => {
        const dom = editor.dom;
        // Fix for #2595 seems that delete removes one extra character on
        // WebKit for some odd reason if you double click select a word
        const rng = normalize(editor.selection.getRng());
        editor.selection.setRng(rng);
        // TINY-1044: Selecting all content in a single table cell will cause the entire table to be deleted
        // when using the native delete command. As such we need to manually delete the cell content instead
        const startCell = dom.getParent(rng.startContainer, isTableCell);
        if (isTableCellContentSelected(dom, rng, startCell)) {
            deleteCellContents(editor, rng, SugarElement.fromDom(startCell));
            // TINY-9193: If the selection is over the whole text node in an element then Firefox incorrectly moves the caret to the previous line
            // TINY-11953: If the selection is over the whole anchor node, then Chrome incorrectly removes parent node alongside with it's child - anchor
        }
        else if (isSelectionOverWholeAnchor(rng) || isSelectionOverWholeTextNode(rng)) {
            rng.deleteContents();
        }
        else {
            editor.getDoc().execCommand('Delete', false);
        }
    };
    const findMarkerNode = (scope) => {
        for (let markerNode = scope; markerNode; markerNode = markerNode.walk()) {
            if (markerNode.attr('id') === 'mce_marker') {
                return Optional.some(markerNode);
            }
        }
        return Optional.none();
    };
    const notHeadingsInSummary = (dom, node, fragment) => {
        return exists(fragment.children(), isHeading) && dom.getParent(node, dom.isBlock)?.nodeName === 'SUMMARY';
    };
    const insertHtmlAtCaret = (editor, value, details) => {
        const selection = editor.selection;
        const dom = editor.dom;
        // Setup parser and serializer
        const parser = editor.parser;
        const merge = details.merge;
        const serializer = HtmlSerializer({
            validate: true
        }, editor.schema);
        const bookmarkHtml = '<span id="mce_marker" data-mce-type="bookmark">&#xFEFF;</span>';
        // TINY-10305: Remove all user-input zwsp to avoid impacting caret removal from content.
        if (!details.preserve_zwsp) {
            value = trim$2(value);
        }
        // Add caret at end of contents if it's missing
        if (value.indexOf('{$caret}') === -1) {
            value += '{$caret}';
        }
        // Replace the caret marker with a span bookmark element
        value = value.replace(/\{\$caret\}/, bookmarkHtml);
        // If selection is at <body>|<p></p> then move it into <body><p>|</p>
        let rng = selection.getRng();
        const caretElement = rng.startContainer;
        const body = editor.getBody();
        if (caretElement === body && selection.isCollapsed()) {
            if (dom.isBlock(body.firstChild) && canHaveChildren(editor, body.firstChild) && dom.isEmpty(body.firstChild)) {
                rng = dom.createRng();
                rng.setStart(body.firstChild, 0);
                rng.setEnd(body.firstChild, 0);
                selection.setRng(rng);
            }
        }
        // Insert node maker where we will insert the new HTML and get it's parent
        if (!selection.isCollapsed()) {
            deleteSelectedContent(editor);
        }
        const parentNode = selection.getNode();
        // Parse the fragment within the context of the parent node
        const parserArgs = { context: parentNode.nodeName.toLowerCase(), data: details.data, insert: true };
        const fragment = parser.parse(value, parserArgs);
        // Custom handling of lists
        if (details.paste === true && isListFragment(editor.schema, fragment) && isParentBlockLi(dom, parentNode)) {
            rng = insertAtCaret$1(serializer, dom, selection.getRng(), fragment);
            if (rng) {
                selection.setRng(rng);
            }
            return value;
        }
        if (details.paste === true && shouldPasteContentOnly(dom, fragment, parentNode, editor.getBody())) {
            fragment.firstChild?.unwrap();
        }
        markFragmentElements(fragment);
        // Move the caret to a more suitable location
        let node = fragment.lastChild;
        if (node && node.attr('id') === 'mce_marker') {
            const marker = node;
            for (node = node.prev; node; node = node.walk(true)) {
                if (node.name === 'table') {
                    break;
                }
                if (node.type === 3 || !dom.isBlock(node.name)) {
                    if (node.parent && editor.schema.isValidChild(node.parent.name, 'span')) {
                        node.parent.insert(marker, node, node.name === 'br');
                    }
                    break;
                }
            }
        }
        editor._selectionOverrides.showBlockCaretContainer(parentNode);
        // If parser says valid we can insert the contents into that parent
        if (!parserArgs.invalid && !notHeadingsInSummary(dom, parentNode, fragment)) {
            value = serializer.serialize(fragment);
            validInsertion(editor, value, parentNode);
        }
        else {
            // If the fragment was invalid within that context then we need
            // to parse and process the parent it's inserted into
            // Insert bookmark node and get the parent
            setContentInternal$1(editor, bookmarkHtml);
            let parentNode = selection.getNode();
            let tempNode;
            const rootNode = editor.getBody();
            // Opera will return the document node when selection is in root
            if (isDocument$1(parentNode)) {
                parentNode = tempNode = rootNode;
            }
            else {
                tempNode = parentNode;
            }
            // Find the ancestor just before the root element
            while (tempNode && tempNode !== rootNode) {
                parentNode = tempNode;
                tempNode = tempNode.parentNode;
            }
            // Get the outer/inner HTML depending on if we are in the root and parser and serialize that
            value = parentNode === rootNode ? rootNode.innerHTML : dom.getOuterHTML(parentNode);
            const root = parser.parse(value);
            const markerNode = findMarkerNode(root);
            const editingHost = markerNode.bind(findClosestEditingHost).getOr(root);
            markerNode.each((marker) => marker.replace(fragment));
            const fragmentNodes = getAllDescendants(fragment);
            fragment.unwrap();
            const invalidChildren = filter$5(fragmentNodes, (node) => isInvalid(editor.schema, node));
            cleanInvalidNodes(invalidChildren, editor.schema, editingHost);
            filter$1(parser.getNodeFilters(), parser.getAttributeFilters(), root);
            value = serializer.serialize(root);
            // Set the inner/outer HTML depending on if we are in the root or not
            if (parentNode === rootNode) {
                dom.setHTML(rootNode, value);
            }
            else {
                dom.setOuterHTML(parentNode, value);
            }
        }
        reduceInlineTextElements(editor, merge);
        moveSelectionToMarker(editor, dom.get('mce_marker'));
        unmarkFragmentElements(editor.getBody());
        trimBrsFromTableCell(dom, selection.getStart(), editor.schema);
        updateCaret(editor.schema, editor.getBody(), selection.getStart());
        return value;
    };

    const moveSelection = (editor) => {
        if (hasFocus(editor)) {
            firstPositionIn(editor.getBody()).each((pos) => {
                const node = pos.getNode();
                const caretPos = isTable$2(node) ? firstPositionIn(node).getOr(pos) : pos;
                editor.selection.setRng(caretPos.toRange());
            });
        }
    };
    const setEditorHtml = (editor, html, noSelection) => {
        editor.dom.setHTML(editor.getBody(), html);
        if (noSelection !== true) {
            moveSelection(editor);
        }
    };
    const setContentString = (editor, body, content, args) => {
        // TINY-10305: Remove all user-input zwsp to avoid impacting caret removal from content.
        content = trim$2(content);
        // Padd empty content in Gecko and Safari. Commands will otherwise fail on the content
        // It will also be impossible to place the caret in the editor unless there is a BR element present
        if (content.length === 0 || /^\s+$/.test(content)) {
            const padd = '<br data-mce-bogus="1">';
            // Todo: There is a lot more root elements that need special padding
            // so separate this and add all of them at some point.
            if (body.nodeName === 'TABLE') {
                content = '<tr><td>' + padd + '</td></tr>';
            }
            else if (/^(UL|OL)$/.test(body.nodeName)) {
                content = '<li>' + padd + '</li>';
            }
            const forcedRootBlockName = getForcedRootBlock(editor);
            // Check if forcedRootBlock is a valid child of the body
            if (editor.schema.isValidChild(body.nodeName.toLowerCase(), forcedRootBlockName.toLowerCase())) {
                content = padd;
                content = editor.dom.createHTML(forcedRootBlockName, getForcedRootBlockAttrs(editor), content);
            }
            else if (!content) {
                content = padd;
            }
            setEditorHtml(editor, content, args.no_selection);
            return { content, html: content };
        }
        else {
            if (args.format !== 'raw') {
                content = HtmlSerializer({ validate: false }, editor.schema).serialize(editor.parser.parse(content, { isRootContent: true, insert: true }));
            }
            const trimmedHtml = isWsPreserveElement(SugarElement.fromDom(body)) ? content : Tools.trim(content);
            setEditorHtml(editor, trimmedHtml, args.no_selection);
            return { content: trimmedHtml, html: trimmedHtml };
        }
    };
    const setContentTree = (editor, body, content, args) => {
        filter$1(editor.parser.getNodeFilters(), editor.parser.getAttributeFilters(), content);
        const html = HtmlSerializer({ validate: false }, editor.schema).serialize(content);
        // TINY-10305: Remove all user-input zwsp to avoid impacting caret removal from content.
        const trimmedHtml = trim$2(isWsPreserveElement(SugarElement.fromDom(body)) ? html : Tools.trim(html));
        setEditorHtml(editor, trimmedHtml, args.no_selection);
        return { content, html: trimmedHtml };
    };
    const setContentInternal = (editor, content, args) => {
        return Optional.from(editor.getBody()).map((body) => {
            if (isTreeNode(content)) {
                return setContentTree(editor, body, content, args);
            }
            else {
                return setContentString(editor, body, content, args);
            }
        }).getOr({ content, html: isTreeNode(args.content) ? '' : args.content });
    };

    const postProcessHooks = {};
    const isPre = matchNodeNames$1(['pre']);
    const addPostProcessHook = (name, hook) => {
        const hooks = postProcessHooks[name];
        if (!hooks) {
            postProcessHooks[name] = [];
        }
        postProcessHooks[name].push(hook);
    };
    const postProcess$1 = (name, editor) => {
        if (has$2(postProcessHooks, name)) {
            each$e(postProcessHooks[name], (hook) => {
                hook(editor);
            });
        }
    };
    addPostProcessHook('pre', (editor) => {
        const rng = editor.selection.getRng();
        const hasPreSibling = (blocks) => (pre) => {
            const prev = pre.previousSibling;
            return isPre(prev) && contains$2(blocks, prev);
        };
        const joinPre = (pre1, pre2) => {
            const sPre2 = SugarElement.fromDom(pre2);
            const doc = documentOrOwner(sPre2).dom;
            remove$8(sPre2);
            append(SugarElement.fromDom(pre1), [
                SugarElement.fromTag('br', doc),
                SugarElement.fromTag('br', doc),
                ...children$1(sPre2)
            ]);
        };
        if (!rng.collapsed) {
            const blocks = editor.selection.getSelectedBlocks();
            const preBlocks = filter$5(filter$5(blocks, isPre), hasPreSibling(blocks));
            each$e(preBlocks, (pre) => {
                joinPre(pre.previousSibling, pre);
            });
        }
    });

    const each$5 = Tools.each;
    const mergeTextDecorationsAndColor = (dom, format, vars, node) => {
        const processTextDecorationsAndColor = (n) => {
            if (isHTMLElement(n) && isElement$7(n.parentNode) && dom.isEditable(n)) {
                const parentTextDecoration = getTextDecoration(dom, n.parentNode);
                if (dom.getStyle(n, 'color') && parentTextDecoration) {
                    dom.setStyle(n, 'text-decoration', parentTextDecoration);
                }
                else if (dom.getStyle(n, 'text-decoration') === parentTextDecoration) {
                    dom.setStyle(n, 'text-decoration', null);
                }
            }
        };
        // Colored nodes should be underlined so that the color of the underline matches the text color.
        if (format.styles && (format.styles.color || format.styles.textDecoration)) {
            Tools.walk(node, processTextDecorationsAndColor, 'childNodes');
            processTextDecorationsAndColor(node);
        }
    };
    const mergeBackgroundColorAndFontSize = (dom, format, vars, node) => {
        // nodes with font-size should have their own background color as well to fit the line-height (see TINY-882)
        if (format.styles && format.styles.backgroundColor) {
            const hasFontSize = hasStyle(dom, 'fontSize');
            processChildElements(node, (elm) => hasFontSize(elm) && dom.isEditable(elm), applyStyle(dom, 'backgroundColor', replaceVars(format.styles.backgroundColor, vars)));
        }
    };
    const mergeSubSup = (dom, format, vars, node) => {
        // Remove font size on all descendants of a sub/sup and remove the inverse elements
        if (isInlineFormat(format) && (format.inline === 'sub' || format.inline === 'sup')) {
            const hasFontSize = hasStyle(dom, 'fontSize');
            processChildElements(node, (elm) => hasFontSize(elm) && dom.isEditable(elm), applyStyle(dom, 'fontSize', ''));
            const inverseTagDescendants = filter$5(dom.select(format.inline === 'sup' ? 'sub' : 'sup', node), dom.isEditable);
            dom.remove(inverseTagDescendants, true);
        }
    };
    const mergeWithChildren = (editor, formatList, vars, node) => {
        // Remove/merge children
        // Note: RemoveFormat.removeFormat will not remove formatting from noneditable nodes
        each$5(formatList, (format) => {
            // Merge all children of similar type will move styles from child to parent
            // this: <span style="color:red"><b><span style="color:red; font-size:10px">text</span></b></span>
            // will become: <span style="color:red"><b><span style="font-size:10px">text</span></b></span>
            if (isInlineFormat(format)) {
                each$5(editor.dom.select(format.inline, node), (child) => {
                    if (isElementNode(child)) {
                        removeNodeFormat(editor, format, vars, child, format.exact ? child : null);
                    }
                });
            }
            clearChildStyles(editor.dom, format, node);
        });
    };
    const mergeWithParents = (editor, format, name, vars, node) => {
        // Remove format if direct parent already has the same format
        // Note: RemoveFormat.removeFormat will not remove formatting from noneditable nodes
        const parentNode = node.parentNode;
        if (matchNode$1(editor, parentNode, name, vars)) {
            if (removeNodeFormat(editor, format, vars, node)) {
                return;
            }
        }
        // Remove format if any ancestor already has the same format
        if (format.merge_with_parents && parentNode) {
            editor.dom.getParent(parentNode, (parent) => {
                if (matchNode$1(editor, parent, name, vars)) {
                    removeNodeFormat(editor, format, vars, node);
                    return true;
                }
                else {
                    return false;
                }
            });
        }
    };

    const each$4 = Tools.each;
    const canFormatBR = (editor, format, node, parentName) => {
        // TINY-6483: Can format 'br' if it is contained in a valid empty block and an inline format is being applied
        if (canFormatEmptyLines(editor) && isInlineFormat(format) && node.parentNode) {
            const validBRParentElements = getTextRootBlockElements(editor.schema);
            // If a caret node is present, the format should apply to that, not the br (applicable to collapsed selections)
            const hasCaretNodeSibling = sibling(SugarElement.fromDom(node), (sibling) => isCaretNode(sibling.dom));
            return hasNonNullableKey(validBRParentElements, parentName) && isEmptyNode(editor.schema, node.parentNode, { skipBogus: false, includeZwsp: true }) && !hasCaretNodeSibling;
        }
        else {
            return false;
        }
    };
    const applyFormatAction = (ed, name, vars, node) => {
        const formatList = ed.formatter.get(name);
        const format = formatList[0];
        const isCollapsed = !node && ed.selection.isCollapsed();
        const dom = ed.dom;
        const selection = ed.selection;
        const applyNodeStyle = (formatList, node) => {
            let found = false;
            // Look for matching formats
            each$4(formatList, (format) => {
                if (!isSelectorFormat(format)) {
                    return false;
                }
                // Check if the node is nonediatble and if the format can override noneditable node
                if (dom.getContentEditable(node) === 'false' && !format.ceFalseOverride) {
                    return true;
                }
                // Check collapsed state if it exists
                if (isNonNullable(format.collapsed) && format.collapsed !== isCollapsed) {
                    return true;
                }
                if (dom.is(node, format.selector) && !isCaretNode(node)) {
                    setElementFormat(ed, node, format, vars, node);
                    found = true;
                    return false;
                }
                return true;
            });
            return found;
        };
        const createWrapElement = (wrapName) => {
            if (isString(wrapName)) {
                const wrapElm = dom.create(wrapName);
                setElementFormat(ed, wrapElm, format, vars, node);
                return wrapElm;
            }
            else {
                return null;
            }
        };
        const applyRngStyle = (dom, rng, nodeSpecific) => {
            const newWrappers = [];
            let contentEditable = true;
            // Setup wrapper element
            const wrapName = format.inline || format.block;
            const wrapElm = createWrapElement(wrapName);
            const isMatchingWrappingBlock = (node) => isWrappingBlockFormat(format) && matchNode$1(ed, node, name, vars);
            const canRenameBlock = (node, parentName, isEditableDescendant) => {
                const isValidBlockFormatForNode = isNonWrappingBlockFormat(format) &&
                    isTextBlock$2(ed.schema, node) &&
                    isValid(ed, parentName, wrapName);
                return isEditableDescendant && isValidBlockFormatForNode;
            };
            const canWrapNode = (node, parentName, isEditableDescendant, isWrappableNoneditableElm) => {
                const nodeName = node.nodeName.toLowerCase();
                const isValidWrapNode = isValid(ed, wrapName, nodeName) &&
                    isValid(ed, parentName, wrapName);
                // If it is not node specific, it means that it was not passed into 'formatter.apply` and is within the editor selection
                const isZwsp$1 = !nodeSpecific && isText$b(node) && isZwsp(node.data);
                const isCaret = isCaretNode(node);
                const isCorrectFormatForNode = !isInlineFormat(format) || !dom.isBlock(node);
                return (isEditableDescendant || isWrappableNoneditableElm) && isValidWrapNode && !isZwsp$1 && !isCaret && isCorrectFormatForNode;
            };
            walk$3(dom, rng, (nodes) => {
                let currentWrapElm;
                /**
                 * Process a list of nodes wrap them.
                 */
                const process = (node) => {
                    let hasContentEditableState = false;
                    let lastContentEditable = contentEditable;
                    let isWrappableNoneditableElm = false;
                    const parentNode = node.parentNode;
                    const parentName = parentNode.nodeName.toLowerCase();
                    // Node has a contentEditable value
                    const contentEditableValue = dom.getContentEditable(node);
                    if (isNonNullable(contentEditableValue)) {
                        lastContentEditable = contentEditable;
                        contentEditable = contentEditableValue === 'true';
                        // Unless the noneditable element is wrappable, we don't want to wrap the container, only it's editable children
                        hasContentEditableState = true;
                        isWrappableNoneditableElm = isWrappableNoneditable(ed, node);
                    }
                    const isEditableDescendant = contentEditable && !hasContentEditableState;
                    // Stop wrapping on br elements except when valid
                    if (isBr$7(node) && !canFormatBR(ed, format, node, parentName)) {
                        currentWrapElm = null;
                        // Remove any br elements when we wrap things
                        if (isBlockFormat(format)) {
                            dom.remove(node);
                        }
                        return;
                    }
                    if (isMatchingWrappingBlock(node)) {
                        currentWrapElm = null;
                        return;
                    }
                    if (canRenameBlock(node, parentName, isEditableDescendant)) {
                        const elm = dom.rename(node, wrapName);
                        setElementFormat(ed, elm, format, vars, node);
                        newWrappers.push(elm);
                        currentWrapElm = null;
                        return;
                    }
                    if (isSelectorFormat(format)) {
                        let found = applyNodeStyle(formatList, node);
                        // TINY-6567/TINY-7393: Include the parent if using an expanded selector format and no match was found for the current node
                        if (!found && isNonNullable(parentNode) && shouldExpandToSelector(format)) {
                            found = applyNodeStyle(formatList, parentNode);
                        }
                        // Continue processing if a selector match wasn't found and a inline element is defined
                        if (!isInlineFormat(format) || found) {
                            currentWrapElm = null;
                            return;
                        }
                    }
                    if (isNonNullable(wrapElm) && canWrapNode(node, parentName, isEditableDescendant, isWrappableNoneditableElm)) {
                        // Start wrapping
                        if (!currentWrapElm) {
                            // Wrap the node
                            currentWrapElm = dom.clone(wrapElm, false);
                            parentNode.insertBefore(currentWrapElm, node);
                            newWrappers.push(currentWrapElm);
                        }
                        // Wrappable noneditable element has been handled so go back to previous state
                        if (isWrappableNoneditableElm && hasContentEditableState) {
                            contentEditable = lastContentEditable;
                        }
                        currentWrapElm.appendChild(node);
                    }
                    else {
                        // Start a new wrapper for possible children
                        currentWrapElm = null;
                        each$e(from(node.childNodes), process);
                        if (hasContentEditableState) {
                            contentEditable = lastContentEditable; // Restore last contentEditable state from stack
                        }
                        // End the last wrapper
                        currentWrapElm = null;
                    }
                };
                each$e(nodes, process);
            });
            // Apply formats to links as well to get the color of the underline to change as well
            if (format.links === true) {
                each$e(newWrappers, (wrapper) => {
                    const process = (target) => {
                        if (target.nodeName === 'A') {
                            setElementFormat(ed, target, format, vars, node);
                        }
                        each$e(from(target.childNodes), process);
                    };
                    process(wrapper);
                });
            }
            normalizeFontSizeElementsAfterApply(ed, name, fromDom$1(newWrappers));
            // Cleanup
            each$e(newWrappers, (node) => {
                const getChildCount = (node) => {
                    let count = 0;
                    each$e(node.childNodes, (node) => {
                        if (!isEmptyTextNode$1(node) && !isBookmarkNode$1(node)) {
                            count++;
                        }
                    });
                    return count;
                };
                const mergeStyles = (node) => {
                    // Check if a child was found and of the same type as the current node
                    const childElement = find$2(node.childNodes, isElementNode$1)
                        .filter((child) => dom.getContentEditable(child) !== 'false' && matchName(dom, child, format));
                    return childElement.map((child) => {
                        const clone = dom.clone(child, false);
                        setElementFormat(ed, clone, format, vars, node);
                        dom.replace(clone, node, true);
                        dom.remove(child, true);
                        return clone;
                    }).getOr(node);
                };
                const childCount = getChildCount(node);
                // Remove empty nodes but only if there is multiple wrappers and they are not block
                // elements so never remove single <h1></h1> since that would remove the
                // current empty block element where the caret is at
                if ((newWrappers.length > 1 || !dom.isBlock(node)) && childCount === 0) {
                    dom.remove(node, true);
                    return;
                }
                if (isInlineFormat(format) || isBlockFormat(format) && format.wrapper) {
                    // Merges the current node with it's children of similar type to reduce the number of elements
                    if (!format.exact && childCount === 1) {
                        node = mergeStyles(node);
                    }
                    mergeWithChildren(ed, formatList, vars, node);
                    mergeWithParents(ed, format, name, vars, node);
                    mergeBackgroundColorAndFontSize(dom, format, vars, node);
                    mergeTextDecorationsAndColor(dom, format, vars, node);
                    mergeSubSup(dom, format, vars, node);
                    mergeSiblings(ed, format, vars, node);
                }
            });
        };
        // TODO: TINY-9142: Remove this to make nested noneditable formatting work
        const targetNode = isNode(node) ? node : selection.getNode();
        if (dom.getContentEditable(targetNode) === 'false' && !isWrappableNoneditable(ed, targetNode)) {
            // node variable is used by other functions above in the same scope so need to set it here
            node = targetNode;
            applyNodeStyle(formatList, node);
            fireFormatApply(ed, name, node, vars);
            return;
        }
        if (format) {
            if (node) {
                if (isNode(node)) {
                    if (!applyNodeStyle(formatList, node)) {
                        const rng = dom.createRng();
                        rng.setStartBefore(node);
                        rng.setEndAfter(node);
                        applyRngStyle(dom, expandRng(dom, rng, formatList), true);
                    }
                }
                else {
                    applyRngStyle(dom, node, true);
                }
            }
            else {
                if (!isCollapsed || !isInlineFormat(format) || getCellsFromEditor(ed).length) {
                    // Apply formatting to selection
                    selection.setRng(normalize(selection.getRng()));
                    preserveSelection(ed, () => {
                        runOnRanges(ed, (selectionRng, fake) => {
                            const expandedRng = fake ? selectionRng : expandRng(dom, selectionRng, formatList);
                            applyRngStyle(dom, expandedRng, false);
                        });
                    }, always);
                    ed.nodeChanged();
                }
                else {
                    applyCaretFormat(ed, name, vars);
                }
                getExpandedListItemFormat(ed.formatter, name).each((liFmt) => {
                    const list = getFullySelectedListItems(ed.selection);
                    each$e(list, (li) => applyStyles(dom, li, liFmt, vars));
                });
            }
            postProcess$1(name, ed);
        }
        fireFormatApply(ed, name, node, vars);
    };
    const applyFormat$1 = (editor, name, vars, node) => {
        if (node || editor.selection.isEditable()) {
            applyFormatAction(editor, name, vars, node);
        }
    };

    const hasVars = (value) => has$2(value, 'vars');
    const setup$A = (registeredFormatListeners, editor) => {
        registeredFormatListeners.set({});
        editor.on('NodeChange', (e) => {
            updateAndFireChangeCallbacks(editor, e.element, registeredFormatListeners.get());
        });
        editor.on('FormatApply FormatRemove', (e) => {
            const element = Optional.from(e.node)
                .map((nodeOrRange) => isNode(nodeOrRange) ? nodeOrRange : nodeOrRange.startContainer)
                .bind((node) => isElement$7(node) ? Optional.some(node) : Optional.from(node.parentElement))
                .getOrThunk(() => fallbackElement(editor));
            updateAndFireChangeCallbacks(editor, element, registeredFormatListeners.get());
        });
    };
    const fallbackElement = (editor) => editor.selection.getStart();
    const matchingNode = (editor, parents, format, similar, vars) => {
        const isMatchingNode = (node) => {
            const matchingFormat = editor.formatter.matchNode(node, format, vars ?? {}, similar);
            return !isUndefined(matchingFormat);
        };
        const isUnableToMatch = (node) => {
            if (matchesUnInheritedFormatSelector(editor, node, format)) {
                return true;
            }
            else {
                if (!similar) {
                    // If we want to find an exact match, then finding a similar match halfway up the parents tree is bad
                    return isNonNullable(editor.formatter.matchNode(node, format, vars, true));
                }
                else {
                    return false;
                }
            }
        };
        return findUntil$1(parents, isMatchingNode, isUnableToMatch);
    };
    const getParents = (editor, elm) => {
        const element = elm ?? fallbackElement(editor);
        return filter$5(getParents$2(editor.dom, element), (node) => isElement$7(node) && !isBogus$1(node));
    };
    const updateAndFireChangeCallbacks = (editor, elm, registeredCallbacks) => {
        // Ignore bogus nodes like the <a> tag created by moveStart()
        const parents = getParents(editor, elm);
        each$d(registeredCallbacks, (data, format) => {
            const runIfChanged = (spec) => {
                const match = matchingNode(editor, parents, format, spec.similar, hasVars(spec) ? spec.vars : undefined);
                const isSet = match.isSome();
                if (spec.state.get() !== isSet) {
                    spec.state.set(isSet);
                    const node = match.getOr(elm);
                    if (hasVars(spec)) {
                        spec.callback(isSet, { node, format, parents });
                    }
                    else {
                        each$e(spec.callbacks, (callback) => callback(isSet, { node, format, parents }));
                    }
                }
            };
            each$e([data.withSimilar, data.withoutSimilar], runIfChanged);
            each$e(data.withVars, runIfChanged);
        });
    };
    const addListeners = (editor, registeredFormatListeners, formats, callback, similar, vars) => {
        const formatChangeItems = registeredFormatListeners.get();
        each$e(formats.split(','), (format) => {
            const group = get$a(formatChangeItems, format).getOrThunk(() => {
                const base = {
                    withSimilar: {
                        state: Cell(false),
                        similar: true,
                        callbacks: []
                    },
                    withoutSimilar: {
                        state: Cell(false),
                        similar: false,
                        callbacks: []
                    },
                    withVars: []
                };
                formatChangeItems[format] = base;
                return base;
            });
            const getCurrent = () => {
                const parents = getParents(editor);
                return matchingNode(editor, parents, format, similar, vars).isSome();
            };
            if (isUndefined(vars)) {
                const toAppendTo = similar ? group.withSimilar : group.withoutSimilar;
                toAppendTo.callbacks.push(callback);
                if (toAppendTo.callbacks.length === 1) {
                    toAppendTo.state.set(getCurrent());
                }
            }
            else {
                group.withVars.push({
                    state: Cell(getCurrent()),
                    similar,
                    vars,
                    callback
                });
            }
        });
        registeredFormatListeners.set(formatChangeItems);
    };
    const removeListeners = (registeredFormatListeners, formats, callback) => {
        const formatChangeItems = registeredFormatListeners.get();
        each$e(formats.split(','), (format) => get$a(formatChangeItems, format).each((group) => {
            formatChangeItems[format] = {
                withSimilar: {
                    ...group.withSimilar,
                    callbacks: filter$5(group.withSimilar.callbacks, (cb) => cb !== callback),
                },
                withoutSimilar: {
                    ...group.withoutSimilar,
                    callbacks: filter$5(group.withoutSimilar.callbacks, (cb) => cb !== callback),
                },
                withVars: filter$5(group.withVars, (item) => item.callback !== callback),
            };
        }));
        registeredFormatListeners.set(formatChangeItems);
    };
    const formatChangedInternal = (editor, registeredFormatListeners, formats, callback, similar, vars) => {
        addListeners(editor, registeredFormatListeners, formats, callback, similar, vars);
        return {
            unbind: () => removeListeners(registeredFormatListeners, formats, callback)
        };
    };

    const toggle = (editor, name, vars, node) => {
        const fmt = editor.formatter.get(name);
        if (fmt) {
            if (match$2(editor, name, vars, node) && (!('toggle' in fmt[0]) || fmt[0].toggle)) {
                removeFormat$1(editor, name, vars, node);
            }
            else {
                applyFormat$1(editor, name, vars, node);
            }
        }
    };

    const tableModel = (element, width, rows) => ({
        element,
        width,
        rows
    });
    const tableRow = (element, cells) => ({
        element,
        cells
    });
    const cellPosition = (x, y) => ({
        x,
        y
    });
    const getSpan = (td, key) => {
        return getOpt(td, key).bind(toInt).getOr(1);
    };
    const fillout = (table, x, y, tr, td) => {
        const rowspan = getSpan(td, 'rowspan');
        const colspan = getSpan(td, 'colspan');
        const rows = table.rows;
        for (let y2 = y; y2 < y + rowspan; y2++) {
            if (!rows[y2]) {
                rows[y2] = tableRow(deep(tr), []);
            }
            for (let x2 = x; x2 < x + colspan; x2++) {
                const cells = rows[y2].cells;
                // not filler td:s are purposely not cloned so that we can
                // find cells in the model by element object references
                cells[x2] = y2 === y && x2 === x ? td : shallow(td);
            }
        }
    };
    const cellExists = (table, x, y) => {
        const rows = table.rows;
        const cells = rows[y] ? rows[y].cells : [];
        return !!cells[x];
    };
    const skipCellsX = (table, x, y) => {
        while (cellExists(table, x, y)) {
            x++;
        }
        return x;
    };
    const getWidth = (rows) => {
        return foldl(rows, (acc, row) => {
            return row.cells.length > acc ? row.cells.length : acc;
        }, 0);
    };
    const findElementPos = (table, element) => {
        const rows = table.rows;
        for (let y = 0; y < rows.length; y++) {
            const cells = rows[y].cells;
            for (let x = 0; x < cells.length; x++) {
                if (eq(cells[x], element)) {
                    return Optional.some(cellPosition(x, y));
                }
            }
        }
        return Optional.none();
    };
    const extractRows = (table, sx, sy, ex, ey) => {
        const newRows = [];
        const rows = table.rows;
        for (let y = sy; y <= ey; y++) {
            const cells = rows[y].cells;
            const slice = sx < ex ? cells.slice(sx, ex + 1) : cells.slice(ex, sx + 1);
            newRows.push(tableRow(rows[y].element, slice));
        }
        return newRows;
    };
    const subTable = (table, startPos, endPos) => {
        const sx = startPos.x, sy = startPos.y;
        const ex = endPos.x, ey = endPos.y;
        const newRows = sy < ey ? extractRows(table, sx, sy, ex, ey) : extractRows(table, sx, ey, ex, sy);
        return tableModel(table.element, getWidth(newRows), newRows);
    };
    const createDomTable = (table, rows) => {
        const tableElement = shallow(table.element);
        const tableBody = SugarElement.fromTag('tbody');
        append(tableBody, rows);
        append$1(tableElement, tableBody);
        return tableElement;
    };
    const modelRowsToDomRows = (table) => {
        return map$3(table.rows, (row) => {
            const cells = map$3(row.cells, (cell) => {
                const td = deep(cell);
                remove$9(td, 'colspan');
                remove$9(td, 'rowspan');
                return td;
            });
            const tr = shallow(row.element);
            append(tr, cells);
            return tr;
        });
    };
    const fromDom = (tableElm) => {
        const table = tableModel(shallow(tableElm), 0, []);
        each$e(descendants(tableElm, 'tr'), (tr, y) => {
            each$e(descendants(tr, 'td,th'), (td, x) => {
                fillout(table, skipCellsX(table, x, y), y, tr, td);
            });
        });
        return tableModel(table.element, getWidth(table.rows), table.rows);
    };
    const toDom = (table) => {
        return createDomTable(table, modelRowsToDomRows(table));
    };
    const subsection = (table, startElement, endElement) => {
        return findElementPos(table, startElement).bind((startPos) => {
            return findElementPos(table, endElement).map((endPos) => {
                return subTable(table, startPos, endPos);
            });
        });
    };

    const findParentListContainer = (parents) => find$2(parents, (elm) => name(elm) === 'ul' || name(elm) === 'ol');
    const getFullySelectedListWrappers = (parents, rng) => find$2(parents, (elm) => name(elm) === 'li' && hasAllContentsSelected(elm, rng)).fold(constant([]), (_li) => findParentListContainer(parents).map((listCont) => {
        const listElm = SugarElement.fromTag(name(listCont));
        // Retain any list-style* styles when generating the new fragment
        const listStyles = filter$4(getAllRaw(listCont), (_style, name) => startsWith(name, 'list-style'));
        setAll(listElm, listStyles);
        return [
            SugarElement.fromTag('li'),
            listElm
        ];
    }).getOr([]));
    const wrap = (innerElm, elms) => {
        const wrapped = foldl(elms, (acc, elm) => {
            append$1(elm, acc);
            return elm;
        }, innerElm);
        return elms.length > 0 ? fromElements([wrapped]) : wrapped;
    };
    const directListWrappers = (commonAnchorContainer) => {
        if (isListItem$2(commonAnchorContainer)) {
            return parent(commonAnchorContainer).filter(isList$1).fold(constant([]), (listElm) => [commonAnchorContainer, listElm]);
        }
        else {
            return isList$1(commonAnchorContainer) ? [commonAnchorContainer] : [];
        }
    };
    const getWrapElements = (rootNode, rng, schema) => {
        const commonAnchorContainer = SugarElement.fromDom(rng.commonAncestorContainer);
        const parents = parentsAndSelf(commonAnchorContainer, rootNode);
        const wrapElements = filter$5(parents, (el) => schema.isWrapper(name(el)));
        const listWrappers = getFullySelectedListWrappers(parents, rng);
        const allWrappers = wrapElements.concat(listWrappers.length ? listWrappers : directListWrappers(commonAnchorContainer));
        return map$3(allWrappers, shallow);
    };
    const emptyFragment = () => fromElements([]);
    const getFragmentFromRange = (rootNode, rng, schema) => wrap(SugarElement.fromDom(rng.cloneContents()), getWrapElements(rootNode, rng, schema));
    const getParentTable = (rootElm, cell) => ancestor$4(cell, 'table', curry(eq, rootElm));
    const getTableFragment = (rootNode, selectedTableCells) => getParentTable(rootNode, selectedTableCells[0]).bind((tableElm) => {
        const firstCell = selectedTableCells[0];
        const lastCell = selectedTableCells[selectedTableCells.length - 1];
        const fullTableModel = fromDom(tableElm);
        return subsection(fullTableModel, firstCell, lastCell).map((sectionedTableModel) => fromElements([toDom(sectionedTableModel)]));
    }).getOrThunk(emptyFragment);
    const getSelectionFragment = (rootNode, ranges, schema) => ranges.length > 0 && ranges[0].collapsed ? emptyFragment() : getFragmentFromRange(rootNode, ranges[0], schema);
    const read$3 = (rootNode, ranges, schema) => {
        const selectedCells = getCellsFromElementOrRanges(ranges, rootNode);
        return selectedCells.length > 0 ? getTableFragment(rootNode, selectedCells) : getSelectionFragment(rootNode, ranges, schema);
    };

    const isCollapsibleWhitespace = (text, index) => index >= 0 && index < text.length && isWhiteSpace(text.charAt(index));
    const getInnerText = (bin) => {
        return trim$2(bin.innerText);
    };
    const getContextNodeName = (parentBlockOpt) => parentBlockOpt.map((block) => block.nodeName).getOr('div').toLowerCase();
    const getTextContent = (editor) => Optional.from(editor.selection.getRng()).map((rng) => {
        const parentBlockOpt = Optional.from(editor.dom.getParent(rng.commonAncestorContainer, editor.dom.isBlock));
        const body = editor.getBody();
        const contextNodeName = getContextNodeName(parentBlockOpt);
        const rangeContentClone = SugarElement.fromDom(rng.cloneContents());
        cleanupBogusElements(rangeContentClone);
        cleanupInputNames(rangeContentClone);
        const bin = editor.dom.add(body, contextNodeName, {
            'data-mce-bogus': 'all',
            'style': 'overflow: hidden; opacity: 0;'
        }, rangeContentClone.dom);
        const text = getInnerText(bin);
        // textContent will not strip leading/trailing spaces since it doesn't consider how it'll render
        const nonRenderedText = trim$2(bin.textContent ?? '');
        editor.dom.remove(bin);
        if (isCollapsibleWhitespace(nonRenderedText, 0) || isCollapsibleWhitespace(nonRenderedText, nonRenderedText.length - 1)) {
            // If the bin contains a trailing/leading space, then we need to inspect the parent block to see if we should include the spaces
            const parentBlock = parentBlockOpt.getOr(body);
            const parentBlockText = getInnerText(parentBlock);
            const textIndex = parentBlockText.indexOf(text);
            if (textIndex === -1) {
                return text;
            }
            else {
                const hasProceedingSpace = isCollapsibleWhitespace(parentBlockText, textIndex - 1);
                const hasTrailingSpace = isCollapsibleWhitespace(parentBlockText, textIndex + text.length);
                return (hasProceedingSpace ? ' ' : '') + text + (hasTrailingSpace ? ' ' : '');
            }
        }
        else {
            return text;
        }
    }).getOr('');
    const getSerializedContent = (editor, args) => {
        const rng = editor.selection.getRng(), tmpElm = editor.dom.create('body');
        const sel = editor.selection.getSel();
        const ranges = processRanges(editor, getRanges(sel));
        const fragment = args.contextual ? read$3(SugarElement.fromDom(editor.getBody()), ranges, editor.schema).dom : rng.cloneContents();
        if (fragment) {
            tmpElm.appendChild(fragment);
        }
        return editor.selection.serializer.serialize(tmpElm, args);
    };
    const extractSelectedContent = (editor, args) => {
        if (args.format === 'text') {
            return getTextContent(editor);
        }
        else {
            const content = getSerializedContent(editor, args);
            if (args.format === 'tree') {
                return content;
            }
            else {
                return editor.selection.isCollapsed() ? '' : content;
            }
        }
    };
    const setupArgs$2 = (args, format) => ({
        ...args,
        format,
        get: true,
        selection: true,
        getInner: true
    });
    const getSelectedContentInternal = (editor, format, args = {}) => {
        const defaultedArgs = setupArgs$2(args, format);
        return preProcessGetContent(editor, defaultedArgs).fold(identity, (updatedArgs) => {
            const content = extractSelectedContent(editor, updatedArgs);
            return postProcessGetContent(editor, content, updatedArgs);
        });
    };

    /**
     * JS Implementation of the O(ND) Difference Algorithm by Eugene W. Myers.
     *
     * @class tinymce.undo.Diff
     * @private
     */
    const KEEP = 0, INSERT = 1, DELETE = 2;
    const diff = (left, right) => {
        const size = left.length + right.length + 2;
        const vDown = new Array(size);
        const vUp = new Array(size);
        const snake = (start, end, diag) => {
            return {
                start,
                end,
                diag
            };
        };
        const buildScript = (start1, end1, start2, end2, script) => {
            const middle = getMiddleSnake(start1, end1, start2, end2);
            if (middle === null || middle.start === end1 && middle.diag === end1 - end2 ||
                middle.end === start1 && middle.diag === start1 - start2) {
                let i = start1;
                let j = start2;
                while (i < end1 || j < end2) {
                    if (i < end1 && j < end2 && left[i] === right[j]) {
                        script.push([KEEP, left[i]]);
                        ++i;
                        ++j;
                    }
                    else {
                        if (end1 - start1 > end2 - start2) {
                            script.push([DELETE, left[i]]);
                            ++i;
                        }
                        else {
                            script.push([INSERT, right[j]]);
                            ++j;
                        }
                    }
                }
            }
            else {
                buildScript(start1, middle.start, start2, middle.start - middle.diag, script);
                for (let i2 = middle.start; i2 < middle.end; ++i2) {
                    script.push([KEEP, left[i2]]);
                }
                buildScript(middle.end, end1, middle.end - middle.diag, end2, script);
            }
        };
        const buildSnake = (start, diag, end1, end2) => {
            let end = start;
            while (end - diag < end2 && end < end1 && left[end] === right[end - diag]) {
                ++end;
            }
            return snake(start, end, diag);
        };
        const getMiddleSnake = (start1, end1, start2, end2) => {
            // Myers Algorithm
            // Initialisations
            const m = end1 - start1;
            const n = end2 - start2;
            if (m === 0 || n === 0) {
                return null;
            }
            const delta = m - n;
            const sum = n + m;
            const offset = (sum % 2 === 0 ? sum : sum + 1) / 2;
            vDown[1 + offset] = start1;
            vUp[1 + offset] = end1 + 1;
            let d, k, i, x, y;
            for (d = 0; d <= offset; ++d) {
                // Down
                for (k = -d; k <= d; k += 2) {
                    // First step
                    i = k + offset;
                    if (k === -d || k !== d && vDown[i - 1] < vDown[i + 1]) {
                        vDown[i] = vDown[i + 1];
                    }
                    else {
                        vDown[i] = vDown[i - 1] + 1;
                    }
                    x = vDown[i];
                    y = x - start1 + start2 - k;
                    while (x < end1 && y < end2 && left[x] === right[y]) {
                        vDown[i] = ++x;
                        ++y;
                    }
                    // Second step
                    if (delta % 2 !== 0 && delta - d <= k && k <= delta + d) {
                        if (vUp[i - delta] <= vDown[i]) {
                            return buildSnake(vUp[i - delta], k + start1 - start2, end1, end2);
                        }
                    }
                }
                // Up
                for (k = delta - d; k <= delta + d; k += 2) {
                    // First step
                    i = k + offset - delta;
                    if (k === delta - d || k !== delta + d && vUp[i + 1] <= vUp[i - 1]) {
                        vUp[i] = vUp[i + 1] - 1;
                    }
                    else {
                        vUp[i] = vUp[i - 1];
                    }
                    x = vUp[i] - 1;
                    y = x - start1 + start2 - k;
                    while (x >= start1 && y >= start2 && left[x] === right[y]) {
                        vUp[i] = x--;
                        y--;
                    }
                    // Second step
                    if (delta % 2 === 0 && -d <= k && k <= d) {
                        if (vUp[i] <= vDown[i + delta]) {
                            return buildSnake(vUp[i], k + start1 - start2, end1, end2);
                        }
                    }
                }
            }
            return null;
        };
        const script = [];
        buildScript(0, left.length, 0, right.length, script);
        return script;
    };

    /**
     * This module reads and applies html fragments from/to dom nodes.
     *
     * @class tinymce.undo.Fragments
     * @private
     */
    const getOuterHtml = (elm) => {
        if (isElement$7(elm)) {
            return elm.outerHTML;
        }
        else if (isText$b(elm)) {
            return Entities.encodeRaw(elm.data, false);
        }
        else if (isComment(elm)) {
            return '<!--' + elm.data + '-->';
        }
        return '';
    };
    const createFragment = (html) => {
        let node;
        const container = document.createElement('div');
        const frag = document.createDocumentFragment();
        if (html) {
            container.innerHTML = html;
        }
        while ((node = container.firstChild)) {
            frag.appendChild(node);
        }
        return frag;
    };
    const insertAt = (elm, html, index) => {
        const fragment = createFragment(html);
        if (elm.hasChildNodes() && index < elm.childNodes.length) {
            const target = elm.childNodes[index];
            elm.insertBefore(fragment, target);
        }
        else {
            elm.appendChild(fragment);
        }
    };
    const removeAt = (elm, index) => {
        if (elm.hasChildNodes() && index < elm.childNodes.length) {
            const target = elm.childNodes[index];
            elm.removeChild(target);
        }
    };
    const applyDiff = (diff, elm) => {
        let index = 0;
        each$e(diff, (action) => {
            if (action[0] === KEEP) {
                index++;
            }
            else if (action[0] === INSERT) {
                insertAt(elm, action[1], index);
                index++;
            }
            else if (action[0] === DELETE) {
                removeAt(elm, index);
            }
        });
    };
    const read$2 = (elm, trimZwsp) => filter$5(map$3(from(elm.childNodes), trimZwsp ? compose(trim$2, getOuterHtml) : getOuterHtml), (item) => {
        return item.length > 0;
    });
    const write = (fragments, elm) => {
        const currentFragments = map$3(from(elm.childNodes), getOuterHtml);
        applyDiff(diff(currentFragments, fragments), elm);
        return elm;
    };

    // We need to create a temporary document instead of using the global document since
    // innerHTML on a detached element will still make http requests to the images
    const lazyTempDocument = cached(() => document.implementation.createHTMLDocument('undo'));
    const hasIframes = (body) => body.querySelector('iframe') !== null;
    const createFragmentedLevel = (fragments) => {
        return {
            type: 'fragmented',
            fragments,
            content: '',
            bookmark: null,
            beforeBookmark: null
        };
    };
    const createCompleteLevel = (content) => {
        return {
            type: 'complete',
            fragments: null,
            content,
            bookmark: null,
            beforeBookmark: null
        };
    };
    const createFromEditor = (editor) => {
        const tempAttrs = editor.serializer.getTempAttrs();
        const body = trim$1(editor.getBody(), tempAttrs);
        return hasIframes(body) ? createFragmentedLevel(read$2(body, true)) : createCompleteLevel(trim$2(body.innerHTML));
    };
    const applyToEditor = (editor, level, before) => {
        const bookmark = before ? level.beforeBookmark : level.bookmark;
        if (level.type === 'fragmented') {
            write(level.fragments, editor.getBody());
        }
        else {
            editor.setContent(level.content, {
                format: 'raw',
                // If we have a path bookmark, we need to check if the bookmark location was a fake caret.
                // If the bookmark was not a fake caret, then we need to ensure that setContent does not move the selection
                // as this can create a new fake caret - particularly if the first element in the body is contenteditable=false.
                // The creation of this new fake caret will cause our path offset to be off by one when restoring the original selection.
                no_selection: isNonNullable(bookmark) && isPathBookmark(bookmark) ? !bookmark.isFakeCaret : true
            });
        }
        if (bookmark) {
            editor.sele