humhub.module('mail.ConversationView', function (module, require, $) {

    var Widget = require('ui.widget').Widget;
    var loader = require('ui.loader');
    var client = require('client');
    var additions = require('ui.additions');
    var object = require('util.object');
    var mail = require('mail.notification');
    var view = require('ui.view');

    var ConversationView = Widget.extend();

    ConversationView.prototype.init = function () {
        additions.observe(this.$);

        var that = this;
        window.onresize = function (evt) {
            that.scrollToBottom();
        };

        if (!this.getActiveMessageId()) {
            this.setActiveMessageId(Widget.instance('#inbox').getFirstMessageId());
        }

        this.reload();
    };

    ConversationView.prototype.onbeforeunloadCheck = function () {
        var replyRichtext = this.getReplyRichtext();
        if (replyRichtext && $(replyRichtext.$[0]).closest('form').find('textarea').val().trim().length) {
            return client.confirmUnload();
        }

        return true;
    };

    ConversationView.prototype.loader = function (load) {
        if (load !== false) {
            loader.set(this.$);
        } else {
            loader.reset(this.$);
        }
    };

    ConversationView.prototype.markSeen = function (id) {
        client.post(this.options.markSeenUrl, {data: {id: id}}).then(function (response) {
            if (object.isDefined(response.messageCount)) {
                mail.setMailMessageCount(response.messageCount);
            }
        }).catch(function (e) {
            module.log.error(e);
        });
    };

    ConversationView.prototype.loadUpdate = function () {
        var $lastEntry = this.$.find('.mail-conversation-entry:not(.own):last');
        var lastEntryId = $lastEntry.data('entry-id');
        var data = {id: this.getActiveMessageId(), from: lastEntryId};

        var that = this;
        client.get(this.options.loadUpdateUrl, {data: data}).then(function (response) {
            if (response.html) {
                $(response.html).each(function () {
                    that.appendEntry($(this));
                });
            }
        })
    };

    ConversationView.prototype.reply = function (evt) {
        var that = this;
        client.submit(evt).then(function (response) {
            if (response.success) {
                that.appendEntry(response.content).then(function () {
                    that.$.find(".time").timeago(); // somehow this is not triggered after reply
                    var richtext = that.getReplyRichtext();
                    if (richtext) {
                        richtext.$.trigger('clear');
                    }
                    var filePreview = that.getReplyFilePreview();
                    if (filePreview.length) {
                        filePreview.hide();
                        filePreview.children('ul.files').html('');
                    }
                    var fileListInputs = that.getReplyFileListInputs();
                    if (fileListInputs.length) {
                        fileListInputs.remove();
                    }
                    that.scrollToBottom();
                    if (!view.isSmall()) { // prevent autofocus on mobile
                        that.focus();
                    }
                    Widget.instance('#inbox').updateEntries([that.getActiveMessageId()]);
                    that.setLivePollInterval();
                });
            } else {
                module.log.error(response, true);
            }
        }).catch(function (e) {
            module.log.error(e, true);
        }).finally(function (e) {
            loader.reset($('.reply-button'));
            evt.finish();
        });
    };

    ConversationView.prototype.setLivePollInterval = function () {
        require('live').setDelay(5);
    };

    ConversationView.prototype.getReplyRichtext = function () {
        return Widget.instance(this.$.find('.ProsemirrorEditor'));
    };

    ConversationView.prototype.getReplyFilePreview = function () {
        return this.$.find('.post-file-list');
    };

    ConversationView.prototype.getReplyFileListInputs = function () {
        return this.$.find('input[name="fileList[]"]');
    };


    ConversationView.prototype.focus = function (evt) {
        var replyRichtext = this.getReplyRichtext();
        if (replyRichtext) {
            replyRichtext.focus();
        }
    };

    ConversationView.prototype.canLoadMore = function () {
        return !this.options.isLast;
    };

    ConversationView.prototype.reload = function () {
        if (this.getActiveMessageId()) {
            this.loadMessage(this.getActiveMessageId());
        }
    };

    ConversationView.prototype.addUser = function (evt) {
        var that = this;

        client.submit(evt).then(function (response) {
            if (response.result) {
                that.$.find('#mail-conversation-header').html(response.result);
            } else if (response.error) {
                module.log.error(response, true);
            }
        }).catch(function (e) {
            module.log.error(e, true);
        });
    };

    ConversationView.prototype.appendEntry = function (html) {
        var that = this;
        var $html = $(html);

        if (that.$.find('[data-entry-id="' + $html.data('entryId') + '"]').length) {
            return Promise.resolve();
        }

        // Filter out all script/links and text nodes
        var $elements = $html.not('script, link').filter(function () {
            return this.nodeType === 1; // filter out text nodes
        });

        // We use opacity because some additions require the actual size of the elements.
        $elements.css('opacity', 0);

        // call insert callback
        this.getListNode().append($html);

        return new Promise(function (resolve, reject) {
            $elements.css('opacity', 1).removeClass('d-none', function () {
                that.onUpdate();
                setTimeout(function () {
                    that.scrollToBottom()
                }, 100);
                resolve();
            });
        })
    };

    ConversationView.prototype.loadMessage = function (evt) {
        if (!this.onbeforeunloadCheck()) {
            return false;
        }

        var messageId = object.isNumber(evt) ? evt : evt.$trigger.data('message-id');
        var that = this;
        this.loader();
        client.get(this.options.loadMessageUrl, {data: {id: messageId}}).then(function (response) {
            that.setActiveMessageId(messageId);
            that.options.isLast = false;

            var inbox = Widget.instance('#inbox');
            inbox.updateActiveItem();
            inbox.hide();

            // Replace history state only if triggered by message preview item
            if (evt.$trigger && history && history.replaceState) {
                var url = evt.$trigger.data('action-url');
                if (url) {
                    history.replaceState(null, null, url);
                }
            }

            that.$.css('visibility', 'hidden');
            return that.updateContent(response.html);
        }).then(function () {
            return that.initScroll();
        }).catch(function (e) {
            module.log.error(e, true);
        }).finally(function () {
            that.loader(false);
            that.$.css('visibility', 'visible');
            that.initReplyRichText();
        });
    };

    ConversationView.prototype.initReplyRichText = function () {
        var that = this;

        if (window.ResizeObserver) {
            var resizeObserver = new ResizeObserver(function (entries) {
                if (that.isScrolledToBottom(100)) {
                    that.scrollToBottom();
                }
            });

            var replyRichtext = that.getReplyRichtext();
            if (replyRichtext) {
                resizeObserver.observe(replyRichtext.$[0]);
            }
        }

        var filePreview = that.getReplyFilePreview();
        var observer = new MutationObserver(function () {
            that.scrollToBottom();
        });
        filePreview.each(function (index, element) {
            observer.observe(element, { childList: true, subtree: true });
        });

        that.focus();
    };

    ConversationView.prototype.isScrolledToBottom = function (tolerance) {
        var $container = this.getContainerNode();

        if (!$container.length) {
            return false;
        }

        tolerance = tolerance || 0;
        var list = this.getContainerNode()[0];
        return list.scrollHeight - list.offsetHeight - list.scrollTop <= tolerance;
    };

    ConversationView.prototype.initScroll = function () {
        if (window.IntersectionObserver) {
            var $container = this.getContainerNode();
            var $streamEnd = $('<div class="conversation-stream-end"></div>');
            $container.prepend($streamEnd);

            var that = this;
            var observer = new IntersectionObserver(function (entries) {
                if (that.preventScrollLoading()) {
                    return;
                }

                if (entries.length && entries[0].isIntersecting) {
                    loader.prepend($container);
                    that.loadMore().finally(function () {
                        loader.reset($container);
                    });
                }

            }, {root: $container[0], rootMargin: "50px"});

            // Assure the conversation list is scrollable by loading more entries until overflow
            return this.assureScroll().then(function () {
                observer.observe($streamEnd[0]);
                if (view.isLarge()) {
                    that.getContainerNode().niceScroll({
                        cursorwidth: "7",
                        cursorborder: "",
                        cursorcolor: "#555",
                        cursoropacitymax: "0.2",
                        nativeparentscrolling: false,
                        railpadding: {top: 0, right: 0, left: 0, bottom: 0}
                    });

                    that.scrollDownButton = undefined;
                    that.getContainerNode().on('scroll', () => that.getScrollDownButton().toggle(!that.isScrolledToBottom()));
                }
            });
        }
    };

    ConversationView.prototype.getScrollDownButton = function () {
        if (typeof this.scrollDownButton === 'object') {
            return this.scrollDownButton;
        }

        this.scrollDownButton = $('<div>')
            .addClass('conversation-scroll-down-button')
            .html('<i class="fa fa-caret-down"></i>')
            .on('click', () => this.scrollToBottom());

        this.getContainerNode().append(this.scrollDownButton);

        return this.scrollDownButton;
    }

    ConversationView.prototype.loadMore = function () {
        var that = this;

        var data = {
            id: this.getActiveMessageId(),
            from: this.$.find('.mail-conversation-entry:first').data('entryId')
        };

        return client.get(this.options.loadMoreUrl, {data: data}).then(function (response) {
            if (response.result) {
                var $result = $(response.result).hide();
                that.getListNode().find('.conversation-stream-end').after($result);
                $result.fadeIn();
            }

            that.options.isLast = !response.result || response.isLast;
        }).catch(function (err) {
            module.log.error(err, true);
        });
    };

    ConversationView.prototype.preventScrollLoading = function () {
        return this.scrollLock || !this.canLoadMore();
    };

    ConversationView.prototype.canLoadMore = function () {
        return !this.options.isLast;
    };

    ConversationView.prototype.assureScroll = function () {
        var that = this;
        var $container = this.getContainerNode();
        if ($container[0].offsetHeight >= $container[0].scrollHeight && this.canLoadMore()) {
            return this.loadMore().then(function () {
                return that.assureScroll();
            }).catch(function () {
                return Promise.resolve();
            })
        }

        return that.scrollToBottom();
    };

    ConversationView.prototype.updateContent = function (html) {
        var that = this;
        return new Promise(function (resolve) {
            that.$.html(html);
            resolve();
        });
    };


    ConversationView.prototype.getActiveMessageId = function () {
        return this.options.messageId;
    };

    ConversationView.prototype.setActiveMessageId = function (id) {
        this.options.messageId = id;
    };

    ConversationView.prototype.scrollToBottom = function () {
        var that = this;

        return new Promise(function (resolve) {
            setTimeout(function () {
                that.$.imagesLoaded(function () {
                    var $container = that.getContainerNode();
                    if (!$container.length) {
                        return;
                    }

                    $container[0].scrollTop = $container[0].scrollHeight;
                    resolve()
                })
            });
        });
    };

    ConversationView.prototype.getContainerNode = function () {
        return this.$.find('.conversation-entry-container');
    };

    ConversationView.prototype.getListNode = function () {
        return this.$.find('.conversation-entry-list');
    };

    ConversationView.prototype.onUpdate = function () {
        if (view.isLarge()) {
            this.getContainerNode().getNiceScroll().resize();
        }
    };

    module.export = ConversationView;
});
